/*
 * Decompiled with CFR 0.152.
 */
package org.elasticsearch.xpack.watcher;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.concurrent.atomic.AtomicReference;
import java.util.stream.Collectors;
import org.apache.logging.log4j.message.ParameterizedMessage;
import org.elasticsearch.ElasticsearchException;
import org.elasticsearch.action.admin.indices.refresh.RefreshRequest;
import org.elasticsearch.action.admin.indices.refresh.RefreshResponse;
import org.elasticsearch.action.search.ClearScrollRequest;
import org.elasticsearch.action.search.SearchRequest;
import org.elasticsearch.action.search.SearchResponse;
import org.elasticsearch.action.search.SearchScrollRequest;
import org.elasticsearch.client.Client;
import org.elasticsearch.cluster.ClusterState;
import org.elasticsearch.cluster.metadata.IndexMetaData;
import org.elasticsearch.cluster.routing.AllocationId;
import org.elasticsearch.cluster.routing.Murmur3HashFunction;
import org.elasticsearch.cluster.routing.Preference;
import org.elasticsearch.cluster.routing.RoutingNode;
import org.elasticsearch.cluster.routing.ShardRouting;
import org.elasticsearch.cluster.routing.ShardRoutingState;
import org.elasticsearch.common.component.AbstractComponent;
import org.elasticsearch.common.settings.Settings;
import org.elasticsearch.common.unit.TimeValue;
import org.elasticsearch.common.util.concurrent.ThreadContext;
import org.elasticsearch.common.xcontent.XContentType;
import org.elasticsearch.search.SearchHit;
import org.elasticsearch.search.builder.SearchSourceBuilder;
import org.elasticsearch.search.sort.SortBuilder;
import org.elasticsearch.search.sort.SortBuilders;
import org.elasticsearch.xpack.ClientHelper;
import org.elasticsearch.xpack.watcher.WatcherState;
import org.elasticsearch.xpack.watcher.execution.ExecutionService;
import org.elasticsearch.xpack.watcher.execution.TriggeredWatch;
import org.elasticsearch.xpack.watcher.execution.TriggeredWatchStore;
import org.elasticsearch.xpack.watcher.support.Exceptions;
import org.elasticsearch.xpack.watcher.trigger.TriggerService;
import org.elasticsearch.xpack.watcher.watch.Watch;
import org.elasticsearch.xpack.watcher.watch.WatchStoreUtils;

public class WatcherService
extends AbstractComponent {
    private final TriggerService triggerService;
    private final TriggeredWatchStore triggeredWatchStore;
    private final ExecutionService executionService;
    private final TimeValue scrollTimeout;
    private final int scrollSize;
    private final Watch.Parser parser;
    private final Client client;
    final AtomicReference<WatcherState> state = new AtomicReference<WatcherState>(WatcherState.STOPPED);
    private final TimeValue defaultSearchTimeout;

    public WatcherService(Settings settings, TriggerService triggerService, TriggeredWatchStore triggeredWatchStore, ExecutionService executionService, Watch.Parser parser, Client client) {
        super(settings);
        this.triggerService = triggerService;
        this.triggeredWatchStore = triggeredWatchStore;
        this.executionService = executionService;
        this.scrollTimeout = settings.getAsTime("xpack.watcher.watch.scroll.timeout", TimeValue.timeValueSeconds((long)30L));
        this.scrollSize = settings.getAsInt("xpack.watcher.watch.scroll.size", Integer.valueOf(100));
        this.defaultSearchTimeout = settings.getAsTime("xpack.watcher.internal.ops.search.default_timeout", TimeValue.timeValueSeconds((long)30L));
        this.parser = parser;
        this.client = client;
    }

    public boolean validate(ClusterState state) {
        boolean executionServiceValid = this.executionService.validate(state);
        if (executionServiceValid) {
            try {
                IndexMetaData indexMetaData = WatchStoreUtils.getConcreteIndex(".watches", state.metaData());
                if (indexMetaData == null) {
                    return true;
                }
                if (indexMetaData.getState() == IndexMetaData.State.CLOSE) {
                    this.logger.debug("watch index [{}] is marked as closed, watcher cannot be started", (Object)indexMetaData.getIndex().getName());
                    return false;
                }
                return state.routingTable().index(indexMetaData.getIndex()).allPrimaryShardsActive();
            }
            catch (IllegalStateException e) {
                this.logger.trace(() -> new ParameterizedMessage("error getting index meta data [{}]: ", (Object)".watches"), (Throwable)e);
                return false;
            }
        }
        return false;
    }

    public void start(ClusterState clusterState) throws Exception {
        WatcherState currentState = this.state.get();
        if (currentState == WatcherState.STARTING || currentState == WatcherState.STARTED) {
            throw new IllegalStateException("watcher is already in state [" + (Object)((Object)currentState) + "]");
        }
        if (this.state.compareAndSet(WatcherState.STOPPED, WatcherState.STARTING)) {
            try {
                this.logger.debug("starting watch service...");
                this.executionService.start(clusterState);
                Collection<Watch> watches = this.loadWatches(clusterState);
                this.triggerService.start(watches);
                Collection<TriggeredWatch> triggeredWatches = this.triggeredWatchStore.findTriggeredWatches(watches, clusterState);
                this.executionService.executeTriggeredWatches(triggeredWatches);
                this.state.set(WatcherState.STARTED);
                this.logger.debug("watch service has started");
            }
            catch (Exception e) {
                this.state.set(WatcherState.STOPPED);
                throw e;
            }
        }
    }

    public void stop(String reason) {
        WatcherState currentState = this.state.get();
        if (currentState == WatcherState.STOPPING || currentState == WatcherState.STOPPED) {
            this.logger.trace("watcher is already in state [{}] not stopping", (Object)currentState);
        } else {
            try {
                if (this.state.compareAndSet(WatcherState.STARTED, WatcherState.STOPPING)) {
                    this.logger.debug("stopping watch service, reason [{}]", (Object)reason);
                    this.triggerService.stop();
                    this.executionService.stop();
                    this.state.set(WatcherState.STOPPED);
                    this.logger.debug("watch service has stopped");
                }
            }
            catch (Exception e) {
                this.state.set(WatcherState.STOPPED);
                this.logger.error("Error stopping watcher", (Throwable)e);
            }
        }
    }

    public void reload(ClusterState clusterState, String reason) {
        this.pauseExecution(reason);
        Collection<Watch> watches = this.loadWatches(clusterState);
        watches.forEach(this.triggerService::add);
        Collection<TriggeredWatch> triggeredWatches = this.triggeredWatchStore.findTriggeredWatches(watches, clusterState);
        this.executionService.executeTriggeredWatches(triggeredWatches);
    }

    public void pauseExecution(String reason) {
        int cancelledTaskCount = this.executionService.pauseExecution();
        this.triggerService.pauseExecution();
        this.logger.debug("paused execution service, reason [{}], cancelled [{}] queued tasks", (Object)reason, (Object)cancelledTaskCount);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     * Enabled aggressive block sorting
     * Enabled unnecessary exception pruning
     * Enabled aggressive exception aggregation
     */
    private Collection<Watch> loadWatches(ClusterState clusterState) {
        IndexMetaData indexMetaData = WatchStoreUtils.getConcreteIndex(".watches", clusterState.metaData());
        if (indexMetaData == null) {
            return Collections.emptyList();
        }
        SearchResponse response = null;
        ArrayList<Watch> watches = new ArrayList<Watch>();
        try (ThreadContext.StoredContext ignore = ClientHelper.stashWithOrigin(this.client.threadPool().getThreadContext(), "watcher");){
            RefreshResponse refreshResponse = (RefreshResponse)this.client.admin().indices().refresh(new RefreshRequest(new String[]{".watches"})).actionGet(TimeValue.timeValueSeconds((long)5L));
            if (refreshResponse.getSuccessfulShards() < indexMetaData.getNumberOfShards()) {
                throw Exceptions.illegalState("not all required shards have been refreshed", new Object[0]);
            }
            String watchIndexName = indexMetaData.getIndex().getName();
            RoutingNode routingNode = clusterState.getRoutingNodes().node(clusterState.nodes().getLocalNodeId());
            if (routingNode == null) {
                List<Watch> list = Collections.emptyList();
                return list;
            }
            List localShards = routingNode.shardsWithState(watchIndexName, new ShardRoutingState[]{ShardRoutingState.RELOCATING, ShardRoutingState.STARTED});
            List watchIndexShardRoutings = clusterState.getRoutingTable().allShards(watchIndexName);
            SearchRequest searchRequest = new SearchRequest(new String[]{".watches"}).scroll(this.scrollTimeout).preference(Preference.ONLY_LOCAL.toString()).source(new SearchSourceBuilder().size(this.scrollSize).sort((SortBuilder)SortBuilders.fieldSort((String)"_doc")).version(Boolean.valueOf(true)));
            response = (SearchResponse)this.client.search(searchRequest).actionGet(this.defaultSearchTimeout);
            if (response.getTotalShards() != response.getSuccessfulShards()) {
                throw new ElasticsearchException("Partial response while loading watches", new Object[0]);
            }
            if (response.getHits().getTotalHits() == 0L) {
                List<Watch> clearScrollRequest = Collections.emptyList();
                return clearScrollRequest;
            }
            HashMap sortedShards = new HashMap(localShards.size());
            for (ShardRouting localShardRouting : localShards) {
                List sortedAllocationIds = watchIndexShardRoutings.stream().filter(sr -> localShardRouting.getId() == sr.getId()).map(ShardRouting::allocationId).filter(Objects::nonNull).map(AllocationId::getId).filter(Objects::nonNull).sorted().collect(Collectors.toList());
                sortedShards.put(localShardRouting.getId(), sortedAllocationIds);
            }
            while (response.getHits().getHits().length != 0) {
                for (SearchHit hit : response.getHits()) {
                    Optional<ShardRouting> correspondingShardOptional = localShards.stream().filter(sr -> sr.shardId().equals((Object)hit.getShard().getShardId())).findFirst();
                    if (!correspondingShardOptional.isPresent()) continue;
                    ShardRouting correspondingShard = correspondingShardOptional.get();
                    List shardAllocationIds = (List)sortedShards.get(hit.getShard().getShardId().id());
                    int bucket = shardAllocationIds.indexOf(correspondingShard.allocationId().getId());
                    String id = hit.getId();
                    if (!this.parseWatchOnThisNode(hit.getId(), shardAllocationIds.size(), bucket)) continue;
                    try {
                        Watch watch = this.parser.parse(id, true, hit.getSourceRef(), XContentType.JSON);
                        watch.version(hit.getVersion());
                        if (!watch.status().state().isActive()) continue;
                        watches.add(watch);
                    }
                    catch (Exception e) {
                        this.logger.error(() -> new ParameterizedMessage("couldn't load watch [{}], ignoring it...", (Object)id), (Throwable)e);
                    }
                }
                SearchScrollRequest request = new SearchScrollRequest(response.getScrollId());
                request.scroll(this.scrollTimeout);
                response = (SearchResponse)this.client.searchScroll(request).actionGet(this.defaultSearchTimeout);
            }
        }
        finally {
            if (response != null) {
                try (ThreadContext.StoredContext ignore2 = ClientHelper.stashWithOrigin(this.client.threadPool().getThreadContext(), "watcher");){
                    ClearScrollRequest clearScrollRequest = new ClearScrollRequest();
                    clearScrollRequest.addScrollId(response.getScrollId());
                    this.client.clearScroll(clearScrollRequest).actionGet(this.scrollTimeout);
                }
            }
        }
        this.logger.debug("Loaded [{}] watches for execution", (Object)watches.size());
        return watches;
    }

    private boolean parseWatchOnThisNode(String id, int totalShardCount, int index) {
        int hash = Murmur3HashFunction.hash((String)id);
        int shardIndex = Math.floorMod(hash, totalShardCount);
        return shardIndex == index;
    }

    public WatcherState state() {
        return this.state.get();
    }

    public Map<String, Object> usageStats() {
        Map<String, Object> innerMap = this.executionService.usageStats();
        return innerMap;
    }
}

