/*
 * Decompiled with CFR 0.152.
 */
package org.elasticsearch.license;

import java.time.Clock;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Locale;
import java.util.UUID;
import java.util.concurrent.atomic.AtomicReference;
import org.apache.logging.log4j.message.Message;
import org.apache.logging.log4j.message.ParameterizedMessage;
import org.elasticsearch.ElasticsearchException;
import org.elasticsearch.Version;
import org.elasticsearch.action.ActionListener;
import org.elasticsearch.cluster.AckedClusterStateUpdateTask;
import org.elasticsearch.cluster.ClusterChangedEvent;
import org.elasticsearch.cluster.ClusterState;
import org.elasticsearch.cluster.ClusterStateListener;
import org.elasticsearch.cluster.ClusterStateTaskConfig;
import org.elasticsearch.cluster.ClusterStateUpdateTask;
import org.elasticsearch.cluster.ack.AckedRequest;
import org.elasticsearch.cluster.ack.ClusterStateUpdateResponse;
import org.elasticsearch.cluster.metadata.MetaData;
import org.elasticsearch.cluster.service.ClusterService;
import org.elasticsearch.common.Nullable;
import org.elasticsearch.common.component.AbstractLifecycleComponent;
import org.elasticsearch.common.component.Lifecycle;
import org.elasticsearch.common.joda.FormatDateTimeFormatter;
import org.elasticsearch.common.joda.Joda;
import org.elasticsearch.common.logging.LoggerMessageFormat;
import org.elasticsearch.common.settings.Setting;
import org.elasticsearch.common.settings.Settings;
import org.elasticsearch.common.unit.TimeValue;
import org.elasticsearch.discovery.DiscoveryModule;
import org.elasticsearch.env.Environment;
import org.elasticsearch.gateway.GatewayService;
import org.elasticsearch.license.DeleteLicenseRequest;
import org.elasticsearch.license.ExpirationCallback;
import org.elasticsearch.license.License;
import org.elasticsearch.license.LicenseVerifier;
import org.elasticsearch.license.LicensesMetaData;
import org.elasticsearch.license.LicensesStatus;
import org.elasticsearch.license.OperationModeFileWatcher;
import org.elasticsearch.license.PostStartTrialResponse;
import org.elasticsearch.license.PutLicenseRequest;
import org.elasticsearch.license.PutLicenseResponse;
import org.elasticsearch.license.SelfGeneratedLicense;
import org.elasticsearch.license.XPackLicenseState;
import org.elasticsearch.watcher.ResourceWatcherService;
import org.elasticsearch.xpack.XPackPlugin;
import org.elasticsearch.xpack.XPackSettings;
import org.elasticsearch.xpack.scheduler.SchedulerEngine;

public class LicenseService
extends AbstractLifecycleComponent
implements ClusterStateListener,
SchedulerEngine.Listener {
    public static final Setting<String> SELF_GENERATED_LICENSE_TYPE = new Setting("xpack.license.self_generated.type", s -> "trial", s -> {
        if (LicenseService.validSelfGeneratedType(s)) {
            return s;
        }
        throw new IllegalArgumentException("Illegal self generated license type [" + s + "]. Must be trial or basic.");
    }, new Setting.Property[]{Setting.Property.NodeScope});
    static final TimeValue SELF_GENERATED_LICENSE_DURATION = TimeValue.timeValueHours((long)720L);
    static final TimeValue GRACE_PERIOD_DURATION = LicenseService.days(7);
    private final ClusterService clusterService;
    private final XPackLicenseState licenseState;
    private final AtomicReference<License> currentLicense = new AtomicReference();
    private SchedulerEngine scheduler;
    private final Clock clock;
    private final OperationModeFileWatcher operationModeFileWatcher;
    private List<ExpirationCallback> expirationCallbacks = new ArrayList<ExpirationCallback>();
    private int selfGeneratedLicenseMaxNodes = 1000;
    public static final String LICENSE_JOB = "licenseJob";
    private static final FormatDateTimeFormatter DATE_FORMATTER = Joda.forPattern((String)"EEEE, MMMMM dd, yyyy", (Locale)Locale.ROOT);
    private static final String ACKNOWLEDGEMENT_HEADER = "This license update requires acknowledgement. To acknowledge the license, please read the following messages and update the license again, this time with the \"acknowledge=true\" parameter:";

    public LicenseService(Settings settings, ClusterService clusterService, Clock clock, Environment env, ResourceWatcherService resourceWatcherService, XPackLicenseState licenseState) {
        super(settings);
        this.clusterService = clusterService;
        this.clock = clock;
        this.scheduler = new SchedulerEngine(clock);
        this.licenseState = licenseState;
        this.operationModeFileWatcher = new OperationModeFileWatcher(resourceWatcherService, XPackPlugin.resolveConfigFile(env, "license_mode"), this.logger, () -> this.updateLicenseState(this.getLicense()));
        this.scheduler.register(this);
        this.populateExpirationCallbacks();
    }

    private void logExpirationWarning(long expirationMillis, boolean expired) {
        String expiredMsg = expired ? "expired" : "will expire";
        String general = LoggerMessageFormat.format(null, (String)"\n#\n# License [{}] on [{}]. If you have a new license, please update it.\n# Otherwise, please reach out to your support contact.\n# ", (Object[])new Object[]{expiredMsg, DATE_FORMATTER.printer().print(expirationMillis)});
        if (expired) {
            general = general.toUpperCase(Locale.ROOT);
        }
        StringBuilder builder = new StringBuilder(general);
        builder.append(System.lineSeparator());
        if (expired) {
            builder.append("# COMMERCIAL PLUGINS OPERATING WITH REDUCED FUNCTIONALITY");
        } else {
            builder.append("# Commercial plugins operate with reduced functionality on license expiration:");
        }
        XPackLicenseState.EXPIRATION_MESSAGES.forEach((feature, messages) -> {
            if (((String[])messages).length > 0) {
                builder.append(System.lineSeparator());
                builder.append("# - ");
                builder.append((String)feature);
                for (String message : messages) {
                    builder.append(System.lineSeparator());
                    builder.append("#  - ");
                    builder.append(message);
                }
            }
        });
        this.logger.warn("{}", (Object)builder);
    }

    private void populateExpirationCallbacks() {
        this.expirationCallbacks.add(new ExpirationCallback.Pre(LicenseService.days(7), LicenseService.days(25), LicenseService.days(1)){

            @Override
            public void on(License license) {
                LicenseService.this.logExpirationWarning(license.expiryDate(), false);
            }
        });
        this.expirationCallbacks.add(new ExpirationCallback.Pre(LicenseService.days(0), LicenseService.days(7), TimeValue.timeValueMinutes((long)10L)){

            @Override
            public void on(License license) {
                LicenseService.this.logExpirationWarning(license.expiryDate(), false);
            }
        });
        this.expirationCallbacks.add(new ExpirationCallback.Post(LicenseService.days(0), null, TimeValue.timeValueMinutes((long)10L)){

            @Override
            public void on(License license) {
                LicenseService.this.logExpirationWarning(license.expiryDate(), true);
            }
        });
    }

    public void registerLicense(PutLicenseRequest request, ActionListener<PutLicenseResponse> listener) {
        final License newLicense = request.license();
        long now = this.clock.millis();
        if (!LicenseVerifier.verifyLicense(newLicense) || newLicense.issueDate() > now || newLicense.startDate() > now) {
            listener.onResponse((Object)new PutLicenseResponse(true, LicensesStatus.INVALID));
        } else if (newLicense.expiryDate() < now) {
            listener.onResponse((Object)new PutLicenseResponse(true, LicensesStatus.EXPIRED));
        } else {
            License currentLicense;
            if (!request.acknowledged() && (currentLicense = this.getLicense()) != null) {
                HashMap<String, String[]> acknowledgeMessages = new HashMap<String, String[]>();
                if (!License.isAutoGeneratedLicense(currentLicense.signature()) && currentLicense.issueDate() > newLicense.issueDate()) {
                    acknowledgeMessages.put("license", new String[]{"The new license is older than the currently installed license. Are you sure you want to override the current license?"});
                }
                XPackLicenseState.ACKNOWLEDGMENT_MESSAGES.forEach((feature, ackMessages) -> {
                    String[] messages = (String[])ackMessages.apply(currentLicense.operationMode(), newLicense.operationMode());
                    if (messages.length > 0) {
                        acknowledgeMessages.put((String)feature, messages);
                    }
                });
                if (!acknowledgeMessages.isEmpty()) {
                    listener.onResponse((Object)new PutLicenseResponse(false, LicensesStatus.VALID, ACKNOWLEDGEMENT_HEADER, acknowledgeMessages));
                    return;
                }
            }
            if (newLicense.isProductionLicense() && ((Boolean)XPackSettings.SECURITY_ENABLED.get(this.settings)).booleanValue() && !((Boolean)XPackSettings.TRANSPORT_SSL_ENABLED.get(this.settings)).booleanValue() && !"single-node".equals(DiscoveryModule.DISCOVERY_TYPE_SETTING.get(this.settings))) {
                throw new IllegalStateException("Can not upgrade to a production license unless TLS is configured or security is disabled");
            }
            this.clusterService.submitStateUpdateTask("register license [" + newLicense.uid() + "]", (ClusterStateTaskConfig)new AckedClusterStateUpdateTask<PutLicenseResponse>((AckedRequest)request, listener){

                protected PutLicenseResponse newResponse(boolean acknowledged) {
                    return new PutLicenseResponse(acknowledged, LicensesStatus.VALID);
                }

                public ClusterState execute(ClusterState currentState) throws Exception {
                    MetaData currentMetadata = currentState.metaData();
                    LicensesMetaData licensesMetaData = (LicensesMetaData)currentMetadata.custom("licenses");
                    Version trialVersion = null;
                    if (licensesMetaData != null) {
                        trialVersion = licensesMetaData.getMostRecentTrialVersion();
                    }
                    MetaData.Builder mdBuilder = MetaData.builder((MetaData)currentMetadata);
                    mdBuilder.putCustom("licenses", (MetaData.Custom)new LicensesMetaData(newLicense, trialVersion));
                    return ClusterState.builder((ClusterState)currentState).metaData(mdBuilder).build();
                }
            });
        }
    }

    private static TimeValue days(int days) {
        return TimeValue.timeValueHours((long)(days * 24));
    }

    @Override
    public void triggered(SchedulerEngine.Event event) {
        LicensesMetaData licensesMetaData = (LicensesMetaData)this.clusterService.state().metaData().custom("licenses");
        if (licensesMetaData != null) {
            License license = licensesMetaData.getLicense();
            if (event.getJobName().equals(LICENSE_JOB)) {
                this.updateLicenseState(license);
            } else if (event.getJobName().startsWith(".license_expiration_job_")) {
                this.expirationCallbacks.stream().filter(expirationCallback -> expirationCallback.getId().equals(event.getJobName())).forEach(expirationCallback -> expirationCallback.on(license));
            }
        }
    }

    public void removeLicense(DeleteLicenseRequest request, ActionListener<ClusterStateUpdateResponse> listener) {
        this.clusterService.submitStateUpdateTask("delete license", (ClusterStateTaskConfig)new AckedClusterStateUpdateTask<ClusterStateUpdateResponse>((AckedRequest)request, listener){

            protected ClusterStateUpdateResponse newResponse(boolean acknowledged) {
                return new ClusterStateUpdateResponse(acknowledged);
            }

            public ClusterState execute(ClusterState currentState) throws Exception {
                MetaData metaData = currentState.metaData();
                LicensesMetaData currentLicenses = (LicensesMetaData)metaData.custom("licenses");
                if (currentLicenses.getLicense() != LicensesMetaData.LICENSE_TOMBSTONE) {
                    MetaData.Builder mdBuilder = MetaData.builder((MetaData)currentState.metaData());
                    LicensesMetaData newMetadata = new LicensesMetaData(LicensesMetaData.LICENSE_TOMBSTONE, currentLicenses.getMostRecentTrialVersion());
                    mdBuilder.putCustom("licenses", (MetaData.Custom)newMetadata);
                    return ClusterState.builder((ClusterState)currentState).metaData(mdBuilder).build();
                }
                return currentState;
            }
        });
    }

    public License getLicense() {
        License license = LicenseService.getLicense(this.clusterService.state().metaData());
        return license == LicensesMetaData.LICENSE_TOMBSTONE ? null : license;
    }

    void upgradeSelfGeneratedLicense(final ActionListener<PostStartTrialResponse> listener) {
        this.clusterService.submitStateUpdateTask("upgrade self generated license", (ClusterStateTaskConfig)new ClusterStateUpdateTask(){

            public void clusterStateProcessed(String source, ClusterState oldState, ClusterState newState) {
                LicensesMetaData licensesMetaData = (LicensesMetaData)oldState.metaData().custom("licenses");
                LicenseService.this.logger.debug("upgraded self generated license: {}", (Object)licensesMetaData);
                if (licensesMetaData == null || licensesMetaData.isEligibleForTrial()) {
                    listener.onResponse((Object)new PostStartTrialResponse(PostStartTrialResponse.STATUS.UPGRADED_TO_TRIAL));
                } else {
                    listener.onResponse((Object)new PostStartTrialResponse(PostStartTrialResponse.STATUS.TRIAL_ALREADY_ACTIVATED));
                }
            }

            public ClusterState execute(ClusterState currentState) throws Exception {
                LicensesMetaData licensesMetaData = (LicensesMetaData)currentState.metaData().custom("licenses");
                if (licensesMetaData == null || licensesMetaData.isEligibleForTrial()) {
                    return LicenseService.this.updateWithLicense(currentState, "trial");
                }
                return currentState;
            }

            public void onFailure(String source, @Nullable Exception e) {
                LicenseService.this.logger.error((Message)new ParameterizedMessage("unexpected failure during [{}]", (Object)source), (Throwable)e);
                listener.onFailure(e);
            }
        });
    }

    private void registerTrialLicense() {
        this.clusterService.submitStateUpdateTask("generate trial license for [" + SELF_GENERATED_LICENSE_DURATION + "]", (ClusterStateTaskConfig)new ClusterStateUpdateTask(){

            public void clusterStateProcessed(String source, ClusterState oldState, ClusterState newState) {
                LicensesMetaData licensesMetaData = (LicensesMetaData)newState.metaData().custom("licenses");
                if (LicenseService.this.logger.isDebugEnabled()) {
                    LicenseService.this.logger.debug("registered self generated license: {}", (Object)licensesMetaData);
                }
            }

            public ClusterState execute(ClusterState currentState) throws Exception {
                MetaData metaData = currentState.metaData();
                LicensesMetaData currentLicensesMetaData = (LicensesMetaData)metaData.custom("licenses");
                if (currentLicensesMetaData == null) {
                    String type = (String)SELF_GENERATED_LICENSE_TYPE.get(LicenseService.this.settings);
                    if (!LicenseService.validSelfGeneratedType(type)) {
                        throw new IllegalArgumentException("Illegal self generated license type [" + type + "]. Must be trial or basic.");
                    }
                    return LicenseService.this.updateWithLicense(currentState, type);
                }
                return currentState;
            }

            public void onFailure(String source, @Nullable Exception e) {
                LicenseService.this.logger.error(() -> new ParameterizedMessage("unexpected failure during [{}]", (Object)source), (Throwable)e);
            }
        });
    }

    private ClusterState updateWithLicense(ClusterState currentState, String type) {
        long issueDate = this.clock.millis();
        MetaData.Builder mdBuilder = MetaData.builder((MetaData)currentState.metaData());
        License.Builder specBuilder = License.builder().uid(UUID.randomUUID().toString()).issuedTo(this.clusterService.getClusterName().value()).maxNodes(this.selfGeneratedLicenseMaxNodes).issueDate(issueDate).type(type).expiryDate(issueDate + SELF_GENERATED_LICENSE_DURATION.getMillis());
        License selfGeneratedLicense = SelfGeneratedLicense.create(specBuilder);
        LicensesMetaData licensesMetaData = "trial".equals(type) ? new LicensesMetaData(selfGeneratedLicense, Version.CURRENT) : new LicensesMetaData(selfGeneratedLicense, null);
        mdBuilder.putCustom("licenses", (MetaData.Custom)licensesMetaData);
        return ClusterState.builder((ClusterState)currentState).metaData(mdBuilder).build();
    }

    protected void doStart() throws ElasticsearchException {
        ClusterState clusterState;
        this.clusterService.addListener((ClusterStateListener)this);
        this.scheduler.start(Collections.emptyList());
        this.logger.debug("initializing license state");
        if (this.clusterService.lifecycleState() == Lifecycle.State.STARTED && !(clusterState = this.clusterService.state()).blocks().hasGlobalBlock(GatewayService.STATE_NOT_RECOVERED_BLOCK) && clusterState.nodes().getMasterNode() != null) {
            LicensesMetaData currentMetaData = (LicensesMetaData)clusterState.metaData().custom("licenses");
            if (clusterState.getNodes().isLocalNodeElectedMaster() && (currentMetaData == null || currentMetaData.getLicense() == null)) {
                this.registerTrialLicense();
            }
        }
    }

    protected void doStop() throws ElasticsearchException {
        this.clusterService.removeListener((ClusterStateListener)this);
        this.scheduler.stop();
        this.currentLicense.set(null);
    }

    protected void doClose() throws ElasticsearchException {
    }

    public void clusterChanged(ClusterChangedEvent event) {
        ClusterState previousClusterState = event.previousState();
        ClusterState currentClusterState = event.state();
        if (!currentClusterState.blocks().hasGlobalBlock(GatewayService.STATE_NOT_RECOVERED_BLOCK)) {
            LicensesMetaData prevLicensesMetaData = (LicensesMetaData)previousClusterState.getMetaData().custom("licenses");
            LicensesMetaData currentLicensesMetaData = (LicensesMetaData)currentClusterState.getMetaData().custom("licenses");
            if (this.logger.isDebugEnabled()) {
                this.logger.debug("previous [{}]", (Object)prevLicensesMetaData);
                this.logger.debug("current [{}]", (Object)currentLicensesMetaData);
            }
            if (previousClusterState.blocks().hasGlobalBlock(GatewayService.STATE_NOT_RECOVERED_BLOCK) || prevLicensesMetaData == null) {
                if (currentLicensesMetaData != null) {
                    this.onUpdate(currentLicensesMetaData);
                }
            } else if (!prevLicensesMetaData.equals((Object)currentLicensesMetaData)) {
                this.onUpdate(currentLicensesMetaData);
            }
            if (currentClusterState.getNodes().isLocalNodeElectedMaster() && prevLicensesMetaData == null && (currentLicensesMetaData == null || currentLicensesMetaData.getLicense() == null)) {
                this.registerTrialLicense();
            }
        } else if (this.logger.isDebugEnabled()) {
            this.logger.debug("skipped license notifications reason: [{}]", (Object)GatewayService.STATE_NOT_RECOVERED_BLOCK);
        }
    }

    protected void updateLicenseState(License license) {
        if (license == LicensesMetaData.LICENSE_TOMBSTONE) {
            this.licenseState.update(License.OperationMode.MISSING, false);
            return;
        }
        if (license != null) {
            long time = this.clock.millis();
            boolean active = time >= license.issueDate() && time < license.expiryDate() + GRACE_PERIOD_DURATION.getMillis();
            this.licenseState.update(license.operationMode(), active);
            if (active) {
                if (time < license.expiryDate()) {
                    this.logger.debug("license [{}] - valid", (Object)license.uid());
                } else {
                    this.logger.warn("license [{}] - grace", (Object)license.uid());
                }
            } else {
                this.logger.warn("license [{}] - expired", (Object)license.uid());
            }
        }
    }

    private void onUpdate(LicensesMetaData currentLicensesMetaData) {
        License license = LicenseService.getLicense(currentLicensesMetaData);
        if (license != null) {
            License previousLicense = this.currentLicense.get();
            if (!license.equals(previousLicense)) {
                this.currentLicense.set(license);
                license.setOperationModeFileWatcher(this.operationModeFileWatcher);
                this.scheduler.add(new SchedulerEngine.Job(LICENSE_JOB, LicenseService.nextLicenseCheck(license)));
                for (ExpirationCallback expirationCallback : this.expirationCallbacks) {
                    this.scheduler.add(new SchedulerEngine.Job(expirationCallback.getId(), (startTime, now) -> expirationCallback.nextScheduledTimeForExpiry(license.expiryDate(), startTime, now)));
                }
                if (previousLicense != null) {
                    previousLicense.removeOperationModeFileWatcher();
                }
                this.logger.info("license [{}] mode [{}] - valid", (Object)license.uid(), (Object)license.operationMode().name().toLowerCase(Locale.ROOT));
            }
            this.updateLicenseState(license);
        }
    }

    static SchedulerEngine.Schedule nextLicenseCheck(License license) {
        return (startTime, time) -> {
            if (time < license.issueDate()) {
                return license.issueDate();
            }
            if (time < license.expiryDate()) {
                return license.expiryDate();
            }
            if (time < license.expiryDate() + GRACE_PERIOD_DURATION.getMillis()) {
                return license.expiryDate() + GRACE_PERIOD_DURATION.getMillis();
            }
            return -1L;
        };
    }

    public static License getLicense(MetaData metaData) {
        LicensesMetaData licensesMetaData = (LicensesMetaData)metaData.custom("licenses");
        return LicenseService.getLicense(licensesMetaData);
    }

    static License getLicense(LicensesMetaData metaData) {
        if (metaData != null) {
            boolean autoGeneratedLicense;
            License license = metaData.getLicense();
            if (license == LicensesMetaData.LICENSE_TOMBSTONE) {
                return license;
            }
            if (license != null && ((autoGeneratedLicense = License.isAutoGeneratedLicense(license.signature())) && SelfGeneratedLicense.verify(license) || !autoGeneratedLicense && LicenseVerifier.verifyLicense(license))) {
                return license;
            }
        }
        return null;
    }

    private static boolean validSelfGeneratedType(String type) {
        return "basic".equals(type) || "trial".equals(type);
    }
}

