/*
 * Decompiled with CFR 0.152.
 */
package org.elasticsearch.xpack.monitoring.exporter.http;

import java.io.IOException;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Set;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.function.Supplier;
import org.apache.http.Header;
import org.apache.http.HttpHost;
import org.apache.http.auth.AuthScope;
import org.apache.http.auth.Credentials;
import org.apache.http.auth.UsernamePasswordCredentials;
import org.apache.http.client.CredentialsProvider;
import org.apache.http.impl.client.BasicCredentialsProvider;
import org.apache.http.message.BasicHeader;
import org.apache.http.nio.conn.ssl.SSLIOSessionStrategy;
import org.apache.logging.log4j.Logger;
import org.elasticsearch.Version;
import org.elasticsearch.client.RestClient;
import org.elasticsearch.client.RestClientBuilder;
import org.elasticsearch.client.sniff.ElasticsearchHostsSniffer;
import org.elasticsearch.client.sniff.HostsSniffer;
import org.elasticsearch.client.sniff.Sniffer;
import org.elasticsearch.cluster.service.ClusterService;
import org.elasticsearch.common.Nullable;
import org.elasticsearch.common.bytes.BytesReference;
import org.elasticsearch.common.collect.MapBuilder;
import org.elasticsearch.common.logging.Loggers;
import org.elasticsearch.common.settings.Settings;
import org.elasticsearch.common.settings.SettingsException;
import org.elasticsearch.common.unit.TimeValue;
import org.elasticsearch.common.util.concurrent.ThreadContext;
import org.elasticsearch.common.util.set.Sets;
import org.elasticsearch.common.xcontent.XContentType;
import org.elasticsearch.xpack.monitoring.exporter.ClusterAlertsUtil;
import org.elasticsearch.xpack.monitoring.exporter.Exporter;
import org.elasticsearch.xpack.monitoring.exporter.MonitoringTemplateUtils;
import org.elasticsearch.xpack.monitoring.exporter.http.ClusterAlertHttpResource;
import org.elasticsearch.xpack.monitoring.exporter.http.HttpExportBulk;
import org.elasticsearch.xpack.monitoring.exporter.http.HttpHostBuilder;
import org.elasticsearch.xpack.monitoring.exporter.http.HttpResource;
import org.elasticsearch.xpack.monitoring.exporter.http.MultiHttpResource;
import org.elasticsearch.xpack.monitoring.exporter.http.NodeFailureListener;
import org.elasticsearch.xpack.monitoring.exporter.http.PipelineHttpResource;
import org.elasticsearch.xpack.monitoring.exporter.http.SecurityHttpClientConfigCallback;
import org.elasticsearch.xpack.monitoring.exporter.http.TemplateHttpResource;
import org.elasticsearch.xpack.monitoring.exporter.http.TimeoutRequestConfigCallback;
import org.elasticsearch.xpack.monitoring.exporter.http.VersionHttpResource;
import org.elasticsearch.xpack.monitoring.exporter.http.WatcherExistsHttpResource;
import org.elasticsearch.xpack.ssl.SSLService;
import org.joda.time.format.DateTimeFormatter;

public class HttpExporter
extends Exporter {
    private static final Logger logger = Loggers.getLogger(HttpExporter.class);
    public static final String TYPE = "http";
    public static final String HOST_SETTING = "host";
    public static final String BULK_TIMEOUT_SETTING = "bulk.timeout";
    public static final String CONNECTION_TIMEOUT_SETTING = "connection.timeout";
    public static final String CONNECTION_READ_TIMEOUT_SETTING = "connection.read_timeout";
    public static final String AUTH_USERNAME_SETTING = "auth.username";
    public static final String AUTH_PASSWORD_SETTING = "auth.password";
    public static final String SSL_SETTING = "ssl";
    public static final String PROXY_BASE_PATH_SETTING = "proxy.base_path";
    public static final String SNIFF_ENABLED_SETTING = "sniff.enabled";
    public static final String HEADERS_SETTING = "headers";
    public static final Set<String> BLACKLISTED_HEADERS = Collections.unmodifiableSet(Sets.newHashSet((Object[])new String[]{"Content-Length", "Content-Type"}));
    public static final String TEMPLATE_CHECK_TIMEOUT_SETTING = "index.template.master_timeout";
    public static final String TEMPLATE_CREATE_LEGACY_VERSIONS_SETTING = "index.template.create_legacy_templates";
    public static final String PIPELINE_CHECK_TIMEOUT_SETTING = "index.pipeline.master_timeout";
    public static final Version MIN_SUPPORTED_CLUSTER_VERSION = Version.V_6_0_0_alpha1;
    private final RestClient client;
    @Nullable
    private final Sniffer sniffer;
    private final Map<String, String> defaultParams;
    private final HttpResource resource;
    private final AtomicBoolean clusterAlertsAllowed = new AtomicBoolean(false);
    private final ThreadContext threadContext;
    private final DateTimeFormatter dateTimeFormatter;

    public HttpExporter(Exporter.Config config, SSLService sslService, ThreadContext threadContext) {
        this(config, sslService, threadContext, new NodeFailureListener(), (HttpResource)HttpExporter.createResources(config));
    }

    HttpExporter(Exporter.Config config, SSLService sslService, ThreadContext threadContext, NodeFailureListener listener, HttpResource resource) {
        this(config, HttpExporter.createRestClient(config, sslService, listener), threadContext, listener, resource);
    }

    HttpExporter(Exporter.Config config, RestClient client, ThreadContext threadContext, NodeFailureListener listener, HttpResource resource) {
        this(config, client, HttpExporter.createSniffer(config, client, listener), threadContext, listener, resource);
    }

    HttpExporter(Exporter.Config config, RestClient client, @Nullable Sniffer sniffer, ThreadContext threadContext, NodeFailureListener listener, HttpResource resource) {
        super(config);
        this.client = Objects.requireNonNull(client);
        this.sniffer = sniffer;
        this.resource = resource;
        this.defaultParams = HttpExporter.createDefaultParams(config);
        this.threadContext = threadContext;
        this.dateTimeFormatter = HttpExporter.dateTimeFormatter(config);
        listener.setResource(resource);
    }

    static RestClient createRestClient(Exporter.Config config, SSLService sslService, NodeFailureListener listener) {
        RestClientBuilder builder = RestClient.builder((HttpHost[])HttpExporter.createHosts(config)).setFailureListener((RestClient.FailureListener)listener);
        String proxyBasePath = config.settings().get(PROXY_BASE_PATH_SETTING);
        if (proxyBasePath != null) {
            try {
                builder.setPathPrefix(proxyBasePath);
            }
            catch (IllegalArgumentException e) {
                throw new SettingsException("[" + HttpExporter.settingFQN(config, PROXY_BASE_PATH_SETTING) + "] is malformed [" + proxyBasePath + "]", (Throwable)e);
            }
        }
        HttpExporter.configureHeaders(builder, config);
        HttpExporter.configureSecurity(builder, config, sslService);
        HttpExporter.configureTimeouts(builder, config);
        return builder.build();
    }

    static Sniffer createSniffer(Exporter.Config config, RestClient client, NodeFailureListener listener) {
        Settings settings = config.settings();
        Sniffer sniffer = null;
        if (settings.getAsBoolean(SNIFF_ENABLED_SETTING, Boolean.valueOf(false)).booleanValue()) {
            List hosts = config.settings().getAsList(HOST_SETTING);
            ElasticsearchHostsSniffer.Scheme scheme = ((String)hosts.get(0)).startsWith("https") ? ElasticsearchHostsSniffer.Scheme.HTTPS : ElasticsearchHostsSniffer.Scheme.HTTP;
            ElasticsearchHostsSniffer hostsSniffer = new ElasticsearchHostsSniffer(client, ElasticsearchHostsSniffer.DEFAULT_SNIFF_REQUEST_TIMEOUT, scheme);
            sniffer = Sniffer.builder((RestClient)client).setHostsSniffer((HostsSniffer)hostsSniffer).build();
            listener.setSniffer(sniffer);
            logger.debug("[" + HttpExporter.settingFQN(config) + "] using host sniffing");
        }
        return sniffer;
    }

    static MultiHttpResource createResources(Exporter.Config config) {
        String resourceOwnerName = HttpExporter.settingFQN(config);
        ArrayList<HttpResource> resources = new ArrayList<HttpResource>();
        resources.add(new VersionHttpResource(resourceOwnerName, MIN_SUPPORTED_CLUSTER_VERSION));
        HttpExporter.configureTemplateResources(config, resourceOwnerName, resources);
        HttpExporter.configurePipelineResources(config, resourceOwnerName, resources);
        HttpExporter.configureClusterAlertsResources(config, resourceOwnerName, resources);
        return new MultiHttpResource(resourceOwnerName, resources);
    }

    private static HttpHost[] createHosts(Exporter.Config config) {
        List hosts = config.settings().getAsList(HOST_SETTING);
        if (hosts.isEmpty()) {
            throw new SettingsException("missing required setting [" + HttpExporter.settingFQN(config, HOST_SETTING) + "]");
        }
        ArrayList<HttpHost> httpHosts = new ArrayList<HttpHost>(hosts.size());
        boolean httpHostFound = false;
        boolean httpsHostFound = false;
        for (String host : hosts) {
            HttpHost httpHost;
            try {
                httpHost = HttpHostBuilder.builder(host).build();
            }
            catch (IllegalArgumentException e) {
                throw new SettingsException("[" + HttpExporter.settingFQN(config, HOST_SETTING) + "] invalid host: [" + host + "]", (Throwable)e);
            }
            if (TYPE.equals(httpHost.getSchemeName())) {
                httpHostFound = true;
            } else {
                httpsHostFound = true;
            }
            if (httpHostFound && httpsHostFound) {
                throw new SettingsException("[" + HttpExporter.settingFQN(config, HOST_SETTING) + "] must use a consistent scheme: http or https");
            }
            httpHosts.add(httpHost);
        }
        if (logger.isDebugEnabled()) {
            logger.debug("[{}] using hosts {}", (Object)HttpExporter.settingFQN(config), (Object)hosts);
        }
        return httpHosts.toArray(new HttpHost[httpHosts.size()]);
    }

    private static void configureHeaders(RestClientBuilder builder, Exporter.Config config) {
        Settings headerSettings = config.settings().getAsSettings(HEADERS_SETTING);
        Set names = headerSettings.names();
        if (names.isEmpty()) {
            return;
        }
        ArrayList<BasicHeader> headers = new ArrayList<BasicHeader>();
        for (String name : names) {
            if (BLACKLISTED_HEADERS.contains(name)) {
                throw new SettingsException("[" + name + "] cannot be overwritten via [" + HttpExporter.settingFQN(config, HEADERS_SETTING) + "]");
            }
            List values = headerSettings.getAsList(name);
            if (values.isEmpty()) {
                String settingName = HttpExporter.settingFQN(config, "headers." + name);
                throw new SettingsException("headers must have values, missing for setting [" + settingName + "]");
            }
            for (String value : values) {
                headers.add(new BasicHeader(name, value));
            }
        }
        builder.setDefaultHeaders(headers.toArray(new Header[headers.size()]));
    }

    private static void configureSecurity(RestClientBuilder builder, Exporter.Config config, SSLService sslService) {
        Settings sslSettings = config.settings().getAsSettings(SSL_SETTING);
        SSLIOSessionStrategy sslStrategy = sslService.sslIOSessionStrategy(sslSettings);
        CredentialsProvider credentialsProvider = HttpExporter.createCredentialsProvider(config);
        List hostList = config.settings().getAsList(HOST_SETTING);
        if (credentialsProvider != null && !hostList.stream().findFirst().orElse("").startsWith("https")) {
            logger.warn("[" + HttpExporter.settingFQN(config) + "] is not using https, but using user authentication with plaintext username/password!");
        }
        builder.setHttpClientConfigCallback((RestClientBuilder.HttpClientConfigCallback)new SecurityHttpClientConfigCallback(sslStrategy, credentialsProvider));
    }

    private static void configureTimeouts(RestClientBuilder builder, Exporter.Config config) {
        Settings settings = config.settings();
        TimeValue connectTimeout = settings.getAsTime(CONNECTION_TIMEOUT_SETTING, TimeValue.timeValueMillis((long)6000L));
        TimeValue socketTimeout = settings.getAsTime(CONNECTION_READ_TIMEOUT_SETTING, TimeValue.timeValueMillis((long)(connectTimeout.millis() * 10L)));
        builder.setRequestConfigCallback((RestClientBuilder.RequestConfigCallback)new TimeoutRequestConfigCallback(connectTimeout, socketTimeout));
    }

    @Nullable
    private static CredentialsProvider createCredentialsProvider(Exporter.Config config) {
        Settings settings = config.settings();
        String username = settings.get(AUTH_USERNAME_SETTING);
        String password = settings.get(AUTH_PASSWORD_SETTING);
        if (username == null) {
            if (password != null) {
                throw new SettingsException("[" + HttpExporter.settingFQN(config, AUTH_PASSWORD_SETTING) + "] without [" + HttpExporter.settingFQN(config, AUTH_USERNAME_SETTING) + "]");
            }
            return null;
        }
        BasicCredentialsProvider credentialsProvider = new BasicCredentialsProvider();
        credentialsProvider.setCredentials(AuthScope.ANY, (Credentials)new UsernamePasswordCredentials(username, password));
        return credentialsProvider;
    }

    static Map<String, String> createDefaultParams(Exporter.Config config) {
        Settings settings = config.settings();
        TimeValue bulkTimeout = settings.getAsTime(BULK_TIMEOUT_SETTING, null);
        MapBuilder params = new MapBuilder();
        if (bulkTimeout != null) {
            params.put((Object)"master_timeout", (Object)bulkTimeout.toString());
        }
        if (settings.getAsBoolean("use_ingest", Boolean.valueOf(true)).booleanValue()) {
            params.put((Object)"pipeline", (Object)MonitoringTemplateUtils.pipelineName("6"));
        }
        params.put((Object)"filter_path", (Object)"errors,items.*.error");
        return params.immutableMap();
    }

    private static void configureTemplateResources(Exporter.Config config, String resourceOwnerName, List<HttpResource> resources) {
        Supplier<String> templateLoader;
        String templateName;
        Settings settings = config.settings();
        TimeValue templateTimeout = settings.getAsTime(TEMPLATE_CHECK_TIMEOUT_SETTING, null);
        for (String templateId : MonitoringTemplateUtils.TEMPLATE_IDS) {
            templateName = MonitoringTemplateUtils.templateName(templateId);
            templateLoader = () -> MonitoringTemplateUtils.loadTemplate(templateId);
            resources.add(new TemplateHttpResource(resourceOwnerName, templateTimeout, templateName, templateLoader));
        }
        if (settings.getAsBoolean(TEMPLATE_CREATE_LEGACY_VERSIONS_SETTING, Boolean.valueOf(true)).booleanValue()) {
            for (String templateId : MonitoringTemplateUtils.OLD_TEMPLATE_IDS) {
                templateName = MonitoringTemplateUtils.oldTemplateName(templateId);
                templateLoader = () -> MonitoringTemplateUtils.createEmptyTemplate(templateId);
                resources.add(new TemplateHttpResource(resourceOwnerName, templateTimeout, templateName, templateLoader));
            }
        }
    }

    private static void configurePipelineResources(Exporter.Config config, String resourceOwnerName, List<HttpResource> resources) {
        Settings settings = config.settings();
        if (settings.getAsBoolean("use_ingest", Boolean.valueOf(true)).booleanValue()) {
            TimeValue pipelineTimeout = settings.getAsTime(PIPELINE_CHECK_TIMEOUT_SETTING, null);
            for (String pipelineId : MonitoringTemplateUtils.PIPELINE_IDS) {
                String pipelineName = MonitoringTemplateUtils.pipelineName(pipelineId);
                Supplier<byte[]> pipeline = () -> BytesReference.toBytes((BytesReference)MonitoringTemplateUtils.loadPipeline(pipelineId, XContentType.JSON).bytes());
                resources.add(new PipelineHttpResource(resourceOwnerName, pipelineTimeout, pipelineName, pipeline));
            }
        }
    }

    private static void configureClusterAlertsResources(Exporter.Config config, String resourceOwnerName, List<HttpResource> resources) {
        Settings settings = config.settings();
        if (settings.getAsBoolean("cluster_alerts.management.enabled", Boolean.valueOf(true)).booleanValue()) {
            ClusterService clusterService = config.clusterService();
            ArrayList<ClusterAlertHttpResource> watchResources = new ArrayList<ClusterAlertHttpResource>();
            List<String> blacklist = ClusterAlertsUtil.getClusterAlertsBlacklist(config);
            for (String watchId : ClusterAlertsUtil.WATCH_IDS) {
                boolean blacklisted = blacklist.contains(watchId);
                Supplier<String> uniqueWatchId = () -> ClusterAlertsUtil.createUniqueWatchId(clusterService, watchId);
                Supplier<String> watch = blacklisted ? null : () -> ClusterAlertsUtil.loadWatch(clusterService, watchId);
                watchResources.add(new ClusterAlertHttpResource(resourceOwnerName, config.licenseState(), uniqueWatchId, watch));
            }
            resources.add(new WatcherExistsHttpResource(resourceOwnerName, clusterService, new MultiHttpResource(resourceOwnerName, watchResources)));
        }
    }

    boolean isExporterReady() {
        boolean canUseClusterAlerts = this.config.licenseState().isMonitoringClusterAlertsAllowed();
        if (this.clusterAlertsAllowed.compareAndSet(!canUseClusterAlerts, canUseClusterAlerts)) {
            this.resource.markDirty();
        }
        return this.resource.checkAndPublishIfDirty(this.client);
    }

    @Override
    public HttpExportBulk openBulk() {
        if (this.isExporterReady()) {
            return new HttpExportBulk(HttpExporter.settingFQN(this.config), this.client, this.defaultParams, this.dateTimeFormatter, this.threadContext);
        }
        return null;
    }

    @Override
    public void doClose() {
        try {
            if (this.sniffer != null) {
                this.sniffer.close();
            }
        }
        catch (IOException | RuntimeException e) {
            logger.error("an error occurred while closing the internal client sniffer", (Throwable)e);
        }
        finally {
            try {
                this.client.close();
            }
            catch (IOException | RuntimeException e) {
                logger.error("an error occurred while closing the internal client", (Throwable)e);
            }
        }
    }
}

