(ns shadow.cljs.devtools.server.worker
  (:refer-clojure :exclude (compile))
  (:require [clojure.core.async :as async :refer (go thread alt!! alt! <!! <! >! >!!)]
            [shadow.cljs.devtools.server.system-bus :as sys-bus]
            [shadow.cljs.devtools.server.system-msg :as sys-msg]
            [shadow.cljs.devtools.server.worker.impl :as impl]
            [shadow.cljs.devtools.server.util :as util]
            [aleph.netty :as netty]
            [aleph.http :as aleph]
            [clojure.tools.logging :as log]
            [clojure.java.io :as io]
            [shadow.cljs.devtools.server.web.common :as common]
            [ring.middleware.file :as ring-file]
            [ring.middleware.file-info :as ring-file-info])
  (:import (java.util UUID)))

(defn get-status [{:keys [status-ref] :as proc}]
  @status-ref)

(defn compile
  "triggers an async compilation, use watch to receive notification about worker state"
  [proc]
  (impl/compile proc))

(defn compile!
  "triggers an async compilation and waits for the compilation result (blocking)"
  [proc]
  (impl/compile! proc))

(defn watch
  "watch all output produced by the worker"
  ([proc chan]
   (watch proc chan true))
  ([proc chan close?]
   (impl/watch proc chan close?)))

(defn start-autobuild
  "automatically compile on file changes"
  [proc]
  (impl/start-autobuild proc))

(defn stop-autobuild [proc]
  (impl/stop-autobuild proc))

(defn sync!
  "ensures that all proc-control commands issued have completed"
  [proc]
  (impl/sync! proc))

(defn repl-eval-connect
  "called by processes that are able to eval repl commands and report their result

   client-out should be a channel that receives things generated by shadow.cljs.repl
   (:repl/invoke, :repl/require, etc)

   returns a channel the results of eval should be put in
   when no more results are coming this channel should be closed"
  [proc client-id client-out]
  (impl/repl-eval-connect proc client-id client-out))

(defn repl-eval
  [{:keys [proc-control] :as worker} client-id input]

  (let [result-chan
        (async/chan 1)]

    (>!! proc-control {:type :repl-eval
                       :client-id client-id
                       :input input
                       :result-chan result-chan})

    (try
      (alt!!
        result-chan
        ([x] x)

        ;; FIXME: things that actually take >10s will timeout and never show their result
        (async/timeout 10000)
        ([_]
          {:type :repl/timeout}))

      (catch InterruptedException e
        {:type :repl/interrupt}))))

;; SERVICE API

(defn disable-all-kinds-of-caching [handler]
  ;; this is strictly a dev server and caching is not wanted for anything
  ;; basically emulates having devtools open with "Disable cache" active
  (fn [req]
    (-> req
        (handler)
        (update-in [:headers] assoc
          "cache-control" "max-age=0, no-cache, no-store, must-revalidate"
          "pragma" "no-cache"
          "expires" "0"))))

(defn start
  [system-bus executor http build-config]
  {:pre [(map? http)
         (map? build-config)
         (keyword? (:id build-config))]}

  (let [proc-id
        (UUID/randomUUID) ;; FIXME: not really unique but unique enough

        _ (log/debug ::start (:id build-config) proc-id)

        ;; closed when the proc-stops
        ;; nothing will ever be written here
        ;; its for linking other processes to the server process
        ;; so they shut down when the worker stops
        proc-stop
        (async/chan)

        ;; controls the worker, registers new clients, etc
        proc-control
        (async/chan)

        ;; we put output here
        output
        (async/chan 100)

        ;; clients tap here to receive output
        output-mult
        (async/mult output)

        ;; FIXME: must use buffer, but can't use 1
        ;; when a notify happens and autobuild is running the process may be busy for a while recompiling
        ;; if another fs update happens in the meantime
        ;; and we don't have a buffer the whole config update will block
        ;; if the buffer is too small we may miss an update
        ;; ideally this would accumulate all updates into one but not sure how to go about that
        ;; (would need to track busy state of worker)
        cljs-watch
        (async/chan (async/sliding-buffer 10))

        ;; same deal here, 1 msg is sent per build so this may produce many messages
        config-watch
        (async/chan (async/sliding-buffer 100))

        channels
        {:proc-stop proc-stop
         :proc-control proc-control
         :output output
         :cljs-watch cljs-watch
         :config-watch config-watch}

        status-ref
        (volatile! {:status :started})

        http-server
        (when-let [http-root (get-in build-config [:devtools :http-root])]
          (let [port (get-in build-config [:devtools :http-port] 0)

                root-dir
                (io/file http-root)]

            (when-not (.exists root-dir)
              (io/make-parents (io/file root-dir "index.html")))

            (let [http-handler
                  (-> common/not-found
                      (ring-file/wrap-file root-dir {:allow-symlinks? true
                                                     :index-files? true})
                      (ring-file-info/wrap-file-info
                        ;; source maps
                        {"map" "application/json"})
                      (disable-all-kinds-of-caching))

                  instance
                  (aleph/start-server http-handler
                    {:port port
                     :executor executor
                     :shutdown-executor? false})

                  port
                  (netty/port instance)]

              ;; FIXME: this should show a proper message somewhere
              ;; worker clients are not listening yet so cannot use output channel
              (log/info ::http-serve {:http-port port :http-root http-root})

              instance)))

        thread-state
        {::impl/worker-state true
         :http http
         :proc-id proc-id
         :build-config build-config
         :status-ref status-ref
         :autobuild false
         :eval-clients {}
         :repl-clients {}
         :pending-results {}
         :channels channels
         :system-bus system-bus
         :executor executor
         :compiler-state nil}

        state-ref
        (volatile! thread-state)

        thread-ref
        (util/server-thread
          state-ref
          thread-state
          {proc-stop nil
           proc-control impl/do-proc-control
           cljs-watch impl/do-cljs-watch
           config-watch impl/do-config-watch}
          {:validate
           impl/worker-state?
           :validate-error
           (fn [state-before state-after msg]
             ;; FIXME: handle this better
             (prn [:invalid-worker-result-after (keys state-after) msg])
             state-before)
           :on-error
           (fn [state-before msg ex]
             ;; FIXME: handle this better
             (prn [:worker-error msg ex])
             state-before)
           :do-shutdown
           (fn [state]
             (>!! output {:type :worker-shutdown :proc-id proc-id})
             state)})

        worker-proc
        {::impl/proc true
         :proc-stop proc-stop
         :proc-id proc-id
         :proc-control proc-control
         :http-server http-server
         :system-bus system-bus
         :cljs-watch cljs-watch
         :output output
         :output-mult output-mult
         :thread-ref thread-ref
         :state-ref state-ref
         :status-ref status-ref}]

    (sys-bus/sub system-bus ::sys-msg/cljs-watch cljs-watch)
    (sys-bus/sub system-bus [::sys-msg/config-watch (:id build-config)] config-watch)

    ;; ensure all channels are cleaned up properly
    (go (<! thread-ref)
        ;; FIXME: I think unsub happens automatically if we close the channel, need to confirm
        (log/debug ::stop (:id build-config) proc-id)
        (sys-bus/unsub system-bus ::sys-msg/cljs-watch cljs-watch)
        (when http-server
          (.close http-server))
        (async/close! output)
        (async/close! proc-stop)
        (async/close! cljs-watch))

    worker-proc))

(defn stop [proc]
  {:pre [(impl/proc? proc)]}
  (async/close! (:proc-stop proc))
  (<!! (:thread-ref proc)))
