(ns tech.ml.dataset.pipeline.base
  (:require [tech.ml.protocols.dataset :as ds-proto]
            [tech.v2.datatype :as dtype]
            [tech.v2.datatype.functional :as dtype-fn]
            [tech.v2.datatype.unary-op :as unary]))



(def ^:dynamic *pipeline-datatype* :float64)

(defn context-datatype
  [context]
  (or (:datatype context) *pipeline-datatype*))



(def ^:dynamic *pipeline-dataset* nil)
(def ^:dynamic *pipeline-column-name* nil)
(def ^:dynamic *pipeline-column-name-seq* nil)


(defmacro with-pipeline-vars
  "Run a body of code with pipeline global variables set."
  [dataset
   column-name
   datatype
   column-name-seq & body]
  (when datatype
    (throw (ex-info "mistake" {})))
  `(with-bindings {#'*pipeline-datatype* (or ~datatype *pipeline-datatype*)
                   #'*pipeline-dataset* (or ~dataset *pipeline-dataset*)
                   #'*pipeline-column-name* (or ~column-name *pipeline-column-name*)
                   #'*pipeline-column-name-seq*
                   (or ~column-name-seq *pipeline-column-name-seq*)}
     (when-not (= *pipeline-datatype* :float64)
       (throw (ex-info "Failed" {})))
     ~@body))


(defmacro with-ds
  [dataset & body]
  `(with-pipeline-vars ~dataset nil nil nil
     ~@body))


(defmacro with-column-name
  [colname & body]
  `(with-pipeline-vars nil ~colname nil nil
     ~@body))


(defmacro with-datatype
  [datatype & body]
  `(with-pipeline-vars nil nil ~datatype nil
     ~@body))


(defmacro with-column-name-seq
  [colname-seq & body]
  `(with-pipeline-vars nil nil nil ~colname-seq
     ~@body))


(defn dtype
  []
  *pipeline-datatype*)


(defn colname
  []
  *pipeline-datatype*)

(defn ds
  []
  *pipeline-dataset*)


(defn eval-math-fn
  [dataset column-name math-fn-or-val]
  (with-bindings {#'*pipeline-dataset* dataset
                  #'*pipeline-column-name* column-name}
    (if (fn? math-fn-or-val)
      (math-fn-or-val)
      math-fn-or-val)))


(defn int-map
  "Perform an integer->integer conversion of a column using a static map.
  The map must be complete; missing entries are errors."
  [table col-data & {:keys [not-strict?]}]
  (-> (if not-strict?
        (unary/unary-reader
         :int32
         (int (table x x))
         col-data)
        (unary/unary-reader
         :int32
         (int (if-let [retval (table x)]
                retval
                (throw (ex-info (format "Int-map failed on value: %s" x)
                                {}))))
         col-data))
      (dtype/->reader *pipeline-datatype*)))


(defn col
  "Return a column.  Only works during 'm=' and the default column
  is the current operating column."
  [& [column-name]]
  (ds-proto/column *pipeline-dataset*
                   (or column-name
                       *pipeline-column-name*)))
