/*
 * Decompiled with CFR 0.152.
 */
package software.amazon.smithy.utils;

import java.io.IOException;
import java.util.ArrayDeque;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Deque;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.function.Function;
import java.util.function.Supplier;
import java.util.regex.Pattern;
import java.util.stream.Collectors;
import software.amazon.smithy.utils.AbstractCodeWriter;
import software.amazon.smithy.utils.CodeSection;
import software.amazon.smithy.utils.CodeWriterFormatterContainer;
import software.amazon.smithy.utils.Pair;
import software.amazon.smithy.utils.SimpleParser;
import software.amazon.smithy.utils.SmithyInternalApi;

@SmithyInternalApi
final class CodeFormatter {
    private CodeFormatter() {
    }

    static void run(StringBuilder sink, AbstractCodeWriter<?> writer, String template, Object[] args) {
        try {
            Sink wrappedSink = Sink.from(sink);
            Operation block = new Parser(writer, template, args).parse();
            block.apply(wrappedSink, writer);
        }
        catch (IOException e) {
            throw new RuntimeException("Error appending to CodeWriter template: " + e, e);
        }
    }

    private static boolean isConditionTruthy(Object value) {
        if (value == null) {
            return false;
        }
        if (value instanceof Optional) {
            return ((Optional)value).isPresent();
        }
        if (value instanceof Boolean) {
            return (Boolean)value;
        }
        if (value instanceof Iterable) {
            return ((Iterable)value).iterator().hasNext();
        }
        if (value instanceof Map) {
            return !((Map)value).isEmpty();
        }
        if (value instanceof String) {
            return !((String)value).isEmpty();
        }
        return true;
    }

    private static Iterator<? extends Map.Entry<?, ?>> getValueIterator(Object value) {
        if (value instanceof Map) {
            return ((Map)value).entrySet().iterator();
        }
        if (value instanceof Iterable) {
            final Iterator iter = ((Iterable)value).iterator();
            return new Iterator<Map.Entry<?, ?>>(){
                int position = 0;

                @Override
                public boolean hasNext() {
                    return iter.hasNext();
                }

                @Override
                public Map.Entry<?, ?> next() {
                    return Pair.of(this.position++, iter.next());
                }
            };
        }
        return Collections.emptyIterator();
    }

    private static final class Parser {
        private static final Pattern NAME_PATTERN = Pattern.compile("^[a-z]+[a-zA-Z0-9_.#$]*$");
        private final String template;
        private final SimpleParser parser;
        private final char expressionStart;
        private final AbstractCodeWriter<?> writer;
        private final Object[] arguments;
        private final boolean[] positionals;
        private int relativeIndex = 0;
        private final Deque<BlockOperation> blocks = new ArrayDeque<BlockOperation>();

        Parser(AbstractCodeWriter<?> writer, String template, Object[] arguments) {
            this.template = template;
            this.writer = writer;
            this.expressionStart = writer.getExpressionStart();
            this.parser = new SimpleParser(template);
            this.arguments = arguments;
            this.positionals = new boolean[arguments.length];
            this.blocks.add(new BlockOperation.Unconditional(""));
        }

        private void pushOperation(Operation op) {
            this.blocks.getFirst().push(op);
        }

        private RuntimeException error(String message) {
            return this.parser.syntax(this.createErrorMessage(message));
        }

        private String createErrorMessage(String message) {
            return message + " (template: " + this.template + ") " + this.writer.getDebugInfo();
        }

        private Operation parse() {
            boolean parsingLiteral = false;
            int literalStartCharacter = 0;
            while (!this.parser.eof()) {
                char c = this.parser.peek();
                this.parser.skip();
                if (c != this.expressionStart) {
                    parsingLiteral = true;
                    continue;
                }
                if (this.parser.peek() == this.expressionStart) {
                    this.pushOperation(Operation.stringSlice(this.template, literalStartCharacter, this.parser.position()));
                    this.parser.expect(this.expressionStart);
                    parsingLiteral = true;
                    literalStartCharacter = this.parser.position();
                    continue;
                }
                int pendingTextStart = parsingLiteral ? literalStartCharacter : -1;
                parsingLiteral = false;
                this.parseArgument(pendingTextStart);
                literalStartCharacter = this.parser.position();
            }
            if (parsingLiteral) {
                this.pushOperation(Operation.stringSlice(this.template, literalStartCharacter, this.parser.position()));
            }
            if (this.relativeIndex == -1) {
                this.ensureAllPositionalArgumentsWereUsed();
            } else if (this.relativeIndex < this.arguments.length) {
                int unusedCount = this.arguments.length - this.relativeIndex;
                throw this.error(String.format("Found %d unused relative format arguments", unusedCount));
            }
            if (this.blocks.size() == 1) {
                return this.blocks.getFirst();
            }
            throw new IllegalArgumentException("Unclosed parse conditional blocks: [" + this.blocks.stream().filter(b -> !b.variable().isEmpty()).map(BlockOperation::variable).collect(Collectors.joining(", ")) + "]");
        }

        private void ensureAllPositionalArgumentsWereUsed() {
            int unused = 0;
            for (boolean b : this.positionals) {
                if (b) continue;
                ++unused;
            }
            if (unused > 0) {
                throw this.error(String.format("Found %d unused positional format arguments", unused));
            }
        }

        private void parseArgument(int pendingTextStart) {
            if (this.parser.peek() == '{') {
                this.parseBracedArgument(pendingTextStart);
            } else {
                if (pendingTextStart > -1) {
                    this.pushOperation(Operation.stringSlice(this.parser.input(), pendingTextStart, this.parser.position() - 1));
                }
                this.pushOperation(this.parseNormalArgument());
            }
        }

        private void parseBracedArgument(int pendingTextStart) {
            int startPosition;
            int startLine = this.parser.line();
            int startColumn = this.parser.column() - 2;
            boolean stripLeading = false;
            this.parser.expect('{');
            if (this.parser.peek() == '~') {
                stripLeading = true;
                this.parser.skip();
                if (pendingTextStart > -1) {
                    for (startPosition = this.parser.position() - 1; startPosition > pendingTextStart && Character.isWhitespace(this.parser.input().charAt(startPosition - 1)); --startPosition) {
                    }
                }
            }
            if (this.parser.peek() == '?' || this.parser.peek() == '^' || this.parser.peek() == '#') {
                this.pushBlock(pendingTextStart, startPosition, startLine, startColumn, this.parser.peek());
                return;
            }
            if (this.parser.peek() == '/') {
                this.popBlock(pendingTextStart, startPosition, startColumn);
                return;
            }
            if (pendingTextStart > -1) {
                this.pushOperation(Operation.stringSlice(this.parser.input(), pendingTextStart, startPosition));
            }
            Operation operation = this.parseNormalArgument();
            if (this.parser.peek() == '@') {
                this.parser.skip();
                int start = this.parser.position();
                this.parser.consumeWhile(this::isNameCharacter);
                String sectionName = this.parser.sliceFrom(start);
                this.ensureNameIsValid(sectionName);
                operation = Operation.inlineSection(sectionName, operation);
            }
            if (this.parser.peek() == '|') {
                if (stripLeading) {
                    throw this.error("Cannot strip leading whitespace with '~' when using inline block alignment '|'");
                }
                String staticWhitespace = this.isAllLeadingWhitespaceOnLine(startPosition, startColumn) ? this.template.substring(startPosition - startColumn, startPosition) : null;
                this.parser.expect('|');
                operation = Operation.block(operation, staticWhitespace);
            }
            boolean skipTrailingWs = this.parseStripTrailingWhitespace();
            this.parser.expect('}');
            this.pushOperation(operation);
            if (skipTrailingWs) {
                this.skipTrailingWhitespaceInParser();
            }
        }

        boolean parseStripTrailingWhitespace() {
            if (this.parser.peek() == '~') {
                this.parser.skip();
                return true;
            }
            return false;
        }

        void skipTrailingWhitespaceInParser() {
            this.parser.consumeWhile(Character::isWhitespace);
        }

        private boolean isAllLeadingWhitespaceOnLine(int startPosition, int startColumn) {
            for (int i = startPosition - startColumn; i < startPosition; ++i) {
                char ch = this.template.charAt(i);
                if (ch == ' ' || ch == '\t') continue;
                return false;
            }
            return true;
        }

        private void pushBlock(int pendingTextStart, int startPosition, int startLine, int startColumn, char c) {
            BlockOperation.Unconditional block;
            this.parser.expect(c);
            String name = this.parseArgumentName();
            switch (c) {
                case '#': {
                    String keyPrefix = "key";
                    String value = "value";
                    if (this.parser.peek() == ' ') {
                        this.parser.sp();
                        this.parser.expect('a');
                        this.parser.expect('s');
                        this.parser.sp();
                        int startPos = this.parser.position();
                        this.parser.consumeWhile(this::isNameCharacter);
                        keyPrefix = this.parser.sliceFrom(startPos);
                        this.ensureNameIsValid(keyPrefix);
                        this.parser.expect(',');
                        this.parser.sp();
                        startPos = this.parser.position();
                        this.parser.consumeWhile(this::isNameCharacter);
                        value = this.parser.sliceFrom(startPos);
                        this.ensureNameIsValid(value);
                    }
                    block = new BlockOperation.Loop(name, keyPrefix, value);
                    break;
                }
                case '^': {
                    block = new BlockOperation.Conditional(name, true);
                    break;
                }
                default: {
                    block = new BlockOperation.Conditional(name, false);
                }
            }
            boolean skipTrailingWs = this.parseStripTrailingWhitespace();
            this.parser.expect('}');
            this.handleConditionalOnLine(pendingTextStart, startPosition, startColumn);
            this.blocks.push(block);
            if (skipTrailingWs) {
                this.skipTrailingWhitespaceInParser();
            }
        }

        private void handleConditionalOnLine(int pendingTextStart, int startPosition, int startColumn) {
            if (this.parser.peek() != '\r' && this.parser.peek() != '\n' && this.parser.peek() != '\u0000') {
                if (pendingTextStart > -1) {
                    this.pushOperation(Operation.stringSlice(this.parser.input(), pendingTextStart, startPosition));
                }
            } else if (this.isAllLeadingWhitespaceOnLine(startPosition, startColumn)) {
                if (pendingTextStart > -1) {
                    this.pushOperation(Operation.stringSlice(this.parser.input(), pendingTextStart, startPosition - startColumn));
                }
                this.parser.skip();
            } else if (pendingTextStart > -1) {
                this.pushOperation(Operation.stringSlice(this.parser.input(), pendingTextStart, startPosition));
            }
        }

        private void popBlock(int pendingTextStart, int startPosition, int startColumn) {
            this.parser.expect('/');
            String name = this.parseArgumentName();
            this.parser.expect('}');
            if (this.blocks.size() == 1) {
                throw new IllegalArgumentException("Attempted to close unopened tag: '" + name + "'");
            }
            this.handleConditionalOnLine(pendingTextStart, startPosition, startColumn);
            BlockOperation block = this.blocks.pop();
            if (!block.variable().equals(name)) {
                throw new IllegalArgumentException("Invalid closing tag: '" + name + "'. Expected: '" + block.variable() + "'");
            }
            this.blocks.getFirst().push(block);
        }

        private Operation parseNormalArgument() {
            Function<AbstractCodeWriter<?>, Object> getter;
            char c = this.parser.peek();
            if (Character.isLowerCase(c)) {
                String name = this.parseNamedArgumentName();
                getter = w -> w.getContext(name);
            } else {
                getter = Character.isDigit(c) ? this.parsePositionalArgumentGetter() : this.parseRelativeArgumentGetter();
            }
            int line = this.parser.line();
            int column = this.parser.column();
            char identifier = this.parser.expect(CodeWriterFormatterContainer.VALID_FORMATTER_CHARS);
            return Operation.formatted(getter, identifier, () -> this.createErrorMessage(String.format("Syntax error at line %d column %d: Unknown formatter `%c` found in format string", line, column, Character.valueOf(identifier))));
        }

        private String parseArgumentName() {
            int start = this.parser.position();
            this.parser.consumeWhile(this::isNameCharacter);
            String name = this.parser.sliceFrom(start);
            this.ensureNameIsValid(name);
            return name;
        }

        private String parseNamedArgumentName() {
            String name = this.parseArgumentName();
            this.parser.expect(':');
            if (this.parser.eof()) {
                throw this.error("Expected an identifier after the ':' in a named argument");
            }
            return name;
        }

        private Function<AbstractCodeWriter<?>, Object> parseRelativeArgumentGetter() {
            if (this.relativeIndex == -1) {
                throw this.error("Cannot mix positional and relative arguments");
            }
            ++this.relativeIndex;
            Object result = this.getPositionalArgument(this.relativeIndex - 1);
            return w -> result;
        }

        private Object getPositionalArgument(int index) {
            if (index >= this.arguments.length) {
                throw this.error(String.format("Given %d arguments but attempted to format index %d", this.arguments.length, index));
            }
            this.positionals[index] = true;
            return this.arguments[index];
        }

        private Function<AbstractCodeWriter<?>, Object> parsePositionalArgumentGetter() {
            if (this.relativeIndex > 0) {
                throw this.error("Cannot mix positional and relative arguments");
            }
            this.relativeIndex = -1;
            int startPosition = this.parser.position();
            this.parser.consumeWhile(Character::isDigit);
            int index = Integer.parseInt(this.parser.sliceFrom(startPosition)) - 1;
            if (index < 0 || index >= this.arguments.length) {
                throw this.error(String.format("Positional argument index %d out of range of provided %d arguments in format string", index, this.arguments.length));
            }
            Object value = this.getPositionalArgument(index);
            return w -> value;
        }

        private void ensureNameIsValid(String name) {
            if (!NAME_PATTERN.matcher(name).matches()) {
                throw this.error(String.format("Invalid format expression name `%s`", name));
            }
        }

        private boolean isNameCharacter(int c) {
            return c >= 97 && c <= 122 || c >= 65 && c <= 90 || c >= 48 && c <= 57 || c == 95 || c == 46 || c == 35 || c == 36;
        }
    }

    private static abstract class BlockOperation
    implements Operation {
        private final String variable;

        BlockOperation(String variable) {
            this.variable = variable;
        }

        final String variable() {
            return this.variable;
        }

        abstract void push(Operation var1);

        static final class Loop
        extends Unconditional {
            private final String keyName;
            private final String valueName;

            Loop(String variableName, String keyName, String valueName) {
                super(variableName);
                this.keyName = keyName;
                this.valueName = valueName;
            }

            @Override
            public void apply(Sink sink, AbstractCodeWriter<?> writer) throws IOException {
                Object value = writer.getContext(this.variable());
                Iterator iterator = CodeFormatter.getValueIterator(value);
                boolean isFirst = true;
                while (iterator.hasNext()) {
                    Map.Entry current = (Map.Entry)iterator.next();
                    writer.pushState();
                    writer.putContext(this.keyName, current.getKey());
                    writer.putContext(this.valueName, current.getValue());
                    writer.putContext(this.keyName + ".first", isFirst);
                    writer.putContext(this.keyName + ".last", !iterator.hasNext());
                    super.apply(sink, writer);
                    writer.popState();
                    isFirst = false;
                }
            }
        }

        static final class Conditional
        extends Unconditional {
            private final boolean negate;

            Conditional(String variable, boolean negate) {
                super(variable);
                this.negate = negate;
            }

            @Override
            public void apply(Sink sink, AbstractCodeWriter<?> writer) throws IOException {
                Object value = writer.getContext(this.variable());
                if (!CodeFormatter.isConditionTruthy(value) == this.negate) {
                    super.apply(sink, writer);
                }
            }
        }

        static class Unconditional
        extends BlockOperation {
            private final List<Operation> operations = new ArrayList<Operation>();

            Unconditional(String variable) {
                super(variable);
            }

            @Override
            public void apply(Sink sink, AbstractCodeWriter<?> writer) throws IOException {
                for (Operation operation : this.operations) {
                    operation.apply(sink, writer);
                }
            }

            @Override
            public void push(Operation operation) {
                this.operations.add(operation);
            }
        }
    }

    private static final class BlockAlignedSink
    implements Sink {
        private final int spaces;
        private final String staticWhitespace;
        private boolean previousIsCarriageReturn;
        private boolean previousIsNewline;
        private final Sink delegate;

        BlockAlignedSink(Sink delegate, String staticWhitespace) {
            this.delegate = delegate;
            this.spaces = delegate.column();
            this.staticWhitespace = staticWhitespace;
        }

        public String toString() {
            return this.delegate.toString();
        }

        @Override
        public int column() {
            return this.delegate.column();
        }

        @Override
        public void append(char c) {
            if (this.previousIsNewline) {
                this.writeSpaces();
                this.previousIsNewline = false;
            }
            if (c == '\n') {
                this.delegate.append('\n');
                this.previousIsNewline = true;
                this.previousIsCarriageReturn = false;
            } else {
                if (this.previousIsCarriageReturn) {
                    this.writeSpaces();
                }
                this.previousIsCarriageReturn = c == '\r';
                this.delegate.append(c);
            }
        }

        private void writeSpaces() {
            if (this.staticWhitespace != null) {
                Sink.writeString(this.delegate, this.staticWhitespace);
            } else {
                for (int i = 0; i < this.spaces; ++i) {
                    this.delegate.append(' ');
                }
            }
        }
    }

    @FunctionalInterface
    private static interface Operation {
        public void apply(Sink var1, AbstractCodeWriter<?> var2) throws IOException;

        public static Operation stringSlice(CharSequence source, int start, int end) {
            return (sink, writer) -> Sink.writeString(sink, source, start, end);
        }

        public static Operation formatted(Function<AbstractCodeWriter<?>, Object> valueGetter, char formatter, Supplier<String> errorMessage) {
            return (sink, writer) -> {
                Object value = valueGetter.apply(writer);
                String result = writer.applyFormatter(formatter, value);
                if (result == null) {
                    throw new RuntimeException((String)errorMessage.get());
                }
                Sink.writeString(sink, result);
            };
        }

        public static Operation inlineSection(String sectionName, Operation delegate) {
            return (sink, writer) -> {
                Sink buffer = Sink.from(new StringBuilder());
                delegate.apply(buffer, writer);
                String defaultValue = buffer.toString();
                CodeSection section = CodeSection.forName(sectionName);
                String expanded = writer.expandSection(section, defaultValue, writer::writeInlineWithNoFormatting);
                expanded = writer.removeTrailingNewline(expanded);
                Sink.writeString(sink, expanded);
            };
        }

        public static Operation block(Operation delegate, String staticWhitespace) {
            return (sink, writer) -> delegate.apply(new BlockAlignedSink(sink, staticWhitespace), writer);
        }
    }

    private static interface Sink {
        public int column();

        public void append(char var1);

        public static void writeString(Sink sink, CharSequence text) {
            Sink.writeString(sink, text, 0, text.length());
        }

        public static void writeString(Sink sink, CharSequence text, int start, int end) {
            for (int i = start; i < end; ++i) {
                sink.append(text.charAt(i));
            }
        }

        public static Sink from(final StringBuilder builder) {
            return new Sink(){
                private int column = 0;

                @Override
                public int column() {
                    return this.column;
                }

                @Override
                public void append(char c) {
                    this.column = c == '\r' || c == '\n' ? 0 : ++this.column;
                    builder.append(c);
                }

                public String toString() {
                    return builder.toString();
                }
            };
        }
    }
}

