/*
 * Decompiled with CFR 0.152.
 */
package software.amazon.smithy.model.loader;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.stream.Collectors;
import software.amazon.smithy.model.Model;
import software.amazon.smithy.model.SourceLocation;
import software.amazon.smithy.model.loader.Prelude;
import software.amazon.smithy.model.loader.ValidationLoader;
import software.amazon.smithy.model.loader.ValidatorDefinition;
import software.amazon.smithy.model.loader.ValidatorFromDefinitionFactory;
import software.amazon.smithy.model.node.ObjectNode;
import software.amazon.smithy.model.shapes.Shape;
import software.amazon.smithy.model.traits.SuppressTrait;
import software.amazon.smithy.model.validation.Severity;
import software.amazon.smithy.model.validation.ValidatedResult;
import software.amazon.smithy.model.validation.ValidationEvent;
import software.amazon.smithy.model.validation.Validator;
import software.amazon.smithy.model.validation.ValidatorFactory;
import software.amazon.smithy.utils.ListUtils;

final class ModelValidator {
    private static final String SUPPRESSIONS = "suppressions";
    private static final String ID = "id";
    private static final String NAMESPACE = "namespace";
    private static final String REASON = "reason";
    private static final String STAR = "*";
    private static final String EMPTY_REASON = "";
    private static final Collection<String> SUPPRESSION_KEYS = ListUtils.of((Object[])new String[]{"id", "namespace", "reason"});
    private final List<Validator> validators;
    private final ArrayList<ValidationEvent> events = new ArrayList();
    private final ValidatorFactory validatorFactory;
    private final Model model;
    private final Map<String, Map<String, String>> namespaceSuppressions = new HashMap<String, Map<String, String>>();

    private ModelValidator(Model model, ValidatorFactory validatorFactory, List<Validator> validators) {
        this.model = model;
        this.validatorFactory = validatorFactory;
        this.validators = new ArrayList<Validator>(validators);
    }

    static List<ValidationEvent> validate(Model model, ValidatorFactory validatorFactory, List<Validator> validators) {
        return new ModelValidator(model, validatorFactory, validators).doValidate();
    }

    private List<ValidationEvent> doValidate() {
        this.assembleNamespaceSuppressions();
        List<ValidatorDefinition> assembledValidatorDefinitions = this.assembleValidatorDefinitions();
        this.assembleValidators(assembledValidatorDefinitions);
        this.events.addAll(this.validators.parallelStream().flatMap(validator -> validator.validate(this.model).stream()).map(this::suppressEvent).filter(ModelValidator::filterPrelude).collect(Collectors.toList()));
        return this.events;
    }

    private static boolean filterPrelude(ValidationEvent event) {
        return event.getSeverity() == Severity.ERROR || !event.getShapeId().filter(Prelude::isPreludeShape).isPresent();
    }

    private List<ValidatorDefinition> assembleValidatorDefinitions() {
        ValidatedResult<List<ValidatorDefinition>> result = ValidationLoader.loadValidators(this.model.getMetadata());
        this.events.addAll(result.getValidationEvents());
        return result.getResult().orElseGet(Collections::emptyList);
    }

    private void assembleValidators(List<ValidatorDefinition> definitions) {
        ValidatorFromDefinitionFactory factory = new ValidatorFromDefinitionFactory(this.validatorFactory);
        for (ValidatorDefinition val : definitions) {
            ValidatedResult<Validator> result = factory.loadValidator(val);
            result.getResult().ifPresent(this.validators::add);
            this.events.addAll(result.getValidationEvents());
            if (!result.getValidationEvents().isEmpty() || result.getResult().isPresent()) continue;
            this.events.add(this.suppressEvent(this.unknownValidatorError(val.name, val.sourceLocation)));
        }
    }

    private ValidationEvent unknownValidatorError(String name, SourceLocation location) {
        return ValidationEvent.builder().id("UnknownValidator_" + name).severity(Severity.WARNING).sourceLocation(location).message("Unable to locate a validator named `" + name + "`").build();
    }

    private void assembleNamespaceSuppressions() {
        this.model.getMetadataProperty(SUPPRESSIONS).ifPresent(value -> {
            List<ObjectNode> values = value.expectArrayNode().getElementsAs(ObjectNode.class);
            for (ObjectNode rule : values) {
                rule.warnIfAdditionalProperties(SUPPRESSION_KEYS);
                String id = rule.expectStringMember(ID).getValue();
                String namespace = rule.expectStringMember(NAMESPACE).getValue();
                String reason = rule.getStringMemberOrDefault(REASON, EMPTY_REASON);
                this.namespaceSuppressions.computeIfAbsent(id, i -> new HashMap()).put(namespace, reason);
            }
        });
    }

    private ValidationEvent suppressEvent(ValidationEvent event) {
        if (!event.getSeverity().canSuppress()) {
            return event;
        }
        String reason = this.resolveReason(event);
        if (reason == null) {
            return event;
        }
        ValidationEvent.Builder builder = event.toBuilder();
        builder.severity(Severity.SUPPRESSED);
        if (!reason.equals(EMPTY_REASON)) {
            builder.suppressionReason(reason);
        }
        return builder.build();
    }

    private String resolveReason(ValidationEvent event) {
        return event.getShapeId().flatMap(this.model::getShape).flatMap(shape -> this.matchSuppression((Shape)shape, event.getId())).orElseGet(() -> this.matchWildcardNamespaceSuppressions(event.getId()));
    }

    private Optional<String> matchSuppression(Shape shape, String eventId) {
        Map<String, String> namespaces;
        if (shape.getTrait(SuppressTrait.class).isPresent() && shape.expectTrait(SuppressTrait.class).getValues().contains(eventId)) {
            return Optional.of(EMPTY_REASON);
        }
        if (this.namespaceSuppressions.containsKey(eventId) && (namespaces = this.namespaceSuppressions.get(eventId)).containsKey(shape.getId().getNamespace())) {
            return Optional.of(namespaces.get(shape.getId().getNamespace()));
        }
        return Optional.empty();
    }

    private String matchWildcardNamespaceSuppressions(String eventId) {
        Map<String, String> namespaces;
        if (this.namespaceSuppressions.containsKey(eventId) && (namespaces = this.namespaceSuppressions.get(eventId)).containsKey(STAR)) {
            return namespaces.get(STAR);
        }
        return null;
    }
}

