/*
 * Decompiled with CFR 0.152.
 */
package software.amazon.smithy.cli.dependencies;

import java.net.URI;
import java.net.URISyntaxException;
import java.nio.file.Paths;
import java.util.ArrayList;
import java.util.List;
import java.util.logging.Logger;
import software.amazon.smithy.build.model.MavenRepository;
import software.amazon.smithy.cli.dependencies.DependencyResolver;
import software.amazon.smithy.cli.dependencies.DependencyResolverException;
import software.amazon.smithy.cli.dependencies.ResolvedArtifact;
import software.amazon.smithy.cli.shaded.apache.maven.repository.internal.MavenRepositorySystemUtils;
import software.amazon.smithy.cli.shaded.eclipse.aether.DefaultRepositorySystemSession;
import software.amazon.smithy.cli.shaded.eclipse.aether.RepositorySystem;
import software.amazon.smithy.cli.shaded.eclipse.aether.artifact.Artifact;
import software.amazon.smithy.cli.shaded.eclipse.aether.artifact.DefaultArtifact;
import software.amazon.smithy.cli.shaded.eclipse.aether.collection.CollectRequest;
import software.amazon.smithy.cli.shaded.eclipse.aether.connector.basic.BasicRepositoryConnectorFactory;
import software.amazon.smithy.cli.shaded.eclipse.aether.graph.Dependency;
import software.amazon.smithy.cli.shaded.eclipse.aether.graph.DependencyFilter;
import software.amazon.smithy.cli.shaded.eclipse.aether.impl.DefaultServiceLocator;
import software.amazon.smithy.cli.shaded.eclipse.aether.repository.LocalRepository;
import software.amazon.smithy.cli.shaded.eclipse.aether.repository.RemoteRepository;
import software.amazon.smithy.cli.shaded.eclipse.aether.resolution.ArtifactResult;
import software.amazon.smithy.cli.shaded.eclipse.aether.resolution.DependencyRequest;
import software.amazon.smithy.cli.shaded.eclipse.aether.resolution.DependencyResolutionException;
import software.amazon.smithy.cli.shaded.eclipse.aether.spi.connector.RepositoryConnectorFactory;
import software.amazon.smithy.cli.shaded.eclipse.aether.spi.connector.transport.TransporterFactory;
import software.amazon.smithy.cli.shaded.eclipse.aether.transport.file.FileTransporterFactory;
import software.amazon.smithy.cli.shaded.eclipse.aether.transport.http.HttpTransporterFactory;
import software.amazon.smithy.cli.shaded.eclipse.aether.util.filter.DependencyFilterUtils;
import software.amazon.smithy.cli.shaded.eclipse.aether.util.repository.AuthenticationBuilder;

public final class MavenDependencyResolver
implements DependencyResolver {
    private static final Logger LOGGER = Logger.getLogger(DependencyResolver.class.getName());
    private final List<RemoteRepository> remoteRepositories = new ArrayList<RemoteRepository>();
    private final List<Dependency> dependencies = new ArrayList<Dependency>();
    private final DefaultRepositorySystemSession session = MavenRepositorySystemUtils.newSession();
    private final DependencyFilter filter = DependencyFilterUtils.classpathFilter("runtime", "compile");
    private final RepositorySystem repositorySystem;

    public MavenDependencyResolver() {
        this(null);
    }

    public MavenDependencyResolver(String cacheLocation) {
        DefaultServiceLocator locator = MavenRepositorySystemUtils.newServiceLocator();
        locator.addService(RepositoryConnectorFactory.class, BasicRepositoryConnectorFactory.class);
        locator.addService(TransporterFactory.class, FileTransporterFactory.class);
        locator.addService(TransporterFactory.class, HttpTransporterFactory.class);
        locator.setErrorHandler(new DefaultServiceLocator.ErrorHandler(){

            @Override
            public void serviceCreationFailed(Class<?> type, Class<?> impl, Throwable exception) {
                throw new DependencyResolverException(exception);
            }
        });
        this.repositorySystem = locator.getService(RepositorySystem.class);
        if (cacheLocation == null) {
            String userHome = System.getProperty("user.home");
            cacheLocation = Paths.get(userHome, ".m2", "repository").toString();
            LOGGER.fine("Set default Maven local cache location to ~/.m2/repository");
        }
        LocalRepository local = new LocalRepository(cacheLocation);
        this.session.setLocalRepositoryManager(this.repositorySystem.newLocalRepositoryManager(this.session, local));
    }

    @Override
    public void addRepository(MavenRepository repository) {
        try {
            URI uri = new URI(repository.getUrl());
            int repositoryIndex = this.remoteRepositories.size() + 1;
            String id = repository.getId().orElseGet(() -> repositoryIndex + "|" + uri.getHost());
            String userInfo = uri.getUserInfo();
            RemoteRepository.Builder builder = new RemoteRepository.Builder(id, "default", repository.getUrl());
            if (userInfo != null) {
                LOGGER.finest(() -> "Setting username and password for " + id + " using URI authority");
                this.addUserInfoAuth(uri, userInfo, builder);
            }
            repository.getHttpCredentials().ifPresent(credentials -> this.addUserInfoAuth(uri, (String)credentials, builder));
            this.remoteRepositories.add(builder.build());
        }
        catch (URISyntaxException e) {
            throw new DependencyResolverException("Invalid Maven repository URL: " + repository.getUrl() + ": " + e.getMessage());
        }
    }

    private void addUserInfoAuth(URI uri, String userInfo, RemoteRepository.Builder builder) {
        String[] parts = userInfo.split(":", 2);
        if (parts.length != 2) {
            throw new DependencyResolverException("Invalid credentials provided for " + uri);
        }
        builder.setAuthentication(new AuthenticationBuilder().addUsername(parts[0]).addPassword(parts[1]).build());
    }

    @Override
    public void addDependency(String coordinates) {
        this.dependencies.add(MavenDependencyResolver.createDependency(coordinates, "compile"));
    }

    @Override
    public List<ResolvedArtifact> resolve() {
        if (this.remoteRepositories.isEmpty()) {
            LOGGER.warning("No Maven repositories are configured, so only the local repository cache is being used");
        }
        List<ArtifactResult> results = this.resolveMavenArtifacts();
        ArrayList<ResolvedArtifact> artifacts = new ArrayList<ResolvedArtifact>(results.size());
        for (ArtifactResult result : results) {
            Artifact artifact = result.getArtifact();
            artifacts.add(new ResolvedArtifact(artifact.getFile().toPath(), artifact.getGroupId(), artifact.getArtifactId(), artifact.getVersion()));
        }
        return artifacts;
    }

    private static Dependency createDependency(String coordinates, String scope) {
        DefaultArtifact artifact;
        try {
            artifact = new DefaultArtifact(coordinates);
        }
        catch (IllegalArgumentException e) {
            throw new DependencyResolverException("Invalid dependency: " + e.getMessage());
        }
        MavenDependencyResolver.validateDependencyVersion(artifact);
        return new Dependency(artifact, scope);
    }

    private static void validateDependencyVersion(Artifact artifact) {
        String version = artifact.getVersion();
        if (version.equals("LATEST")) {
            throw new DependencyResolverException("LATEST dependencies are not supported: " + artifact);
        }
        if (version.equals("latest-status") || version.startsWith("latest.")) {
            throw new DependencyResolverException("Gradle style latest dependencies are not supported: " + artifact);
        }
        if (version.equals("RELEASE")) {
            throw new DependencyResolverException("RELEASE dependencies are not supported: " + artifact);
        }
        if (version.contains("+")) {
            throw new DependencyResolverException("'+' dependencies are not supported: " + artifact);
        }
    }

    private List<ArtifactResult> resolveMavenArtifacts() {
        LOGGER.fine(() -> "Resolving Maven dependencies for Smithy CLI; repos: " + this.remoteRepositories + "; dependencies: " + this.dependencies);
        CollectRequest collectRequest = new CollectRequest();
        collectRequest.setRepositories(this.remoteRepositories);
        collectRequest.setDependencies(this.dependencies);
        DependencyRequest dependencyRequest = new DependencyRequest(collectRequest, this.filter);
        try {
            List<ArtifactResult> results = this.repositorySystem.resolveDependencies(this.session, dependencyRequest).getArtifactResults();
            LOGGER.fine(() -> "Resolved Maven dependencies: " + results);
            return results;
        }
        catch (DependencyResolutionException e) {
            throw new DependencyResolverException(e);
        }
    }
}

