/*
 * Decompiled with CFR 0.152.
 */
package software.amazon.awssdk.s3accessgrants.cache;

import com.github.benmanes.caffeine.cache.AsyncCache;
import com.github.benmanes.caffeine.cache.Caffeine;
import com.github.benmanes.caffeine.cache.Expiry;
import com.github.benmanes.caffeine.cache.stats.CacheStats;
import java.time.Instant;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.CompletionStage;
import java.util.concurrent.TimeUnit;
import org.assertj.core.util.VisibleForTesting;
import software.amazon.awssdk.annotations.NotNull;
import software.amazon.awssdk.auth.credentials.AwsSessionCredentials;
import software.amazon.awssdk.identity.spi.AwsCredentialsIdentity;
import software.amazon.awssdk.s3accessgrants.cache.CacheKey;
import software.amazon.awssdk.s3accessgrants.cache.S3AccessGrantsAccessDeniedCache;
import software.amazon.awssdk.s3accessgrants.cache.S3AccessGrantsCachedAccountIdResolver;
import software.amazon.awssdk.services.s3control.S3ControlAsyncClient;
import software.amazon.awssdk.services.s3control.model.Credentials;
import software.amazon.awssdk.services.s3control.model.GetDataAccessRequest;
import software.amazon.awssdk.services.s3control.model.GetDataAccessResponse;
import software.amazon.awssdk.services.s3control.model.Permission;
import software.amazon.awssdk.services.s3control.model.Privilege;
import software.amazon.awssdk.services.s3control.model.S3ControlException;
import software.amazon.awssdk.utils.Logger;

public class S3AccessGrantsCache {
    private AsyncCache<CacheKey, AwsCredentialsIdentity> cache;
    private final S3ControlAsyncClient s3ControlAsyncClient;
    private int maxCacheSize;
    private final S3AccessGrantsCachedAccountIdResolver s3AccessGrantsCachedAccountIdResolver;
    private final int cacheExpirationTimePercentage;
    private static final Logger logger = Logger.loggerFor(S3AccessGrantsCache.class);

    private S3AccessGrantsCache(@NotNull S3ControlAsyncClient s3ControlAsyncClient, S3AccessGrantsCachedAccountIdResolver resolver, int maxCacheSize, int cacheExpirationTimePercentage) {
        if (s3ControlAsyncClient == null) {
            throw new IllegalArgumentException("S3ControlAsyncClient is required");
        }
        this.s3ControlAsyncClient = s3ControlAsyncClient;
        this.s3AccessGrantsCachedAccountIdResolver = resolver;
        this.cacheExpirationTimePercentage = cacheExpirationTimePercentage;
        this.maxCacheSize = maxCacheSize;
        this.cache = Caffeine.newBuilder().maximumSize((long)maxCacheSize).expireAfter(new CustomCacheExpiry()).recordStats().buildAsync();
    }

    protected S3AccessGrantsCachedAccountIdResolver getS3AccessGrantsCachedAccountIdResolver() {
        return this.s3AccessGrantsCachedAccountIdResolver;
    }

    protected static Builder builder() {
        return new BuilderImpl();
    }

    protected CompletableFuture<AwsCredentialsIdentity> getCredentials(CacheKey cacheKey, String accountId, S3AccessGrantsAccessDeniedCache s3AccessGrantsAccessDeniedCache) throws S3ControlException {
        logger.debug(() -> "Fetching credentials from Access Grants for s3Prefix: " + cacheKey.s3Prefix);
        CompletionStage<Object> credentials = this.searchKeyInCacheAtPrefixLevel(cacheKey);
        if (credentials == null && (cacheKey.permission == Permission.READ || cacheKey.permission == Permission.WRITE)) {
            credentials = this.searchKeyInCacheAtPrefixLevel(cacheKey.toBuilder().permission(Permission.READWRITE).build());
        }
        if (credentials == null) {
            credentials = this.searchKeyInCacheAtCharacterLevel(cacheKey);
        }
        if (credentials == null && (cacheKey.permission == Permission.READ || cacheKey.permission == Permission.WRITE)) {
            credentials = this.searchKeyInCacheAtCharacterLevel(cacheKey.toBuilder().permission(Permission.READWRITE).build());
        }
        if (credentials == null) {
            try {
                logger.debug(() -> "Credentials not available in the cache. Fetching credentials from Access Grants service.");
                credentials = this.getCredentialsFromService(cacheKey, accountId).thenApply(getDataAccessResponse -> {
                    Credentials accessGrantsCredentials = getDataAccessResponse.credentials();
                    long duration = this.getTTL(accessGrantsCredentials.expiration());
                    AwsSessionCredentials sessionCredentials = AwsSessionCredentials.builder().accessKeyId(accessGrantsCredentials.accessKeyId()).secretAccessKey(accessGrantsCredentials.secretAccessKey()).sessionToken(accessGrantsCredentials.sessionToken()).build();
                    String accessGrantsTarget = getDataAccessResponse.matchedGrantTarget();
                    if (accessGrantsTarget.endsWith("*")) {
                        this.putValueInCache(cacheKey.toBuilder().s3Prefix(this.processMatchedGrantTarget(accessGrantsTarget)).build(), CompletableFuture.supplyAsync(() -> sessionCredentials), duration);
                    }
                    logger.debug(() -> "Successfully retrieved the credentials from Access Grants service");
                    return sessionCredentials;
                });
            }
            catch (S3ControlException s3ControlException) {
                logger.error(() -> "Exception occurred while fetching the credentials: " + (Object)((Object)s3ControlException));
                if (s3ControlException.statusCode() == 403) {
                    logger.debug(() -> "Caching the Access Denied request.");
                    s3AccessGrantsAccessDeniedCache.putValueInCache(cacheKey, s3ControlException);
                }
                throw s3ControlException;
            }
        }
        return credentials;
    }

    @VisibleForTesting
    long getTTL(Instant expirationTime) {
        Instant now = Instant.now();
        return (long)((float)(expirationTime.getEpochSecond() - now.getEpochSecond()) * ((float)this.cacheExpirationTimePercentage / 100.0f));
    }

    private CompletableFuture<GetDataAccessResponse> getCredentialsFromService(CacheKey cacheKey, String accountId) throws S3ControlException {
        String resolvedAccountId = this.s3AccessGrantsCachedAccountIdResolver.resolve(accountId, cacheKey.s3Prefix);
        logger.debug(() -> "Fetching credentials from Access Grants for accountId: " + resolvedAccountId + ", s3Prefix: " + cacheKey.s3Prefix + ", permission: " + cacheKey.permission + ", privilege: " + Privilege.DEFAULT);
        GetDataAccessRequest dataAccessRequest = (GetDataAccessRequest)GetDataAccessRequest.builder().accountId(resolvedAccountId).target(cacheKey.s3Prefix).permission(cacheKey.permission).privilege(Privilege.DEFAULT).build();
        return this.s3ControlAsyncClient.getDataAccess(dataAccessRequest);
    }

    private CompletableFuture<AwsCredentialsIdentity> searchKeyInCacheAtPrefixLevel(CacheKey cacheKey) {
        String prefix = cacheKey.s3Prefix;
        while (!prefix.equals("s3:")) {
            CompletableFuture cacheValue = this.cache.getIfPresent((Object)cacheKey.toBuilder().s3Prefix(prefix).build());
            if (cacheValue != null) {
                logger.debug(() -> "Successfully retrieved credentials from the cache.");
                return cacheValue;
            }
            prefix = this.getNextPrefix(prefix);
        }
        return null;
    }

    private CompletableFuture<AwsCredentialsIdentity> searchKeyInCacheAtCharacterLevel(CacheKey cacheKey) {
        String prefix = cacheKey.s3Prefix;
        while (!prefix.equals("s3://")) {
            CompletableFuture cacheValue = this.cache.getIfPresent((Object)cacheKey.toBuilder().s3Prefix(prefix + "*").build());
            if (cacheValue != null) {
                logger.debug(() -> "Successfully retrieved credentials from the cache.");
                return cacheValue;
            }
            prefix = this.getNextPrefixByChar(prefix);
        }
        return null;
    }

    @VisibleForTesting
    void putValueInCache(CacheKey cacheKey, CompletableFuture<AwsCredentialsIdentity> credentials, long duration) {
        logger.debug(() -> "Caching the credentials for s3Prefix:" + cacheKey.s3Prefix + " and permission: " + cacheKey.permission);
        this.cache.put((Object)cacheKey, credentials);
        this.cache.synchronous().policy().expireVariably().ifPresent(ev -> ev.setExpiresAfter((Object)cacheKey, duration, TimeUnit.SECONDS));
    }

    private String getNextPrefix(String prefix) {
        return prefix.substring(0, prefix.lastIndexOf("/"));
    }

    private String getNextPrefixByChar(String prefix) {
        return prefix.substring(0, prefix.length() - 1);
    }

    @VisibleForTesting
    String processMatchedGrantTarget(String matchedGrantTarget) {
        if (matchedGrantTarget.substring(matchedGrantTarget.length() - 2).equals("/*")) {
            return matchedGrantTarget.substring(0, matchedGrantTarget.length() - 2);
        }
        return matchedGrantTarget;
    }

    protected CacheStats getCacheStats() {
        return this.cache.synchronous().stats();
    }

    @VisibleForTesting
    void invalidateCache() {
        this.cache.synchronous().invalidateAll();
    }

    private static class CustomCacheExpiry<K, V>
    implements Expiry<K, V> {
        private CustomCacheExpiry() {
        }

        public long expireAfterCreate(K key, V value, long currentTime) {
            return Long.MIN_VALUE;
        }

        public long expireAfterUpdate(K key, V value, long currentTime, long currentDuration) {
            return currentDuration;
        }

        public long expireAfterRead(K key, V value, long currentTime, long currentDuration) {
            return currentDuration;
        }
    }

    static final class BuilderImpl
    implements Builder {
        private S3ControlAsyncClient s3ControlAsyncClient;
        private int maxCacheSize = 30000;
        private S3AccessGrantsCachedAccountIdResolver s3AccessGrantsCachedAccountIdResolver;
        private int cacheExpirationTimePercentage;

        private BuilderImpl() {
        }

        @Override
        public S3AccessGrantsCache build() {
            S3AccessGrantsCachedAccountIdResolver s3AccessGrantsCachedAccountIdResolver = S3AccessGrantsCachedAccountIdResolver.builder().S3ControlAsyncClient(this.s3ControlAsyncClient).build();
            return new S3AccessGrantsCache(this.s3ControlAsyncClient, s3AccessGrantsCachedAccountIdResolver, this.maxCacheSize, this.cacheExpirationTimePercentage);
        }

        @Override
        public S3AccessGrantsCache buildWithAccountIdResolver() {
            return new S3AccessGrantsCache(this.s3ControlAsyncClient, this.s3AccessGrantsCachedAccountIdResolver, this.maxCacheSize, this.cacheExpirationTimePercentage);
        }

        @Override
        public Builder s3ControlAsyncClient(S3ControlAsyncClient s3ControlAsyncClient) {
            this.s3ControlAsyncClient = s3ControlAsyncClient;
            return this;
        }

        @Override
        public Builder maxCacheSize(int maxCacheSize) {
            this.maxCacheSize = maxCacheSize;
            return this;
        }

        @Override
        public Builder cacheExpirationTimePercentage(int cacheExpirationTimePrecentage) {
            this.cacheExpirationTimePercentage = cacheExpirationTimePrecentage;
            return this;
        }

        @Override
        public Builder s3AccessGrantsCachedAccountIdResolver(S3AccessGrantsCachedAccountIdResolver s3AccessGrantsCachedAccountIdResolver) {
            this.s3AccessGrantsCachedAccountIdResolver = s3AccessGrantsCachedAccountIdResolver;
            return this;
        }
    }

    public static interface Builder {
        public S3AccessGrantsCache build();

        public S3AccessGrantsCache buildWithAccountIdResolver();

        public Builder s3ControlAsyncClient(S3ControlAsyncClient var1);

        public Builder maxCacheSize(int var1);

        public Builder cacheExpirationTimePercentage(int var1);

        public Builder s3AccessGrantsCachedAccountIdResolver(S3AccessGrantsCachedAccountIdResolver var1);
    }
}

