/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.transfer.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.Mutable;
import software.amazon.awssdk.annotations.NotThreadSafe;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Each step type has its own <code>StepDetails</code> structure.
 * </p>
 * <p>
 * The key/value pairs used to tag a file during the execution of a workflow step.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class TagStepDetails implements SdkPojo, Serializable, ToCopyableBuilder<TagStepDetails.Builder, TagStepDetails> {
    private static final SdkField<String> NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Name")
            .getter(getter(TagStepDetails::name)).setter(setter(Builder::name))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Name").build()).build();

    private static final SdkField<List<S3Tag>> TAGS_FIELD = SdkField
            .<List<S3Tag>> builder(MarshallingType.LIST)
            .memberName("Tags")
            .getter(getter(TagStepDetails::tags))
            .setter(setter(Builder::tags))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Tags").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<S3Tag> builder(MarshallingType.SDK_POJO)
                                            .constructor(S3Tag::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<String> SOURCE_FILE_LOCATION_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("SourceFileLocation").getter(getter(TagStepDetails::sourceFileLocation))
            .setter(setter(Builder::sourceFileLocation))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("SourceFileLocation").build())
            .build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(NAME_FIELD, TAGS_FIELD,
            SOURCE_FILE_LOCATION_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = memberNameToFieldInitializer();

    private static final long serialVersionUID = 1L;

    private final String name;

    private final List<S3Tag> tags;

    private final String sourceFileLocation;

    private TagStepDetails(BuilderImpl builder) {
        this.name = builder.name;
        this.tags = builder.tags;
        this.sourceFileLocation = builder.sourceFileLocation;
    }

    /**
     * <p>
     * The name of the step, used as an identifier.
     * </p>
     * 
     * @return The name of the step, used as an identifier.
     */
    public final String name() {
        return name;
    }

    /**
     * For responses, this returns true if the service returned a value for the Tags property. This DOES NOT check that
     * the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is useful
     * because the SDK will never return a null collection or map, but you may need to differentiate between the service
     * returning nothing (or null) and the service returning an empty collection or map. For requests, this returns true
     * if a value for the property was specified in the request builder, and false if a value was not specified.
     */
    public final boolean hasTags() {
        return tags != null && !(tags instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * Array that contains from 1 to 10 key/value pairs.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasTags} method.
     * </p>
     * 
     * @return Array that contains from 1 to 10 key/value pairs.
     */
    public final List<S3Tag> tags() {
        return tags;
    }

    /**
     * <p>
     * Specifies which file to use as input to the workflow step: either the output from the previous step, or the
     * originally uploaded file for the workflow.
     * </p>
     * <ul>
     * <li>
     * <p>
     * To use the previous file as the input, enter <code>${previous.file}</code>. In this case, this workflow step uses
     * the output file from the previous workflow step as input. This is the default value.
     * </p>
     * </li>
     * <li>
     * <p>
     * To use the originally uploaded file location as input for this step, enter <code>${original.file}</code>.
     * </p>
     * </li>
     * </ul>
     * 
     * @return Specifies which file to use as input to the workflow step: either the output from the previous step, or
     *         the originally uploaded file for the workflow.</p>
     *         <ul>
     *         <li>
     *         <p>
     *         To use the previous file as the input, enter <code>${previous.file}</code>. In this case, this workflow
     *         step uses the output file from the previous workflow step as input. This is the default value.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         To use the originally uploaded file location as input for this step, enter <code>${original.file}</code>.
     *         </p>
     *         </li>
     */
    public final String sourceFileLocation() {
        return sourceFileLocation;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(name());
        hashCode = 31 * hashCode + Objects.hashCode(hasTags() ? tags() : null);
        hashCode = 31 * hashCode + Objects.hashCode(sourceFileLocation());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof TagStepDetails)) {
            return false;
        }
        TagStepDetails other = (TagStepDetails) obj;
        return Objects.equals(name(), other.name()) && hasTags() == other.hasTags() && Objects.equals(tags(), other.tags())
                && Objects.equals(sourceFileLocation(), other.sourceFileLocation());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("TagStepDetails").add("Name", name()).add("Tags", hasTags() ? tags() : null)
                .add("SourceFileLocation", sourceFileLocation()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "Name":
            return Optional.ofNullable(clazz.cast(name()));
        case "Tags":
            return Optional.ofNullable(clazz.cast(tags()));
        case "SourceFileLocation":
            return Optional.ofNullable(clazz.cast(sourceFileLocation()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static Map<String, SdkField<?>> memberNameToFieldInitializer() {
        Map<String, SdkField<?>> map = new HashMap<>();
        map.put("Name", NAME_FIELD);
        map.put("Tags", TAGS_FIELD);
        map.put("SourceFileLocation", SOURCE_FILE_LOCATION_FIELD);
        return Collections.unmodifiableMap(map);
    }

    private static <T> Function<Object, T> getter(Function<TagStepDetails, T> g) {
        return obj -> g.apply((TagStepDetails) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    @Mutable
    @NotThreadSafe
    public interface Builder extends SdkPojo, CopyableBuilder<Builder, TagStepDetails> {
        /**
         * <p>
         * The name of the step, used as an identifier.
         * </p>
         * 
         * @param name
         *        The name of the step, used as an identifier.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder name(String name);

        /**
         * <p>
         * Array that contains from 1 to 10 key/value pairs.
         * </p>
         * 
         * @param tags
         *        Array that contains from 1 to 10 key/value pairs.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder tags(Collection<S3Tag> tags);

        /**
         * <p>
         * Array that contains from 1 to 10 key/value pairs.
         * </p>
         * 
         * @param tags
         *        Array that contains from 1 to 10 key/value pairs.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder tags(S3Tag... tags);

        /**
         * <p>
         * Array that contains from 1 to 10 key/value pairs.
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link software.amazon.awssdk.services.transfer.model.S3Tag.Builder} avoiding the need to create one manually
         * via {@link software.amazon.awssdk.services.transfer.model.S3Tag#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes,
         * {@link software.amazon.awssdk.services.transfer.model.S3Tag.Builder#build()} is called immediately and its
         * result is passed to {@link #tags(List<S3Tag>)}.
         * 
         * @param tags
         *        a consumer that will call methods on
         *        {@link software.amazon.awssdk.services.transfer.model.S3Tag.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #tags(java.util.Collection<S3Tag>)
         */
        Builder tags(Consumer<S3Tag.Builder>... tags);

        /**
         * <p>
         * Specifies which file to use as input to the workflow step: either the output from the previous step, or the
         * originally uploaded file for the workflow.
         * </p>
         * <ul>
         * <li>
         * <p>
         * To use the previous file as the input, enter <code>${previous.file}</code>. In this case, this workflow step
         * uses the output file from the previous workflow step as input. This is the default value.
         * </p>
         * </li>
         * <li>
         * <p>
         * To use the originally uploaded file location as input for this step, enter <code>${original.file}</code>.
         * </p>
         * </li>
         * </ul>
         * 
         * @param sourceFileLocation
         *        Specifies which file to use as input to the workflow step: either the output from the previous step,
         *        or the originally uploaded file for the workflow.</p>
         *        <ul>
         *        <li>
         *        <p>
         *        To use the previous file as the input, enter <code>${previous.file}</code>. In this case, this
         *        workflow step uses the output file from the previous workflow step as input. This is the default
         *        value.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        To use the originally uploaded file location as input for this step, enter
         *        <code>${original.file}</code>.
         *        </p>
         *        </li>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder sourceFileLocation(String sourceFileLocation);
    }

    static final class BuilderImpl implements Builder {
        private String name;

        private List<S3Tag> tags = DefaultSdkAutoConstructList.getInstance();

        private String sourceFileLocation;

        private BuilderImpl() {
        }

        private BuilderImpl(TagStepDetails model) {
            name(model.name);
            tags(model.tags);
            sourceFileLocation(model.sourceFileLocation);
        }

        public final String getName() {
            return name;
        }

        public final void setName(String name) {
            this.name = name;
        }

        @Override
        public final Builder name(String name) {
            this.name = name;
            return this;
        }

        public final List<S3Tag.Builder> getTags() {
            List<S3Tag.Builder> result = S3TagsCopier.copyToBuilder(this.tags);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setTags(Collection<S3Tag.BuilderImpl> tags) {
            this.tags = S3TagsCopier.copyFromBuilder(tags);
        }

        @Override
        public final Builder tags(Collection<S3Tag> tags) {
            this.tags = S3TagsCopier.copy(tags);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder tags(S3Tag... tags) {
            tags(Arrays.asList(tags));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder tags(Consumer<S3Tag.Builder>... tags) {
            tags(Stream.of(tags).map(c -> S3Tag.builder().applyMutation(c).build()).collect(Collectors.toList()));
            return this;
        }

        public final String getSourceFileLocation() {
            return sourceFileLocation;
        }

        public final void setSourceFileLocation(String sourceFileLocation) {
            this.sourceFileLocation = sourceFileLocation;
        }

        @Override
        public final Builder sourceFileLocation(String sourceFileLocation) {
            this.sourceFileLocation = sourceFileLocation;
            return this;
        }

        @Override
        public TagStepDetails build() {
            return new TagStepDetails(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
