/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.shield;

import java.util.concurrent.CompletableFuture;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.client.handler.AwsAsyncClientHandler;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.core.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.client.handler.AsyncClientHandler;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.protocols.core.ExceptionMetadata;
import software.amazon.awssdk.protocols.json.AwsJsonProtocol;
import software.amazon.awssdk.protocols.json.AwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.BaseAwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.JsonOperationMetadata;
import software.amazon.awssdk.services.shield.model.AccessDeniedException;
import software.amazon.awssdk.services.shield.model.AccessDeniedForDependencyException;
import software.amazon.awssdk.services.shield.model.AssociateDrtLogBucketRequest;
import software.amazon.awssdk.services.shield.model.AssociateDrtLogBucketResponse;
import software.amazon.awssdk.services.shield.model.AssociateDrtRoleRequest;
import software.amazon.awssdk.services.shield.model.AssociateDrtRoleResponse;
import software.amazon.awssdk.services.shield.model.AssociateHealthCheckRequest;
import software.amazon.awssdk.services.shield.model.AssociateHealthCheckResponse;
import software.amazon.awssdk.services.shield.model.CreateProtectionRequest;
import software.amazon.awssdk.services.shield.model.CreateProtectionResponse;
import software.amazon.awssdk.services.shield.model.CreateSubscriptionRequest;
import software.amazon.awssdk.services.shield.model.CreateSubscriptionResponse;
import software.amazon.awssdk.services.shield.model.DeleteProtectionRequest;
import software.amazon.awssdk.services.shield.model.DeleteProtectionResponse;
import software.amazon.awssdk.services.shield.model.DescribeAttackRequest;
import software.amazon.awssdk.services.shield.model.DescribeAttackResponse;
import software.amazon.awssdk.services.shield.model.DescribeDrtAccessRequest;
import software.amazon.awssdk.services.shield.model.DescribeDrtAccessResponse;
import software.amazon.awssdk.services.shield.model.DescribeEmergencyContactSettingsRequest;
import software.amazon.awssdk.services.shield.model.DescribeEmergencyContactSettingsResponse;
import software.amazon.awssdk.services.shield.model.DescribeProtectionRequest;
import software.amazon.awssdk.services.shield.model.DescribeProtectionResponse;
import software.amazon.awssdk.services.shield.model.DescribeSubscriptionRequest;
import software.amazon.awssdk.services.shield.model.DescribeSubscriptionResponse;
import software.amazon.awssdk.services.shield.model.DisassociateDrtLogBucketRequest;
import software.amazon.awssdk.services.shield.model.DisassociateDrtLogBucketResponse;
import software.amazon.awssdk.services.shield.model.DisassociateDrtRoleRequest;
import software.amazon.awssdk.services.shield.model.DisassociateDrtRoleResponse;
import software.amazon.awssdk.services.shield.model.DisassociateHealthCheckRequest;
import software.amazon.awssdk.services.shield.model.DisassociateHealthCheckResponse;
import software.amazon.awssdk.services.shield.model.GetSubscriptionStateRequest;
import software.amazon.awssdk.services.shield.model.GetSubscriptionStateResponse;
import software.amazon.awssdk.services.shield.model.InternalErrorException;
import software.amazon.awssdk.services.shield.model.InvalidOperationException;
import software.amazon.awssdk.services.shield.model.InvalidPaginationTokenException;
import software.amazon.awssdk.services.shield.model.InvalidParameterException;
import software.amazon.awssdk.services.shield.model.InvalidResourceException;
import software.amazon.awssdk.services.shield.model.LimitsExceededException;
import software.amazon.awssdk.services.shield.model.ListAttacksRequest;
import software.amazon.awssdk.services.shield.model.ListAttacksResponse;
import software.amazon.awssdk.services.shield.model.ListProtectionsRequest;
import software.amazon.awssdk.services.shield.model.ListProtectionsResponse;
import software.amazon.awssdk.services.shield.model.LockedSubscriptionException;
import software.amazon.awssdk.services.shield.model.NoAssociatedRoleException;
import software.amazon.awssdk.services.shield.model.OptimisticLockException;
import software.amazon.awssdk.services.shield.model.ResourceAlreadyExistsException;
import software.amazon.awssdk.services.shield.model.ResourceNotFoundException;
import software.amazon.awssdk.services.shield.model.ShieldException;
import software.amazon.awssdk.services.shield.model.UpdateEmergencyContactSettingsRequest;
import software.amazon.awssdk.services.shield.model.UpdateEmergencyContactSettingsResponse;
import software.amazon.awssdk.services.shield.model.UpdateSubscriptionRequest;
import software.amazon.awssdk.services.shield.model.UpdateSubscriptionResponse;
import software.amazon.awssdk.services.shield.transform.AssociateDrtLogBucketRequestMarshaller;
import software.amazon.awssdk.services.shield.transform.AssociateDrtRoleRequestMarshaller;
import software.amazon.awssdk.services.shield.transform.AssociateHealthCheckRequestMarshaller;
import software.amazon.awssdk.services.shield.transform.CreateProtectionRequestMarshaller;
import software.amazon.awssdk.services.shield.transform.CreateSubscriptionRequestMarshaller;
import software.amazon.awssdk.services.shield.transform.DeleteProtectionRequestMarshaller;
import software.amazon.awssdk.services.shield.transform.DescribeAttackRequestMarshaller;
import software.amazon.awssdk.services.shield.transform.DescribeDrtAccessRequestMarshaller;
import software.amazon.awssdk.services.shield.transform.DescribeEmergencyContactSettingsRequestMarshaller;
import software.amazon.awssdk.services.shield.transform.DescribeProtectionRequestMarshaller;
import software.amazon.awssdk.services.shield.transform.DescribeSubscriptionRequestMarshaller;
import software.amazon.awssdk.services.shield.transform.DisassociateDrtLogBucketRequestMarshaller;
import software.amazon.awssdk.services.shield.transform.DisassociateDrtRoleRequestMarshaller;
import software.amazon.awssdk.services.shield.transform.DisassociateHealthCheckRequestMarshaller;
import software.amazon.awssdk.services.shield.transform.GetSubscriptionStateRequestMarshaller;
import software.amazon.awssdk.services.shield.transform.ListAttacksRequestMarshaller;
import software.amazon.awssdk.services.shield.transform.ListProtectionsRequestMarshaller;
import software.amazon.awssdk.services.shield.transform.UpdateEmergencyContactSettingsRequestMarshaller;
import software.amazon.awssdk.services.shield.transform.UpdateSubscriptionRequestMarshaller;
import software.amazon.awssdk.utils.CompletableFutureUtils;

/**
 * Internal implementation of {@link ShieldAsyncClient}.
 *
 * @see ShieldAsyncClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultShieldAsyncClient implements ShieldAsyncClient {
    private static final Logger log = LoggerFactory.getLogger(DefaultShieldAsyncClient.class);

    private final AsyncClientHandler clientHandler;

    private final AwsJsonProtocolFactory protocolFactory;

    private final SdkClientConfiguration clientConfiguration;

    protected DefaultShieldAsyncClient(SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsAsyncClientHandler(clientConfiguration);
        this.clientConfiguration = clientConfiguration;
        this.protocolFactory = init(AwsJsonProtocolFactory.builder()).build();
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    /**
     * <p>
     * Authorizes the DDoS Response team (DRT) to access the specified Amazon S3 bucket containing your AWS WAF logs.
     * You can associate up to 10 Amazon S3 buckets with your subscription.
     * </p>
     * <p>
     * To use the services of the DRT and make an <code>AssociateDRTLogBucket</code> request, you must be subscribed to
     * the <a href="https://aws.amazon.com/premiumsupport/business-support/">Business Support plan</a> or the <a
     * href="https://aws.amazon.com/premiumsupport/enterprise-support/">Enterprise Support plan</a>.
     * </p>
     *
     * @param associateDrtLogBucketRequest
     * @return A Java Future containing the result of the AssociateDRTLogBucket operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalErrorException Exception that indicates that a problem occurred with the service
     *         infrastructure. You can retry the request.</li>
     *         <li>InvalidOperationException Exception that indicates that the operation would not cause any change to
     *         occur.</li>
     *         <li>NoAssociatedRoleException The ARN of the role that you specifed does not exist.</li>
     *         <li>LimitsExceededException Exception that indicates that the operation would exceed a limit.</p>
     *         <p>
     *         <code>Type</code> is the type of limit that would be exceeded.
     *         </p>
     *         <p>
     *         <code>Limit</code> is the threshold that would be exceeded.</li>
     *         <li>InvalidParameterException Exception that indicates that the parameters passed to the API are invalid.
     *         </li>
     *         <li>AccessDeniedForDependencyException In order to grant the necessary access to the DDoS Response Team,
     *         the user submitting the request must have the <code>iam:PassRole</code> permission. This error indicates
     *         the user did not have the appropriate permissions. For more information, see <a
     *         href="https://docs.aws.amazon.com/IAM/latest/UserGuide/id_roles_use_passrole.html">Granting a User
     *         Permissions to Pass a Role to an AWS Service</a>.</li>
     *         <li>OptimisticLockException Exception that indicates that the protection state has been modified by
     *         another client. You can retry the request.</li>
     *         <li>ResourceNotFoundException Exception indicating the specified resource does not exist.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ShieldException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample ShieldAsyncClient.AssociateDRTLogBucket
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/shield-2016-06-02/AssociateDRTLogBucket" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<AssociateDrtLogBucketResponse> associateDRTLogBucket(
            AssociateDrtLogBucketRequest associateDrtLogBucketRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<AssociateDrtLogBucketResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, AssociateDrtLogBucketResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<AssociateDrtLogBucketResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<AssociateDrtLogBucketRequest, AssociateDrtLogBucketResponse>()
                            .withOperationName("AssociateDRTLogBucket")
                            .withMarshaller(new AssociateDrtLogBucketRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(associateDrtLogBucketRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Authorizes the DDoS Response team (DRT), using the specified role, to access your AWS account to assist with DDoS
     * attack mitigation during potential attacks. This enables the DRT to inspect your AWS WAF configuration and create
     * or update AWS WAF rules and web ACLs.
     * </p>
     * <p>
     * You can associate only one <code>RoleArn</code> with your subscription. If you submit an
     * <code>AssociateDRTRole</code> request for an account that already has an associated role, the new
     * <code>RoleArn</code> will replace the existing <code>RoleArn</code>.
     * </p>
     * <p>
     * Prior to making the <code>AssociateDRTRole</code> request, you must attach the <a href=
     * "https://console.aws.amazon.com/iam/home?#/policies/arn:aws:iam::aws:policy/service-role/AWSShieldDRTAccessPolicy"
     * >AWSShieldDRTAccessPolicy</a> managed policy to the role you will specify in the request. For more information
     * see <a
     * href=" https://docs.aws.amazon.com/IAM/latest/UserGuide/access_policies_manage-attach-detach.html">Attaching and
     * Detaching IAM Policies</a>. The role must also trust the service principal <code> drt.shield.amazonaws.com</code>
     * . For more information, see <a
     * href="https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_principal.html">IAM JSON
     * Policy Elements: Principal</a>.
     * </p>
     * <p>
     * The DRT will have access only to your AWS WAF and Shield resources. By submitting this request, you authorize the
     * DRT to inspect your AWS WAF and Shield configuration and create and update AWS WAF rules and web ACLs on your
     * behalf. The DRT takes these actions only if explicitly authorized by you.
     * </p>
     * <p>
     * You must have the <code>iam:PassRole</code> permission to make an <code>AssociateDRTRole</code> request. For more
     * information, see <a href="https://docs.aws.amazon.com/IAM/latest/UserGuide/id_roles_use_passrole.html">Granting a
     * User Permissions to Pass a Role to an AWS Service</a>.
     * </p>
     * <p>
     * To use the services of the DRT and make an <code>AssociateDRTRole</code> request, you must be subscribed to the
     * <a href="https://aws.amazon.com/premiumsupport/business-support/">Business Support plan</a> or the <a
     * href="https://aws.amazon.com/premiumsupport/enterprise-support/">Enterprise Support plan</a>.
     * </p>
     *
     * @param associateDrtRoleRequest
     * @return A Java Future containing the result of the AssociateDRTRole operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalErrorException Exception that indicates that a problem occurred with the service
     *         infrastructure. You can retry the request.</li>
     *         <li>InvalidOperationException Exception that indicates that the operation would not cause any change to
     *         occur.</li>
     *         <li>InvalidParameterException Exception that indicates that the parameters passed to the API are invalid.
     *         </li>
     *         <li>AccessDeniedForDependencyException In order to grant the necessary access to the DDoS Response Team,
     *         the user submitting the request must have the <code>iam:PassRole</code> permission. This error indicates
     *         the user did not have the appropriate permissions. For more information, see <a
     *         href="https://docs.aws.amazon.com/IAM/latest/UserGuide/id_roles_use_passrole.html">Granting a User
     *         Permissions to Pass a Role to an AWS Service</a>.</li>
     *         <li>OptimisticLockException Exception that indicates that the protection state has been modified by
     *         another client. You can retry the request.</li>
     *         <li>ResourceNotFoundException Exception indicating the specified resource does not exist.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ShieldException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample ShieldAsyncClient.AssociateDRTRole
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/shield-2016-06-02/AssociateDRTRole" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<AssociateDrtRoleResponse> associateDRTRole(AssociateDrtRoleRequest associateDrtRoleRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<AssociateDrtRoleResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, AssociateDrtRoleResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<AssociateDrtRoleResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<AssociateDrtRoleRequest, AssociateDrtRoleResponse>()
                            .withOperationName("AssociateDRTRole")
                            .withMarshaller(new AssociateDrtRoleRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(associateDrtRoleRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Adds health-based detection to the Shield Advanced protection for a resource. Shield Advanced health-based
     * detection uses the health of your AWS resource to improve responsiveness and accuracy in attack detection and
     * mitigation.
     * </p>
     * <p>
     * You define the health check in Route 53 and then associate it with your Shield Advanced protection. For more
     * information, see <a href=
     * "https://docs.aws.amazon.com/waf/latest/developerguide/ddos-overview.html#ddos-advanced-health-check-option"
     * >Shield Advanced Health-Based Detection</a> in the <a
     * href="https://docs.aws.amazon.com/waf/latest/developerguide/">AWS WAF and AWS Shield Developer Guide</a>.
     * </p>
     *
     * @param associateHealthCheckRequest
     * @return A Java Future containing the result of the AssociateHealthCheck operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalErrorException Exception that indicates that a problem occurred with the service
     *         infrastructure. You can retry the request.</li>
     *         <li>LimitsExceededException Exception that indicates that the operation would exceed a limit.</p>
     *         <p>
     *         <code>Type</code> is the type of limit that would be exceeded.
     *         </p>
     *         <p>
     *         <code>Limit</code> is the threshold that would be exceeded.</li>
     *         <li>ResourceNotFoundException Exception indicating the specified resource does not exist.</li>
     *         <li>InvalidParameterException Exception that indicates that the parameters passed to the API are invalid.
     *         </li>
     *         <li>OptimisticLockException Exception that indicates that the protection state has been modified by
     *         another client. You can retry the request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ShieldException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample ShieldAsyncClient.AssociateHealthCheck
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/shield-2016-06-02/AssociateHealthCheck" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<AssociateHealthCheckResponse> associateHealthCheck(
            AssociateHealthCheckRequest associateHealthCheckRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<AssociateHealthCheckResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, AssociateHealthCheckResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<AssociateHealthCheckResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<AssociateHealthCheckRequest, AssociateHealthCheckResponse>()
                            .withOperationName("AssociateHealthCheck")
                            .withMarshaller(new AssociateHealthCheckRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(associateHealthCheckRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Enables AWS Shield Advanced for a specific AWS resource. The resource can be an Amazon CloudFront distribution,
     * Elastic Load Balancing load balancer, AWS Global Accelerator accelerator, Elastic IP Address, or an Amazon Route
     * 53 hosted zone.
     * </p>
     * <p>
     * You can add protection to only a single resource with each CreateProtection request. If you want to add
     * protection to multiple resources at once, use the <a href="https://console.aws.amazon.com/waf/">AWS WAF
     * console</a>. For more information see <a
     * href="https://docs.aws.amazon.com/waf/latest/developerguide/getting-started-ddos.html">Getting Started with AWS
     * Shield Advanced</a> and <a
     * href="https://docs.aws.amazon.com/waf/latest/developerguide/configure-new-protection.html">Add AWS Shield
     * Advanced Protection to more AWS Resources</a>.
     * </p>
     *
     * @param createProtectionRequest
     * @return A Java Future containing the result of the CreateProtection operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalErrorException Exception that indicates that a problem occurred with the service
     *         infrastructure. You can retry the request.</li>
     *         <li>InvalidResourceException Exception that indicates that the resource is invalid. You might not have
     *         access to the resource, or the resource might not exist.</li>
     *         <li>InvalidOperationException Exception that indicates that the operation would not cause any change to
     *         occur.</li>
     *         <li>LimitsExceededException Exception that indicates that the operation would exceed a limit.</p>
     *         <p>
     *         <code>Type</code> is the type of limit that would be exceeded.
     *         </p>
     *         <p>
     *         <code>Limit</code> is the threshold that would be exceeded.</li>
     *         <li>ResourceAlreadyExistsException Exception indicating the specified resource already exists.</li>
     *         <li>OptimisticLockException Exception that indicates that the protection state has been modified by
     *         another client. You can retry the request.</li>
     *         <li>ResourceNotFoundException Exception indicating the specified resource does not exist.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ShieldException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample ShieldAsyncClient.CreateProtection
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/shield-2016-06-02/CreateProtection" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<CreateProtectionResponse> createProtection(CreateProtectionRequest createProtectionRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateProtectionResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, CreateProtectionResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateProtectionResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateProtectionRequest, CreateProtectionResponse>()
                            .withOperationName("CreateProtection")
                            .withMarshaller(new CreateProtectionRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(createProtectionRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Activates AWS Shield Advanced for an account.
     * </p>
     * <p>
     * As part of this request you can specify <code>EmergencySettings</code> that automaticaly grant the DDoS response
     * team (DRT) needed permissions to assist you during a suspected DDoS attack. For more information see <a
     * href="https://docs.aws.amazon.com/waf/latest/developerguide/authorize-DRT.html">Authorize the DDoS Response Team
     * to Create Rules and Web ACLs on Your Behalf</a>.
     * </p>
     * <p>
     * To use the services of the DRT, you must be subscribed to the <a
     * href="https://aws.amazon.com/premiumsupport/business-support/">Business Support plan</a> or the <a
     * href="https://aws.amazon.com/premiumsupport/enterprise-support/">Enterprise Support plan</a>.
     * </p>
     * <p>
     * When you initally create a subscription, your subscription is set to be automatically renewed at the end of the
     * existing subscription period. You can change this by submitting an <code>UpdateSubscription</code> request.
     * </p>
     *
     * @param createSubscriptionRequest
     * @return A Java Future containing the result of the CreateSubscription operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalErrorException Exception that indicates that a problem occurred with the service
     *         infrastructure. You can retry the request.</li>
     *         <li>ResourceAlreadyExistsException Exception indicating the specified resource already exists.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ShieldException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample ShieldAsyncClient.CreateSubscription
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/shield-2016-06-02/CreateSubscription" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<CreateSubscriptionResponse> createSubscription(CreateSubscriptionRequest createSubscriptionRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateSubscriptionResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, CreateSubscriptionResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateSubscriptionResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateSubscriptionRequest, CreateSubscriptionResponse>()
                            .withOperationName("CreateSubscription")
                            .withMarshaller(new CreateSubscriptionRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(createSubscriptionRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes an AWS Shield Advanced <a>Protection</a>.
     * </p>
     *
     * @param deleteProtectionRequest
     * @return A Java Future containing the result of the DeleteProtection operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalErrorException Exception that indicates that a problem occurred with the service
     *         infrastructure. You can retry the request.</li>
     *         <li>ResourceNotFoundException Exception indicating the specified resource does not exist.</li>
     *         <li>OptimisticLockException Exception that indicates that the protection state has been modified by
     *         another client. You can retry the request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ShieldException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample ShieldAsyncClient.DeleteProtection
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/shield-2016-06-02/DeleteProtection" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteProtectionResponse> deleteProtection(DeleteProtectionRequest deleteProtectionRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteProtectionResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DeleteProtectionResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteProtectionResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteProtectionRequest, DeleteProtectionResponse>()
                            .withOperationName("DeleteProtection")
                            .withMarshaller(new DeleteProtectionRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(deleteProtectionRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Describes the details of a DDoS attack.
     * </p>
     *
     * @param describeAttackRequest
     * @return A Java Future containing the result of the DescribeAttack operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalErrorException Exception that indicates that a problem occurred with the service
     *         infrastructure. You can retry the request.</li>
     *         <li>AccessDeniedException Exception that indicates the specified <code>AttackId</code> does not exist, or
     *         the requester does not have the appropriate permissions to access the <code>AttackId</code>.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ShieldException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample ShieldAsyncClient.DescribeAttack
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/shield-2016-06-02/DescribeAttack" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<DescribeAttackResponse> describeAttack(DescribeAttackRequest describeAttackRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DescribeAttackResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DescribeAttackResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DescribeAttackResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribeAttackRequest, DescribeAttackResponse>()
                            .withOperationName("DescribeAttack")
                            .withMarshaller(new DescribeAttackRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(describeAttackRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns the current role and list of Amazon S3 log buckets used by the DDoS Response team (DRT) to access your
     * AWS account while assisting with attack mitigation.
     * </p>
     *
     * @param describeDrtAccessRequest
     * @return A Java Future containing the result of the DescribeDRTAccess operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalErrorException Exception that indicates that a problem occurred with the service
     *         infrastructure. You can retry the request.</li>
     *         <li>ResourceNotFoundException Exception indicating the specified resource does not exist.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ShieldException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample ShieldAsyncClient.DescribeDRTAccess
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/shield-2016-06-02/DescribeDRTAccess" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<DescribeDrtAccessResponse> describeDRTAccess(DescribeDrtAccessRequest describeDrtAccessRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DescribeDrtAccessResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DescribeDrtAccessResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DescribeDrtAccessResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribeDrtAccessRequest, DescribeDrtAccessResponse>()
                            .withOperationName("DescribeDRTAccess")
                            .withMarshaller(new DescribeDrtAccessRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(describeDrtAccessRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists the email addresses that the DRT can use to contact you during a suspected attack.
     * </p>
     *
     * @param describeEmergencyContactSettingsRequest
     * @return A Java Future containing the result of the DescribeEmergencyContactSettings operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalErrorException Exception that indicates that a problem occurred with the service
     *         infrastructure. You can retry the request.</li>
     *         <li>ResourceNotFoundException Exception indicating the specified resource does not exist.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ShieldException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample ShieldAsyncClient.DescribeEmergencyContactSettings
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/shield-2016-06-02/DescribeEmergencyContactSettings"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DescribeEmergencyContactSettingsResponse> describeEmergencyContactSettings(
            DescribeEmergencyContactSettingsRequest describeEmergencyContactSettingsRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DescribeEmergencyContactSettingsResponse> responseHandler = protocolFactory
                    .createResponseHandler(operationMetadata, DescribeEmergencyContactSettingsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DescribeEmergencyContactSettingsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribeEmergencyContactSettingsRequest, DescribeEmergencyContactSettingsResponse>()
                            .withOperationName("DescribeEmergencyContactSettings")
                            .withMarshaller(new DescribeEmergencyContactSettingsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(describeEmergencyContactSettingsRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists the details of a <a>Protection</a> object.
     * </p>
     *
     * @param describeProtectionRequest
     * @return A Java Future containing the result of the DescribeProtection operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalErrorException Exception that indicates that a problem occurred with the service
     *         infrastructure. You can retry the request.</li>
     *         <li>InvalidParameterException Exception that indicates that the parameters passed to the API are invalid.
     *         </li>
     *         <li>ResourceNotFoundException Exception indicating the specified resource does not exist.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ShieldException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample ShieldAsyncClient.DescribeProtection
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/shield-2016-06-02/DescribeProtection" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<DescribeProtectionResponse> describeProtection(DescribeProtectionRequest describeProtectionRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DescribeProtectionResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DescribeProtectionResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DescribeProtectionResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribeProtectionRequest, DescribeProtectionResponse>()
                            .withOperationName("DescribeProtection")
                            .withMarshaller(new DescribeProtectionRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(describeProtectionRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Provides details about the AWS Shield Advanced subscription for an account.
     * </p>
     *
     * @param describeSubscriptionRequest
     * @return A Java Future containing the result of the DescribeSubscription operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalErrorException Exception that indicates that a problem occurred with the service
     *         infrastructure. You can retry the request.</li>
     *         <li>ResourceNotFoundException Exception indicating the specified resource does not exist.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ShieldException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample ShieldAsyncClient.DescribeSubscription
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/shield-2016-06-02/DescribeSubscription" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<DescribeSubscriptionResponse> describeSubscription(
            DescribeSubscriptionRequest describeSubscriptionRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DescribeSubscriptionResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DescribeSubscriptionResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DescribeSubscriptionResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribeSubscriptionRequest, DescribeSubscriptionResponse>()
                            .withOperationName("DescribeSubscription")
                            .withMarshaller(new DescribeSubscriptionRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(describeSubscriptionRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Removes the DDoS Response team's (DRT) access to the specified Amazon S3 bucket containing your AWS WAF logs.
     * </p>
     * <p>
     * To make a <code>DisassociateDRTLogBucket</code> request, you must be subscribed to the <a
     * href="https://aws.amazon.com/premiumsupport/business-support/">Business Support plan</a> or the <a
     * href="https://aws.amazon.com/premiumsupport/enterprise-support/">Enterprise Support plan</a>. However, if you are
     * not subscribed to one of these support plans, but had been previously and had granted the DRT access to your
     * account, you can submit a <code>DisassociateDRTLogBucket</code> request to remove this access.
     * </p>
     *
     * @param disassociateDrtLogBucketRequest
     * @return A Java Future containing the result of the DisassociateDRTLogBucket operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalErrorException Exception that indicates that a problem occurred with the service
     *         infrastructure. You can retry the request.</li>
     *         <li>InvalidOperationException Exception that indicates that the operation would not cause any change to
     *         occur.</li>
     *         <li>NoAssociatedRoleException The ARN of the role that you specifed does not exist.</li>
     *         <li>AccessDeniedForDependencyException In order to grant the necessary access to the DDoS Response Team,
     *         the user submitting the request must have the <code>iam:PassRole</code> permission. This error indicates
     *         the user did not have the appropriate permissions. For more information, see <a
     *         href="https://docs.aws.amazon.com/IAM/latest/UserGuide/id_roles_use_passrole.html">Granting a User
     *         Permissions to Pass a Role to an AWS Service</a>.</li>
     *         <li>OptimisticLockException Exception that indicates that the protection state has been modified by
     *         another client. You can retry the request.</li>
     *         <li>ResourceNotFoundException Exception indicating the specified resource does not exist.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ShieldException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample ShieldAsyncClient.DisassociateDRTLogBucket
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/shield-2016-06-02/DisassociateDRTLogBucket"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DisassociateDrtLogBucketResponse> disassociateDRTLogBucket(
            DisassociateDrtLogBucketRequest disassociateDrtLogBucketRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DisassociateDrtLogBucketResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DisassociateDrtLogBucketResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DisassociateDrtLogBucketResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DisassociateDrtLogBucketRequest, DisassociateDrtLogBucketResponse>()
                            .withOperationName("DisassociateDRTLogBucket")
                            .withMarshaller(new DisassociateDrtLogBucketRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(disassociateDrtLogBucketRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Removes the DDoS Response team's (DRT) access to your AWS account.
     * </p>
     * <p>
     * To make a <code>DisassociateDRTRole</code> request, you must be subscribed to the <a
     * href="https://aws.amazon.com/premiumsupport/business-support/">Business Support plan</a> or the <a
     * href="https://aws.amazon.com/premiumsupport/enterprise-support/">Enterprise Support plan</a>. However, if you are
     * not subscribed to one of these support plans, but had been previously and had granted the DRT access to your
     * account, you can submit a <code>DisassociateDRTRole</code> request to remove this access.
     * </p>
     *
     * @param disassociateDrtRoleRequest
     * @return A Java Future containing the result of the DisassociateDRTRole operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalErrorException Exception that indicates that a problem occurred with the service
     *         infrastructure. You can retry the request.</li>
     *         <li>InvalidOperationException Exception that indicates that the operation would not cause any change to
     *         occur.</li>
     *         <li>OptimisticLockException Exception that indicates that the protection state has been modified by
     *         another client. You can retry the request.</li>
     *         <li>ResourceNotFoundException Exception indicating the specified resource does not exist.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ShieldException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample ShieldAsyncClient.DisassociateDRTRole
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/shield-2016-06-02/DisassociateDRTRole" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<DisassociateDrtRoleResponse> disassociateDRTRole(
            DisassociateDrtRoleRequest disassociateDrtRoleRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DisassociateDrtRoleResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DisassociateDrtRoleResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DisassociateDrtRoleResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DisassociateDrtRoleRequest, DisassociateDrtRoleResponse>()
                            .withOperationName("DisassociateDRTRole")
                            .withMarshaller(new DisassociateDrtRoleRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(disassociateDrtRoleRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Removes health-based detection from the Shield Advanced protection for a resource. Shield Advanced health-based
     * detection uses the health of your AWS resource to improve responsiveness and accuracy in attack detection and
     * mitigation.
     * </p>
     * <p>
     * You define the health check in Route 53 and then associate or disassociate it with your Shield Advanced
     * protection. For more information, see <a href=
     * "https://docs.aws.amazon.com/waf/latest/developerguide/ddos-overview.html#ddos-advanced-health-check-option"
     * >Shield Advanced Health-Based Detection</a> in the <a
     * href="https://docs.aws.amazon.com/waf/latest/developerguide/">AWS WAF and AWS Shield Developer Guide</a>.
     * </p>
     *
     * @param disassociateHealthCheckRequest
     * @return A Java Future containing the result of the DisassociateHealthCheck operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalErrorException Exception that indicates that a problem occurred with the service
     *         infrastructure. You can retry the request.</li>
     *         <li>InvalidParameterException Exception that indicates that the parameters passed to the API are invalid.
     *         </li>
     *         <li>ResourceNotFoundException Exception indicating the specified resource does not exist.</li>
     *         <li>OptimisticLockException Exception that indicates that the protection state has been modified by
     *         another client. You can retry the request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ShieldException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample ShieldAsyncClient.DisassociateHealthCheck
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/shield-2016-06-02/DisassociateHealthCheck" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<DisassociateHealthCheckResponse> disassociateHealthCheck(
            DisassociateHealthCheckRequest disassociateHealthCheckRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DisassociateHealthCheckResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DisassociateHealthCheckResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DisassociateHealthCheckResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DisassociateHealthCheckRequest, DisassociateHealthCheckResponse>()
                            .withOperationName("DisassociateHealthCheck")
                            .withMarshaller(new DisassociateHealthCheckRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(disassociateHealthCheckRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns the <code>SubscriptionState</code>, either <code>Active</code> or <code>Inactive</code>.
     * </p>
     *
     * @param getSubscriptionStateRequest
     * @return A Java Future containing the result of the GetSubscriptionState operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalErrorException Exception that indicates that a problem occurred with the service
     *         infrastructure. You can retry the request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ShieldException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample ShieldAsyncClient.GetSubscriptionState
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/shield-2016-06-02/GetSubscriptionState" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<GetSubscriptionStateResponse> getSubscriptionState(
            GetSubscriptionStateRequest getSubscriptionStateRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetSubscriptionStateResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GetSubscriptionStateResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetSubscriptionStateResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetSubscriptionStateRequest, GetSubscriptionStateResponse>()
                            .withOperationName("GetSubscriptionState")
                            .withMarshaller(new GetSubscriptionStateRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(getSubscriptionStateRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns all ongoing DDoS attacks or all DDoS attacks during a specified time period.
     * </p>
     *
     * @param listAttacksRequest
     * @return A Java Future containing the result of the ListAttacks operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalErrorException Exception that indicates that a problem occurred with the service
     *         infrastructure. You can retry the request.</li>
     *         <li>InvalidParameterException Exception that indicates that the parameters passed to the API are invalid.
     *         </li>
     *         <li>InvalidOperationException Exception that indicates that the operation would not cause any change to
     *         occur.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ShieldException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample ShieldAsyncClient.ListAttacks
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/shield-2016-06-02/ListAttacks" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<ListAttacksResponse> listAttacks(ListAttacksRequest listAttacksRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListAttacksResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    ListAttacksResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListAttacksResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListAttacksRequest, ListAttacksResponse>()
                            .withOperationName("ListAttacks").withMarshaller(new ListAttacksRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(listAttacksRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists all <a>Protection</a> objects for the account.
     * </p>
     *
     * @param listProtectionsRequest
     * @return A Java Future containing the result of the ListProtections operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalErrorException Exception that indicates that a problem occurred with the service
     *         infrastructure. You can retry the request.</li>
     *         <li>ResourceNotFoundException Exception indicating the specified resource does not exist.</li>
     *         <li>InvalidPaginationTokenException Exception that indicates that the NextToken specified in the request
     *         is invalid. Submit the request using the NextToken value that was returned in the response.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ShieldException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample ShieldAsyncClient.ListProtections
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/shield-2016-06-02/ListProtections" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<ListProtectionsResponse> listProtections(ListProtectionsRequest listProtectionsRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListProtectionsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListProtectionsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListProtectionsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListProtectionsRequest, ListProtectionsResponse>()
                            .withOperationName("ListProtections")
                            .withMarshaller(new ListProtectionsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(listProtectionsRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Updates the details of the list of email addresses that the DRT can use to contact you during a suspected attack.
     * </p>
     *
     * @param updateEmergencyContactSettingsRequest
     * @return A Java Future containing the result of the UpdateEmergencyContactSettings operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalErrorException Exception that indicates that a problem occurred with the service
     *         infrastructure. You can retry the request.</li>
     *         <li>InvalidParameterException Exception that indicates that the parameters passed to the API are invalid.
     *         </li>
     *         <li>OptimisticLockException Exception that indicates that the protection state has been modified by
     *         another client. You can retry the request.</li>
     *         <li>ResourceNotFoundException Exception indicating the specified resource does not exist.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ShieldException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample ShieldAsyncClient.UpdateEmergencyContactSettings
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/shield-2016-06-02/UpdateEmergencyContactSettings"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateEmergencyContactSettingsResponse> updateEmergencyContactSettings(
            UpdateEmergencyContactSettingsRequest updateEmergencyContactSettingsRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateEmergencyContactSettingsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, UpdateEmergencyContactSettingsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UpdateEmergencyContactSettingsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateEmergencyContactSettingsRequest, UpdateEmergencyContactSettingsResponse>()
                            .withOperationName("UpdateEmergencyContactSettings")
                            .withMarshaller(new UpdateEmergencyContactSettingsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(updateEmergencyContactSettingsRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Updates the details of an existing subscription. Only enter values for parameters you want to change. Empty
     * parameters are not updated.
     * </p>
     *
     * @param updateSubscriptionRequest
     * @return A Java Future containing the result of the UpdateSubscription operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalErrorException Exception that indicates that a problem occurred with the service
     *         infrastructure. You can retry the request.</li>
     *         <li>LockedSubscriptionException You are trying to update a subscription that has not yet completed the
     *         1-year commitment. You can change the <code>AutoRenew</code> parameter during the last 30 days of your
     *         subscription. This exception indicates that you are attempting to change <code>AutoRenew</code> prior to
     *         that period.</li>
     *         <li>ResourceNotFoundException Exception indicating the specified resource does not exist.</li>
     *         <li>InvalidParameterException Exception that indicates that the parameters passed to the API are invalid.
     *         </li>
     *         <li>OptimisticLockException Exception that indicates that the protection state has been modified by
     *         another client. You can retry the request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ShieldException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample ShieldAsyncClient.UpdateSubscription
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/shield-2016-06-02/UpdateSubscription" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateSubscriptionResponse> updateSubscription(UpdateSubscriptionRequest updateSubscriptionRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateSubscriptionResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, UpdateSubscriptionResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UpdateSubscriptionResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateSubscriptionRequest, UpdateSubscriptionResponse>()
                            .withOperationName("UpdateSubscription")
                            .withMarshaller(new UpdateSubscriptionRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(updateSubscriptionRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    @Override
    public void close() {
        clientHandler.close();
    }

    private <T extends BaseAwsJsonProtocolFactory.Builder<T>> T init(T builder) {
        return builder
                .clientConfiguration(clientConfiguration)
                .defaultServiceExceptionSupplier(ShieldException::builder)
                .protocol(AwsJsonProtocol.AWS_JSON)
                .protocolVersion("1.1")
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("AccessDeniedException")
                                .exceptionBuilderSupplier(AccessDeniedException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("OptimisticLockException")
                                .exceptionBuilderSupplier(OptimisticLockException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("LockedSubscriptionException")
                                .exceptionBuilderSupplier(LockedSubscriptionException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidResourceException")
                                .exceptionBuilderSupplier(InvalidResourceException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidParameterException")
                                .exceptionBuilderSupplier(InvalidParameterException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidOperationException")
                                .exceptionBuilderSupplier(InvalidOperationException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ResourceNotFoundException")
                                .exceptionBuilderSupplier(ResourceNotFoundException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ResourceAlreadyExistsException")
                                .exceptionBuilderSupplier(ResourceAlreadyExistsException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InternalErrorException")
                                .exceptionBuilderSupplier(InternalErrorException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidPaginationTokenException")
                                .exceptionBuilderSupplier(InvalidPaginationTokenException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("LimitsExceededException")
                                .exceptionBuilderSupplier(LimitsExceededException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("NoAssociatedRoleException")
                                .exceptionBuilderSupplier(NoAssociatedRoleException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("AccessDeniedForDependencyException")
                                .exceptionBuilderSupplier(AccessDeniedForDependencyException::builder).build());
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata) {
        return protocolFactory.createErrorResponseHandler(operationMetadata);
    }
}
