/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.serverlessapplicationrepository;

import java.util.Collections;
import java.util.List;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.client.handler.AwsSyncClientHandler;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.core.RequestOverrideConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientOption;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.client.handler.SyncClientHandler;
import software.amazon.awssdk.core.exception.SdkClientException;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.core.metrics.CoreMetric;
import software.amazon.awssdk.metrics.MetricCollector;
import software.amazon.awssdk.metrics.MetricPublisher;
import software.amazon.awssdk.metrics.NoOpMetricCollector;
import software.amazon.awssdk.protocols.core.ExceptionMetadata;
import software.amazon.awssdk.protocols.json.AwsJsonProtocol;
import software.amazon.awssdk.protocols.json.AwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.BaseAwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.JsonOperationMetadata;
import software.amazon.awssdk.services.serverlessapplicationrepository.model.BadRequestException;
import software.amazon.awssdk.services.serverlessapplicationrepository.model.ConflictException;
import software.amazon.awssdk.services.serverlessapplicationrepository.model.CreateApplicationRequest;
import software.amazon.awssdk.services.serverlessapplicationrepository.model.CreateApplicationResponse;
import software.amazon.awssdk.services.serverlessapplicationrepository.model.CreateApplicationVersionRequest;
import software.amazon.awssdk.services.serverlessapplicationrepository.model.CreateApplicationVersionResponse;
import software.amazon.awssdk.services.serverlessapplicationrepository.model.CreateCloudFormationChangeSetRequest;
import software.amazon.awssdk.services.serverlessapplicationrepository.model.CreateCloudFormationChangeSetResponse;
import software.amazon.awssdk.services.serverlessapplicationrepository.model.CreateCloudFormationTemplateRequest;
import software.amazon.awssdk.services.serverlessapplicationrepository.model.CreateCloudFormationTemplateResponse;
import software.amazon.awssdk.services.serverlessapplicationrepository.model.DeleteApplicationRequest;
import software.amazon.awssdk.services.serverlessapplicationrepository.model.DeleteApplicationResponse;
import software.amazon.awssdk.services.serverlessapplicationrepository.model.ForbiddenException;
import software.amazon.awssdk.services.serverlessapplicationrepository.model.GetApplicationPolicyRequest;
import software.amazon.awssdk.services.serverlessapplicationrepository.model.GetApplicationPolicyResponse;
import software.amazon.awssdk.services.serverlessapplicationrepository.model.GetApplicationRequest;
import software.amazon.awssdk.services.serverlessapplicationrepository.model.GetApplicationResponse;
import software.amazon.awssdk.services.serverlessapplicationrepository.model.GetCloudFormationTemplateRequest;
import software.amazon.awssdk.services.serverlessapplicationrepository.model.GetCloudFormationTemplateResponse;
import software.amazon.awssdk.services.serverlessapplicationrepository.model.InternalServerErrorException;
import software.amazon.awssdk.services.serverlessapplicationrepository.model.ListApplicationDependenciesRequest;
import software.amazon.awssdk.services.serverlessapplicationrepository.model.ListApplicationDependenciesResponse;
import software.amazon.awssdk.services.serverlessapplicationrepository.model.ListApplicationVersionsRequest;
import software.amazon.awssdk.services.serverlessapplicationrepository.model.ListApplicationVersionsResponse;
import software.amazon.awssdk.services.serverlessapplicationrepository.model.ListApplicationsRequest;
import software.amazon.awssdk.services.serverlessapplicationrepository.model.ListApplicationsResponse;
import software.amazon.awssdk.services.serverlessapplicationrepository.model.NotFoundException;
import software.amazon.awssdk.services.serverlessapplicationrepository.model.PutApplicationPolicyRequest;
import software.amazon.awssdk.services.serverlessapplicationrepository.model.PutApplicationPolicyResponse;
import software.amazon.awssdk.services.serverlessapplicationrepository.model.ServerlessApplicationRepositoryException;
import software.amazon.awssdk.services.serverlessapplicationrepository.model.TooManyRequestsException;
import software.amazon.awssdk.services.serverlessapplicationrepository.model.UnshareApplicationRequest;
import software.amazon.awssdk.services.serverlessapplicationrepository.model.UnshareApplicationResponse;
import software.amazon.awssdk.services.serverlessapplicationrepository.model.UpdateApplicationRequest;
import software.amazon.awssdk.services.serverlessapplicationrepository.model.UpdateApplicationResponse;
import software.amazon.awssdk.services.serverlessapplicationrepository.transform.CreateApplicationRequestMarshaller;
import software.amazon.awssdk.services.serverlessapplicationrepository.transform.CreateApplicationVersionRequestMarshaller;
import software.amazon.awssdk.services.serverlessapplicationrepository.transform.CreateCloudFormationChangeSetRequestMarshaller;
import software.amazon.awssdk.services.serverlessapplicationrepository.transform.CreateCloudFormationTemplateRequestMarshaller;
import software.amazon.awssdk.services.serverlessapplicationrepository.transform.DeleteApplicationRequestMarshaller;
import software.amazon.awssdk.services.serverlessapplicationrepository.transform.GetApplicationPolicyRequestMarshaller;
import software.amazon.awssdk.services.serverlessapplicationrepository.transform.GetApplicationRequestMarshaller;
import software.amazon.awssdk.services.serverlessapplicationrepository.transform.GetCloudFormationTemplateRequestMarshaller;
import software.amazon.awssdk.services.serverlessapplicationrepository.transform.ListApplicationDependenciesRequestMarshaller;
import software.amazon.awssdk.services.serverlessapplicationrepository.transform.ListApplicationVersionsRequestMarshaller;
import software.amazon.awssdk.services.serverlessapplicationrepository.transform.ListApplicationsRequestMarshaller;
import software.amazon.awssdk.services.serverlessapplicationrepository.transform.PutApplicationPolicyRequestMarshaller;
import software.amazon.awssdk.services.serverlessapplicationrepository.transform.UnshareApplicationRequestMarshaller;
import software.amazon.awssdk.services.serverlessapplicationrepository.transform.UpdateApplicationRequestMarshaller;
import software.amazon.awssdk.utils.Logger;

/**
 * Internal implementation of {@link ServerlessApplicationRepositoryClient}.
 *
 * @see ServerlessApplicationRepositoryClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultServerlessApplicationRepositoryClient implements ServerlessApplicationRepositoryClient {
    private static final Logger log = Logger.loggerFor(DefaultServerlessApplicationRepositoryClient.class);

    private final SyncClientHandler clientHandler;

    private final AwsJsonProtocolFactory protocolFactory;

    private final SdkClientConfiguration clientConfiguration;

    private final ServerlessApplicationRepositoryServiceClientConfiguration serviceClientConfiguration;

    protected DefaultServerlessApplicationRepositoryClient(
            ServerlessApplicationRepositoryServiceClientConfiguration serviceClientConfiguration,
            SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsSyncClientHandler(clientConfiguration);
        this.clientConfiguration = clientConfiguration;
        this.serviceClientConfiguration = serviceClientConfiguration;
        this.protocolFactory = init(AwsJsonProtocolFactory.builder()).build();
    }

    /**
     * <p>
     * Creates an application, optionally including an AWS SAM file to create the first application version in the same
     * call.
     * </p>
     *
     * @param createApplicationRequest
     * @return Result of the CreateApplication operation returned by the service.
     * @throws TooManyRequestsException
     *         The client is sending more than the allowed number of requests per unit of time.
     * @throws BadRequestException
     *         One of the parameters in the request is invalid.
     * @throws InternalServerErrorException
     *         The AWS Serverless Application Repository service encountered an internal error.
     * @throws ConflictException
     *         The resource already exists.
     * @throws ForbiddenException
     *         The client is not authenticated.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ServerlessApplicationRepositoryException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ServerlessApplicationRepositoryClient.CreateApplication
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/serverlessrepo-2017-09-08/CreateApplication"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CreateApplicationResponse createApplication(CreateApplicationRequest createApplicationRequest)
            throws TooManyRequestsException, BadRequestException, InternalServerErrorException, ConflictException,
            ForbiddenException, AwsServiceException, SdkClientException, ServerlessApplicationRepositoryException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<CreateApplicationResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                CreateApplicationResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createApplicationRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ServerlessApplicationRepository");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateApplication");

            return clientHandler.execute(new ClientExecutionParams<CreateApplicationRequest, CreateApplicationResponse>()
                    .withOperationName("CreateApplication").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(createApplicationRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new CreateApplicationRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Creates an application version.
     * </p>
     *
     * @param createApplicationVersionRequest
     * @return Result of the CreateApplicationVersion operation returned by the service.
     * @throws TooManyRequestsException
     *         The client is sending more than the allowed number of requests per unit of time.
     * @throws BadRequestException
     *         One of the parameters in the request is invalid.
     * @throws InternalServerErrorException
     *         The AWS Serverless Application Repository service encountered an internal error.
     * @throws ConflictException
     *         The resource already exists.
     * @throws ForbiddenException
     *         The client is not authenticated.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ServerlessApplicationRepositoryException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ServerlessApplicationRepositoryClient.CreateApplicationVersion
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/serverlessrepo-2017-09-08/CreateApplicationVersion"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CreateApplicationVersionResponse createApplicationVersion(
            CreateApplicationVersionRequest createApplicationVersionRequest) throws TooManyRequestsException,
            BadRequestException, InternalServerErrorException, ConflictException, ForbiddenException, AwsServiceException,
            SdkClientException, ServerlessApplicationRepositoryException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<CreateApplicationVersionResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, CreateApplicationVersionResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createApplicationVersionRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ServerlessApplicationRepository");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateApplicationVersion");

            return clientHandler
                    .execute(new ClientExecutionParams<CreateApplicationVersionRequest, CreateApplicationVersionResponse>()
                            .withOperationName("CreateApplicationVersion").withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withInput(createApplicationVersionRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new CreateApplicationVersionRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Creates an AWS CloudFormation change set for the given application.
     * </p>
     *
     * @param createCloudFormationChangeSetRequest
     * @return Result of the CreateCloudFormationChangeSet operation returned by the service.
     * @throws TooManyRequestsException
     *         The client is sending more than the allowed number of requests per unit of time.
     * @throws BadRequestException
     *         One of the parameters in the request is invalid.
     * @throws InternalServerErrorException
     *         The AWS Serverless Application Repository service encountered an internal error.
     * @throws ForbiddenException
     *         The client is not authenticated.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ServerlessApplicationRepositoryException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ServerlessApplicationRepositoryClient.CreateCloudFormationChangeSet
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/serverlessrepo-2017-09-08/CreateCloudFormationChangeSet"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CreateCloudFormationChangeSetResponse createCloudFormationChangeSet(
            CreateCloudFormationChangeSetRequest createCloudFormationChangeSetRequest) throws TooManyRequestsException,
            BadRequestException, InternalServerErrorException, ForbiddenException, AwsServiceException, SdkClientException,
            ServerlessApplicationRepositoryException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<CreateCloudFormationChangeSetResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, CreateCloudFormationChangeSetResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                createCloudFormationChangeSetRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ServerlessApplicationRepository");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateCloudFormationChangeSet");

            return clientHandler
                    .execute(new ClientExecutionParams<CreateCloudFormationChangeSetRequest, CreateCloudFormationChangeSetResponse>()
                            .withOperationName("CreateCloudFormationChangeSet").withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withInput(createCloudFormationChangeSetRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new CreateCloudFormationChangeSetRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Creates an AWS CloudFormation template.
     * </p>
     *
     * @param createCloudFormationTemplateRequest
     * @return Result of the CreateCloudFormationTemplate operation returned by the service.
     * @throws NotFoundException
     *         The resource (for example, an access policy statement) specified in the request doesn't exist.
     * @throws TooManyRequestsException
     *         The client is sending more than the allowed number of requests per unit of time.
     * @throws BadRequestException
     *         One of the parameters in the request is invalid.
     * @throws InternalServerErrorException
     *         The AWS Serverless Application Repository service encountered an internal error.
     * @throws ForbiddenException
     *         The client is not authenticated.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ServerlessApplicationRepositoryException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ServerlessApplicationRepositoryClient.CreateCloudFormationTemplate
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/serverlessrepo-2017-09-08/CreateCloudFormationTemplate"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CreateCloudFormationTemplateResponse createCloudFormationTemplate(
            CreateCloudFormationTemplateRequest createCloudFormationTemplateRequest) throws NotFoundException,
            TooManyRequestsException, BadRequestException, InternalServerErrorException, ForbiddenException, AwsServiceException,
            SdkClientException, ServerlessApplicationRepositoryException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<CreateCloudFormationTemplateResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, CreateCloudFormationTemplateResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createCloudFormationTemplateRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ServerlessApplicationRepository");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateCloudFormationTemplate");

            return clientHandler
                    .execute(new ClientExecutionParams<CreateCloudFormationTemplateRequest, CreateCloudFormationTemplateResponse>()
                            .withOperationName("CreateCloudFormationTemplate").withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withInput(createCloudFormationTemplateRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new CreateCloudFormationTemplateRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Deletes the specified application.
     * </p>
     *
     * @param deleteApplicationRequest
     * @return Result of the DeleteApplication operation returned by the service.
     * @throws BadRequestException
     *         One of the parameters in the request is invalid.
     * @throws InternalServerErrorException
     *         The AWS Serverless Application Repository service encountered an internal error.
     * @throws ForbiddenException
     *         The client is not authenticated.
     * @throws NotFoundException
     *         The resource (for example, an access policy statement) specified in the request doesn't exist.
     * @throws TooManyRequestsException
     *         The client is sending more than the allowed number of requests per unit of time.
     * @throws ConflictException
     *         The resource already exists.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ServerlessApplicationRepositoryException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ServerlessApplicationRepositoryClient.DeleteApplication
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/serverlessrepo-2017-09-08/DeleteApplication"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DeleteApplicationResponse deleteApplication(DeleteApplicationRequest deleteApplicationRequest)
            throws BadRequestException, InternalServerErrorException, ForbiddenException, NotFoundException,
            TooManyRequestsException, ConflictException, AwsServiceException, SdkClientException,
            ServerlessApplicationRepositoryException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DeleteApplicationResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                DeleteApplicationResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteApplicationRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ServerlessApplicationRepository");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteApplication");

            return clientHandler.execute(new ClientExecutionParams<DeleteApplicationRequest, DeleteApplicationResponse>()
                    .withOperationName("DeleteApplication").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(deleteApplicationRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DeleteApplicationRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Gets the specified application.
     * </p>
     *
     * @param getApplicationRequest
     * @return Result of the GetApplication operation returned by the service.
     * @throws NotFoundException
     *         The resource (for example, an access policy statement) specified in the request doesn't exist.
     * @throws TooManyRequestsException
     *         The client is sending more than the allowed number of requests per unit of time.
     * @throws BadRequestException
     *         One of the parameters in the request is invalid.
     * @throws InternalServerErrorException
     *         The AWS Serverless Application Repository service encountered an internal error.
     * @throws ForbiddenException
     *         The client is not authenticated.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ServerlessApplicationRepositoryException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ServerlessApplicationRepositoryClient.GetApplication
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/serverlessrepo-2017-09-08/GetApplication" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public GetApplicationResponse getApplication(GetApplicationRequest getApplicationRequest) throws NotFoundException,
            TooManyRequestsException, BadRequestException, InternalServerErrorException, ForbiddenException, AwsServiceException,
            SdkClientException, ServerlessApplicationRepositoryException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetApplicationResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                GetApplicationResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getApplicationRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ServerlessApplicationRepository");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetApplication");

            return clientHandler.execute(new ClientExecutionParams<GetApplicationRequest, GetApplicationResponse>()
                    .withOperationName("GetApplication").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(getApplicationRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new GetApplicationRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Retrieves the policy for the application.
     * </p>
     *
     * @param getApplicationPolicyRequest
     * @return Result of the GetApplicationPolicy operation returned by the service.
     * @throws NotFoundException
     *         The resource (for example, an access policy statement) specified in the request doesn't exist.
     * @throws TooManyRequestsException
     *         The client is sending more than the allowed number of requests per unit of time.
     * @throws BadRequestException
     *         One of the parameters in the request is invalid.
     * @throws InternalServerErrorException
     *         The AWS Serverless Application Repository service encountered an internal error.
     * @throws ForbiddenException
     *         The client is not authenticated.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ServerlessApplicationRepositoryException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ServerlessApplicationRepositoryClient.GetApplicationPolicy
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/serverlessrepo-2017-09-08/GetApplicationPolicy"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public GetApplicationPolicyResponse getApplicationPolicy(GetApplicationPolicyRequest getApplicationPolicyRequest)
            throws NotFoundException, TooManyRequestsException, BadRequestException, InternalServerErrorException,
            ForbiddenException, AwsServiceException, SdkClientException, ServerlessApplicationRepositoryException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetApplicationPolicyResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, GetApplicationPolicyResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getApplicationPolicyRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ServerlessApplicationRepository");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetApplicationPolicy");

            return clientHandler.execute(new ClientExecutionParams<GetApplicationPolicyRequest, GetApplicationPolicyResponse>()
                    .withOperationName("GetApplicationPolicy").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(getApplicationPolicyRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new GetApplicationPolicyRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Gets the specified AWS CloudFormation template.
     * </p>
     *
     * @param getCloudFormationTemplateRequest
     * @return Result of the GetCloudFormationTemplate operation returned by the service.
     * @throws NotFoundException
     *         The resource (for example, an access policy statement) specified in the request doesn't exist.
     * @throws TooManyRequestsException
     *         The client is sending more than the allowed number of requests per unit of time.
     * @throws BadRequestException
     *         One of the parameters in the request is invalid.
     * @throws InternalServerErrorException
     *         The AWS Serverless Application Repository service encountered an internal error.
     * @throws ForbiddenException
     *         The client is not authenticated.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ServerlessApplicationRepositoryException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ServerlessApplicationRepositoryClient.GetCloudFormationTemplate
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/serverlessrepo-2017-09-08/GetCloudFormationTemplate"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public GetCloudFormationTemplateResponse getCloudFormationTemplate(
            GetCloudFormationTemplateRequest getCloudFormationTemplateRequest) throws NotFoundException,
            TooManyRequestsException, BadRequestException, InternalServerErrorException, ForbiddenException, AwsServiceException,
            SdkClientException, ServerlessApplicationRepositoryException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetCloudFormationTemplateResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, GetCloudFormationTemplateResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getCloudFormationTemplateRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ServerlessApplicationRepository");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetCloudFormationTemplate");

            return clientHandler
                    .execute(new ClientExecutionParams<GetCloudFormationTemplateRequest, GetCloudFormationTemplateResponse>()
                            .withOperationName("GetCloudFormationTemplate").withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withInput(getCloudFormationTemplateRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new GetCloudFormationTemplateRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Retrieves the list of applications nested in the containing application.
     * </p>
     *
     * @param listApplicationDependenciesRequest
     * @return Result of the ListApplicationDependencies operation returned by the service.
     * @throws NotFoundException
     *         The resource (for example, an access policy statement) specified in the request doesn't exist.
     * @throws TooManyRequestsException
     *         The client is sending more than the allowed number of requests per unit of time.
     * @throws BadRequestException
     *         One of the parameters in the request is invalid.
     * @throws InternalServerErrorException
     *         The AWS Serverless Application Repository service encountered an internal error.
     * @throws ForbiddenException
     *         The client is not authenticated.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ServerlessApplicationRepositoryException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ServerlessApplicationRepositoryClient.ListApplicationDependencies
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/serverlessrepo-2017-09-08/ListApplicationDependencies"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListApplicationDependenciesResponse listApplicationDependencies(
            ListApplicationDependenciesRequest listApplicationDependenciesRequest) throws NotFoundException,
            TooManyRequestsException, BadRequestException, InternalServerErrorException, ForbiddenException, AwsServiceException,
            SdkClientException, ServerlessApplicationRepositoryException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListApplicationDependenciesResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, ListApplicationDependenciesResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listApplicationDependenciesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ServerlessApplicationRepository");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListApplicationDependencies");

            return clientHandler
                    .execute(new ClientExecutionParams<ListApplicationDependenciesRequest, ListApplicationDependenciesResponse>()
                            .withOperationName("ListApplicationDependencies").withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withInput(listApplicationDependenciesRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new ListApplicationDependenciesRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Lists versions for the specified application.
     * </p>
     *
     * @param listApplicationVersionsRequest
     * @return Result of the ListApplicationVersions operation returned by the service.
     * @throws NotFoundException
     *         The resource (for example, an access policy statement) specified in the request doesn't exist.
     * @throws TooManyRequestsException
     *         The client is sending more than the allowed number of requests per unit of time.
     * @throws BadRequestException
     *         One of the parameters in the request is invalid.
     * @throws InternalServerErrorException
     *         The AWS Serverless Application Repository service encountered an internal error.
     * @throws ForbiddenException
     *         The client is not authenticated.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ServerlessApplicationRepositoryException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ServerlessApplicationRepositoryClient.ListApplicationVersions
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/serverlessrepo-2017-09-08/ListApplicationVersions"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListApplicationVersionsResponse listApplicationVersions(ListApplicationVersionsRequest listApplicationVersionsRequest)
            throws NotFoundException, TooManyRequestsException, BadRequestException, InternalServerErrorException,
            ForbiddenException, AwsServiceException, SdkClientException, ServerlessApplicationRepositoryException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListApplicationVersionsResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, ListApplicationVersionsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listApplicationVersionsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ServerlessApplicationRepository");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListApplicationVersions");

            return clientHandler
                    .execute(new ClientExecutionParams<ListApplicationVersionsRequest, ListApplicationVersionsResponse>()
                            .withOperationName("ListApplicationVersions").withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withInput(listApplicationVersionsRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new ListApplicationVersionsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Lists applications owned by the requester.
     * </p>
     *
     * @param listApplicationsRequest
     * @return Result of the ListApplications operation returned by the service.
     * @throws NotFoundException
     *         The resource (for example, an access policy statement) specified in the request doesn't exist.
     * @throws BadRequestException
     *         One of the parameters in the request is invalid.
     * @throws InternalServerErrorException
     *         The AWS Serverless Application Repository service encountered an internal error.
     * @throws ForbiddenException
     *         The client is not authenticated.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ServerlessApplicationRepositoryException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ServerlessApplicationRepositoryClient.ListApplications
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/serverlessrepo-2017-09-08/ListApplications"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListApplicationsResponse listApplications(ListApplicationsRequest listApplicationsRequest) throws NotFoundException,
            BadRequestException, InternalServerErrorException, ForbiddenException, AwsServiceException, SdkClientException,
            ServerlessApplicationRepositoryException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListApplicationsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                ListApplicationsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listApplicationsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ServerlessApplicationRepository");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListApplications");

            return clientHandler.execute(new ClientExecutionParams<ListApplicationsRequest, ListApplicationsResponse>()
                    .withOperationName("ListApplications").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(listApplicationsRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListApplicationsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Sets the permission policy for an application. For the list of actions supported for this operation, see <a href=
     * "https://docs.aws.amazon.com/serverlessrepo/latest/devguide/access-control-resource-based.html#application-permissions"
     * >Application Permissions</a> .
     * </p>
     *
     * @param putApplicationPolicyRequest
     * @return Result of the PutApplicationPolicy operation returned by the service.
     * @throws NotFoundException
     *         The resource (for example, an access policy statement) specified in the request doesn't exist.
     * @throws TooManyRequestsException
     *         The client is sending more than the allowed number of requests per unit of time.
     * @throws BadRequestException
     *         One of the parameters in the request is invalid.
     * @throws InternalServerErrorException
     *         The AWS Serverless Application Repository service encountered an internal error.
     * @throws ForbiddenException
     *         The client is not authenticated.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ServerlessApplicationRepositoryException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ServerlessApplicationRepositoryClient.PutApplicationPolicy
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/serverlessrepo-2017-09-08/PutApplicationPolicy"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public PutApplicationPolicyResponse putApplicationPolicy(PutApplicationPolicyRequest putApplicationPolicyRequest)
            throws NotFoundException, TooManyRequestsException, BadRequestException, InternalServerErrorException,
            ForbiddenException, AwsServiceException, SdkClientException, ServerlessApplicationRepositoryException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<PutApplicationPolicyResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, PutApplicationPolicyResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, putApplicationPolicyRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ServerlessApplicationRepository");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "PutApplicationPolicy");

            return clientHandler.execute(new ClientExecutionParams<PutApplicationPolicyRequest, PutApplicationPolicyResponse>()
                    .withOperationName("PutApplicationPolicy").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(putApplicationPolicyRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new PutApplicationPolicyRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Unshares an application from an AWS Organization.
     * </p>
     * <p>
     * This operation can be called only from the organization's master account.
     * </p>
     *
     * @param unshareApplicationRequest
     * @return Result of the UnshareApplication operation returned by the service.
     * @throws NotFoundException
     *         The resource (for example, an access policy statement) specified in the request doesn't exist.
     * @throws TooManyRequestsException
     *         The client is sending more than the allowed number of requests per unit of time.
     * @throws BadRequestException
     *         One of the parameters in the request is invalid.
     * @throws InternalServerErrorException
     *         The AWS Serverless Application Repository service encountered an internal error.
     * @throws ForbiddenException
     *         The client is not authenticated.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ServerlessApplicationRepositoryException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ServerlessApplicationRepositoryClient.UnshareApplication
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/serverlessrepo-2017-09-08/UnshareApplication"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public UnshareApplicationResponse unshareApplication(UnshareApplicationRequest unshareApplicationRequest)
            throws NotFoundException, TooManyRequestsException, BadRequestException, InternalServerErrorException,
            ForbiddenException, AwsServiceException, SdkClientException, ServerlessApplicationRepositoryException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UnshareApplicationResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, UnshareApplicationResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, unshareApplicationRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ServerlessApplicationRepository");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UnshareApplication");

            return clientHandler.execute(new ClientExecutionParams<UnshareApplicationRequest, UnshareApplicationResponse>()
                    .withOperationName("UnshareApplication").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(unshareApplicationRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new UnshareApplicationRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Updates the specified application.
     * </p>
     *
     * @param updateApplicationRequest
     * @return Result of the UpdateApplication operation returned by the service.
     * @throws BadRequestException
     *         One of the parameters in the request is invalid.
     * @throws InternalServerErrorException
     *         The AWS Serverless Application Repository service encountered an internal error.
     * @throws ForbiddenException
     *         The client is not authenticated.
     * @throws NotFoundException
     *         The resource (for example, an access policy statement) specified in the request doesn't exist.
     * @throws TooManyRequestsException
     *         The client is sending more than the allowed number of requests per unit of time.
     * @throws ConflictException
     *         The resource already exists.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ServerlessApplicationRepositoryException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ServerlessApplicationRepositoryClient.UpdateApplication
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/serverlessrepo-2017-09-08/UpdateApplication"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public UpdateApplicationResponse updateApplication(UpdateApplicationRequest updateApplicationRequest)
            throws BadRequestException, InternalServerErrorException, ForbiddenException, NotFoundException,
            TooManyRequestsException, ConflictException, AwsServiceException, SdkClientException,
            ServerlessApplicationRepositoryException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UpdateApplicationResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                UpdateApplicationResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateApplicationRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ServerlessApplicationRepository");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateApplication");

            return clientHandler.execute(new ClientExecutionParams<UpdateApplicationRequest, UpdateApplicationResponse>()
                    .withOperationName("UpdateApplication").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(updateApplicationRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new UpdateApplicationRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    private static List<MetricPublisher> resolveMetricPublishers(SdkClientConfiguration clientConfiguration,
            RequestOverrideConfiguration requestOverrideConfiguration) {
        List<MetricPublisher> publishers = null;
        if (requestOverrideConfiguration != null) {
            publishers = requestOverrideConfiguration.metricPublishers();
        }
        if (publishers == null || publishers.isEmpty()) {
            publishers = clientConfiguration.option(SdkClientOption.METRIC_PUBLISHERS);
        }
        if (publishers == null) {
            publishers = Collections.emptyList();
        }
        return publishers;
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata) {
        return protocolFactory.createErrorResponseHandler(operationMetadata);
    }

    private <T extends BaseAwsJsonProtocolFactory.Builder<T>> T init(T builder) {
        return builder
                .clientConfiguration(clientConfiguration)
                .defaultServiceExceptionSupplier(ServerlessApplicationRepositoryException::builder)
                .protocol(AwsJsonProtocol.REST_JSON)
                .protocolVersion("1.1")
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ConflictException")
                                .exceptionBuilderSupplier(ConflictException::builder).httpStatusCode(409).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("NotFoundException")
                                .exceptionBuilderSupplier(NotFoundException::builder).httpStatusCode(404).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ForbiddenException")
                                .exceptionBuilderSupplier(ForbiddenException::builder).httpStatusCode(403).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("TooManyRequestsException")
                                .exceptionBuilderSupplier(TooManyRequestsException::builder).httpStatusCode(429).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("BadRequestException")
                                .exceptionBuilderSupplier(BadRequestException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InternalServerErrorException")
                                .exceptionBuilderSupplier(InternalServerErrorException::builder).httpStatusCode(500).build());
    }

    @Override
    public final ServerlessApplicationRepositoryServiceClientConfiguration serviceClientConfiguration() {
        return this.serviceClientConfiguration;
    }

    @Override
    public void close() {
        clientHandler.close();
    }
}
