/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.secretsmanager.model;

import java.nio.ByteBuffer;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.awscore.AwsRequestOverrideConfiguration;
import software.amazon.awssdk.core.SdkBytes;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.DefaultValueTrait;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 */
@Generated("software.amazon.awssdk:codegen")
public final class PutSecretValueRequest extends SecretsManagerRequest implements
        ToCopyableBuilder<PutSecretValueRequest.Builder, PutSecretValueRequest> {
    private static final SdkField<String> SECRET_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("SecretId").getter(getter(PutSecretValueRequest::secretId)).setter(setter(Builder::secretId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("SecretId").build()).build();

    private static final SdkField<String> CLIENT_REQUEST_TOKEN_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("ClientRequestToken")
            .getter(getter(PutSecretValueRequest::clientRequestToken))
            .setter(setter(Builder::clientRequestToken))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ClientRequestToken").build(),
                    DefaultValueTrait.idempotencyToken()).build();

    private static final SdkField<SdkBytes> SECRET_BINARY_FIELD = SdkField.<SdkBytes> builder(MarshallingType.SDK_BYTES)
            .memberName("SecretBinary").getter(getter(PutSecretValueRequest::secretBinary)).setter(setter(Builder::secretBinary))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("SecretBinary").build()).build();

    private static final SdkField<String> SECRET_STRING_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("SecretString").getter(getter(PutSecretValueRequest::secretString)).setter(setter(Builder::secretString))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("SecretString").build()).build();

    private static final SdkField<List<String>> VERSION_STAGES_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("VersionStages")
            .getter(getter(PutSecretValueRequest::versionStages))
            .setter(setter(Builder::versionStages))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("VersionStages").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<String> ROTATION_TOKEN_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("RotationToken").getter(getter(PutSecretValueRequest::rotationToken))
            .setter(setter(Builder::rotationToken))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("RotationToken").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(SECRET_ID_FIELD,
            CLIENT_REQUEST_TOKEN_FIELD, SECRET_BINARY_FIELD, SECRET_STRING_FIELD, VERSION_STAGES_FIELD, ROTATION_TOKEN_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = Collections
            .unmodifiableMap(new HashMap<String, SdkField<?>>() {
                {
                    put("SecretId", SECRET_ID_FIELD);
                    put("ClientRequestToken", CLIENT_REQUEST_TOKEN_FIELD);
                    put("SecretBinary", SECRET_BINARY_FIELD);
                    put("SecretString", SECRET_STRING_FIELD);
                    put("VersionStages", VERSION_STAGES_FIELD);
                    put("RotationToken", ROTATION_TOKEN_FIELD);
                }
            });

    private final String secretId;

    private final String clientRequestToken;

    private final SdkBytes secretBinary;

    private final String secretString;

    private final List<String> versionStages;

    private final String rotationToken;

    private PutSecretValueRequest(BuilderImpl builder) {
        super(builder);
        this.secretId = builder.secretId;
        this.clientRequestToken = builder.clientRequestToken;
        this.secretBinary = builder.secretBinary;
        this.secretString = builder.secretString;
        this.versionStages = builder.versionStages;
        this.rotationToken = builder.rotationToken;
    }

    /**
     * <p>
     * The ARN or name of the secret to add a new version to.
     * </p>
     * <p>
     * For an ARN, we recommend that you specify a complete ARN rather than a partial ARN. See <a
     * href="https://docs.aws.amazon.com/secretsmanager/latest/userguide/troubleshoot.html#ARN_secretnamehyphen">Finding
     * a secret from a partial ARN</a>.
     * </p>
     * <p>
     * If the secret doesn't already exist, use <code>CreateSecret</code> instead.
     * </p>
     * 
     * @return The ARN or name of the secret to add a new version to.</p>
     *         <p>
     *         For an ARN, we recommend that you specify a complete ARN rather than a partial ARN. See <a href=
     *         "https://docs.aws.amazon.com/secretsmanager/latest/userguide/troubleshoot.html#ARN_secretnamehyphen"
     *         >Finding a secret from a partial ARN</a>.
     *         </p>
     *         <p>
     *         If the secret doesn't already exist, use <code>CreateSecret</code> instead.
     */
    public final String secretId() {
        return secretId;
    }

    /**
     * <p>
     * A unique identifier for the new version of the secret.
     * </p>
     * <note>
     * <p>
     * If you use the Amazon Web Services CLI or one of the Amazon Web Services SDKs to call this operation, then you
     * can leave this parameter empty. The CLI or SDK generates a random UUID for you and includes it as the value for
     * this parameter in the request.
     * </p>
     * </note>
     * <p>
     * If you generate a raw HTTP request to the Secrets Manager service endpoint, then you must generate a
     * <code>ClientRequestToken</code> and include it in the request.
     * </p>
     * <p>
     * This value helps ensure idempotency. Secrets Manager uses this value to prevent the accidental creation of
     * duplicate versions if there are failures and retries during a rotation. We recommend that you generate a <a
     * href="https://wikipedia.org/wiki/Universally_unique_identifier">UUID-type</a> value to ensure uniqueness of your
     * versions within the specified secret.
     * </p>
     * <ul>
     * <li>
     * <p>
     * If the <code>ClientRequestToken</code> value isn't already associated with a version of the secret then a new
     * version of the secret is created.
     * </p>
     * </li>
     * <li>
     * <p>
     * If a version with this value already exists and that version's <code>SecretString</code> or
     * <code>SecretBinary</code> values are the same as those in the request then the request is ignored. The operation
     * is idempotent.
     * </p>
     * </li>
     * <li>
     * <p>
     * If a version with this value already exists and the version of the <code>SecretString</code> and
     * <code>SecretBinary</code> values are different from those in the request, then the request fails because you
     * can't modify a secret version. You can only create new versions to store new secret values.
     * </p>
     * </li>
     * </ul>
     * <p>
     * This value becomes the <code>VersionId</code> of the new version.
     * </p>
     * 
     * @return A unique identifier for the new version of the secret. </p> <note>
     *         <p>
     *         If you use the Amazon Web Services CLI or one of the Amazon Web Services SDKs to call this operation,
     *         then you can leave this parameter empty. The CLI or SDK generates a random UUID for you and includes it
     *         as the value for this parameter in the request.
     *         </p>
     *         </note>
     *         <p>
     *         If you generate a raw HTTP request to the Secrets Manager service endpoint, then you must generate a
     *         <code>ClientRequestToken</code> and include it in the request.
     *         </p>
     *         <p>
     *         This value helps ensure idempotency. Secrets Manager uses this value to prevent the accidental creation
     *         of duplicate versions if there are failures and retries during a rotation. We recommend that you generate
     *         a <a href="https://wikipedia.org/wiki/Universally_unique_identifier">UUID-type</a> value to ensure
     *         uniqueness of your versions within the specified secret.
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         If the <code>ClientRequestToken</code> value isn't already associated with a version of the secret then a
     *         new version of the secret is created.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         If a version with this value already exists and that version's <code>SecretString</code> or
     *         <code>SecretBinary</code> values are the same as those in the request then the request is ignored. The
     *         operation is idempotent.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         If a version with this value already exists and the version of the <code>SecretString</code> and
     *         <code>SecretBinary</code> values are different from those in the request, then the request fails because
     *         you can't modify a secret version. You can only create new versions to store new secret values.
     *         </p>
     *         </li>
     *         </ul>
     *         <p>
     *         This value becomes the <code>VersionId</code> of the new version.
     */
    public final String clientRequestToken() {
        return clientRequestToken;
    }

    /**
     * <p>
     * The binary data to encrypt and store in the new version of the secret. To use this parameter in the command-line
     * tools, we recommend that you store your binary data in a file and then pass the contents of the file as a
     * parameter.
     * </p>
     * <p>
     * You must include <code>SecretBinary</code> or <code>SecretString</code>, but not both.
     * </p>
     * <p>
     * You can't access this value from the Secrets Manager console.
     * </p>
     * <p>
     * Sensitive: This field contains sensitive information, so the service does not include it in CloudTrail log
     * entries. If you create your own log entries, you must also avoid logging the information in this field.
     * </p>
     * 
     * @return The binary data to encrypt and store in the new version of the secret. To use this parameter in the
     *         command-line tools, we recommend that you store your binary data in a file and then pass the contents of
     *         the file as a parameter. </p>
     *         <p>
     *         You must include <code>SecretBinary</code> or <code>SecretString</code>, but not both.
     *         </p>
     *         <p>
     *         You can't access this value from the Secrets Manager console.
     *         </p>
     *         <p>
     *         Sensitive: This field contains sensitive information, so the service does not include it in CloudTrail
     *         log entries. If you create your own log entries, you must also avoid logging the information in this
     *         field.
     */
    public final SdkBytes secretBinary() {
        return secretBinary;
    }

    /**
     * <p>
     * The text to encrypt and store in the new version of the secret.
     * </p>
     * <p>
     * You must include <code>SecretBinary</code> or <code>SecretString</code>, but not both.
     * </p>
     * <p>
     * We recommend you create the secret string as JSON key/value pairs, as shown in the example.
     * </p>
     * <p>
     * Sensitive: This field contains sensitive information, so the service does not include it in CloudTrail log
     * entries. If you create your own log entries, you must also avoid logging the information in this field.
     * </p>
     * 
     * @return The text to encrypt and store in the new version of the secret. </p>
     *         <p>
     *         You must include <code>SecretBinary</code> or <code>SecretString</code>, but not both.
     *         </p>
     *         <p>
     *         We recommend you create the secret string as JSON key/value pairs, as shown in the example.
     *         </p>
     *         <p>
     *         Sensitive: This field contains sensitive information, so the service does not include it in CloudTrail
     *         log entries. If you create your own log entries, you must also avoid logging the information in this
     *         field.
     */
    public final String secretString() {
        return secretString;
    }

    /**
     * For responses, this returns true if the service returned a value for the VersionStages property. This DOES NOT
     * check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasVersionStages() {
        return versionStages != null && !(versionStages instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * A list of staging labels to attach to this version of the secret. Secrets Manager uses staging labels to track
     * versions of a secret through the rotation process.
     * </p>
     * <p>
     * If you specify a staging label that's already associated with a different version of the same secret, then
     * Secrets Manager removes the label from the other version and attaches it to this version. If you specify
     * <code>AWSCURRENT</code>, and it is already attached to another version, then Secrets Manager also moves the
     * staging label <code>AWSPREVIOUS</code> to the version that <code>AWSCURRENT</code> was removed from.
     * </p>
     * <p>
     * If you don't include <code>VersionStages</code>, then Secrets Manager automatically moves the staging label
     * <code>AWSCURRENT</code> to this version.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasVersionStages} method.
     * </p>
     * 
     * @return A list of staging labels to attach to this version of the secret. Secrets Manager uses staging labels to
     *         track versions of a secret through the rotation process.</p>
     *         <p>
     *         If you specify a staging label that's already associated with a different version of the same secret,
     *         then Secrets Manager removes the label from the other version and attaches it to this version. If you
     *         specify <code>AWSCURRENT</code>, and it is already attached to another version, then Secrets Manager also
     *         moves the staging label <code>AWSPREVIOUS</code> to the version that <code>AWSCURRENT</code> was removed
     *         from.
     *         </p>
     *         <p>
     *         If you don't include <code>VersionStages</code>, then Secrets Manager automatically moves the staging
     *         label <code>AWSCURRENT</code> to this version.
     */
    public final List<String> versionStages() {
        return versionStages;
    }

    /**
     * <p>
     * A unique identifier that indicates the source of the request. For cross-account rotation (when you rotate a
     * secret in one account by using a Lambda rotation function in another account) and the Lambda rotation function
     * assumes an IAM role to call Secrets Manager, Secrets Manager validates the identity with the rotation token. For
     * more information, see <a
     * href="https://docs.aws.amazon.com/secretsmanager/latest/userguide/rotating-secrets.html">How rotation works</a>.
     * </p>
     * <p>
     * Sensitive: This field contains sensitive information, so the service does not include it in CloudTrail log
     * entries. If you create your own log entries, you must also avoid logging the information in this field.
     * </p>
     * 
     * @return A unique identifier that indicates the source of the request. For cross-account rotation (when you rotate
     *         a secret in one account by using a Lambda rotation function in another account) and the Lambda rotation
     *         function assumes an IAM role to call Secrets Manager, Secrets Manager validates the identity with the
     *         rotation token. For more information, see <a
     *         href="https://docs.aws.amazon.com/secretsmanager/latest/userguide/rotating-secrets.html">How rotation
     *         works</a>.</p>
     *         <p>
     *         Sensitive: This field contains sensitive information, so the service does not include it in CloudTrail
     *         log entries. If you create your own log entries, you must also avoid logging the information in this
     *         field.
     */
    public final String rotationToken() {
        return rotationToken;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + super.hashCode();
        hashCode = 31 * hashCode + Objects.hashCode(secretId());
        hashCode = 31 * hashCode + Objects.hashCode(clientRequestToken());
        hashCode = 31 * hashCode + Objects.hashCode(secretBinary());
        hashCode = 31 * hashCode + Objects.hashCode(secretString());
        hashCode = 31 * hashCode + Objects.hashCode(hasVersionStages() ? versionStages() : null);
        hashCode = 31 * hashCode + Objects.hashCode(rotationToken());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return super.equals(obj) && equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof PutSecretValueRequest)) {
            return false;
        }
        PutSecretValueRequest other = (PutSecretValueRequest) obj;
        return Objects.equals(secretId(), other.secretId()) && Objects.equals(clientRequestToken(), other.clientRequestToken())
                && Objects.equals(secretBinary(), other.secretBinary()) && Objects.equals(secretString(), other.secretString())
                && hasVersionStages() == other.hasVersionStages() && Objects.equals(versionStages(), other.versionStages())
                && Objects.equals(rotationToken(), other.rotationToken());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("PutSecretValueRequest").add("SecretId", secretId())
                .add("ClientRequestToken", clientRequestToken())
                .add("SecretBinary", secretBinary() == null ? null : "*** Sensitive Data Redacted ***")
                .add("SecretString", secretString() == null ? null : "*** Sensitive Data Redacted ***")
                .add("VersionStages", hasVersionStages() ? versionStages() : null)
                .add("RotationToken", rotationToken() == null ? null : "*** Sensitive Data Redacted ***").build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "SecretId":
            return Optional.ofNullable(clazz.cast(secretId()));
        case "ClientRequestToken":
            return Optional.ofNullable(clazz.cast(clientRequestToken()));
        case "SecretBinary":
            return Optional.ofNullable(clazz.cast(secretBinary()));
        case "SecretString":
            return Optional.ofNullable(clazz.cast(secretString()));
        case "VersionStages":
            return Optional.ofNullable(clazz.cast(versionStages()));
        case "RotationToken":
            return Optional.ofNullable(clazz.cast(rotationToken()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static <T> Function<Object, T> getter(Function<PutSecretValueRequest, T> g) {
        return obj -> g.apply((PutSecretValueRequest) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SecretsManagerRequest.Builder, SdkPojo, CopyableBuilder<Builder, PutSecretValueRequest> {
        /**
         * <p>
         * The ARN or name of the secret to add a new version to.
         * </p>
         * <p>
         * For an ARN, we recommend that you specify a complete ARN rather than a partial ARN. See <a
         * href="https://docs.aws.amazon.com/secretsmanager/latest/userguide/troubleshoot.html#ARN_secretnamehyphen"
         * >Finding a secret from a partial ARN</a>.
         * </p>
         * <p>
         * If the secret doesn't already exist, use <code>CreateSecret</code> instead.
         * </p>
         * 
         * @param secretId
         *        The ARN or name of the secret to add a new version to.</p>
         *        <p>
         *        For an ARN, we recommend that you specify a complete ARN rather than a partial ARN. See <a href=
         *        "https://docs.aws.amazon.com/secretsmanager/latest/userguide/troubleshoot.html#ARN_secretnamehyphen"
         *        >Finding a secret from a partial ARN</a>.
         *        </p>
         *        <p>
         *        If the secret doesn't already exist, use <code>CreateSecret</code> instead.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder secretId(String secretId);

        /**
         * <p>
         * A unique identifier for the new version of the secret.
         * </p>
         * <note>
         * <p>
         * If you use the Amazon Web Services CLI or one of the Amazon Web Services SDKs to call this operation, then
         * you can leave this parameter empty. The CLI or SDK generates a random UUID for you and includes it as the
         * value for this parameter in the request.
         * </p>
         * </note>
         * <p>
         * If you generate a raw HTTP request to the Secrets Manager service endpoint, then you must generate a
         * <code>ClientRequestToken</code> and include it in the request.
         * </p>
         * <p>
         * This value helps ensure idempotency. Secrets Manager uses this value to prevent the accidental creation of
         * duplicate versions if there are failures and retries during a rotation. We recommend that you generate a <a
         * href="https://wikipedia.org/wiki/Universally_unique_identifier">UUID-type</a> value to ensure uniqueness of
         * your versions within the specified secret.
         * </p>
         * <ul>
         * <li>
         * <p>
         * If the <code>ClientRequestToken</code> value isn't already associated with a version of the secret then a new
         * version of the secret is created.
         * </p>
         * </li>
         * <li>
         * <p>
         * If a version with this value already exists and that version's <code>SecretString</code> or
         * <code>SecretBinary</code> values are the same as those in the request then the request is ignored. The
         * operation is idempotent.
         * </p>
         * </li>
         * <li>
         * <p>
         * If a version with this value already exists and the version of the <code>SecretString</code> and
         * <code>SecretBinary</code> values are different from those in the request, then the request fails because you
         * can't modify a secret version. You can only create new versions to store new secret values.
         * </p>
         * </li>
         * </ul>
         * <p>
         * This value becomes the <code>VersionId</code> of the new version.
         * </p>
         * 
         * @param clientRequestToken
         *        A unique identifier for the new version of the secret. </p> <note>
         *        <p>
         *        If you use the Amazon Web Services CLI or one of the Amazon Web Services SDKs to call this operation,
         *        then you can leave this parameter empty. The CLI or SDK generates a random UUID for you and includes
         *        it as the value for this parameter in the request.
         *        </p>
         *        </note>
         *        <p>
         *        If you generate a raw HTTP request to the Secrets Manager service endpoint, then you must generate a
         *        <code>ClientRequestToken</code> and include it in the request.
         *        </p>
         *        <p>
         *        This value helps ensure idempotency. Secrets Manager uses this value to prevent the accidental
         *        creation of duplicate versions if there are failures and retries during a rotation. We recommend that
         *        you generate a <a href="https://wikipedia.org/wiki/Universally_unique_identifier">UUID-type</a> value
         *        to ensure uniqueness of your versions within the specified secret.
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        If the <code>ClientRequestToken</code> value isn't already associated with a version of the secret
         *        then a new version of the secret is created.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        If a version with this value already exists and that version's <code>SecretString</code> or
         *        <code>SecretBinary</code> values are the same as those in the request then the request is ignored. The
         *        operation is idempotent.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        If a version with this value already exists and the version of the <code>SecretString</code> and
         *        <code>SecretBinary</code> values are different from those in the request, then the request fails
         *        because you can't modify a secret version. You can only create new versions to store new secret
         *        values.
         *        </p>
         *        </li>
         *        </ul>
         *        <p>
         *        This value becomes the <code>VersionId</code> of the new version.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder clientRequestToken(String clientRequestToken);

        /**
         * <p>
         * The binary data to encrypt and store in the new version of the secret. To use this parameter in the
         * command-line tools, we recommend that you store your binary data in a file and then pass the contents of the
         * file as a parameter.
         * </p>
         * <p>
         * You must include <code>SecretBinary</code> or <code>SecretString</code>, but not both.
         * </p>
         * <p>
         * You can't access this value from the Secrets Manager console.
         * </p>
         * <p>
         * Sensitive: This field contains sensitive information, so the service does not include it in CloudTrail log
         * entries. If you create your own log entries, you must also avoid logging the information in this field.
         * </p>
         * 
         * @param secretBinary
         *        The binary data to encrypt and store in the new version of the secret. To use this parameter in the
         *        command-line tools, we recommend that you store your binary data in a file and then pass the contents
         *        of the file as a parameter. </p>
         *        <p>
         *        You must include <code>SecretBinary</code> or <code>SecretString</code>, but not both.
         *        </p>
         *        <p>
         *        You can't access this value from the Secrets Manager console.
         *        </p>
         *        <p>
         *        Sensitive: This field contains sensitive information, so the service does not include it in CloudTrail
         *        log entries. If you create your own log entries, you must also avoid logging the information in this
         *        field.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder secretBinary(SdkBytes secretBinary);

        /**
         * <p>
         * The text to encrypt and store in the new version of the secret.
         * </p>
         * <p>
         * You must include <code>SecretBinary</code> or <code>SecretString</code>, but not both.
         * </p>
         * <p>
         * We recommend you create the secret string as JSON key/value pairs, as shown in the example.
         * </p>
         * <p>
         * Sensitive: This field contains sensitive information, so the service does not include it in CloudTrail log
         * entries. If you create your own log entries, you must also avoid logging the information in this field.
         * </p>
         * 
         * @param secretString
         *        The text to encrypt and store in the new version of the secret. </p>
         *        <p>
         *        You must include <code>SecretBinary</code> or <code>SecretString</code>, but not both.
         *        </p>
         *        <p>
         *        We recommend you create the secret string as JSON key/value pairs, as shown in the example.
         *        </p>
         *        <p>
         *        Sensitive: This field contains sensitive information, so the service does not include it in CloudTrail
         *        log entries. If you create your own log entries, you must also avoid logging the information in this
         *        field.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder secretString(String secretString);

        /**
         * <p>
         * A list of staging labels to attach to this version of the secret. Secrets Manager uses staging labels to
         * track versions of a secret through the rotation process.
         * </p>
         * <p>
         * If you specify a staging label that's already associated with a different version of the same secret, then
         * Secrets Manager removes the label from the other version and attaches it to this version. If you specify
         * <code>AWSCURRENT</code>, and it is already attached to another version, then Secrets Manager also moves the
         * staging label <code>AWSPREVIOUS</code> to the version that <code>AWSCURRENT</code> was removed from.
         * </p>
         * <p>
         * If you don't include <code>VersionStages</code>, then Secrets Manager automatically moves the staging label
         * <code>AWSCURRENT</code> to this version.
         * </p>
         * 
         * @param versionStages
         *        A list of staging labels to attach to this version of the secret. Secrets Manager uses staging labels
         *        to track versions of a secret through the rotation process.</p>
         *        <p>
         *        If you specify a staging label that's already associated with a different version of the same secret,
         *        then Secrets Manager removes the label from the other version and attaches it to this version. If you
         *        specify <code>AWSCURRENT</code>, and it is already attached to another version, then Secrets Manager
         *        also moves the staging label <code>AWSPREVIOUS</code> to the version that <code>AWSCURRENT</code> was
         *        removed from.
         *        </p>
         *        <p>
         *        If you don't include <code>VersionStages</code>, then Secrets Manager automatically moves the staging
         *        label <code>AWSCURRENT</code> to this version.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder versionStages(Collection<String> versionStages);

        /**
         * <p>
         * A list of staging labels to attach to this version of the secret. Secrets Manager uses staging labels to
         * track versions of a secret through the rotation process.
         * </p>
         * <p>
         * If you specify a staging label that's already associated with a different version of the same secret, then
         * Secrets Manager removes the label from the other version and attaches it to this version. If you specify
         * <code>AWSCURRENT</code>, and it is already attached to another version, then Secrets Manager also moves the
         * staging label <code>AWSPREVIOUS</code> to the version that <code>AWSCURRENT</code> was removed from.
         * </p>
         * <p>
         * If you don't include <code>VersionStages</code>, then Secrets Manager automatically moves the staging label
         * <code>AWSCURRENT</code> to this version.
         * </p>
         * 
         * @param versionStages
         *        A list of staging labels to attach to this version of the secret. Secrets Manager uses staging labels
         *        to track versions of a secret through the rotation process.</p>
         *        <p>
         *        If you specify a staging label that's already associated with a different version of the same secret,
         *        then Secrets Manager removes the label from the other version and attaches it to this version. If you
         *        specify <code>AWSCURRENT</code>, and it is already attached to another version, then Secrets Manager
         *        also moves the staging label <code>AWSPREVIOUS</code> to the version that <code>AWSCURRENT</code> was
         *        removed from.
         *        </p>
         *        <p>
         *        If you don't include <code>VersionStages</code>, then Secrets Manager automatically moves the staging
         *        label <code>AWSCURRENT</code> to this version.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder versionStages(String... versionStages);

        /**
         * <p>
         * A unique identifier that indicates the source of the request. For cross-account rotation (when you rotate a
         * secret in one account by using a Lambda rotation function in another account) and the Lambda rotation
         * function assumes an IAM role to call Secrets Manager, Secrets Manager validates the identity with the
         * rotation token. For more information, see <a
         * href="https://docs.aws.amazon.com/secretsmanager/latest/userguide/rotating-secrets.html">How rotation
         * works</a>.
         * </p>
         * <p>
         * Sensitive: This field contains sensitive information, so the service does not include it in CloudTrail log
         * entries. If you create your own log entries, you must also avoid logging the information in this field.
         * </p>
         * 
         * @param rotationToken
         *        A unique identifier that indicates the source of the request. For cross-account rotation (when you
         *        rotate a secret in one account by using a Lambda rotation function in another account) and the Lambda
         *        rotation function assumes an IAM role to call Secrets Manager, Secrets Manager validates the identity
         *        with the rotation token. For more information, see <a
         *        href="https://docs.aws.amazon.com/secretsmanager/latest/userguide/rotating-secrets.html">How rotation
         *        works</a>.</p>
         *        <p>
         *        Sensitive: This field contains sensitive information, so the service does not include it in CloudTrail
         *        log entries. If you create your own log entries, you must also avoid logging the information in this
         *        field.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder rotationToken(String rotationToken);

        @Override
        Builder overrideConfiguration(AwsRequestOverrideConfiguration overrideConfiguration);

        @Override
        Builder overrideConfiguration(Consumer<AwsRequestOverrideConfiguration.Builder> builderConsumer);
    }

    static final class BuilderImpl extends SecretsManagerRequest.BuilderImpl implements Builder {
        private String secretId;

        private String clientRequestToken;

        private SdkBytes secretBinary;

        private String secretString;

        private List<String> versionStages = DefaultSdkAutoConstructList.getInstance();

        private String rotationToken;

        private BuilderImpl() {
        }

        private BuilderImpl(PutSecretValueRequest model) {
            super(model);
            secretId(model.secretId);
            clientRequestToken(model.clientRequestToken);
            secretBinary(model.secretBinary);
            secretString(model.secretString);
            versionStages(model.versionStages);
            rotationToken(model.rotationToken);
        }

        public final String getSecretId() {
            return secretId;
        }

        public final void setSecretId(String secretId) {
            this.secretId = secretId;
        }

        @Override
        public final Builder secretId(String secretId) {
            this.secretId = secretId;
            return this;
        }

        public final String getClientRequestToken() {
            return clientRequestToken;
        }

        public final void setClientRequestToken(String clientRequestToken) {
            this.clientRequestToken = clientRequestToken;
        }

        @Override
        public final Builder clientRequestToken(String clientRequestToken) {
            this.clientRequestToken = clientRequestToken;
            return this;
        }

        public final ByteBuffer getSecretBinary() {
            return secretBinary == null ? null : secretBinary.asByteBuffer();
        }

        public final void setSecretBinary(ByteBuffer secretBinary) {
            secretBinary(secretBinary == null ? null : SdkBytes.fromByteBuffer(secretBinary));
        }

        @Override
        public final Builder secretBinary(SdkBytes secretBinary) {
            this.secretBinary = secretBinary;
            return this;
        }

        public final String getSecretString() {
            return secretString;
        }

        public final void setSecretString(String secretString) {
            this.secretString = secretString;
        }

        @Override
        public final Builder secretString(String secretString) {
            this.secretString = secretString;
            return this;
        }

        public final Collection<String> getVersionStages() {
            if (versionStages instanceof SdkAutoConstructList) {
                return null;
            }
            return versionStages;
        }

        public final void setVersionStages(Collection<String> versionStages) {
            this.versionStages = SecretVersionStagesTypeCopier.copy(versionStages);
        }

        @Override
        public final Builder versionStages(Collection<String> versionStages) {
            this.versionStages = SecretVersionStagesTypeCopier.copy(versionStages);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder versionStages(String... versionStages) {
            versionStages(Arrays.asList(versionStages));
            return this;
        }

        public final String getRotationToken() {
            return rotationToken;
        }

        public final void setRotationToken(String rotationToken) {
            this.rotationToken = rotationToken;
        }

        @Override
        public final Builder rotationToken(String rotationToken) {
            this.rotationToken = rotationToken;
            return this;
        }

        @Override
        public Builder overrideConfiguration(AwsRequestOverrideConfiguration overrideConfiguration) {
            super.overrideConfiguration(overrideConfiguration);
            return this;
        }

        @Override
        public Builder overrideConfiguration(Consumer<AwsRequestOverrideConfiguration.Builder> builderConsumer) {
            super.overrideConfiguration(builderConsumer);
            return this;
        }

        @Override
        public PutSecretValueRequest build() {
            return new PutSecretValueRequest(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
