/*
 * Copyright 2015-2020 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.sagemaker.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Describes the S3 data source.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class S3DataSource implements SdkPojo, Serializable, ToCopyableBuilder<S3DataSource.Builder, S3DataSource> {
    private static final SdkField<String> S3_DATA_TYPE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(S3DataSource::s3DataTypeAsString)).setter(setter(Builder::s3DataType))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("S3DataType").build()).build();

    private static final SdkField<String> S3_URI_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(S3DataSource::s3Uri)).setter(setter(Builder::s3Uri))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("S3Uri").build()).build();

    private static final SdkField<String> S3_DATA_DISTRIBUTION_TYPE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(S3DataSource::s3DataDistributionTypeAsString)).setter(setter(Builder::s3DataDistributionType))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("S3DataDistributionType").build())
            .build();

    private static final SdkField<List<String>> ATTRIBUTE_NAMES_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .getter(getter(S3DataSource::attributeNames))
            .setter(setter(Builder::attributeNames))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("AttributeNames").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(S3_DATA_TYPE_FIELD,
            S3_URI_FIELD, S3_DATA_DISTRIBUTION_TYPE_FIELD, ATTRIBUTE_NAMES_FIELD));

    private static final long serialVersionUID = 1L;

    private final String s3DataType;

    private final String s3Uri;

    private final String s3DataDistributionType;

    private final List<String> attributeNames;

    private S3DataSource(BuilderImpl builder) {
        this.s3DataType = builder.s3DataType;
        this.s3Uri = builder.s3Uri;
        this.s3DataDistributionType = builder.s3DataDistributionType;
        this.attributeNames = builder.attributeNames;
    }

    /**
     * <p>
     * If you choose <code>S3Prefix</code>, <code>S3Uri</code> identifies a key name prefix. Amazon SageMaker uses all
     * objects that match the specified key name prefix for model training.
     * </p>
     * <p>
     * If you choose <code>ManifestFile</code>, <code>S3Uri</code> identifies an object that is a manifest file
     * containing a list of object keys that you want Amazon SageMaker to use for model training.
     * </p>
     * <p>
     * If you choose <code>AugmentedManifestFile</code>, S3Uri identifies an object that is an augmented manifest file
     * in JSON lines format. This file contains the data you want to use for model training.
     * <code>AugmentedManifestFile</code> can only be used if the Channel's input mode is <code>Pipe</code>.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #s3DataType} will
     * return {@link S3DataType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #s3DataTypeAsString}.
     * </p>
     * 
     * @return If you choose <code>S3Prefix</code>, <code>S3Uri</code> identifies a key name prefix. Amazon SageMaker
     *         uses all objects that match the specified key name prefix for model training. </p>
     *         <p>
     *         If you choose <code>ManifestFile</code>, <code>S3Uri</code> identifies an object that is a manifest file
     *         containing a list of object keys that you want Amazon SageMaker to use for model training.
     *         </p>
     *         <p>
     *         If you choose <code>AugmentedManifestFile</code>, S3Uri identifies an object that is an augmented
     *         manifest file in JSON lines format. This file contains the data you want to use for model training.
     *         <code>AugmentedManifestFile</code> can only be used if the Channel's input mode is <code>Pipe</code>.
     * @see S3DataType
     */
    public S3DataType s3DataType() {
        return S3DataType.fromValue(s3DataType);
    }

    /**
     * <p>
     * If you choose <code>S3Prefix</code>, <code>S3Uri</code> identifies a key name prefix. Amazon SageMaker uses all
     * objects that match the specified key name prefix for model training.
     * </p>
     * <p>
     * If you choose <code>ManifestFile</code>, <code>S3Uri</code> identifies an object that is a manifest file
     * containing a list of object keys that you want Amazon SageMaker to use for model training.
     * </p>
     * <p>
     * If you choose <code>AugmentedManifestFile</code>, S3Uri identifies an object that is an augmented manifest file
     * in JSON lines format. This file contains the data you want to use for model training.
     * <code>AugmentedManifestFile</code> can only be used if the Channel's input mode is <code>Pipe</code>.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #s3DataType} will
     * return {@link S3DataType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #s3DataTypeAsString}.
     * </p>
     * 
     * @return If you choose <code>S3Prefix</code>, <code>S3Uri</code> identifies a key name prefix. Amazon SageMaker
     *         uses all objects that match the specified key name prefix for model training. </p>
     *         <p>
     *         If you choose <code>ManifestFile</code>, <code>S3Uri</code> identifies an object that is a manifest file
     *         containing a list of object keys that you want Amazon SageMaker to use for model training.
     *         </p>
     *         <p>
     *         If you choose <code>AugmentedManifestFile</code>, S3Uri identifies an object that is an augmented
     *         manifest file in JSON lines format. This file contains the data you want to use for model training.
     *         <code>AugmentedManifestFile</code> can only be used if the Channel's input mode is <code>Pipe</code>.
     * @see S3DataType
     */
    public String s3DataTypeAsString() {
        return s3DataType;
    }

    /**
     * <p>
     * Depending on the value specified for the <code>S3DataType</code>, identifies either a key name prefix or a
     * manifest. For example:
     * </p>
     * <ul>
     * <li>
     * <p>
     * A key name prefix might look like this: <code>s3://bucketname/exampleprefix</code>.
     * </p>
     * </li>
     * <li>
     * <p>
     * A manifest might look like this: <code>s3://bucketname/example.manifest</code>
     * </p>
     * <p>
     * The manifest is an S3 object which is a JSON file with the following format:
     * </p>
     * <p>
     * The preceding JSON matches the following <code>s3Uris</code>:
     * </p>
     * <p>
     * <code>[ {"prefix": "s3://customer_bucket/some/prefix/"},</code>
     * </p>
     * <p>
     * <code>"relative/path/to/custdata-1",</code>
     * </p>
     * <p>
     * <code>"relative/path/custdata-2",</code>
     * </p>
     * <p>
     * <code>...</code>
     * </p>
     * <p>
     * <code>"relative/path/custdata-N"</code>
     * </p>
     * <p>
     * <code>]</code>
     * </p>
     * <p>
     * The preceding JSON matches the following <code>s3Uris</code>:
     * </p>
     * <p>
     * <code>s3://customer_bucket/some/prefix/relative/path/to/custdata-1</code>
     * </p>
     * <p>
     * <code>s3://customer_bucket/some/prefix/relative/path/custdata-2</code>
     * </p>
     * <p>
     * <code>...</code>
     * </p>
     * <p>
     * <code>s3://customer_bucket/some/prefix/relative/path/custdata-N</code>
     * </p>
     * <p>
     * The complete set of <code>s3uris</code> in this manifest is the input data for the channel for this datasource.
     * The object that each <code>s3uris</code> points to must be readable by the IAM role that Amazon SageMaker uses to
     * perform tasks on your behalf.
     * </p>
     * </li>
     * </ul>
     * 
     * @return Depending on the value specified for the <code>S3DataType</code>, identifies either a key name prefix or
     *         a manifest. For example: </p>
     *         <ul>
     *         <li>
     *         <p>
     *         A key name prefix might look like this: <code>s3://bucketname/exampleprefix</code>.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         A manifest might look like this: <code>s3://bucketname/example.manifest</code>
     *         </p>
     *         <p>
     *         The manifest is an S3 object which is a JSON file with the following format:
     *         </p>
     *         <p>
     *         The preceding JSON matches the following <code>s3Uris</code>:
     *         </p>
     *         <p>
     *         <code>[ {"prefix": "s3://customer_bucket/some/prefix/"},</code>
     *         </p>
     *         <p>
     *         <code>"relative/path/to/custdata-1",</code>
     *         </p>
     *         <p>
     *         <code>"relative/path/custdata-2",</code>
     *         </p>
     *         <p>
     *         <code>...</code>
     *         </p>
     *         <p>
     *         <code>"relative/path/custdata-N"</code>
     *         </p>
     *         <p>
     *         <code>]</code>
     *         </p>
     *         <p>
     *         The preceding JSON matches the following <code>s3Uris</code>:
     *         </p>
     *         <p>
     *         <code>s3://customer_bucket/some/prefix/relative/path/to/custdata-1</code>
     *         </p>
     *         <p>
     *         <code>s3://customer_bucket/some/prefix/relative/path/custdata-2</code>
     *         </p>
     *         <p>
     *         <code>...</code>
     *         </p>
     *         <p>
     *         <code>s3://customer_bucket/some/prefix/relative/path/custdata-N</code>
     *         </p>
     *         <p>
     *         The complete set of <code>s3uris</code> in this manifest is the input data for the channel for this
     *         datasource. The object that each <code>s3uris</code> points to must be readable by the IAM role that
     *         Amazon SageMaker uses to perform tasks on your behalf.
     *         </p>
     *         </li>
     */
    public String s3Uri() {
        return s3Uri;
    }

    /**
     * <p>
     * If you want Amazon SageMaker to replicate the entire dataset on each ML compute instance that is launched for
     * model training, specify <code>FullyReplicated</code>.
     * </p>
     * <p>
     * If you want Amazon SageMaker to replicate a subset of data on each ML compute instance that is launched for model
     * training, specify <code>ShardedByS3Key</code>. If there are <i>n</i> ML compute instances launched for a training
     * job, each instance gets approximately 1/<i>n</i> of the number of S3 objects. In this case, model training on
     * each machine uses only the subset of training data.
     * </p>
     * <p>
     * Don't choose more ML compute instances for training than available S3 objects. If you do, some nodes won't get
     * any data and you will pay for nodes that aren't getting any training data. This applies in both File and Pipe
     * modes. Keep this in mind when developing algorithms.
     * </p>
     * <p>
     * In distributed training, where you use multiple ML compute EC2 instances, you might choose
     * <code>ShardedByS3Key</code>. If the algorithm requires copying training data to the ML storage volume (when
     * <code>TrainingInputMode</code> is set to <code>File</code>), this copies 1/<i>n</i> of the number of objects.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version,
     * {@link #s3DataDistributionType} will return {@link S3DataDistribution#UNKNOWN_TO_SDK_VERSION}. The raw value
     * returned by the service is available from {@link #s3DataDistributionTypeAsString}.
     * </p>
     * 
     * @return If you want Amazon SageMaker to replicate the entire dataset on each ML compute instance that is launched
     *         for model training, specify <code>FullyReplicated</code>. </p>
     *         <p>
     *         If you want Amazon SageMaker to replicate a subset of data on each ML compute instance that is launched
     *         for model training, specify <code>ShardedByS3Key</code>. If there are <i>n</i> ML compute instances
     *         launched for a training job, each instance gets approximately 1/<i>n</i> of the number of S3 objects. In
     *         this case, model training on each machine uses only the subset of training data.
     *         </p>
     *         <p>
     *         Don't choose more ML compute instances for training than available S3 objects. If you do, some nodes
     *         won't get any data and you will pay for nodes that aren't getting any training data. This applies in both
     *         File and Pipe modes. Keep this in mind when developing algorithms.
     *         </p>
     *         <p>
     *         In distributed training, where you use multiple ML compute EC2 instances, you might choose
     *         <code>ShardedByS3Key</code>. If the algorithm requires copying training data to the ML storage volume
     *         (when <code>TrainingInputMode</code> is set to <code>File</code>), this copies 1/<i>n</i> of the number
     *         of objects.
     * @see S3DataDistribution
     */
    public S3DataDistribution s3DataDistributionType() {
        return S3DataDistribution.fromValue(s3DataDistributionType);
    }

    /**
     * <p>
     * If you want Amazon SageMaker to replicate the entire dataset on each ML compute instance that is launched for
     * model training, specify <code>FullyReplicated</code>.
     * </p>
     * <p>
     * If you want Amazon SageMaker to replicate a subset of data on each ML compute instance that is launched for model
     * training, specify <code>ShardedByS3Key</code>. If there are <i>n</i> ML compute instances launched for a training
     * job, each instance gets approximately 1/<i>n</i> of the number of S3 objects. In this case, model training on
     * each machine uses only the subset of training data.
     * </p>
     * <p>
     * Don't choose more ML compute instances for training than available S3 objects. If you do, some nodes won't get
     * any data and you will pay for nodes that aren't getting any training data. This applies in both File and Pipe
     * modes. Keep this in mind when developing algorithms.
     * </p>
     * <p>
     * In distributed training, where you use multiple ML compute EC2 instances, you might choose
     * <code>ShardedByS3Key</code>. If the algorithm requires copying training data to the ML storage volume (when
     * <code>TrainingInputMode</code> is set to <code>File</code>), this copies 1/<i>n</i> of the number of objects.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version,
     * {@link #s3DataDistributionType} will return {@link S3DataDistribution#UNKNOWN_TO_SDK_VERSION}. The raw value
     * returned by the service is available from {@link #s3DataDistributionTypeAsString}.
     * </p>
     * 
     * @return If you want Amazon SageMaker to replicate the entire dataset on each ML compute instance that is launched
     *         for model training, specify <code>FullyReplicated</code>. </p>
     *         <p>
     *         If you want Amazon SageMaker to replicate a subset of data on each ML compute instance that is launched
     *         for model training, specify <code>ShardedByS3Key</code>. If there are <i>n</i> ML compute instances
     *         launched for a training job, each instance gets approximately 1/<i>n</i> of the number of S3 objects. In
     *         this case, model training on each machine uses only the subset of training data.
     *         </p>
     *         <p>
     *         Don't choose more ML compute instances for training than available S3 objects. If you do, some nodes
     *         won't get any data and you will pay for nodes that aren't getting any training data. This applies in both
     *         File and Pipe modes. Keep this in mind when developing algorithms.
     *         </p>
     *         <p>
     *         In distributed training, where you use multiple ML compute EC2 instances, you might choose
     *         <code>ShardedByS3Key</code>. If the algorithm requires copying training data to the ML storage volume
     *         (when <code>TrainingInputMode</code> is set to <code>File</code>), this copies 1/<i>n</i> of the number
     *         of objects.
     * @see S3DataDistribution
     */
    public String s3DataDistributionTypeAsString() {
        return s3DataDistributionType;
    }

    /**
     * Returns true if the AttributeNames property was specified by the sender (it may be empty), or false if the sender
     * did not specify the value (it will be empty). For responses returned by the SDK, the sender is the AWS service.
     */
    public boolean hasAttributeNames() {
        return attributeNames != null && !(attributeNames instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * A list of one or more attribute names to use that are found in a specified augmented manifest file.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasAttributeNames()} to see if a value was sent in this field.
     * </p>
     * 
     * @return A list of one or more attribute names to use that are found in a specified augmented manifest file.
     */
    public List<String> attributeNames() {
        return attributeNames;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(s3DataTypeAsString());
        hashCode = 31 * hashCode + Objects.hashCode(s3Uri());
        hashCode = 31 * hashCode + Objects.hashCode(s3DataDistributionTypeAsString());
        hashCode = 31 * hashCode + Objects.hashCode(attributeNames());
        return hashCode;
    }

    @Override
    public boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof S3DataSource)) {
            return false;
        }
        S3DataSource other = (S3DataSource) obj;
        return Objects.equals(s3DataTypeAsString(), other.s3DataTypeAsString()) && Objects.equals(s3Uri(), other.s3Uri())
                && Objects.equals(s3DataDistributionTypeAsString(), other.s3DataDistributionTypeAsString())
                && Objects.equals(attributeNames(), other.attributeNames());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public String toString() {
        return ToString.builder("S3DataSource").add("S3DataType", s3DataTypeAsString()).add("S3Uri", s3Uri())
                .add("S3DataDistributionType", s3DataDistributionTypeAsString()).add("AttributeNames", attributeNames()).build();
    }

    public <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "S3DataType":
            return Optional.ofNullable(clazz.cast(s3DataTypeAsString()));
        case "S3Uri":
            return Optional.ofNullable(clazz.cast(s3Uri()));
        case "S3DataDistributionType":
            return Optional.ofNullable(clazz.cast(s3DataDistributionTypeAsString()));
        case "AttributeNames":
            return Optional.ofNullable(clazz.cast(attributeNames()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<S3DataSource, T> g) {
        return obj -> g.apply((S3DataSource) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, S3DataSource> {
        /**
         * <p>
         * If you choose <code>S3Prefix</code>, <code>S3Uri</code> identifies a key name prefix. Amazon SageMaker uses
         * all objects that match the specified key name prefix for model training.
         * </p>
         * <p>
         * If you choose <code>ManifestFile</code>, <code>S3Uri</code> identifies an object that is a manifest file
         * containing a list of object keys that you want Amazon SageMaker to use for model training.
         * </p>
         * <p>
         * If you choose <code>AugmentedManifestFile</code>, S3Uri identifies an object that is an augmented manifest
         * file in JSON lines format. This file contains the data you want to use for model training.
         * <code>AugmentedManifestFile</code> can only be used if the Channel's input mode is <code>Pipe</code>.
         * </p>
         * 
         * @param s3DataType
         *        If you choose <code>S3Prefix</code>, <code>S3Uri</code> identifies a key name prefix. Amazon SageMaker
         *        uses all objects that match the specified key name prefix for model training. </p>
         *        <p>
         *        If you choose <code>ManifestFile</code>, <code>S3Uri</code> identifies an object that is a manifest
         *        file containing a list of object keys that you want Amazon SageMaker to use for model training.
         *        </p>
         *        <p>
         *        If you choose <code>AugmentedManifestFile</code>, S3Uri identifies an object that is an augmented
         *        manifest file in JSON lines format. This file contains the data you want to use for model training.
         *        <code>AugmentedManifestFile</code> can only be used if the Channel's input mode is <code>Pipe</code>.
         * @see S3DataType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see S3DataType
         */
        Builder s3DataType(String s3DataType);

        /**
         * <p>
         * If you choose <code>S3Prefix</code>, <code>S3Uri</code> identifies a key name prefix. Amazon SageMaker uses
         * all objects that match the specified key name prefix for model training.
         * </p>
         * <p>
         * If you choose <code>ManifestFile</code>, <code>S3Uri</code> identifies an object that is a manifest file
         * containing a list of object keys that you want Amazon SageMaker to use for model training.
         * </p>
         * <p>
         * If you choose <code>AugmentedManifestFile</code>, S3Uri identifies an object that is an augmented manifest
         * file in JSON lines format. This file contains the data you want to use for model training.
         * <code>AugmentedManifestFile</code> can only be used if the Channel's input mode is <code>Pipe</code>.
         * </p>
         * 
         * @param s3DataType
         *        If you choose <code>S3Prefix</code>, <code>S3Uri</code> identifies a key name prefix. Amazon SageMaker
         *        uses all objects that match the specified key name prefix for model training. </p>
         *        <p>
         *        If you choose <code>ManifestFile</code>, <code>S3Uri</code> identifies an object that is a manifest
         *        file containing a list of object keys that you want Amazon SageMaker to use for model training.
         *        </p>
         *        <p>
         *        If you choose <code>AugmentedManifestFile</code>, S3Uri identifies an object that is an augmented
         *        manifest file in JSON lines format. This file contains the data you want to use for model training.
         *        <code>AugmentedManifestFile</code> can only be used if the Channel's input mode is <code>Pipe</code>.
         * @see S3DataType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see S3DataType
         */
        Builder s3DataType(S3DataType s3DataType);

        /**
         * <p>
         * Depending on the value specified for the <code>S3DataType</code>, identifies either a key name prefix or a
         * manifest. For example:
         * </p>
         * <ul>
         * <li>
         * <p>
         * A key name prefix might look like this: <code>s3://bucketname/exampleprefix</code>.
         * </p>
         * </li>
         * <li>
         * <p>
         * A manifest might look like this: <code>s3://bucketname/example.manifest</code>
         * </p>
         * <p>
         * The manifest is an S3 object which is a JSON file with the following format:
         * </p>
         * <p>
         * The preceding JSON matches the following <code>s3Uris</code>:
         * </p>
         * <p>
         * <code>[ {"prefix": "s3://customer_bucket/some/prefix/"},</code>
         * </p>
         * <p>
         * <code>"relative/path/to/custdata-1",</code>
         * </p>
         * <p>
         * <code>"relative/path/custdata-2",</code>
         * </p>
         * <p>
         * <code>...</code>
         * </p>
         * <p>
         * <code>"relative/path/custdata-N"</code>
         * </p>
         * <p>
         * <code>]</code>
         * </p>
         * <p>
         * The preceding JSON matches the following <code>s3Uris</code>:
         * </p>
         * <p>
         * <code>s3://customer_bucket/some/prefix/relative/path/to/custdata-1</code>
         * </p>
         * <p>
         * <code>s3://customer_bucket/some/prefix/relative/path/custdata-2</code>
         * </p>
         * <p>
         * <code>...</code>
         * </p>
         * <p>
         * <code>s3://customer_bucket/some/prefix/relative/path/custdata-N</code>
         * </p>
         * <p>
         * The complete set of <code>s3uris</code> in this manifest is the input data for the channel for this
         * datasource. The object that each <code>s3uris</code> points to must be readable by the IAM role that Amazon
         * SageMaker uses to perform tasks on your behalf.
         * </p>
         * </li>
         * </ul>
         * 
         * @param s3Uri
         *        Depending on the value specified for the <code>S3DataType</code>, identifies either a key name prefix
         *        or a manifest. For example: </p>
         *        <ul>
         *        <li>
         *        <p>
         *        A key name prefix might look like this: <code>s3://bucketname/exampleprefix</code>.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        A manifest might look like this: <code>s3://bucketname/example.manifest</code>
         *        </p>
         *        <p>
         *        The manifest is an S3 object which is a JSON file with the following format:
         *        </p>
         *        <p>
         *        The preceding JSON matches the following <code>s3Uris</code>:
         *        </p>
         *        <p>
         *        <code>[ {"prefix": "s3://customer_bucket/some/prefix/"},</code>
         *        </p>
         *        <p>
         *        <code>"relative/path/to/custdata-1",</code>
         *        </p>
         *        <p>
         *        <code>"relative/path/custdata-2",</code>
         *        </p>
         *        <p>
         *        <code>...</code>
         *        </p>
         *        <p>
         *        <code>"relative/path/custdata-N"</code>
         *        </p>
         *        <p>
         *        <code>]</code>
         *        </p>
         *        <p>
         *        The preceding JSON matches the following <code>s3Uris</code>:
         *        </p>
         *        <p>
         *        <code>s3://customer_bucket/some/prefix/relative/path/to/custdata-1</code>
         *        </p>
         *        <p>
         *        <code>s3://customer_bucket/some/prefix/relative/path/custdata-2</code>
         *        </p>
         *        <p>
         *        <code>...</code>
         *        </p>
         *        <p>
         *        <code>s3://customer_bucket/some/prefix/relative/path/custdata-N</code>
         *        </p>
         *        <p>
         *        The complete set of <code>s3uris</code> in this manifest is the input data for the channel for this
         *        datasource. The object that each <code>s3uris</code> points to must be readable by the IAM role that
         *        Amazon SageMaker uses to perform tasks on your behalf.
         *        </p>
         *        </li>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder s3Uri(String s3Uri);

        /**
         * <p>
         * If you want Amazon SageMaker to replicate the entire dataset on each ML compute instance that is launched for
         * model training, specify <code>FullyReplicated</code>.
         * </p>
         * <p>
         * If you want Amazon SageMaker to replicate a subset of data on each ML compute instance that is launched for
         * model training, specify <code>ShardedByS3Key</code>. If there are <i>n</i> ML compute instances launched for
         * a training job, each instance gets approximately 1/<i>n</i> of the number of S3 objects. In this case, model
         * training on each machine uses only the subset of training data.
         * </p>
         * <p>
         * Don't choose more ML compute instances for training than available S3 objects. If you do, some nodes won't
         * get any data and you will pay for nodes that aren't getting any training data. This applies in both File and
         * Pipe modes. Keep this in mind when developing algorithms.
         * </p>
         * <p>
         * In distributed training, where you use multiple ML compute EC2 instances, you might choose
         * <code>ShardedByS3Key</code>. If the algorithm requires copying training data to the ML storage volume (when
         * <code>TrainingInputMode</code> is set to <code>File</code>), this copies 1/<i>n</i> of the number of objects.
         * </p>
         * 
         * @param s3DataDistributionType
         *        If you want Amazon SageMaker to replicate the entire dataset on each ML compute instance that is
         *        launched for model training, specify <code>FullyReplicated</code>. </p>
         *        <p>
         *        If you want Amazon SageMaker to replicate a subset of data on each ML compute instance that is
         *        launched for model training, specify <code>ShardedByS3Key</code>. If there are <i>n</i> ML compute
         *        instances launched for a training job, each instance gets approximately 1/<i>n</i> of the number of S3
         *        objects. In this case, model training on each machine uses only the subset of training data.
         *        </p>
         *        <p>
         *        Don't choose more ML compute instances for training than available S3 objects. If you do, some nodes
         *        won't get any data and you will pay for nodes that aren't getting any training data. This applies in
         *        both File and Pipe modes. Keep this in mind when developing algorithms.
         *        </p>
         *        <p>
         *        In distributed training, where you use multiple ML compute EC2 instances, you might choose
         *        <code>ShardedByS3Key</code>. If the algorithm requires copying training data to the ML storage volume
         *        (when <code>TrainingInputMode</code> is set to <code>File</code>), this copies 1/<i>n</i> of the
         *        number of objects.
         * @see S3DataDistribution
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see S3DataDistribution
         */
        Builder s3DataDistributionType(String s3DataDistributionType);

        /**
         * <p>
         * If you want Amazon SageMaker to replicate the entire dataset on each ML compute instance that is launched for
         * model training, specify <code>FullyReplicated</code>.
         * </p>
         * <p>
         * If you want Amazon SageMaker to replicate a subset of data on each ML compute instance that is launched for
         * model training, specify <code>ShardedByS3Key</code>. If there are <i>n</i> ML compute instances launched for
         * a training job, each instance gets approximately 1/<i>n</i> of the number of S3 objects. In this case, model
         * training on each machine uses only the subset of training data.
         * </p>
         * <p>
         * Don't choose more ML compute instances for training than available S3 objects. If you do, some nodes won't
         * get any data and you will pay for nodes that aren't getting any training data. This applies in both File and
         * Pipe modes. Keep this in mind when developing algorithms.
         * </p>
         * <p>
         * In distributed training, where you use multiple ML compute EC2 instances, you might choose
         * <code>ShardedByS3Key</code>. If the algorithm requires copying training data to the ML storage volume (when
         * <code>TrainingInputMode</code> is set to <code>File</code>), this copies 1/<i>n</i> of the number of objects.
         * </p>
         * 
         * @param s3DataDistributionType
         *        If you want Amazon SageMaker to replicate the entire dataset on each ML compute instance that is
         *        launched for model training, specify <code>FullyReplicated</code>. </p>
         *        <p>
         *        If you want Amazon SageMaker to replicate a subset of data on each ML compute instance that is
         *        launched for model training, specify <code>ShardedByS3Key</code>. If there are <i>n</i> ML compute
         *        instances launched for a training job, each instance gets approximately 1/<i>n</i> of the number of S3
         *        objects. In this case, model training on each machine uses only the subset of training data.
         *        </p>
         *        <p>
         *        Don't choose more ML compute instances for training than available S3 objects. If you do, some nodes
         *        won't get any data and you will pay for nodes that aren't getting any training data. This applies in
         *        both File and Pipe modes. Keep this in mind when developing algorithms.
         *        </p>
         *        <p>
         *        In distributed training, where you use multiple ML compute EC2 instances, you might choose
         *        <code>ShardedByS3Key</code>. If the algorithm requires copying training data to the ML storage volume
         *        (when <code>TrainingInputMode</code> is set to <code>File</code>), this copies 1/<i>n</i> of the
         *        number of objects.
         * @see S3DataDistribution
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see S3DataDistribution
         */
        Builder s3DataDistributionType(S3DataDistribution s3DataDistributionType);

        /**
         * <p>
         * A list of one or more attribute names to use that are found in a specified augmented manifest file.
         * </p>
         * 
         * @param attributeNames
         *        A list of one or more attribute names to use that are found in a specified augmented manifest file.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder attributeNames(Collection<String> attributeNames);

        /**
         * <p>
         * A list of one or more attribute names to use that are found in a specified augmented manifest file.
         * </p>
         * 
         * @param attributeNames
         *        A list of one or more attribute names to use that are found in a specified augmented manifest file.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder attributeNames(String... attributeNames);
    }

    static final class BuilderImpl implements Builder {
        private String s3DataType;

        private String s3Uri;

        private String s3DataDistributionType;

        private List<String> attributeNames = DefaultSdkAutoConstructList.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(S3DataSource model) {
            s3DataType(model.s3DataType);
            s3Uri(model.s3Uri);
            s3DataDistributionType(model.s3DataDistributionType);
            attributeNames(model.attributeNames);
        }

        public final String getS3DataTypeAsString() {
            return s3DataType;
        }

        @Override
        public final Builder s3DataType(String s3DataType) {
            this.s3DataType = s3DataType;
            return this;
        }

        @Override
        public final Builder s3DataType(S3DataType s3DataType) {
            this.s3DataType(s3DataType == null ? null : s3DataType.toString());
            return this;
        }

        public final void setS3DataType(String s3DataType) {
            this.s3DataType = s3DataType;
        }

        public final String getS3Uri() {
            return s3Uri;
        }

        @Override
        public final Builder s3Uri(String s3Uri) {
            this.s3Uri = s3Uri;
            return this;
        }

        public final void setS3Uri(String s3Uri) {
            this.s3Uri = s3Uri;
        }

        public final String getS3DataDistributionTypeAsString() {
            return s3DataDistributionType;
        }

        @Override
        public final Builder s3DataDistributionType(String s3DataDistributionType) {
            this.s3DataDistributionType = s3DataDistributionType;
            return this;
        }

        @Override
        public final Builder s3DataDistributionType(S3DataDistribution s3DataDistributionType) {
            this.s3DataDistributionType(s3DataDistributionType == null ? null : s3DataDistributionType.toString());
            return this;
        }

        public final void setS3DataDistributionType(String s3DataDistributionType) {
            this.s3DataDistributionType = s3DataDistributionType;
        }

        public final Collection<String> getAttributeNames() {
            return attributeNames;
        }

        @Override
        public final Builder attributeNames(Collection<String> attributeNames) {
            this.attributeNames = AttributeNamesCopier.copy(attributeNames);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder attributeNames(String... attributeNames) {
            attributeNames(Arrays.asList(attributeNames));
            return this;
        }

        public final void setAttributeNames(Collection<String> attributeNames) {
            this.attributeNames = AttributeNamesCopier.copy(attributeNames);
        }

        @Override
        public S3DataSource build() {
            return new S3DataSource(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
