/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.s3.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.Mutable;
import software.amazon.awssdk.annotations.NotThreadSafe;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * A bucket-level setting for Amazon S3 general purpose buckets used to prevent the upload of new objects encrypted with
 * the specified server-side encryption type. For example, blocking an encryption type will block <code>PutObject</code>, <code>CopyObject</code>, <code>PostObject</code>, multipart upload, and replication requests to the bucket for
 * objects with the specified encryption type. However, you can continue to read and list any pre-existing objects
 * already encrypted with the specified encryption type. For more information, see <a
 * href="https://docs.aws.amazon.com/AmazonS3/userguide/block-encryption-type.html">Blocking an encryption type for a
 * general purpose bucket</a>.
 * </p>
 * <p>
 * This data type is used with the following actions:
 * </p>
 * <ul>
 * <li>
 * <p>
 * <a href="https://docs.aws.amazon.com/AmazonS3/latest/API/API_PutBucketEncryption.html">PutBucketEncryption</a>
 * </p>
 * </li>
 * <li>
 * <p>
 * <a href="https://docs.aws.amazon.com/AmazonS3/latest/API/API_GetBucketEncryption.html">GetBucketEncryption</a>
 * </p>
 * </li>
 * <li>
 * <p>
 * <a href="https://docs.aws.amazon.com/AmazonS3/latest/API/API_DeleteBucketEncryption.html">DeleteBucketEncryption</a>
 * </p>
 * </li>
 * </ul>
 * <dl>
 * <dt>Permissions</dt>
 * <dd>
 * <p>
 * You must have the <code>s3:PutEncryptionConfiguration</code> permission to block or unblock an encryption type for a
 * bucket.
 * </p>
 * <p>
 * You must have the <code>s3:GetEncryptionConfiguration</code> permission to view a bucket's encryption type.
 * </p>
 * </dd>
 * </dl>
 */
@Generated("software.amazon.awssdk:codegen")
public final class BlockedEncryptionTypes implements SdkPojo, Serializable,
        ToCopyableBuilder<BlockedEncryptionTypes.Builder, BlockedEncryptionTypes> {
    private static final SdkField<List<String>> ENCRYPTION_TYPE_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("EncryptionType")
            .getter(getter(BlockedEncryptionTypes::encryptionTypeAsStrings))
            .setter(setter(Builder::encryptionTypeWithStrings))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("EncryptionType")
                    .unmarshallLocationName("EncryptionType").build(),
                    ListTrait
                            .builder()
                            .memberLocationName("EncryptionType")
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("EncryptionType").unmarshallLocationName("EncryptionType")
                                                    .build()).build()).isFlattened(true).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(ENCRYPTION_TYPE_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = memberNameToFieldInitializer();

    private static final long serialVersionUID = 1L;

    private final List<String> encryptionType;

    private BlockedEncryptionTypes(BuilderImpl builder) {
        this.encryptionType = builder.encryptionType;
    }

    /**
     * <p>
     * The object encryption type that you want to block or unblock for an Amazon S3 general purpose bucket.
     * </p>
     * <note>
     * <p>
     * Currently, this parameter only supports blocking or unblocking server side encryption with customer-provided keys
     * (SSE-C). For more information about SSE-C, see <a
     * href="https://docs.aws.amazon.com/AmazonS3/latest/userguide/ServerSideEncryptionCustomerKeys.html">Using
     * server-side encryption with customer-provided keys (SSE-C)</a>.
     * </p>
     * </note>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasEncryptionType} method.
     * </p>
     * 
     * @return The object encryption type that you want to block or unblock for an Amazon S3 general purpose bucket.</p>
     *         <note>
     *         <p>
     *         Currently, this parameter only supports blocking or unblocking server side encryption with
     *         customer-provided keys (SSE-C). For more information about SSE-C, see <a
     *         href="https://docs.aws.amazon.com/AmazonS3/latest/userguide/ServerSideEncryptionCustomerKeys.html">Using
     *         server-side encryption with customer-provided keys (SSE-C)</a>.
     *         </p>
     */
    public final List<EncryptionType> encryptionType() {
        return EncryptionTypeListCopier.copyStringToEnum(encryptionType);
    }

    /**
     * For responses, this returns true if the service returned a value for the EncryptionType property. This DOES NOT
     * check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasEncryptionType() {
        return encryptionType != null && !(encryptionType instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The object encryption type that you want to block or unblock for an Amazon S3 general purpose bucket.
     * </p>
     * <note>
     * <p>
     * Currently, this parameter only supports blocking or unblocking server side encryption with customer-provided keys
     * (SSE-C). For more information about SSE-C, see <a
     * href="https://docs.aws.amazon.com/AmazonS3/latest/userguide/ServerSideEncryptionCustomerKeys.html">Using
     * server-side encryption with customer-provided keys (SSE-C)</a>.
     * </p>
     * </note>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasEncryptionType} method.
     * </p>
     * 
     * @return The object encryption type that you want to block or unblock for an Amazon S3 general purpose bucket.</p>
     *         <note>
     *         <p>
     *         Currently, this parameter only supports blocking or unblocking server side encryption with
     *         customer-provided keys (SSE-C). For more information about SSE-C, see <a
     *         href="https://docs.aws.amazon.com/AmazonS3/latest/userguide/ServerSideEncryptionCustomerKeys.html">Using
     *         server-side encryption with customer-provided keys (SSE-C)</a>.
     *         </p>
     */
    public final List<String> encryptionTypeAsStrings() {
        return encryptionType;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(hasEncryptionType() ? encryptionTypeAsStrings() : null);
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof BlockedEncryptionTypes)) {
            return false;
        }
        BlockedEncryptionTypes other = (BlockedEncryptionTypes) obj;
        return hasEncryptionType() == other.hasEncryptionType()
                && Objects.equals(encryptionTypeAsStrings(), other.encryptionTypeAsStrings());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("BlockedEncryptionTypes")
                .add("EncryptionType", hasEncryptionType() ? encryptionTypeAsStrings() : null).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "EncryptionType":
            return Optional.ofNullable(clazz.cast(encryptionTypeAsStrings()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static Map<String, SdkField<?>> memberNameToFieldInitializer() {
        Map<String, SdkField<?>> map = new HashMap<>();
        map.put("EncryptionType", ENCRYPTION_TYPE_FIELD);
        return Collections.unmodifiableMap(map);
    }

    private static <T> Function<Object, T> getter(Function<BlockedEncryptionTypes, T> g) {
        return obj -> g.apply((BlockedEncryptionTypes) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    @Mutable
    @NotThreadSafe
    public interface Builder extends SdkPojo, CopyableBuilder<Builder, BlockedEncryptionTypes> {
        /**
         * <p>
         * The object encryption type that you want to block or unblock for an Amazon S3 general purpose bucket.
         * </p>
         * <note>
         * <p>
         * Currently, this parameter only supports blocking or unblocking server side encryption with customer-provided
         * keys (SSE-C). For more information about SSE-C, see <a
         * href="https://docs.aws.amazon.com/AmazonS3/latest/userguide/ServerSideEncryptionCustomerKeys.html">Using
         * server-side encryption with customer-provided keys (SSE-C)</a>.
         * </p>
         * </note>
         * 
         * @param encryptionType
         *        The object encryption type that you want to block or unblock for an Amazon S3 general purpose
         *        bucket.</p> <note>
         *        <p>
         *        Currently, this parameter only supports blocking or unblocking server side encryption with
         *        customer-provided keys (SSE-C). For more information about SSE-C, see <a
         *        href="https://docs.aws.amazon.com/AmazonS3/latest/userguide/ServerSideEncryptionCustomerKeys.html"
         *        >Using server-side encryption with customer-provided keys (SSE-C)</a>.
         *        </p>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder encryptionTypeWithStrings(Collection<String> encryptionType);

        /**
         * <p>
         * The object encryption type that you want to block or unblock for an Amazon S3 general purpose bucket.
         * </p>
         * <note>
         * <p>
         * Currently, this parameter only supports blocking or unblocking server side encryption with customer-provided
         * keys (SSE-C). For more information about SSE-C, see <a
         * href="https://docs.aws.amazon.com/AmazonS3/latest/userguide/ServerSideEncryptionCustomerKeys.html">Using
         * server-side encryption with customer-provided keys (SSE-C)</a>.
         * </p>
         * </note>
         * 
         * @param encryptionType
         *        The object encryption type that you want to block or unblock for an Amazon S3 general purpose
         *        bucket.</p> <note>
         *        <p>
         *        Currently, this parameter only supports blocking or unblocking server side encryption with
         *        customer-provided keys (SSE-C). For more information about SSE-C, see <a
         *        href="https://docs.aws.amazon.com/AmazonS3/latest/userguide/ServerSideEncryptionCustomerKeys.html"
         *        >Using server-side encryption with customer-provided keys (SSE-C)</a>.
         *        </p>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder encryptionTypeWithStrings(String... encryptionType);

        /**
         * <p>
         * The object encryption type that you want to block or unblock for an Amazon S3 general purpose bucket.
         * </p>
         * <note>
         * <p>
         * Currently, this parameter only supports blocking or unblocking server side encryption with customer-provided
         * keys (SSE-C). For more information about SSE-C, see <a
         * href="https://docs.aws.amazon.com/AmazonS3/latest/userguide/ServerSideEncryptionCustomerKeys.html">Using
         * server-side encryption with customer-provided keys (SSE-C)</a>.
         * </p>
         * </note>
         * 
         * @param encryptionType
         *        The object encryption type that you want to block or unblock for an Amazon S3 general purpose
         *        bucket.</p> <note>
         *        <p>
         *        Currently, this parameter only supports blocking or unblocking server side encryption with
         *        customer-provided keys (SSE-C). For more information about SSE-C, see <a
         *        href="https://docs.aws.amazon.com/AmazonS3/latest/userguide/ServerSideEncryptionCustomerKeys.html"
         *        >Using server-side encryption with customer-provided keys (SSE-C)</a>.
         *        </p>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder encryptionType(Collection<EncryptionType> encryptionType);

        /**
         * <p>
         * The object encryption type that you want to block or unblock for an Amazon S3 general purpose bucket.
         * </p>
         * <note>
         * <p>
         * Currently, this parameter only supports blocking or unblocking server side encryption with customer-provided
         * keys (SSE-C). For more information about SSE-C, see <a
         * href="https://docs.aws.amazon.com/AmazonS3/latest/userguide/ServerSideEncryptionCustomerKeys.html">Using
         * server-side encryption with customer-provided keys (SSE-C)</a>.
         * </p>
         * </note>
         * 
         * @param encryptionType
         *        The object encryption type that you want to block or unblock for an Amazon S3 general purpose
         *        bucket.</p> <note>
         *        <p>
         *        Currently, this parameter only supports blocking or unblocking server side encryption with
         *        customer-provided keys (SSE-C). For more information about SSE-C, see <a
         *        href="https://docs.aws.amazon.com/AmazonS3/latest/userguide/ServerSideEncryptionCustomerKeys.html"
         *        >Using server-side encryption with customer-provided keys (SSE-C)</a>.
         *        </p>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder encryptionType(EncryptionType... encryptionType);
    }

    static final class BuilderImpl implements Builder {
        private List<String> encryptionType = DefaultSdkAutoConstructList.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(BlockedEncryptionTypes model) {
            encryptionTypeWithStrings(model.encryptionType);
        }

        public final Collection<String> getEncryptionType() {
            if (encryptionType instanceof SdkAutoConstructList) {
                return null;
            }
            return encryptionType;
        }

        public final void setEncryptionType(Collection<String> encryptionType) {
            this.encryptionType = EncryptionTypeListCopier.copy(encryptionType);
        }

        @Override
        public final Builder encryptionTypeWithStrings(Collection<String> encryptionType) {
            this.encryptionType = EncryptionTypeListCopier.copy(encryptionType);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder encryptionTypeWithStrings(String... encryptionType) {
            encryptionTypeWithStrings(Arrays.asList(encryptionType));
            return this;
        }

        @Override
        public final Builder encryptionType(Collection<EncryptionType> encryptionType) {
            this.encryptionType = EncryptionTypeListCopier.copyEnumToString(encryptionType);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder encryptionType(EncryptionType... encryptionType) {
            encryptionType(Arrays.asList(encryptionType));
            return this;
        }

        @Override
        public BlockedEncryptionTypes build() {
            return new BlockedEncryptionTypes(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
