/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.robomaker.model;

import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.awscore.AwsRequestOverrideConfiguration;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 */
@Generated("software.amazon.awssdk:codegen")
public final class CreateSimulationApplicationVersionRequest extends RoboMakerRequest implements
        ToCopyableBuilder<CreateSimulationApplicationVersionRequest.Builder, CreateSimulationApplicationVersionRequest> {
    private static final SdkField<String> APPLICATION_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("application").getter(getter(CreateSimulationApplicationVersionRequest::application))
            .setter(setter(Builder::application))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("application").build()).build();

    private static final SdkField<String> CURRENT_REVISION_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("currentRevisionId").getter(getter(CreateSimulationApplicationVersionRequest::currentRevisionId))
            .setter(setter(Builder::currentRevisionId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("currentRevisionId").build()).build();

    private static final SdkField<List<String>> S3_ETAGS_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("s3Etags")
            .getter(getter(CreateSimulationApplicationVersionRequest::s3Etags))
            .setter(setter(Builder::s3Etags))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("s3Etags").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<String> IMAGE_DIGEST_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("imageDigest").getter(getter(CreateSimulationApplicationVersionRequest::imageDigest))
            .setter(setter(Builder::imageDigest))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("imageDigest").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(APPLICATION_FIELD,
            CURRENT_REVISION_ID_FIELD, S3_ETAGS_FIELD, IMAGE_DIGEST_FIELD));

    private final String application;

    private final String currentRevisionId;

    private final List<String> s3Etags;

    private final String imageDigest;

    private CreateSimulationApplicationVersionRequest(BuilderImpl builder) {
        super(builder);
        this.application = builder.application;
        this.currentRevisionId = builder.currentRevisionId;
        this.s3Etags = builder.s3Etags;
        this.imageDigest = builder.imageDigest;
    }

    /**
     * <p>
     * The application information for the simulation application.
     * </p>
     * 
     * @return The application information for the simulation application.
     */
    public final String application() {
        return application;
    }

    /**
     * <p>
     * The current revision id for the simulation application. If you provide a value and it matches the latest revision
     * ID, a new version will be created.
     * </p>
     * 
     * @return The current revision id for the simulation application. If you provide a value and it matches the latest
     *         revision ID, a new version will be created.
     */
    public final String currentRevisionId() {
        return currentRevisionId;
    }

    /**
     * For responses, this returns true if the service returned a value for the S3Etags property. This DOES NOT check
     * that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is
     * useful because the SDK will never return a null collection or map, but you may need to differentiate between the
     * service returning nothing (or null) and the service returning an empty collection or map. For requests, this
     * returns true if a value for the property was specified in the request builder, and false if a value was not
     * specified.
     */
    public final boolean hasS3Etags() {
        return s3Etags != null && !(s3Etags instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The Amazon S3 eTag identifier for the zip file bundle that you use to create the simulation application.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasS3Etags} method.
     * </p>
     * 
     * @return The Amazon S3 eTag identifier for the zip file bundle that you use to create the simulation application.
     */
    public final List<String> s3Etags() {
        return s3Etags;
    }

    /**
     * <p>
     * The SHA256 digest used to identify the Docker image URI used to created the simulation application.
     * </p>
     * 
     * @return The SHA256 digest used to identify the Docker image URI used to created the simulation application.
     */
    public final String imageDigest() {
        return imageDigest;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + super.hashCode();
        hashCode = 31 * hashCode + Objects.hashCode(application());
        hashCode = 31 * hashCode + Objects.hashCode(currentRevisionId());
        hashCode = 31 * hashCode + Objects.hashCode(hasS3Etags() ? s3Etags() : null);
        hashCode = 31 * hashCode + Objects.hashCode(imageDigest());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return super.equals(obj) && equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof CreateSimulationApplicationVersionRequest)) {
            return false;
        }
        CreateSimulationApplicationVersionRequest other = (CreateSimulationApplicationVersionRequest) obj;
        return Objects.equals(application(), other.application())
                && Objects.equals(currentRevisionId(), other.currentRevisionId()) && hasS3Etags() == other.hasS3Etags()
                && Objects.equals(s3Etags(), other.s3Etags()) && Objects.equals(imageDigest(), other.imageDigest());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("CreateSimulationApplicationVersionRequest").add("Application", application())
                .add("CurrentRevisionId", currentRevisionId()).add("S3Etags", hasS3Etags() ? s3Etags() : null)
                .add("ImageDigest", imageDigest()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "application":
            return Optional.ofNullable(clazz.cast(application()));
        case "currentRevisionId":
            return Optional.ofNullable(clazz.cast(currentRevisionId()));
        case "s3Etags":
            return Optional.ofNullable(clazz.cast(s3Etags()));
        case "imageDigest":
            return Optional.ofNullable(clazz.cast(imageDigest()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<CreateSimulationApplicationVersionRequest, T> g) {
        return obj -> g.apply((CreateSimulationApplicationVersionRequest) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends RoboMakerRequest.Builder, SdkPojo,
            CopyableBuilder<Builder, CreateSimulationApplicationVersionRequest> {
        /**
         * <p>
         * The application information for the simulation application.
         * </p>
         * 
         * @param application
         *        The application information for the simulation application.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder application(String application);

        /**
         * <p>
         * The current revision id for the simulation application. If you provide a value and it matches the latest
         * revision ID, a new version will be created.
         * </p>
         * 
         * @param currentRevisionId
         *        The current revision id for the simulation application. If you provide a value and it matches the
         *        latest revision ID, a new version will be created.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder currentRevisionId(String currentRevisionId);

        /**
         * <p>
         * The Amazon S3 eTag identifier for the zip file bundle that you use to create the simulation application.
         * </p>
         * 
         * @param s3Etags
         *        The Amazon S3 eTag identifier for the zip file bundle that you use to create the simulation
         *        application.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder s3Etags(Collection<String> s3Etags);

        /**
         * <p>
         * The Amazon S3 eTag identifier for the zip file bundle that you use to create the simulation application.
         * </p>
         * 
         * @param s3Etags
         *        The Amazon S3 eTag identifier for the zip file bundle that you use to create the simulation
         *        application.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder s3Etags(String... s3Etags);

        /**
         * <p>
         * The SHA256 digest used to identify the Docker image URI used to created the simulation application.
         * </p>
         * 
         * @param imageDigest
         *        The SHA256 digest used to identify the Docker image URI used to created the simulation application.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder imageDigest(String imageDigest);

        @Override
        Builder overrideConfiguration(AwsRequestOverrideConfiguration overrideConfiguration);

        @Override
        Builder overrideConfiguration(Consumer<AwsRequestOverrideConfiguration.Builder> builderConsumer);
    }

    static final class BuilderImpl extends RoboMakerRequest.BuilderImpl implements Builder {
        private String application;

        private String currentRevisionId;

        private List<String> s3Etags = DefaultSdkAutoConstructList.getInstance();

        private String imageDigest;

        private BuilderImpl() {
        }

        private BuilderImpl(CreateSimulationApplicationVersionRequest model) {
            super(model);
            application(model.application);
            currentRevisionId(model.currentRevisionId);
            s3Etags(model.s3Etags);
            imageDigest(model.imageDigest);
        }

        public final String getApplication() {
            return application;
        }

        public final void setApplication(String application) {
            this.application = application;
        }

        @Override
        public final Builder application(String application) {
            this.application = application;
            return this;
        }

        public final String getCurrentRevisionId() {
            return currentRevisionId;
        }

        public final void setCurrentRevisionId(String currentRevisionId) {
            this.currentRevisionId = currentRevisionId;
        }

        @Override
        public final Builder currentRevisionId(String currentRevisionId) {
            this.currentRevisionId = currentRevisionId;
            return this;
        }

        public final Collection<String> getS3Etags() {
            if (s3Etags instanceof SdkAutoConstructList) {
                return null;
            }
            return s3Etags;
        }

        public final void setS3Etags(Collection<String> s3Etags) {
            this.s3Etags = S3EtagsCopier.copy(s3Etags);
        }

        @Override
        public final Builder s3Etags(Collection<String> s3Etags) {
            this.s3Etags = S3EtagsCopier.copy(s3Etags);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder s3Etags(String... s3Etags) {
            s3Etags(Arrays.asList(s3Etags));
            return this;
        }

        public final String getImageDigest() {
            return imageDigest;
        }

        public final void setImageDigest(String imageDigest) {
            this.imageDigest = imageDigest;
        }

        @Override
        public final Builder imageDigest(String imageDigest) {
            this.imageDigest = imageDigest;
            return this;
        }

        @Override
        public Builder overrideConfiguration(AwsRequestOverrideConfiguration overrideConfiguration) {
            super.overrideConfiguration(overrideConfiguration);
            return this;
        }

        @Override
        public Builder overrideConfiguration(Consumer<AwsRequestOverrideConfiguration.Builder> builderConsumer) {
            super.overrideConfiguration(builderConsumer);
            return this;
        }

        @Override
        public CreateSimulationApplicationVersionRequest build() {
            return new CreateSimulationApplicationVersionRequest(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
