/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.redshift.model;

import java.io.Serializable;
import java.time.Instant;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Describes an event.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class Event implements SdkPojo, Serializable, ToCopyableBuilder<Event.Builder, Event> {
    private static final SdkField<String> SOURCE_IDENTIFIER_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("SourceIdentifier").getter(getter(Event::sourceIdentifier)).setter(setter(Builder::sourceIdentifier))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("SourceIdentifier").build()).build();

    private static final SdkField<String> SOURCE_TYPE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("SourceType").getter(getter(Event::sourceTypeAsString)).setter(setter(Builder::sourceType))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("SourceType").build()).build();

    private static final SdkField<String> MESSAGE_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Message")
            .getter(getter(Event::message)).setter(setter(Builder::message))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Message").build()).build();

    private static final SdkField<List<String>> EVENT_CATEGORIES_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("EventCategories")
            .getter(getter(Event::eventCategories))
            .setter(setter(Builder::eventCategories))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("EventCategories").build(),
                    ListTrait
                            .builder()
                            .memberLocationName("EventCategory")
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("EventCategory").build()).build()).build()).build();

    private static final SdkField<String> SEVERITY_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("Severity").getter(getter(Event::severity)).setter(setter(Builder::severity))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Severity").build()).build();

    private static final SdkField<Instant> DATE_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT).memberName("Date")
            .getter(getter(Event::date)).setter(setter(Builder::date))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Date").build()).build();

    private static final SdkField<String> EVENT_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("EventId").getter(getter(Event::eventId)).setter(setter(Builder::eventId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("EventId").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(SOURCE_IDENTIFIER_FIELD,
            SOURCE_TYPE_FIELD, MESSAGE_FIELD, EVENT_CATEGORIES_FIELD, SEVERITY_FIELD, DATE_FIELD, EVENT_ID_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = memberNameToFieldInitializer();

    private static final long serialVersionUID = 1L;

    private final String sourceIdentifier;

    private final String sourceType;

    private final String message;

    private final List<String> eventCategories;

    private final String severity;

    private final Instant date;

    private final String eventId;

    private Event(BuilderImpl builder) {
        this.sourceIdentifier = builder.sourceIdentifier;
        this.sourceType = builder.sourceType;
        this.message = builder.message;
        this.eventCategories = builder.eventCategories;
        this.severity = builder.severity;
        this.date = builder.date;
        this.eventId = builder.eventId;
    }

    /**
     * <p>
     * The identifier for the source of the event.
     * </p>
     * 
     * @return The identifier for the source of the event.
     */
    public final String sourceIdentifier() {
        return sourceIdentifier;
    }

    /**
     * <p>
     * The source type for this event.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #sourceType} will
     * return {@link SourceType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #sourceTypeAsString}.
     * </p>
     * 
     * @return The source type for this event.
     * @see SourceType
     */
    public final SourceType sourceType() {
        return SourceType.fromValue(sourceType);
    }

    /**
     * <p>
     * The source type for this event.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #sourceType} will
     * return {@link SourceType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #sourceTypeAsString}.
     * </p>
     * 
     * @return The source type for this event.
     * @see SourceType
     */
    public final String sourceTypeAsString() {
        return sourceType;
    }

    /**
     * <p>
     * The text of this event.
     * </p>
     * 
     * @return The text of this event.
     */
    public final String message() {
        return message;
    }

    /**
     * For responses, this returns true if the service returned a value for the EventCategories property. This DOES NOT
     * check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasEventCategories() {
        return eventCategories != null && !(eventCategories instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * A list of the event categories.
     * </p>
     * <p>
     * Values: Configuration, Management, Monitoring, Security, Pending
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasEventCategories} method.
     * </p>
     * 
     * @return A list of the event categories.</p>
     *         <p>
     *         Values: Configuration, Management, Monitoring, Security, Pending
     */
    public final List<String> eventCategories() {
        return eventCategories;
    }

    /**
     * <p>
     * The severity of the event.
     * </p>
     * <p>
     * Values: ERROR, INFO
     * </p>
     * 
     * @return The severity of the event.</p>
     *         <p>
     *         Values: ERROR, INFO
     */
    public final String severity() {
        return severity;
    }

    /**
     * <p>
     * The date and time of the event.
     * </p>
     * 
     * @return The date and time of the event.
     */
    public final Instant date() {
        return date;
    }

    /**
     * <p>
     * The identifier of the event.
     * </p>
     * 
     * @return The identifier of the event.
     */
    public final String eventId() {
        return eventId;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(sourceIdentifier());
        hashCode = 31 * hashCode + Objects.hashCode(sourceTypeAsString());
        hashCode = 31 * hashCode + Objects.hashCode(message());
        hashCode = 31 * hashCode + Objects.hashCode(hasEventCategories() ? eventCategories() : null);
        hashCode = 31 * hashCode + Objects.hashCode(severity());
        hashCode = 31 * hashCode + Objects.hashCode(date());
        hashCode = 31 * hashCode + Objects.hashCode(eventId());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof Event)) {
            return false;
        }
        Event other = (Event) obj;
        return Objects.equals(sourceIdentifier(), other.sourceIdentifier())
                && Objects.equals(sourceTypeAsString(), other.sourceTypeAsString()) && Objects.equals(message(), other.message())
                && hasEventCategories() == other.hasEventCategories()
                && Objects.equals(eventCategories(), other.eventCategories()) && Objects.equals(severity(), other.severity())
                && Objects.equals(date(), other.date()) && Objects.equals(eventId(), other.eventId());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("Event").add("SourceIdentifier", sourceIdentifier()).add("SourceType", sourceTypeAsString())
                .add("Message", message()).add("EventCategories", hasEventCategories() ? eventCategories() : null)
                .add("Severity", severity()).add("Date", date()).add("EventId", eventId()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "SourceIdentifier":
            return Optional.ofNullable(clazz.cast(sourceIdentifier()));
        case "SourceType":
            return Optional.ofNullable(clazz.cast(sourceTypeAsString()));
        case "Message":
            return Optional.ofNullable(clazz.cast(message()));
        case "EventCategories":
            return Optional.ofNullable(clazz.cast(eventCategories()));
        case "Severity":
            return Optional.ofNullable(clazz.cast(severity()));
        case "Date":
            return Optional.ofNullable(clazz.cast(date()));
        case "EventId":
            return Optional.ofNullable(clazz.cast(eventId()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static Map<String, SdkField<?>> memberNameToFieldInitializer() {
        Map<String, SdkField<?>> map = new HashMap<>();
        map.put("SourceIdentifier", SOURCE_IDENTIFIER_FIELD);
        map.put("SourceType", SOURCE_TYPE_FIELD);
        map.put("Message", MESSAGE_FIELD);
        map.put("EventCategories", EVENT_CATEGORIES_FIELD);
        map.put("Severity", SEVERITY_FIELD);
        map.put("Date", DATE_FIELD);
        map.put("EventId", EVENT_ID_FIELD);
        return Collections.unmodifiableMap(map);
    }

    private static <T> Function<Object, T> getter(Function<Event, T> g) {
        return obj -> g.apply((Event) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, Event> {
        /**
         * <p>
         * The identifier for the source of the event.
         * </p>
         * 
         * @param sourceIdentifier
         *        The identifier for the source of the event.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder sourceIdentifier(String sourceIdentifier);

        /**
         * <p>
         * The source type for this event.
         * </p>
         * 
         * @param sourceType
         *        The source type for this event.
         * @see SourceType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see SourceType
         */
        Builder sourceType(String sourceType);

        /**
         * <p>
         * The source type for this event.
         * </p>
         * 
         * @param sourceType
         *        The source type for this event.
         * @see SourceType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see SourceType
         */
        Builder sourceType(SourceType sourceType);

        /**
         * <p>
         * The text of this event.
         * </p>
         * 
         * @param message
         *        The text of this event.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder message(String message);

        /**
         * <p>
         * A list of the event categories.
         * </p>
         * <p>
         * Values: Configuration, Management, Monitoring, Security, Pending
         * </p>
         * 
         * @param eventCategories
         *        A list of the event categories.</p>
         *        <p>
         *        Values: Configuration, Management, Monitoring, Security, Pending
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder eventCategories(Collection<String> eventCategories);

        /**
         * <p>
         * A list of the event categories.
         * </p>
         * <p>
         * Values: Configuration, Management, Monitoring, Security, Pending
         * </p>
         * 
         * @param eventCategories
         *        A list of the event categories.</p>
         *        <p>
         *        Values: Configuration, Management, Monitoring, Security, Pending
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder eventCategories(String... eventCategories);

        /**
         * <p>
         * The severity of the event.
         * </p>
         * <p>
         * Values: ERROR, INFO
         * </p>
         * 
         * @param severity
         *        The severity of the event.</p>
         *        <p>
         *        Values: ERROR, INFO
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder severity(String severity);

        /**
         * <p>
         * The date and time of the event.
         * </p>
         * 
         * @param date
         *        The date and time of the event.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder date(Instant date);

        /**
         * <p>
         * The identifier of the event.
         * </p>
         * 
         * @param eventId
         *        The identifier of the event.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder eventId(String eventId);
    }

    static final class BuilderImpl implements Builder {
        private String sourceIdentifier;

        private String sourceType;

        private String message;

        private List<String> eventCategories = DefaultSdkAutoConstructList.getInstance();

        private String severity;

        private Instant date;

        private String eventId;

        private BuilderImpl() {
        }

        private BuilderImpl(Event model) {
            sourceIdentifier(model.sourceIdentifier);
            sourceType(model.sourceType);
            message(model.message);
            eventCategories(model.eventCategories);
            severity(model.severity);
            date(model.date);
            eventId(model.eventId);
        }

        public final String getSourceIdentifier() {
            return sourceIdentifier;
        }

        public final void setSourceIdentifier(String sourceIdentifier) {
            this.sourceIdentifier = sourceIdentifier;
        }

        @Override
        public final Builder sourceIdentifier(String sourceIdentifier) {
            this.sourceIdentifier = sourceIdentifier;
            return this;
        }

        public final String getSourceType() {
            return sourceType;
        }

        public final void setSourceType(String sourceType) {
            this.sourceType = sourceType;
        }

        @Override
        public final Builder sourceType(String sourceType) {
            this.sourceType = sourceType;
            return this;
        }

        @Override
        public final Builder sourceType(SourceType sourceType) {
            this.sourceType(sourceType == null ? null : sourceType.toString());
            return this;
        }

        public final String getMessage() {
            return message;
        }

        public final void setMessage(String message) {
            this.message = message;
        }

        @Override
        public final Builder message(String message) {
            this.message = message;
            return this;
        }

        public final Collection<String> getEventCategories() {
            if (eventCategories instanceof SdkAutoConstructList) {
                return null;
            }
            return eventCategories;
        }

        public final void setEventCategories(Collection<String> eventCategories) {
            this.eventCategories = EventCategoriesListCopier.copy(eventCategories);
        }

        @Override
        public final Builder eventCategories(Collection<String> eventCategories) {
            this.eventCategories = EventCategoriesListCopier.copy(eventCategories);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder eventCategories(String... eventCategories) {
            eventCategories(Arrays.asList(eventCategories));
            return this;
        }

        public final String getSeverity() {
            return severity;
        }

        public final void setSeverity(String severity) {
            this.severity = severity;
        }

        @Override
        public final Builder severity(String severity) {
            this.severity = severity;
            return this;
        }

        public final Instant getDate() {
            return date;
        }

        public final void setDate(Instant date) {
            this.date = date;
        }

        @Override
        public final Builder date(Instant date) {
            this.date = date;
            return this;
        }

        public final String getEventId() {
            return eventId;
        }

        public final void setEventId(String eventId) {
            this.eventId = eventId;
        }

        @Override
        public final Builder eventId(String eventId) {
            this.eventId = eventId;
            return this;
        }

        @Override
        public Event build() {
            return new Event(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
