/*
 * Copyright 2013-2018 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.protocolrestjson;

import static software.amazon.awssdk.utils.FunctionalUtils.runAndLogError;

import java.util.concurrent.CompletableFuture;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.client.handler.AwsAsyncClientHandler;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.awscore.internal.protocol.json.AwsJsonProtocol;
import software.amazon.awssdk.awscore.protocol.json.AwsJsonProtocolFactory;
import software.amazon.awssdk.awscore.protocol.json.AwsJsonProtocolMetadata;
import software.amazon.awssdk.core.async.AsyncRequestBody;
import software.amazon.awssdk.core.async.AsyncResponseTransformer;
import software.amazon.awssdk.core.client.handler.AsyncClientHandler;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.core.internal.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.protocol.json.JsonClientMetadata;
import software.amazon.awssdk.core.protocol.json.JsonErrorResponseMetadata;
import software.amazon.awssdk.core.protocol.json.JsonErrorShapeMetadata;
import software.amazon.awssdk.core.protocol.json.JsonOperationMetadata;
import software.amazon.awssdk.services.protocolrestjson.model.AllTypesRequest;
import software.amazon.awssdk.services.protocolrestjson.model.AllTypesResponse;
import software.amazon.awssdk.services.protocolrestjson.model.DeleteOperationRequest;
import software.amazon.awssdk.services.protocolrestjson.model.DeleteOperationResponse;
import software.amazon.awssdk.services.protocolrestjson.model.EmptyModeledException;
import software.amazon.awssdk.services.protocolrestjson.model.HeadOperationRequest;
import software.amazon.awssdk.services.protocolrestjson.model.HeadOperationResponse;
import software.amazon.awssdk.services.protocolrestjson.model.IdempotentOperationRequest;
import software.amazon.awssdk.services.protocolrestjson.model.IdempotentOperationResponse;
import software.amazon.awssdk.services.protocolrestjson.model.JsonValuesOperationRequest;
import software.amazon.awssdk.services.protocolrestjson.model.JsonValuesOperationResponse;
import software.amazon.awssdk.services.protocolrestjson.model.MapOfStringToListOfStringInQueryParamsRequest;
import software.amazon.awssdk.services.protocolrestjson.model.MapOfStringToListOfStringInQueryParamsResponse;
import software.amazon.awssdk.services.protocolrestjson.model.MembersInHeadersRequest;
import software.amazon.awssdk.services.protocolrestjson.model.MembersInHeadersResponse;
import software.amazon.awssdk.services.protocolrestjson.model.MembersInQueryParamsRequest;
import software.amazon.awssdk.services.protocolrestjson.model.MembersInQueryParamsResponse;
import software.amazon.awssdk.services.protocolrestjson.model.MultiLocationOperationRequest;
import software.amazon.awssdk.services.protocolrestjson.model.MultiLocationOperationResponse;
import software.amazon.awssdk.services.protocolrestjson.model.NestedContainersRequest;
import software.amazon.awssdk.services.protocolrestjson.model.NestedContainersResponse;
import software.amazon.awssdk.services.protocolrestjson.model.OperationWithExplicitPayloadBlobRequest;
import software.amazon.awssdk.services.protocolrestjson.model.OperationWithExplicitPayloadBlobResponse;
import software.amazon.awssdk.services.protocolrestjson.model.OperationWithExplicitPayloadStructureRequest;
import software.amazon.awssdk.services.protocolrestjson.model.OperationWithExplicitPayloadStructureResponse;
import software.amazon.awssdk.services.protocolrestjson.model.OperationWithGreedyLabelRequest;
import software.amazon.awssdk.services.protocolrestjson.model.OperationWithGreedyLabelResponse;
import software.amazon.awssdk.services.protocolrestjson.model.OperationWithModeledContentTypeRequest;
import software.amazon.awssdk.services.protocolrestjson.model.OperationWithModeledContentTypeResponse;
import software.amazon.awssdk.services.protocolrestjson.model.OperationWithNoInputOrOutputRequest;
import software.amazon.awssdk.services.protocolrestjson.model.OperationWithNoInputOrOutputResponse;
import software.amazon.awssdk.services.protocolrestjson.model.QueryParamWithoutValueRequest;
import software.amazon.awssdk.services.protocolrestjson.model.QueryParamWithoutValueResponse;
import software.amazon.awssdk.services.protocolrestjson.model.StreamingInputOperationRequest;
import software.amazon.awssdk.services.protocolrestjson.model.StreamingInputOperationResponse;
import software.amazon.awssdk.services.protocolrestjson.model.StreamingOutputOperationRequest;
import software.amazon.awssdk.services.protocolrestjson.model.StreamingOutputOperationResponse;
import software.amazon.awssdk.services.protocolrestjson.transform.AllTypesRequestMarshaller;
import software.amazon.awssdk.services.protocolrestjson.transform.AllTypesResponseUnmarshaller;
import software.amazon.awssdk.services.protocolrestjson.transform.DeleteOperationRequestMarshaller;
import software.amazon.awssdk.services.protocolrestjson.transform.DeleteOperationResponseUnmarshaller;
import software.amazon.awssdk.services.protocolrestjson.transform.HeadOperationRequestMarshaller;
import software.amazon.awssdk.services.protocolrestjson.transform.HeadOperationResponseUnmarshaller;
import software.amazon.awssdk.services.protocolrestjson.transform.IdempotentOperationRequestMarshaller;
import software.amazon.awssdk.services.protocolrestjson.transform.IdempotentOperationResponseUnmarshaller;
import software.amazon.awssdk.services.protocolrestjson.transform.JsonValuesOperationRequestMarshaller;
import software.amazon.awssdk.services.protocolrestjson.transform.JsonValuesOperationResponseUnmarshaller;
import software.amazon.awssdk.services.protocolrestjson.transform.MapOfStringToListOfStringInQueryParamsRequestMarshaller;
import software.amazon.awssdk.services.protocolrestjson.transform.MapOfStringToListOfStringInQueryParamsResponseUnmarshaller;
import software.amazon.awssdk.services.protocolrestjson.transform.MembersInHeadersRequestMarshaller;
import software.amazon.awssdk.services.protocolrestjson.transform.MembersInHeadersResponseUnmarshaller;
import software.amazon.awssdk.services.protocolrestjson.transform.MembersInQueryParamsRequestMarshaller;
import software.amazon.awssdk.services.protocolrestjson.transform.MembersInQueryParamsResponseUnmarshaller;
import software.amazon.awssdk.services.protocolrestjson.transform.MultiLocationOperationRequestMarshaller;
import software.amazon.awssdk.services.protocolrestjson.transform.MultiLocationOperationResponseUnmarshaller;
import software.amazon.awssdk.services.protocolrestjson.transform.NestedContainersRequestMarshaller;
import software.amazon.awssdk.services.protocolrestjson.transform.NestedContainersResponseUnmarshaller;
import software.amazon.awssdk.services.protocolrestjson.transform.OperationWithExplicitPayloadBlobRequestMarshaller;
import software.amazon.awssdk.services.protocolrestjson.transform.OperationWithExplicitPayloadBlobResponseUnmarshaller;
import software.amazon.awssdk.services.protocolrestjson.transform.OperationWithExplicitPayloadStructureRequestMarshaller;
import software.amazon.awssdk.services.protocolrestjson.transform.OperationWithExplicitPayloadStructureResponseUnmarshaller;
import software.amazon.awssdk.services.protocolrestjson.transform.OperationWithGreedyLabelRequestMarshaller;
import software.amazon.awssdk.services.protocolrestjson.transform.OperationWithGreedyLabelResponseUnmarshaller;
import software.amazon.awssdk.services.protocolrestjson.transform.OperationWithModeledContentTypeRequestMarshaller;
import software.amazon.awssdk.services.protocolrestjson.transform.OperationWithModeledContentTypeResponseUnmarshaller;
import software.amazon.awssdk.services.protocolrestjson.transform.OperationWithNoInputOrOutputRequestMarshaller;
import software.amazon.awssdk.services.protocolrestjson.transform.OperationWithNoInputOrOutputResponseUnmarshaller;
import software.amazon.awssdk.services.protocolrestjson.transform.QueryParamWithoutValueRequestMarshaller;
import software.amazon.awssdk.services.protocolrestjson.transform.QueryParamWithoutValueResponseUnmarshaller;
import software.amazon.awssdk.services.protocolrestjson.transform.StreamingInputOperationRequestMarshaller;
import software.amazon.awssdk.services.protocolrestjson.transform.StreamingInputOperationResponseUnmarshaller;
import software.amazon.awssdk.services.protocolrestjson.transform.StreamingOutputOperationRequestMarshaller;
import software.amazon.awssdk.services.protocolrestjson.transform.StreamingOutputOperationResponseUnmarshaller;
import software.amazon.awssdk.utils.CompletableFutureUtils;

/**
 * Internal implementation of {@link ProtocolRestJsonAsyncClient}.
 *
 * @see ProtocolRestJsonAsyncClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultProtocolRestJsonAsyncClient implements ProtocolRestJsonAsyncClient {
    private static final Logger log = LoggerFactory.getLogger(DefaultProtocolRestJsonAsyncClient.class);

    private final AsyncClientHandler clientHandler;

    private final AwsJsonProtocolFactory protocolFactory;

    protected DefaultProtocolRestJsonAsyncClient(SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsAsyncClientHandler(clientConfiguration);
        this.protocolFactory = init(false);
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    /**
     * Invokes the AllTypes operation asynchronously.
     *
     * @param allTypesRequest
     * @return A Java Future containing the result of the AllTypes operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>EmptyModeledException</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ProtocolRestJsonException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample ProtocolRestJsonAsyncClient.AllTypes
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/restjson-2016-03-11/AllTypes" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<AllTypesResponse> allTypes(AllTypesRequest allTypesRequest) {
        try {

            HttpResponseHandler<AllTypesResponse> responseHandler = protocolFactory.createResponseHandler(
                    new JsonOperationMetadata().withPayloadJson(true).withHasStreamingSuccessResponse(false),
                    new AllTypesResponseUnmarshaller());

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory);

            return clientHandler.execute(new ClientExecutionParams<AllTypesRequest, AllTypesResponse>()
                    .withMarshaller(new AllTypesRequestMarshaller(protocolFactory)).withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(allTypesRequest));
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * Invokes the DeleteOperation operation asynchronously.
     *
     * @param deleteOperationRequest
     * @return A Java Future containing the result of the DeleteOperation operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ProtocolRestJsonException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample ProtocolRestJsonAsyncClient.DeleteOperation
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/restjson-2016-03-11/DeleteOperation" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteOperationResponse> deleteOperation(DeleteOperationRequest deleteOperationRequest) {
        try {

            HttpResponseHandler<DeleteOperationResponse> responseHandler = protocolFactory.createResponseHandler(
                    new JsonOperationMetadata().withPayloadJson(true).withHasStreamingSuccessResponse(false),
                    new DeleteOperationResponseUnmarshaller());

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory);

            return clientHandler.execute(new ClientExecutionParams<DeleteOperationRequest, DeleteOperationResponse>()
                    .withMarshaller(new DeleteOperationRequestMarshaller(protocolFactory)).withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(deleteOperationRequest));
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * Invokes the HeadOperation operation asynchronously.
     *
     * @param headOperationRequest
     * @return A Java Future containing the result of the HeadOperation operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ProtocolRestJsonException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample ProtocolRestJsonAsyncClient.HeadOperation
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/restjson-2016-03-11/HeadOperation" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<HeadOperationResponse> headOperation(HeadOperationRequest headOperationRequest) {
        try {

            HttpResponseHandler<HeadOperationResponse> responseHandler = protocolFactory.createResponseHandler(
                    new JsonOperationMetadata().withPayloadJson(true).withHasStreamingSuccessResponse(false),
                    new HeadOperationResponseUnmarshaller());

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory);

            return clientHandler.execute(new ClientExecutionParams<HeadOperationRequest, HeadOperationResponse>()
                    .withMarshaller(new HeadOperationRequestMarshaller(protocolFactory)).withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(headOperationRequest));
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * Invokes the IdempotentOperation operation asynchronously.
     *
     * @param idempotentOperationRequest
     * @return A Java Future containing the result of the IdempotentOperation operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ProtocolRestJsonException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample ProtocolRestJsonAsyncClient.IdempotentOperation
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/restjson-2016-03-11/IdempotentOperation" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<IdempotentOperationResponse> idempotentOperation(
            IdempotentOperationRequest idempotentOperationRequest) {
        try {

            HttpResponseHandler<IdempotentOperationResponse> responseHandler = protocolFactory.createResponseHandler(
                    new JsonOperationMetadata().withPayloadJson(true).withHasStreamingSuccessResponse(false),
                    new IdempotentOperationResponseUnmarshaller());

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory);

            return clientHandler.execute(new ClientExecutionParams<IdempotentOperationRequest, IdempotentOperationResponse>()
                    .withMarshaller(new IdempotentOperationRequestMarshaller(protocolFactory))
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withInput(idempotentOperationRequest));
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * Invokes the JsonValuesOperation operation asynchronously.
     *
     * @param jsonValuesOperationRequest
     * @return A Java Future containing the result of the JsonValuesOperation operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>EmptyModeledException</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ProtocolRestJsonException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample ProtocolRestJsonAsyncClient.JsonValuesOperation
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/restjson-2016-03-11/JsonValuesOperation" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<JsonValuesOperationResponse> jsonValuesOperation(
            JsonValuesOperationRequest jsonValuesOperationRequest) {
        try {

            HttpResponseHandler<JsonValuesOperationResponse> responseHandler = protocolFactory.createResponseHandler(
                    new JsonOperationMetadata().withPayloadJson(true).withHasStreamingSuccessResponse(false),
                    new JsonValuesOperationResponseUnmarshaller());

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory);

            return clientHandler.execute(new ClientExecutionParams<JsonValuesOperationRequest, JsonValuesOperationResponse>()
                    .withMarshaller(new JsonValuesOperationRequestMarshaller(protocolFactory))
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withInput(jsonValuesOperationRequest));
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * Invokes the MapOfStringToListOfStringInQueryParams operation asynchronously.
     *
     * @param mapOfStringToListOfStringInQueryParamsRequest
     * @return A Java Future containing the result of the MapOfStringToListOfStringInQueryParams operation returned by
     *         the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ProtocolRestJsonException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample ProtocolRestJsonAsyncClient.MapOfStringToListOfStringInQueryParams
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/restjson-2016-03-11/MapOfStringToListOfStringInQueryParams"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<MapOfStringToListOfStringInQueryParamsResponse> mapOfStringToListOfStringInQueryParams(
            MapOfStringToListOfStringInQueryParamsRequest mapOfStringToListOfStringInQueryParamsRequest) {
        try {

            HttpResponseHandler<MapOfStringToListOfStringInQueryParamsResponse> responseHandler = protocolFactory
                    .createResponseHandler(
                            new JsonOperationMetadata().withPayloadJson(true).withHasStreamingSuccessResponse(false),
                            new MapOfStringToListOfStringInQueryParamsResponseUnmarshaller());

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory);

            return clientHandler
                    .execute(new ClientExecutionParams<MapOfStringToListOfStringInQueryParamsRequest, MapOfStringToListOfStringInQueryParamsResponse>()
                            .withMarshaller(new MapOfStringToListOfStringInQueryParamsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(mapOfStringToListOfStringInQueryParamsRequest));
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * Invokes the MembersInHeaders operation asynchronously.
     *
     * @param membersInHeadersRequest
     * @return A Java Future containing the result of the MembersInHeaders operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ProtocolRestJsonException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample ProtocolRestJsonAsyncClient.MembersInHeaders
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/restjson-2016-03-11/MembersInHeaders" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<MembersInHeadersResponse> membersInHeaders(MembersInHeadersRequest membersInHeadersRequest) {
        try {

            HttpResponseHandler<MembersInHeadersResponse> responseHandler = protocolFactory.createResponseHandler(
                    new JsonOperationMetadata().withPayloadJson(true).withHasStreamingSuccessResponse(false),
                    new MembersInHeadersResponseUnmarshaller());

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory);

            return clientHandler.execute(new ClientExecutionParams<MembersInHeadersRequest, MembersInHeadersResponse>()
                    .withMarshaller(new MembersInHeadersRequestMarshaller(protocolFactory)).withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(membersInHeadersRequest));
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * Invokes the MembersInQueryParams operation asynchronously.
     *
     * @param membersInQueryParamsRequest
     * @return A Java Future containing the result of the MembersInQueryParams operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ProtocolRestJsonException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample ProtocolRestJsonAsyncClient.MembersInQueryParams
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/restjson-2016-03-11/MembersInQueryParams" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<MembersInQueryParamsResponse> membersInQueryParams(
            MembersInQueryParamsRequest membersInQueryParamsRequest) {
        try {

            HttpResponseHandler<MembersInQueryParamsResponse> responseHandler = protocolFactory.createResponseHandler(
                    new JsonOperationMetadata().withPayloadJson(true).withHasStreamingSuccessResponse(false),
                    new MembersInQueryParamsResponseUnmarshaller());

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory);

            return clientHandler.execute(new ClientExecutionParams<MembersInQueryParamsRequest, MembersInQueryParamsResponse>()
                    .withMarshaller(new MembersInQueryParamsRequestMarshaller(protocolFactory))
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withInput(membersInQueryParamsRequest));
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * Invokes the MultiLocationOperation operation asynchronously.
     *
     * @param multiLocationOperationRequest
     * @return A Java Future containing the result of the MultiLocationOperation operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ProtocolRestJsonException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample ProtocolRestJsonAsyncClient.MultiLocationOperation
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/restjson-2016-03-11/MultiLocationOperation"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<MultiLocationOperationResponse> multiLocationOperation(
            MultiLocationOperationRequest multiLocationOperationRequest) {
        try {

            HttpResponseHandler<MultiLocationOperationResponse> responseHandler = protocolFactory.createResponseHandler(
                    new JsonOperationMetadata().withPayloadJson(true).withHasStreamingSuccessResponse(false),
                    new MultiLocationOperationResponseUnmarshaller());

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory);

            return clientHandler
                    .execute(new ClientExecutionParams<MultiLocationOperationRequest, MultiLocationOperationResponse>()
                            .withMarshaller(new MultiLocationOperationRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(multiLocationOperationRequest));
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * Invokes the NestedContainers operation asynchronously.
     *
     * @param nestedContainersRequest
     * @return A Java Future containing the result of the NestedContainers operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ProtocolRestJsonException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample ProtocolRestJsonAsyncClient.NestedContainers
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/restjson-2016-03-11/NestedContainers" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<NestedContainersResponse> nestedContainers(NestedContainersRequest nestedContainersRequest) {
        try {

            HttpResponseHandler<NestedContainersResponse> responseHandler = protocolFactory.createResponseHandler(
                    new JsonOperationMetadata().withPayloadJson(true).withHasStreamingSuccessResponse(false),
                    new NestedContainersResponseUnmarshaller());

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory);

            return clientHandler.execute(new ClientExecutionParams<NestedContainersRequest, NestedContainersResponse>()
                    .withMarshaller(new NestedContainersRequestMarshaller(protocolFactory)).withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(nestedContainersRequest));
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * Invokes the OperationWithExplicitPayloadBlob operation asynchronously.
     *
     * @param operationWithExplicitPayloadBlobRequest
     * @return A Java Future containing the result of the OperationWithExplicitPayloadBlob operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ProtocolRestJsonException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample ProtocolRestJsonAsyncClient.OperationWithExplicitPayloadBlob
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/restjson-2016-03-11/OperationWithExplicitPayloadBlob"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<OperationWithExplicitPayloadBlobResponse> operationWithExplicitPayloadBlob(
            OperationWithExplicitPayloadBlobRequest operationWithExplicitPayloadBlobRequest) {
        try {

            HttpResponseHandler<OperationWithExplicitPayloadBlobResponse> responseHandler = protocolFactory
                    .createResponseHandler(
                            new JsonOperationMetadata().withPayloadJson(false).withHasStreamingSuccessResponse(false),
                            new OperationWithExplicitPayloadBlobResponseUnmarshaller());

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory);

            return clientHandler
                    .execute(new ClientExecutionParams<OperationWithExplicitPayloadBlobRequest, OperationWithExplicitPayloadBlobResponse>()
                            .withMarshaller(new OperationWithExplicitPayloadBlobRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(operationWithExplicitPayloadBlobRequest));
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * Invokes the OperationWithExplicitPayloadStructure operation asynchronously.
     *
     * @param operationWithExplicitPayloadStructureRequest
     * @return A Java Future containing the result of the OperationWithExplicitPayloadStructure operation returned by
     *         the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ProtocolRestJsonException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample ProtocolRestJsonAsyncClient.OperationWithExplicitPayloadStructure
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/restjson-2016-03-11/OperationWithExplicitPayloadStructure"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<OperationWithExplicitPayloadStructureResponse> operationWithExplicitPayloadStructure(
            OperationWithExplicitPayloadStructureRequest operationWithExplicitPayloadStructureRequest) {
        try {

            HttpResponseHandler<OperationWithExplicitPayloadStructureResponse> responseHandler = protocolFactory
                    .createResponseHandler(
                            new JsonOperationMetadata().withPayloadJson(true).withHasStreamingSuccessResponse(false),
                            new OperationWithExplicitPayloadStructureResponseUnmarshaller());

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory);

            return clientHandler
                    .execute(new ClientExecutionParams<OperationWithExplicitPayloadStructureRequest, OperationWithExplicitPayloadStructureResponse>()
                            .withMarshaller(new OperationWithExplicitPayloadStructureRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(operationWithExplicitPayloadStructureRequest));
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * Invokes the OperationWithGreedyLabel operation asynchronously.
     *
     * @param operationWithGreedyLabelRequest
     * @return A Java Future containing the result of the OperationWithGreedyLabel operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ProtocolRestJsonException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample ProtocolRestJsonAsyncClient.OperationWithGreedyLabel
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/restjson-2016-03-11/OperationWithGreedyLabel"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<OperationWithGreedyLabelResponse> operationWithGreedyLabel(
            OperationWithGreedyLabelRequest operationWithGreedyLabelRequest) {
        try {

            HttpResponseHandler<OperationWithGreedyLabelResponse> responseHandler = protocolFactory.createResponseHandler(
                    new JsonOperationMetadata().withPayloadJson(true).withHasStreamingSuccessResponse(false),
                    new OperationWithGreedyLabelResponseUnmarshaller());

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory);

            return clientHandler
                    .execute(new ClientExecutionParams<OperationWithGreedyLabelRequest, OperationWithGreedyLabelResponse>()
                            .withMarshaller(new OperationWithGreedyLabelRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(operationWithGreedyLabelRequest));
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * Invokes the OperationWithModeledContentType operation asynchronously.
     *
     * @param operationWithModeledContentTypeRequest
     * @return A Java Future containing the result of the OperationWithModeledContentType operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ProtocolRestJsonException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample ProtocolRestJsonAsyncClient.OperationWithModeledContentType
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/restjson-2016-03-11/OperationWithModeledContentType"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<OperationWithModeledContentTypeResponse> operationWithModeledContentType(
            OperationWithModeledContentTypeRequest operationWithModeledContentTypeRequest) {
        try {

            HttpResponseHandler<OperationWithModeledContentTypeResponse> responseHandler = protocolFactory.createResponseHandler(
                    new JsonOperationMetadata().withPayloadJson(true).withHasStreamingSuccessResponse(false),
                    new OperationWithModeledContentTypeResponseUnmarshaller());

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory);

            return clientHandler
                    .execute(new ClientExecutionParams<OperationWithModeledContentTypeRequest, OperationWithModeledContentTypeResponse>()
                            .withMarshaller(new OperationWithModeledContentTypeRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(operationWithModeledContentTypeRequest));
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * Invokes the OperationWithNoInputOrOutput operation asynchronously.
     *
     * @param operationWithNoInputOrOutputRequest
     * @return A Java Future containing the result of the OperationWithNoInputOrOutput operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ProtocolRestJsonException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample ProtocolRestJsonAsyncClient.OperationWithNoInputOrOutput
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/restjson-2016-03-11/OperationWithNoInputOrOutput"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<OperationWithNoInputOrOutputResponse> operationWithNoInputOrOutput(
            OperationWithNoInputOrOutputRequest operationWithNoInputOrOutputRequest) {
        try {

            HttpResponseHandler<OperationWithNoInputOrOutputResponse> responseHandler = protocolFactory.createResponseHandler(
                    new JsonOperationMetadata().withPayloadJson(true).withHasStreamingSuccessResponse(false),
                    new OperationWithNoInputOrOutputResponseUnmarshaller());

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory);

            return clientHandler
                    .execute(new ClientExecutionParams<OperationWithNoInputOrOutputRequest, OperationWithNoInputOrOutputResponse>()
                            .withMarshaller(new OperationWithNoInputOrOutputRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(operationWithNoInputOrOutputRequest));
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * Invokes the QueryParamWithoutValue operation asynchronously.
     *
     * @param queryParamWithoutValueRequest
     * @return A Java Future containing the result of the QueryParamWithoutValue operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ProtocolRestJsonException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample ProtocolRestJsonAsyncClient.QueryParamWithoutValue
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/restjson-2016-03-11/QueryParamWithoutValue"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<QueryParamWithoutValueResponse> queryParamWithoutValue(
            QueryParamWithoutValueRequest queryParamWithoutValueRequest) {
        try {

            HttpResponseHandler<QueryParamWithoutValueResponse> responseHandler = protocolFactory.createResponseHandler(
                    new JsonOperationMetadata().withPayloadJson(true).withHasStreamingSuccessResponse(false),
                    new QueryParamWithoutValueResponseUnmarshaller());

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory);

            return clientHandler
                    .execute(new ClientExecutionParams<QueryParamWithoutValueRequest, QueryParamWithoutValueResponse>()
                            .withMarshaller(new QueryParamWithoutValueRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(queryParamWithoutValueRequest));
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * Invokes the StreamingInputOperation operation asynchronously.
     *
     * @param streamingInputOperationRequest
     * @param requestBody
     *        Functional interface that can be implemented to produce the request content in a non-blocking manner. The
     *        size of the content is expected to be known up front. See {@link AsyncRequestBody} for specific details on
     *        implementing this interface as well as links to precanned implementations for common scenarios like
     *        uploading from a file. The service documentation for the request content is as follows ''
     * @return A Java Future containing the result of the StreamingInputOperation operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ProtocolRestJsonException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample ProtocolRestJsonAsyncClient.StreamingInputOperation
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/restjson-2016-03-11/StreamingInputOperation"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<StreamingInputOperationResponse> streamingInputOperation(
            StreamingInputOperationRequest streamingInputOperationRequest, AsyncRequestBody requestBody) {
        try {

            HttpResponseHandler<StreamingInputOperationResponse> responseHandler = protocolFactory.createResponseHandler(
                    new JsonOperationMetadata().withPayloadJson(true).withHasStreamingSuccessResponse(false),
                    new StreamingInputOperationResponseUnmarshaller());

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory);

            return clientHandler
                    .execute(new ClientExecutionParams<StreamingInputOperationRequest, StreamingInputOperationResponse>()
                            .withMarshaller(new StreamingInputOperationRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withAsyncRequestBody(requestBody).withInput(streamingInputOperationRequest));
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * Invokes the StreamingOutputOperation operation asynchronously.
     *
     * @param streamingOutputOperationRequest
     * @param asyncResponseTransformer
     *        The response transformer for processing the streaming response in a non-blocking manner. See
     *        {@link AsyncResponseTransformer} for details on how this callback should be implemented and for links to
     *        precanned implementations for common scenarios like downloading to a file. The service documentation for
     *        the response content is as follows ''.
     * @return A future to the transformed result of the AsyncResponseTransformer.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ProtocolRestJsonException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample ProtocolRestJsonAsyncClient.StreamingOutputOperation
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/restjson-2016-03-11/StreamingOutputOperation"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public <ReturnT> CompletableFuture<ReturnT> streamingOutputOperation(
            StreamingOutputOperationRequest streamingOutputOperationRequest,
            AsyncResponseTransformer<StreamingOutputOperationResponse, ReturnT> asyncResponseTransformer) {
        try {

            HttpResponseHandler<StreamingOutputOperationResponse> responseHandler = protocolFactory.createResponseHandler(
                    new JsonOperationMetadata().withPayloadJson(false).withHasStreamingSuccessResponse(true),
                    new StreamingOutputOperationResponseUnmarshaller());

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory);

            return clientHandler.execute(
                    new ClientExecutionParams<StreamingOutputOperationRequest, StreamingOutputOperationResponse>()
                            .withMarshaller(new StreamingOutputOperationRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(streamingOutputOperationRequest), asyncResponseTransformer).whenComplete((r, e) -> {
                if (e != null) {
                    asyncResponseTransformer.exceptionOccurred(e);
                }
            });
        } catch (Throwable t) {
            runAndLogError(log, "Exception thrown in exceptionOccurred callback, ignoring",
                    () -> asyncResponseTransformer.exceptionOccurred(t));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    @Override
    public void close() {
        clientHandler.close();
    }

    private software.amazon.awssdk.awscore.protocol.json.AwsJsonProtocolFactory init(boolean supportsCbor) {
        return new AwsJsonProtocolFactory(new JsonClientMetadata()
                .withSupportsCbor(supportsCbor)
                .withSupportsIon(false)
                .withBaseServiceExceptionClass(
                        software.amazon.awssdk.services.protocolrestjson.model.ProtocolRestJsonException.class)
                .withContentTypeOverride("")
                .addErrorMetadata(
                        new JsonErrorShapeMetadata().withErrorCode("EmptyModeledException").withModeledClass(
                                EmptyModeledException.class)), AwsJsonProtocolMetadata.builder().protocolVersion("1.1")
                .protocol(AwsJsonProtocol.REST_JSON).build());
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(AwsJsonProtocolFactory protocolFactory) {
        return protocolFactory.createErrorResponseHandler(new JsonErrorResponseMetadata());
    }
}
