/*
 * Copyright 2013-2018 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.protocolec2;

import java.util.concurrent.CompletableFuture;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.client.handler.AwsAsyncClientHandler;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.core.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.client.handler.AsyncClientHandler;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.protocols.core.ExceptionMetadata;
import software.amazon.awssdk.protocols.query.AwsEc2ProtocolFactory;
import software.amazon.awssdk.services.protocolec2.model.AllTypesRequest;
import software.amazon.awssdk.services.protocolec2.model.AllTypesResponse;
import software.amazon.awssdk.services.protocolec2.model.Ec2TypesRequest;
import software.amazon.awssdk.services.protocolec2.model.Ec2TypesResponse;
import software.amazon.awssdk.services.protocolec2.model.EmptyModeledException;
import software.amazon.awssdk.services.protocolec2.model.IdempotentOperationRequest;
import software.amazon.awssdk.services.protocolec2.model.IdempotentOperationResponse;
import software.amazon.awssdk.services.protocolec2.model.ImplicitPayloadException;
import software.amazon.awssdk.services.protocolec2.model.ProtocolEc2Exception;
import software.amazon.awssdk.services.protocolec2.transform.AllTypesRequestMarshaller;
import software.amazon.awssdk.services.protocolec2.transform.Ec2TypesRequestMarshaller;
import software.amazon.awssdk.services.protocolec2.transform.IdempotentOperationRequestMarshaller;
import software.amazon.awssdk.utils.CompletableFutureUtils;

/**
 * Internal implementation of {@link ProtocolEc2AsyncClient}.
 *
 * @see ProtocolEc2AsyncClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultProtocolEc2AsyncClient implements ProtocolEc2AsyncClient {
    private static final Logger log = LoggerFactory.getLogger(DefaultProtocolEc2AsyncClient.class);

    private final AsyncClientHandler clientHandler;

    private final AwsEc2ProtocolFactory protocolFactory;

    private final SdkClientConfiguration clientConfiguration;

    protected DefaultProtocolEc2AsyncClient(SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsAsyncClientHandler(clientConfiguration);
        this.clientConfiguration = clientConfiguration;
        this.protocolFactory = init();
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    /**
     * Invokes the AllTypes operation asynchronously.
     *
     * @param allTypesRequest
     * @return A Java Future containing the result of the AllTypes operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>EmptyModeledException</li>
     *         <li>ImplicitPayloadException</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ProtocolEc2Exception Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample ProtocolEc2AsyncClient.AllTypes
     */
    @Override
    public CompletableFuture<AllTypesResponse> allTypes(AllTypesRequest allTypesRequest) {
        try {

            HttpResponseHandler<AllTypesResponse> responseHandler = protocolFactory
                    .createResponseHandler(AllTypesResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = protocolFactory.createErrorResponseHandler();

            return clientHandler.execute(new ClientExecutionParams<AllTypesRequest, AllTypesResponse>()
                    .withMarshaller(new AllTypesRequestMarshaller(protocolFactory)).withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(allTypesRequest));
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * Invokes the Ec2Types operation asynchronously.
     *
     * @param ec2TypesRequest
     * @return A Java Future containing the result of the Ec2Types operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ProtocolEc2Exception Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample ProtocolEc2AsyncClient.Ec2Types
     */
    @Override
    public CompletableFuture<Ec2TypesResponse> ec2Types(Ec2TypesRequest ec2TypesRequest) {
        try {

            HttpResponseHandler<Ec2TypesResponse> responseHandler = protocolFactory
                    .createResponseHandler(Ec2TypesResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = protocolFactory.createErrorResponseHandler();

            return clientHandler.execute(new ClientExecutionParams<Ec2TypesRequest, Ec2TypesResponse>()
                    .withMarshaller(new Ec2TypesRequestMarshaller(protocolFactory)).withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(ec2TypesRequest));
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * Invokes the IdempotentOperation operation asynchronously.
     *
     * @param idempotentOperationRequest
     * @return A Java Future containing the result of the IdempotentOperation operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ProtocolEc2Exception Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample ProtocolEc2AsyncClient.IdempotentOperation
     */
    @Override
    public CompletableFuture<IdempotentOperationResponse> idempotentOperation(
            IdempotentOperationRequest idempotentOperationRequest) {
        try {

            HttpResponseHandler<IdempotentOperationResponse> responseHandler = protocolFactory
                    .createResponseHandler(IdempotentOperationResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = protocolFactory.createErrorResponseHandler();

            return clientHandler.execute(new ClientExecutionParams<IdempotentOperationRequest, IdempotentOperationResponse>()
                    .withMarshaller(new IdempotentOperationRequestMarshaller(protocolFactory))
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withInput(idempotentOperationRequest));
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    @Override
    public void close() {
        clientHandler.close();
    }

    private AwsEc2ProtocolFactory init() {
        return AwsEc2ProtocolFactory
                .builder()
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("EmptyModeledException")
                                .exceptionBuilderSupplier(EmptyModeledException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ImplicitPayloadException")
                                .exceptionBuilderSupplier(ImplicitPayloadException::builder).build())
                .clientConfiguration(clientConfiguration).defaultServiceExceptionSupplier(ProtocolEc2Exception::builder).build();
    }
}
