/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.paymentcryptography.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.Mutable;
import software.amazon.awssdk.annotations.NotThreadSafe;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Represents the replication status information for a key in a replication region for <a
 * href="https://docs.aws.amazon.com/payment-cryptography/latest/userguide/keys-multi-region-replication.html"
 * >Multi-Region key replication</a>.
 * </p>
 * <p>
 * This structure contains details about the current state of key replication, including any status messages and
 * operational information about the replication process.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class ReplicationStatusType implements SdkPojo, Serializable,
        ToCopyableBuilder<ReplicationStatusType.Builder, ReplicationStatusType> {
    private static final SdkField<String> STATUS_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Status")
            .getter(getter(ReplicationStatusType::statusAsString)).setter(setter(Builder::status))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Status").build()).build();

    private static final SdkField<String> STATUS_MESSAGE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("StatusMessage").getter(getter(ReplicationStatusType::statusMessage))
            .setter(setter(Builder::statusMessage))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("StatusMessage").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(STATUS_FIELD,
            STATUS_MESSAGE_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = memberNameToFieldInitializer();

    private static final long serialVersionUID = 1L;

    private final String status;

    private final String statusMessage;

    private ReplicationStatusType(BuilderImpl builder) {
        this.status = builder.status;
        this.statusMessage = builder.statusMessage;
    }

    /**
     * <p>
     * The current status of key replication in this Amazon Web Services Region.
     * </p>
     * <p>
     * This field indicates whether the key replication is in progress, completed successfully, or has encountered an
     * error. Possible values include states such as <code>SYNCRHONIZED</code>, <code>IN_PROGRESS</code>,
     * <code>DELETE_IN_PROGRESS</code>, or <code>FAILED</code>. This provides visibility into the replication process
     * for monitoring and troubleshooting purposes.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #status} will
     * return {@link KeyReplicationState#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #statusAsString}.
     * </p>
     * 
     * @return The current status of key replication in this Amazon Web Services Region.</p>
     *         <p>
     *         This field indicates whether the key replication is in progress, completed successfully, or has
     *         encountered an error. Possible values include states such as <code>SYNCRHONIZED</code>,
     *         <code>IN_PROGRESS</code>, <code>DELETE_IN_PROGRESS</code>, or <code>FAILED</code>. This provides
     *         visibility into the replication process for monitoring and troubleshooting purposes.
     * @see KeyReplicationState
     */
    public final KeyReplicationState status() {
        return KeyReplicationState.fromValue(status);
    }

    /**
     * <p>
     * The current status of key replication in this Amazon Web Services Region.
     * </p>
     * <p>
     * This field indicates whether the key replication is in progress, completed successfully, or has encountered an
     * error. Possible values include states such as <code>SYNCRHONIZED</code>, <code>IN_PROGRESS</code>,
     * <code>DELETE_IN_PROGRESS</code>, or <code>FAILED</code>. This provides visibility into the replication process
     * for monitoring and troubleshooting purposes.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #status} will
     * return {@link KeyReplicationState#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #statusAsString}.
     * </p>
     * 
     * @return The current status of key replication in this Amazon Web Services Region.</p>
     *         <p>
     *         This field indicates whether the key replication is in progress, completed successfully, or has
     *         encountered an error. Possible values include states such as <code>SYNCRHONIZED</code>,
     *         <code>IN_PROGRESS</code>, <code>DELETE_IN_PROGRESS</code>, or <code>FAILED</code>. This provides
     *         visibility into the replication process for monitoring and troubleshooting purposes.
     * @see KeyReplicationState
     */
    public final String statusAsString() {
        return status;
    }

    /**
     * <p>
     * A message that provides additional information about the current replication status of the key.
     * </p>
     * <p>
     * This field contains details about any issues or progress updates related to key replication operations. It may
     * include information about replication failures, synchronization status, or other operational details.
     * </p>
     * 
     * @return A message that provides additional information about the current replication status of the key.</p>
     *         <p>
     *         This field contains details about any issues or progress updates related to key replication operations.
     *         It may include information about replication failures, synchronization status, or other operational
     *         details.
     */
    public final String statusMessage() {
        return statusMessage;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(statusAsString());
        hashCode = 31 * hashCode + Objects.hashCode(statusMessage());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof ReplicationStatusType)) {
            return false;
        }
        ReplicationStatusType other = (ReplicationStatusType) obj;
        return Objects.equals(statusAsString(), other.statusAsString()) && Objects.equals(statusMessage(), other.statusMessage());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("ReplicationStatusType").add("Status", statusAsString()).add("StatusMessage", statusMessage())
                .build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "Status":
            return Optional.ofNullable(clazz.cast(statusAsString()));
        case "StatusMessage":
            return Optional.ofNullable(clazz.cast(statusMessage()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static Map<String, SdkField<?>> memberNameToFieldInitializer() {
        Map<String, SdkField<?>> map = new HashMap<>();
        map.put("Status", STATUS_FIELD);
        map.put("StatusMessage", STATUS_MESSAGE_FIELD);
        return Collections.unmodifiableMap(map);
    }

    private static <T> Function<Object, T> getter(Function<ReplicationStatusType, T> g) {
        return obj -> g.apply((ReplicationStatusType) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    @Mutable
    @NotThreadSafe
    public interface Builder extends SdkPojo, CopyableBuilder<Builder, ReplicationStatusType> {
        /**
         * <p>
         * The current status of key replication in this Amazon Web Services Region.
         * </p>
         * <p>
         * This field indicates whether the key replication is in progress, completed successfully, or has encountered
         * an error. Possible values include states such as <code>SYNCRHONIZED</code>, <code>IN_PROGRESS</code>,
         * <code>DELETE_IN_PROGRESS</code>, or <code>FAILED</code>. This provides visibility into the replication
         * process for monitoring and troubleshooting purposes.
         * </p>
         * 
         * @param status
         *        The current status of key replication in this Amazon Web Services Region.</p>
         *        <p>
         *        This field indicates whether the key replication is in progress, completed successfully, or has
         *        encountered an error. Possible values include states such as <code>SYNCRHONIZED</code>,
         *        <code>IN_PROGRESS</code>, <code>DELETE_IN_PROGRESS</code>, or <code>FAILED</code>. This provides
         *        visibility into the replication process for monitoring and troubleshooting purposes.
         * @see KeyReplicationState
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see KeyReplicationState
         */
        Builder status(String status);

        /**
         * <p>
         * The current status of key replication in this Amazon Web Services Region.
         * </p>
         * <p>
         * This field indicates whether the key replication is in progress, completed successfully, or has encountered
         * an error. Possible values include states such as <code>SYNCRHONIZED</code>, <code>IN_PROGRESS</code>,
         * <code>DELETE_IN_PROGRESS</code>, or <code>FAILED</code>. This provides visibility into the replication
         * process for monitoring and troubleshooting purposes.
         * </p>
         * 
         * @param status
         *        The current status of key replication in this Amazon Web Services Region.</p>
         *        <p>
         *        This field indicates whether the key replication is in progress, completed successfully, or has
         *        encountered an error. Possible values include states such as <code>SYNCRHONIZED</code>,
         *        <code>IN_PROGRESS</code>, <code>DELETE_IN_PROGRESS</code>, or <code>FAILED</code>. This provides
         *        visibility into the replication process for monitoring and troubleshooting purposes.
         * @see KeyReplicationState
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see KeyReplicationState
         */
        Builder status(KeyReplicationState status);

        /**
         * <p>
         * A message that provides additional information about the current replication status of the key.
         * </p>
         * <p>
         * This field contains details about any issues or progress updates related to key replication operations. It
         * may include information about replication failures, synchronization status, or other operational details.
         * </p>
         * 
         * @param statusMessage
         *        A message that provides additional information about the current replication status of the key.</p>
         *        <p>
         *        This field contains details about any issues or progress updates related to key replication
         *        operations. It may include information about replication failures, synchronization status, or other
         *        operational details.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder statusMessage(String statusMessage);
    }

    static final class BuilderImpl implements Builder {
        private String status;

        private String statusMessage;

        private BuilderImpl() {
        }

        private BuilderImpl(ReplicationStatusType model) {
            status(model.status);
            statusMessage(model.statusMessage);
        }

        public final String getStatus() {
            return status;
        }

        public final void setStatus(String status) {
            this.status = status;
        }

        @Override
        public final Builder status(String status) {
            this.status = status;
            return this;
        }

        @Override
        public final Builder status(KeyReplicationState status) {
            this.status(status == null ? null : status.toString());
            return this;
        }

        public final String getStatusMessage() {
            return statusMessage;
        }

        public final void setStatusMessage(String statusMessage) {
            this.statusMessage = statusMessage;
        }

        @Override
        public final Builder statusMessage(String statusMessage) {
            this.statusMessage = statusMessage;
            return this;
        }

        @Override
        public ReplicationStatusType build() {
            return new ReplicationStatusType(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
