/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.paymentcryptography.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.Mutable;
import software.amazon.awssdk.annotations.NotThreadSafe;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.traits.MapTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructMap;
import software.amazon.awssdk.core.util.SdkAutoConstructMap;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Optional metadata for export associated with the key material. This data is signed but transmitted in clear text.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class KeyBlockHeaders implements SdkPojo, Serializable, ToCopyableBuilder<KeyBlockHeaders.Builder, KeyBlockHeaders> {
    private static final SdkField<KeyModesOfUse> KEY_MODES_OF_USE_FIELD = SdkField
            .<KeyModesOfUse> builder(MarshallingType.SDK_POJO).memberName("KeyModesOfUse")
            .getter(getter(KeyBlockHeaders::keyModesOfUse)).setter(setter(Builder::keyModesOfUse))
            .constructor(KeyModesOfUse::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("KeyModesOfUse").build()).build();

    private static final SdkField<String> KEY_EXPORTABILITY_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("KeyExportability").getter(getter(KeyBlockHeaders::keyExportabilityAsString))
            .setter(setter(Builder::keyExportability))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("KeyExportability").build()).build();

    private static final SdkField<String> KEY_VERSION_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("KeyVersion").getter(getter(KeyBlockHeaders::keyVersion)).setter(setter(Builder::keyVersion))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("KeyVersion").build()).build();

    private static final SdkField<Map<String, String>> OPTIONAL_BLOCKS_FIELD = SdkField
            .<Map<String, String>> builder(MarshallingType.MAP)
            .memberName("OptionalBlocks")
            .getter(getter(KeyBlockHeaders::optionalBlocks))
            .setter(setter(Builder::optionalBlocks))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("OptionalBlocks").build(),
                    MapTrait.builder()
                            .keyLocationName("key")
                            .valueLocationName("value")
                            .valueFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("value").build()).build()).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(KEY_MODES_OF_USE_FIELD,
            KEY_EXPORTABILITY_FIELD, KEY_VERSION_FIELD, OPTIONAL_BLOCKS_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = memberNameToFieldInitializer();

    private static final long serialVersionUID = 1L;

    private final KeyModesOfUse keyModesOfUse;

    private final String keyExportability;

    private final String keyVersion;

    private final Map<String, String> optionalBlocks;

    private KeyBlockHeaders(BuilderImpl builder) {
        this.keyModesOfUse = builder.keyModesOfUse;
        this.keyExportability = builder.keyExportability;
        this.keyVersion = builder.keyVersion;
        this.optionalBlocks = builder.optionalBlocks;
    }

    /**
     * Returns the value of the KeyModesOfUse property for this object.
     * 
     * @return The value of the KeyModesOfUse property for this object.
     */
    public final KeyModesOfUse keyModesOfUse() {
        return keyModesOfUse;
    }

    /**
     * <p>
     * Specifies subsequent exportability of the key within the key block after it is received by the receiving party.
     * It can be used to further restrict exportability of the key after export from Amazon Web Services Payment
     * Cryptography.
     * </p>
     * <p>
     * When set to <code>EXPORTABLE</code>, the key can be subsequently exported by the receiver under a KEK using TR-31
     * or TR-34 key block export only. When set to <code>NON_EXPORTABLE</code>, the key cannot be subsequently exported
     * by the receiver. When set to <code>SENSITIVE</code>, the key can be exported by the receiver under a KEK using
     * TR-31, TR-34, RSA wrap and unwrap cryptogram or using a symmetric cryptogram key export method. For further
     * information refer to <a href="https://webstore.ansi.org/standards/ascx9/ansix91432022">ANSI X9.143-2022</a>.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #keyExportability}
     * will return {@link KeyExportability#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #keyExportabilityAsString}.
     * </p>
     * 
     * @return Specifies subsequent exportability of the key within the key block after it is received by the receiving
     *         party. It can be used to further restrict exportability of the key after export from Amazon Web Services
     *         Payment Cryptography.</p>
     *         <p>
     *         When set to <code>EXPORTABLE</code>, the key can be subsequently exported by the receiver under a KEK
     *         using TR-31 or TR-34 key block export only. When set to <code>NON_EXPORTABLE</code>, the key cannot be
     *         subsequently exported by the receiver. When set to <code>SENSITIVE</code>, the key can be exported by the
     *         receiver under a KEK using TR-31, TR-34, RSA wrap and unwrap cryptogram or using a symmetric cryptogram
     *         key export method. For further information refer to <a
     *         href="https://webstore.ansi.org/standards/ascx9/ansix91432022">ANSI X9.143-2022</a>.
     * @see KeyExportability
     */
    public final KeyExportability keyExportability() {
        return KeyExportability.fromValue(keyExportability);
    }

    /**
     * <p>
     * Specifies subsequent exportability of the key within the key block after it is received by the receiving party.
     * It can be used to further restrict exportability of the key after export from Amazon Web Services Payment
     * Cryptography.
     * </p>
     * <p>
     * When set to <code>EXPORTABLE</code>, the key can be subsequently exported by the receiver under a KEK using TR-31
     * or TR-34 key block export only. When set to <code>NON_EXPORTABLE</code>, the key cannot be subsequently exported
     * by the receiver. When set to <code>SENSITIVE</code>, the key can be exported by the receiver under a KEK using
     * TR-31, TR-34, RSA wrap and unwrap cryptogram or using a symmetric cryptogram key export method. For further
     * information refer to <a href="https://webstore.ansi.org/standards/ascx9/ansix91432022">ANSI X9.143-2022</a>.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #keyExportability}
     * will return {@link KeyExportability#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #keyExportabilityAsString}.
     * </p>
     * 
     * @return Specifies subsequent exportability of the key within the key block after it is received by the receiving
     *         party. It can be used to further restrict exportability of the key after export from Amazon Web Services
     *         Payment Cryptography.</p>
     *         <p>
     *         When set to <code>EXPORTABLE</code>, the key can be subsequently exported by the receiver under a KEK
     *         using TR-31 or TR-34 key block export only. When set to <code>NON_EXPORTABLE</code>, the key cannot be
     *         subsequently exported by the receiver. When set to <code>SENSITIVE</code>, the key can be exported by the
     *         receiver under a KEK using TR-31, TR-34, RSA wrap and unwrap cryptogram or using a symmetric cryptogram
     *         key export method. For further information refer to <a
     *         href="https://webstore.ansi.org/standards/ascx9/ansix91432022">ANSI X9.143-2022</a>.
     * @see KeyExportability
     */
    public final String keyExportabilityAsString() {
        return keyExportability;
    }

    /**
     * <p>
     * Parameter used to indicate the version of the key carried in the key block or indicate the value carried in the
     * key block is a component of a key.
     * </p>
     * 
     * @return Parameter used to indicate the version of the key carried in the key block or indicate the value carried
     *         in the key block is a component of a key.
     */
    public final String keyVersion() {
        return keyVersion;
    }

    /**
     * For responses, this returns true if the service returned a value for the OptionalBlocks property. This DOES NOT
     * check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasOptionalBlocks() {
        return optionalBlocks != null && !(optionalBlocks instanceof SdkAutoConstructMap);
    }

    /**
     * <p>
     * Parameter used to indicate the type of optional data in key block headers. Refer to <a
     * href="https://webstore.ansi.org/standards/ascx9/ansix91432022">ANSI X9.143-2022</a> for information on allowed
     * data type for optional blocks.
     * </p>
     * <p>
     * Optional block character limit is 112 characters. For each optional block, 2 characters are reserved for optional
     * block ID and 2 characters reserved for optional block length. More than one optional blocks can be included as
     * long as the combined length does not increase 112 characters.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasOptionalBlocks} method.
     * </p>
     * 
     * @return Parameter used to indicate the type of optional data in key block headers. Refer to <a
     *         href="https://webstore.ansi.org/standards/ascx9/ansix91432022">ANSI X9.143-2022</a> for information on
     *         allowed data type for optional blocks.</p>
     *         <p>
     *         Optional block character limit is 112 characters. For each optional block, 2 characters are reserved for
     *         optional block ID and 2 characters reserved for optional block length. More than one optional blocks can
     *         be included as long as the combined length does not increase 112 characters.
     */
    public final Map<String, String> optionalBlocks() {
        return optionalBlocks;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(keyModesOfUse());
        hashCode = 31 * hashCode + Objects.hashCode(keyExportabilityAsString());
        hashCode = 31 * hashCode + Objects.hashCode(keyVersion());
        hashCode = 31 * hashCode + Objects.hashCode(hasOptionalBlocks() ? optionalBlocks() : null);
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof KeyBlockHeaders)) {
            return false;
        }
        KeyBlockHeaders other = (KeyBlockHeaders) obj;
        return Objects.equals(keyModesOfUse(), other.keyModesOfUse())
                && Objects.equals(keyExportabilityAsString(), other.keyExportabilityAsString())
                && Objects.equals(keyVersion(), other.keyVersion()) && hasOptionalBlocks() == other.hasOptionalBlocks()
                && Objects.equals(optionalBlocks(), other.optionalBlocks());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("KeyBlockHeaders").add("KeyModesOfUse", keyModesOfUse())
                .add("KeyExportability", keyExportabilityAsString()).add("KeyVersion", keyVersion())
                .add("OptionalBlocks", optionalBlocks() == null ? null : "*** Sensitive Data Redacted ***").build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "KeyModesOfUse":
            return Optional.ofNullable(clazz.cast(keyModesOfUse()));
        case "KeyExportability":
            return Optional.ofNullable(clazz.cast(keyExportabilityAsString()));
        case "KeyVersion":
            return Optional.ofNullable(clazz.cast(keyVersion()));
        case "OptionalBlocks":
            return Optional.ofNullable(clazz.cast(optionalBlocks()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static Map<String, SdkField<?>> memberNameToFieldInitializer() {
        Map<String, SdkField<?>> map = new HashMap<>();
        map.put("KeyModesOfUse", KEY_MODES_OF_USE_FIELD);
        map.put("KeyExportability", KEY_EXPORTABILITY_FIELD);
        map.put("KeyVersion", KEY_VERSION_FIELD);
        map.put("OptionalBlocks", OPTIONAL_BLOCKS_FIELD);
        return Collections.unmodifiableMap(map);
    }

    private static <T> Function<Object, T> getter(Function<KeyBlockHeaders, T> g) {
        return obj -> g.apply((KeyBlockHeaders) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    @Mutable
    @NotThreadSafe
    public interface Builder extends SdkPojo, CopyableBuilder<Builder, KeyBlockHeaders> {
        /**
         * Sets the value of the KeyModesOfUse property for this object.
         *
         * @param keyModesOfUse
         *        The new value for the KeyModesOfUse property for this object.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder keyModesOfUse(KeyModesOfUse keyModesOfUse);

        /**
         * Sets the value of the KeyModesOfUse property for this object.
         *
         * This is a convenience method that creates an instance of the {@link KeyModesOfUse.Builder} avoiding the need
         * to create one manually via {@link KeyModesOfUse#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link KeyModesOfUse.Builder#build()} is called immediately and its
         * result is passed to {@link #keyModesOfUse(KeyModesOfUse)}.
         * 
         * @param keyModesOfUse
         *        a consumer that will call methods on {@link KeyModesOfUse.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #keyModesOfUse(KeyModesOfUse)
         */
        default Builder keyModesOfUse(Consumer<KeyModesOfUse.Builder> keyModesOfUse) {
            return keyModesOfUse(KeyModesOfUse.builder().applyMutation(keyModesOfUse).build());
        }

        /**
         * <p>
         * Specifies subsequent exportability of the key within the key block after it is received by the receiving
         * party. It can be used to further restrict exportability of the key after export from Amazon Web Services
         * Payment Cryptography.
         * </p>
         * <p>
         * When set to <code>EXPORTABLE</code>, the key can be subsequently exported by the receiver under a KEK using
         * TR-31 or TR-34 key block export only. When set to <code>NON_EXPORTABLE</code>, the key cannot be subsequently
         * exported by the receiver. When set to <code>SENSITIVE</code>, the key can be exported by the receiver under a
         * KEK using TR-31, TR-34, RSA wrap and unwrap cryptogram or using a symmetric cryptogram key export method. For
         * further information refer to <a href="https://webstore.ansi.org/standards/ascx9/ansix91432022">ANSI
         * X9.143-2022</a>.
         * </p>
         * 
         * @param keyExportability
         *        Specifies subsequent exportability of the key within the key block after it is received by the
         *        receiving party. It can be used to further restrict exportability of the key after export from Amazon
         *        Web Services Payment Cryptography.</p>
         *        <p>
         *        When set to <code>EXPORTABLE</code>, the key can be subsequently exported by the receiver under a KEK
         *        using TR-31 or TR-34 key block export only. When set to <code>NON_EXPORTABLE</code>, the key cannot be
         *        subsequently exported by the receiver. When set to <code>SENSITIVE</code>, the key can be exported by
         *        the receiver under a KEK using TR-31, TR-34, RSA wrap and unwrap cryptogram or using a symmetric
         *        cryptogram key export method. For further information refer to <a
         *        href="https://webstore.ansi.org/standards/ascx9/ansix91432022">ANSI X9.143-2022</a>.
         * @see KeyExportability
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see KeyExportability
         */
        Builder keyExportability(String keyExportability);

        /**
         * <p>
         * Specifies subsequent exportability of the key within the key block after it is received by the receiving
         * party. It can be used to further restrict exportability of the key after export from Amazon Web Services
         * Payment Cryptography.
         * </p>
         * <p>
         * When set to <code>EXPORTABLE</code>, the key can be subsequently exported by the receiver under a KEK using
         * TR-31 or TR-34 key block export only. When set to <code>NON_EXPORTABLE</code>, the key cannot be subsequently
         * exported by the receiver. When set to <code>SENSITIVE</code>, the key can be exported by the receiver under a
         * KEK using TR-31, TR-34, RSA wrap and unwrap cryptogram or using a symmetric cryptogram key export method. For
         * further information refer to <a href="https://webstore.ansi.org/standards/ascx9/ansix91432022">ANSI
         * X9.143-2022</a>.
         * </p>
         * 
         * @param keyExportability
         *        Specifies subsequent exportability of the key within the key block after it is received by the
         *        receiving party. It can be used to further restrict exportability of the key after export from Amazon
         *        Web Services Payment Cryptography.</p>
         *        <p>
         *        When set to <code>EXPORTABLE</code>, the key can be subsequently exported by the receiver under a KEK
         *        using TR-31 or TR-34 key block export only. When set to <code>NON_EXPORTABLE</code>, the key cannot be
         *        subsequently exported by the receiver. When set to <code>SENSITIVE</code>, the key can be exported by
         *        the receiver under a KEK using TR-31, TR-34, RSA wrap and unwrap cryptogram or using a symmetric
         *        cryptogram key export method. For further information refer to <a
         *        href="https://webstore.ansi.org/standards/ascx9/ansix91432022">ANSI X9.143-2022</a>.
         * @see KeyExportability
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see KeyExportability
         */
        Builder keyExportability(KeyExportability keyExportability);

        /**
         * <p>
         * Parameter used to indicate the version of the key carried in the key block or indicate the value carried in
         * the key block is a component of a key.
         * </p>
         * 
         * @param keyVersion
         *        Parameter used to indicate the version of the key carried in the key block or indicate the value
         *        carried in the key block is a component of a key.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder keyVersion(String keyVersion);

        /**
         * <p>
         * Parameter used to indicate the type of optional data in key block headers. Refer to <a
         * href="https://webstore.ansi.org/standards/ascx9/ansix91432022">ANSI X9.143-2022</a> for information on
         * allowed data type for optional blocks.
         * </p>
         * <p>
         * Optional block character limit is 112 characters. For each optional block, 2 characters are reserved for
         * optional block ID and 2 characters reserved for optional block length. More than one optional blocks can be
         * included as long as the combined length does not increase 112 characters.
         * </p>
         * 
         * @param optionalBlocks
         *        Parameter used to indicate the type of optional data in key block headers. Refer to <a
         *        href="https://webstore.ansi.org/standards/ascx9/ansix91432022">ANSI X9.143-2022</a> for information on
         *        allowed data type for optional blocks.</p>
         *        <p>
         *        Optional block character limit is 112 characters. For each optional block, 2 characters are reserved
         *        for optional block ID and 2 characters reserved for optional block length. More than one optional
         *        blocks can be included as long as the combined length does not increase 112 characters.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder optionalBlocks(Map<String, String> optionalBlocks);
    }

    static final class BuilderImpl implements Builder {
        private KeyModesOfUse keyModesOfUse;

        private String keyExportability;

        private String keyVersion;

        private Map<String, String> optionalBlocks = DefaultSdkAutoConstructMap.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(KeyBlockHeaders model) {
            keyModesOfUse(model.keyModesOfUse);
            keyExportability(model.keyExportability);
            keyVersion(model.keyVersion);
            optionalBlocks(model.optionalBlocks);
        }

        public final KeyModesOfUse.Builder getKeyModesOfUse() {
            return keyModesOfUse != null ? keyModesOfUse.toBuilder() : null;
        }

        public final void setKeyModesOfUse(KeyModesOfUse.BuilderImpl keyModesOfUse) {
            this.keyModesOfUse = keyModesOfUse != null ? keyModesOfUse.build() : null;
        }

        @Override
        public final Builder keyModesOfUse(KeyModesOfUse keyModesOfUse) {
            this.keyModesOfUse = keyModesOfUse;
            return this;
        }

        public final String getKeyExportability() {
            return keyExportability;
        }

        public final void setKeyExportability(String keyExportability) {
            this.keyExportability = keyExportability;
        }

        @Override
        public final Builder keyExportability(String keyExportability) {
            this.keyExportability = keyExportability;
            return this;
        }

        @Override
        public final Builder keyExportability(KeyExportability keyExportability) {
            this.keyExportability(keyExportability == null ? null : keyExportability.toString());
            return this;
        }

        public final String getKeyVersion() {
            return keyVersion;
        }

        public final void setKeyVersion(String keyVersion) {
            this.keyVersion = keyVersion;
        }

        @Override
        public final Builder keyVersion(String keyVersion) {
            this.keyVersion = keyVersion;
            return this;
        }

        public final Map<String, String> getOptionalBlocks() {
            if (optionalBlocks instanceof SdkAutoConstructMap) {
                return null;
            }
            return optionalBlocks;
        }

        public final void setOptionalBlocks(Map<String, String> optionalBlocks) {
            this.optionalBlocks = OptionalBlocksCopier.copy(optionalBlocks);
        }

        @Override
        public final Builder optionalBlocks(Map<String, String> optionalBlocks) {
            this.optionalBlocks = OptionalBlocksCopier.copy(optionalBlocks);
            return this;
        }

        @Override
        public KeyBlockHeaders build() {
            return new KeyBlockHeaders(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
