/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.paymentcryptography.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.Mutable;
import software.amazon.awssdk.annotations.NotThreadSafe;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Parameter information for key material export using asymmetric RSA wrap and unwrap key exchange method.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class ExportKeyCryptogram implements SdkPojo, Serializable,
        ToCopyableBuilder<ExportKeyCryptogram.Builder, ExportKeyCryptogram> {
    private static final SdkField<String> CERTIFICATE_AUTHORITY_PUBLIC_KEY_IDENTIFIER_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("CertificateAuthorityPublicKeyIdentifier")
            .getter(getter(ExportKeyCryptogram::certificateAuthorityPublicKeyIdentifier))
            .setter(setter(Builder::certificateAuthorityPublicKeyIdentifier))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                    .locationName("CertificateAuthorityPublicKeyIdentifier").build()).build();

    private static final SdkField<String> WRAPPING_KEY_CERTIFICATE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("WrappingKeyCertificate").getter(getter(ExportKeyCryptogram::wrappingKeyCertificate))
            .setter(setter(Builder::wrappingKeyCertificate))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("WrappingKeyCertificate").build())
            .build();

    private static final SdkField<String> WRAPPING_SPEC_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("WrappingSpec").getter(getter(ExportKeyCryptogram::wrappingSpecAsString))
            .setter(setter(Builder::wrappingSpec))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("WrappingSpec").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(
            CERTIFICATE_AUTHORITY_PUBLIC_KEY_IDENTIFIER_FIELD, WRAPPING_KEY_CERTIFICATE_FIELD, WRAPPING_SPEC_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = memberNameToFieldInitializer();

    private static final long serialVersionUID = 1L;

    private final String certificateAuthorityPublicKeyIdentifier;

    private final String wrappingKeyCertificate;

    private final String wrappingSpec;

    private ExportKeyCryptogram(BuilderImpl builder) {
        this.certificateAuthorityPublicKeyIdentifier = builder.certificateAuthorityPublicKeyIdentifier;
        this.wrappingKeyCertificate = builder.wrappingKeyCertificate;
        this.wrappingSpec = builder.wrappingSpec;
    }

    /**
     * <p>
     * The <code>KeyARN</code> of the certificate chain that signs the wrapping key certificate during RSA wrap and
     * unwrap key export.
     * </p>
     * 
     * @return The <code>KeyARN</code> of the certificate chain that signs the wrapping key certificate during RSA wrap
     *         and unwrap key export.
     */
    public final String certificateAuthorityPublicKeyIdentifier() {
        return certificateAuthorityPublicKeyIdentifier;
    }

    /**
     * <p>
     * The wrapping key certificate in PEM format (base64 encoded). Amazon Web Services Payment Cryptography uses this
     * certificate to wrap the key under export.
     * </p>
     * 
     * @return The wrapping key certificate in PEM format (base64 encoded). Amazon Web Services Payment Cryptography
     *         uses this certificate to wrap the key under export.
     */
    public final String wrappingKeyCertificate() {
        return wrappingKeyCertificate;
    }

    /**
     * <p>
     * The wrapping spec for the key under export.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #wrappingSpec} will
     * return {@link WrappingKeySpec#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #wrappingSpecAsString}.
     * </p>
     * 
     * @return The wrapping spec for the key under export.
     * @see WrappingKeySpec
     */
    public final WrappingKeySpec wrappingSpec() {
        return WrappingKeySpec.fromValue(wrappingSpec);
    }

    /**
     * <p>
     * The wrapping spec for the key under export.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #wrappingSpec} will
     * return {@link WrappingKeySpec#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #wrappingSpecAsString}.
     * </p>
     * 
     * @return The wrapping spec for the key under export.
     * @see WrappingKeySpec
     */
    public final String wrappingSpecAsString() {
        return wrappingSpec;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(certificateAuthorityPublicKeyIdentifier());
        hashCode = 31 * hashCode + Objects.hashCode(wrappingKeyCertificate());
        hashCode = 31 * hashCode + Objects.hashCode(wrappingSpecAsString());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof ExportKeyCryptogram)) {
            return false;
        }
        ExportKeyCryptogram other = (ExportKeyCryptogram) obj;
        return Objects.equals(certificateAuthorityPublicKeyIdentifier(), other.certificateAuthorityPublicKeyIdentifier())
                && Objects.equals(wrappingKeyCertificate(), other.wrappingKeyCertificate())
                && Objects.equals(wrappingSpecAsString(), other.wrappingSpecAsString());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("ExportKeyCryptogram")
                .add("CertificateAuthorityPublicKeyIdentifier", certificateAuthorityPublicKeyIdentifier())
                .add("WrappingKeyCertificate", wrappingKeyCertificate()).add("WrappingSpec", wrappingSpecAsString()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "CertificateAuthorityPublicKeyIdentifier":
            return Optional.ofNullable(clazz.cast(certificateAuthorityPublicKeyIdentifier()));
        case "WrappingKeyCertificate":
            return Optional.ofNullable(clazz.cast(wrappingKeyCertificate()));
        case "WrappingSpec":
            return Optional.ofNullable(clazz.cast(wrappingSpecAsString()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static Map<String, SdkField<?>> memberNameToFieldInitializer() {
        Map<String, SdkField<?>> map = new HashMap<>();
        map.put("CertificateAuthorityPublicKeyIdentifier", CERTIFICATE_AUTHORITY_PUBLIC_KEY_IDENTIFIER_FIELD);
        map.put("WrappingKeyCertificate", WRAPPING_KEY_CERTIFICATE_FIELD);
        map.put("WrappingSpec", WRAPPING_SPEC_FIELD);
        return Collections.unmodifiableMap(map);
    }

    private static <T> Function<Object, T> getter(Function<ExportKeyCryptogram, T> g) {
        return obj -> g.apply((ExportKeyCryptogram) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    @Mutable
    @NotThreadSafe
    public interface Builder extends SdkPojo, CopyableBuilder<Builder, ExportKeyCryptogram> {
        /**
         * <p>
         * The <code>KeyARN</code> of the certificate chain that signs the wrapping key certificate during RSA wrap and
         * unwrap key export.
         * </p>
         * 
         * @param certificateAuthorityPublicKeyIdentifier
         *        The <code>KeyARN</code> of the certificate chain that signs the wrapping key certificate during RSA
         *        wrap and unwrap key export.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder certificateAuthorityPublicKeyIdentifier(String certificateAuthorityPublicKeyIdentifier);

        /**
         * <p>
         * The wrapping key certificate in PEM format (base64 encoded). Amazon Web Services Payment Cryptography uses
         * this certificate to wrap the key under export.
         * </p>
         * 
         * @param wrappingKeyCertificate
         *        The wrapping key certificate in PEM format (base64 encoded). Amazon Web Services Payment Cryptography
         *        uses this certificate to wrap the key under export.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder wrappingKeyCertificate(String wrappingKeyCertificate);

        /**
         * <p>
         * The wrapping spec for the key under export.
         * </p>
         * 
         * @param wrappingSpec
         *        The wrapping spec for the key under export.
         * @see WrappingKeySpec
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see WrappingKeySpec
         */
        Builder wrappingSpec(String wrappingSpec);

        /**
         * <p>
         * The wrapping spec for the key under export.
         * </p>
         * 
         * @param wrappingSpec
         *        The wrapping spec for the key under export.
         * @see WrappingKeySpec
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see WrappingKeySpec
         */
        Builder wrappingSpec(WrappingKeySpec wrappingSpec);
    }

    static final class BuilderImpl implements Builder {
        private String certificateAuthorityPublicKeyIdentifier;

        private String wrappingKeyCertificate;

        private String wrappingSpec;

        private BuilderImpl() {
        }

        private BuilderImpl(ExportKeyCryptogram model) {
            certificateAuthorityPublicKeyIdentifier(model.certificateAuthorityPublicKeyIdentifier);
            wrappingKeyCertificate(model.wrappingKeyCertificate);
            wrappingSpec(model.wrappingSpec);
        }

        public final String getCertificateAuthorityPublicKeyIdentifier() {
            return certificateAuthorityPublicKeyIdentifier;
        }

        public final void setCertificateAuthorityPublicKeyIdentifier(String certificateAuthorityPublicKeyIdentifier) {
            this.certificateAuthorityPublicKeyIdentifier = certificateAuthorityPublicKeyIdentifier;
        }

        @Override
        public final Builder certificateAuthorityPublicKeyIdentifier(String certificateAuthorityPublicKeyIdentifier) {
            this.certificateAuthorityPublicKeyIdentifier = certificateAuthorityPublicKeyIdentifier;
            return this;
        }

        public final String getWrappingKeyCertificate() {
            return wrappingKeyCertificate;
        }

        public final void setWrappingKeyCertificate(String wrappingKeyCertificate) {
            this.wrappingKeyCertificate = wrappingKeyCertificate;
        }

        @Override
        public final Builder wrappingKeyCertificate(String wrappingKeyCertificate) {
            this.wrappingKeyCertificate = wrappingKeyCertificate;
            return this;
        }

        public final String getWrappingSpec() {
            return wrappingSpec;
        }

        public final void setWrappingSpec(String wrappingSpec) {
            this.wrappingSpec = wrappingSpec;
        }

        @Override
        public final Builder wrappingSpec(String wrappingSpec) {
            this.wrappingSpec = wrappingSpec;
            return this;
        }

        @Override
        public final Builder wrappingSpec(WrappingKeySpec wrappingSpec) {
            this.wrappingSpec(wrappingSpec == null ? null : wrappingSpec.toString());
            return this;
        }

        @Override
        public ExportKeyCryptogram build() {
            return new ExportKeyCryptogram(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
