/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.lambda.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.Mutable;
import software.amazon.awssdk.annotations.NotThreadSafe;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * The <a href=
 * "https://docs.aws.amazon.com/lambda/latest/dg/invocation-eventsourcemapping.html#invocation-eventsourcemapping-provisioned-mode"
 * > provisioned mode</a> configuration for the event source. Use Provisioned Mode to customize the minimum and maximum
 * number of event pollers for your event source.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class ProvisionedPollerConfig implements SdkPojo, Serializable,
        ToCopyableBuilder<ProvisionedPollerConfig.Builder, ProvisionedPollerConfig> {
    private static final SdkField<Integer> MINIMUM_POLLERS_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER)
            .memberName("MinimumPollers").getter(getter(ProvisionedPollerConfig::minimumPollers))
            .setter(setter(Builder::minimumPollers))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("MinimumPollers").build()).build();

    private static final SdkField<Integer> MAXIMUM_POLLERS_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER)
            .memberName("MaximumPollers").getter(getter(ProvisionedPollerConfig::maximumPollers))
            .setter(setter(Builder::maximumPollers))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("MaximumPollers").build()).build();

    private static final SdkField<String> POLLER_GROUP_NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("PollerGroupName").getter(getter(ProvisionedPollerConfig::pollerGroupName))
            .setter(setter(Builder::pollerGroupName))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("PollerGroupName").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(MINIMUM_POLLERS_FIELD,
            MAXIMUM_POLLERS_FIELD, POLLER_GROUP_NAME_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = memberNameToFieldInitializer();

    private static final long serialVersionUID = 1L;

    private final Integer minimumPollers;

    private final Integer maximumPollers;

    private final String pollerGroupName;

    private ProvisionedPollerConfig(BuilderImpl builder) {
        this.minimumPollers = builder.minimumPollers;
        this.maximumPollers = builder.maximumPollers;
        this.pollerGroupName = builder.pollerGroupName;
    }

    /**
     * <p>
     * The minimum number of event pollers this event source can scale down to. For Amazon SQS events source mappings,
     * default is 2, and minimum 2 required. For Amazon MSK and self-managed Apache Kafka event source mappings, default
     * is 1.
     * </p>
     * 
     * @return The minimum number of event pollers this event source can scale down to. For Amazon SQS events source
     *         mappings, default is 2, and minimum 2 required. For Amazon MSK and self-managed Apache Kafka event source
     *         mappings, default is 1.
     */
    public final Integer minimumPollers() {
        return minimumPollers;
    }

    /**
     * <p>
     * The maximum number of event pollers this event source can scale up to. For Amazon SQS events source mappings,
     * default is 200, and minimum value allowed is 2. For Amazon MSK and self-managed Apache Kafka event source
     * mappings, default is 200, and minimum value allowed is 1.
     * </p>
     * 
     * @return The maximum number of event pollers this event source can scale up to. For Amazon SQS events source
     *         mappings, default is 200, and minimum value allowed is 2. For Amazon MSK and self-managed Apache Kafka
     *         event source mappings, default is 200, and minimum value allowed is 1.
     */
    public final Integer maximumPollers() {
        return maximumPollers;
    }

    /**
     * <p>
     * (Amazon MSK and self-managed Apache Kafka) The name of the provisioned poller group. Use this option to group
     * multiple ESMs within the VPC to share Event Poller Unit (EPU) capacity. This option is used to optimize
     * Provisioned mode costs for your ESMs. You can group up to 100 ESMs per poller group and aggregate maximum pollers
     * across all ESMs in a group cannot exceed 2000.
     * </p>
     * 
     * @return (Amazon MSK and self-managed Apache Kafka) The name of the provisioned poller group. Use this option to
     *         group multiple ESMs within the VPC to share Event Poller Unit (EPU) capacity. This option is used to
     *         optimize Provisioned mode costs for your ESMs. You can group up to 100 ESMs per poller group and
     *         aggregate maximum pollers across all ESMs in a group cannot exceed 2000.
     */
    public final String pollerGroupName() {
        return pollerGroupName;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(minimumPollers());
        hashCode = 31 * hashCode + Objects.hashCode(maximumPollers());
        hashCode = 31 * hashCode + Objects.hashCode(pollerGroupName());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof ProvisionedPollerConfig)) {
            return false;
        }
        ProvisionedPollerConfig other = (ProvisionedPollerConfig) obj;
        return Objects.equals(minimumPollers(), other.minimumPollers())
                && Objects.equals(maximumPollers(), other.maximumPollers())
                && Objects.equals(pollerGroupName(), other.pollerGroupName());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("ProvisionedPollerConfig").add("MinimumPollers", minimumPollers())
                .add("MaximumPollers", maximumPollers()).add("PollerGroupName", pollerGroupName()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "MinimumPollers":
            return Optional.ofNullable(clazz.cast(minimumPollers()));
        case "MaximumPollers":
            return Optional.ofNullable(clazz.cast(maximumPollers()));
        case "PollerGroupName":
            return Optional.ofNullable(clazz.cast(pollerGroupName()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static Map<String, SdkField<?>> memberNameToFieldInitializer() {
        Map<String, SdkField<?>> map = new HashMap<>();
        map.put("MinimumPollers", MINIMUM_POLLERS_FIELD);
        map.put("MaximumPollers", MAXIMUM_POLLERS_FIELD);
        map.put("PollerGroupName", POLLER_GROUP_NAME_FIELD);
        return Collections.unmodifiableMap(map);
    }

    private static <T> Function<Object, T> getter(Function<ProvisionedPollerConfig, T> g) {
        return obj -> g.apply((ProvisionedPollerConfig) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    @Mutable
    @NotThreadSafe
    public interface Builder extends SdkPojo, CopyableBuilder<Builder, ProvisionedPollerConfig> {
        /**
         * <p>
         * The minimum number of event pollers this event source can scale down to. For Amazon SQS events source
         * mappings, default is 2, and minimum 2 required. For Amazon MSK and self-managed Apache Kafka event source
         * mappings, default is 1.
         * </p>
         * 
         * @param minimumPollers
         *        The minimum number of event pollers this event source can scale down to. For Amazon SQS events source
         *        mappings, default is 2, and minimum 2 required. For Amazon MSK and self-managed Apache Kafka event
         *        source mappings, default is 1.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder minimumPollers(Integer minimumPollers);

        /**
         * <p>
         * The maximum number of event pollers this event source can scale up to. For Amazon SQS events source mappings,
         * default is 200, and minimum value allowed is 2. For Amazon MSK and self-managed Apache Kafka event source
         * mappings, default is 200, and minimum value allowed is 1.
         * </p>
         * 
         * @param maximumPollers
         *        The maximum number of event pollers this event source can scale up to. For Amazon SQS events source
         *        mappings, default is 200, and minimum value allowed is 2. For Amazon MSK and self-managed Apache Kafka
         *        event source mappings, default is 200, and minimum value allowed is 1.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder maximumPollers(Integer maximumPollers);

        /**
         * <p>
         * (Amazon MSK and self-managed Apache Kafka) The name of the provisioned poller group. Use this option to group
         * multiple ESMs within the VPC to share Event Poller Unit (EPU) capacity. This option is used to optimize
         * Provisioned mode costs for your ESMs. You can group up to 100 ESMs per poller group and aggregate maximum
         * pollers across all ESMs in a group cannot exceed 2000.
         * </p>
         * 
         * @param pollerGroupName
         *        (Amazon MSK and self-managed Apache Kafka) The name of the provisioned poller group. Use this option
         *        to group multiple ESMs within the VPC to share Event Poller Unit (EPU) capacity. This option is used
         *        to optimize Provisioned mode costs for your ESMs. You can group up to 100 ESMs per poller group and
         *        aggregate maximum pollers across all ESMs in a group cannot exceed 2000.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder pollerGroupName(String pollerGroupName);
    }

    static final class BuilderImpl implements Builder {
        private Integer minimumPollers;

        private Integer maximumPollers;

        private String pollerGroupName;

        private BuilderImpl() {
        }

        private BuilderImpl(ProvisionedPollerConfig model) {
            minimumPollers(model.minimumPollers);
            maximumPollers(model.maximumPollers);
            pollerGroupName(model.pollerGroupName);
        }

        public final Integer getMinimumPollers() {
            return minimumPollers;
        }

        public final void setMinimumPollers(Integer minimumPollers) {
            this.minimumPollers = minimumPollers;
        }

        @Override
        public final Builder minimumPollers(Integer minimumPollers) {
            this.minimumPollers = minimumPollers;
            return this;
        }

        public final Integer getMaximumPollers() {
            return maximumPollers;
        }

        public final void setMaximumPollers(Integer maximumPollers) {
            this.maximumPollers = maximumPollers;
        }

        @Override
        public final Builder maximumPollers(Integer maximumPollers) {
            this.maximumPollers = maximumPollers;
            return this;
        }

        public final String getPollerGroupName() {
            return pollerGroupName;
        }

        public final void setPollerGroupName(String pollerGroupName) {
            this.pollerGroupName = pollerGroupName;
        }

        @Override
        public final Builder pollerGroupName(String pollerGroupName) {
            this.pollerGroupName = pollerGroupName;
            return this;
        }

        @Override
        public ProvisionedPollerConfig build() {
            return new ProvisionedPollerConfig(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
