/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.iottwinmaker.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * An object that specifies information about time series property values. This object is used and consumed by the <a
 * href="https://docs.aws.amazon.com/iot-twinmaker/latest/apireference/API_BatchPutPropertyValues.html">
 * BatchPutPropertyValues</a> action.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class PropertyValueEntry implements SdkPojo, Serializable,
        ToCopyableBuilder<PropertyValueEntry.Builder, PropertyValueEntry> {
    private static final SdkField<EntityPropertyReference> ENTITY_PROPERTY_REFERENCE_FIELD = SdkField
            .<EntityPropertyReference> builder(MarshallingType.SDK_POJO).memberName("entityPropertyReference")
            .getter(getter(PropertyValueEntry::entityPropertyReference)).setter(setter(Builder::entityPropertyReference))
            .constructor(EntityPropertyReference::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("entityPropertyReference").build())
            .build();

    private static final SdkField<List<PropertyValue>> PROPERTY_VALUES_FIELD = SdkField
            .<List<PropertyValue>> builder(MarshallingType.LIST)
            .memberName("propertyValues")
            .getter(getter(PropertyValueEntry::propertyValues))
            .setter(setter(Builder::propertyValues))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("propertyValues").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<PropertyValue> builder(MarshallingType.SDK_POJO)
                                            .constructor(PropertyValue::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(
            ENTITY_PROPERTY_REFERENCE_FIELD, PROPERTY_VALUES_FIELD));

    private static final long serialVersionUID = 1L;

    private final EntityPropertyReference entityPropertyReference;

    private final List<PropertyValue> propertyValues;

    private PropertyValueEntry(BuilderImpl builder) {
        this.entityPropertyReference = builder.entityPropertyReference;
        this.propertyValues = builder.propertyValues;
    }

    /**
     * <p>
     * An object that contains information about the entity that has the property.
     * </p>
     * 
     * @return An object that contains information about the entity that has the property.
     */
    public final EntityPropertyReference entityPropertyReference() {
        return entityPropertyReference;
    }

    /**
     * For responses, this returns true if the service returned a value for the PropertyValues property. This DOES NOT
     * check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasPropertyValues() {
        return propertyValues != null && !(propertyValues instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * A list of objects that specify time series property values.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasPropertyValues} method.
     * </p>
     * 
     * @return A list of objects that specify time series property values.
     */
    public final List<PropertyValue> propertyValues() {
        return propertyValues;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(entityPropertyReference());
        hashCode = 31 * hashCode + Objects.hashCode(hasPropertyValues() ? propertyValues() : null);
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof PropertyValueEntry)) {
            return false;
        }
        PropertyValueEntry other = (PropertyValueEntry) obj;
        return Objects.equals(entityPropertyReference(), other.entityPropertyReference())
                && hasPropertyValues() == other.hasPropertyValues() && Objects.equals(propertyValues(), other.propertyValues());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("PropertyValueEntry").add("EntityPropertyReference", entityPropertyReference())
                .add("PropertyValues", hasPropertyValues() ? propertyValues() : null).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "entityPropertyReference":
            return Optional.ofNullable(clazz.cast(entityPropertyReference()));
        case "propertyValues":
            return Optional.ofNullable(clazz.cast(propertyValues()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<PropertyValueEntry, T> g) {
        return obj -> g.apply((PropertyValueEntry) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, PropertyValueEntry> {
        /**
         * <p>
         * An object that contains information about the entity that has the property.
         * </p>
         * 
         * @param entityPropertyReference
         *        An object that contains information about the entity that has the property.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder entityPropertyReference(EntityPropertyReference entityPropertyReference);

        /**
         * <p>
         * An object that contains information about the entity that has the property.
         * </p>
         * This is a convenience method that creates an instance of the {@link EntityPropertyReference.Builder} avoiding
         * the need to create one manually via {@link EntityPropertyReference#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link EntityPropertyReference.Builder#build()} is called immediately
         * and its result is passed to {@link #entityPropertyReference(EntityPropertyReference)}.
         * 
         * @param entityPropertyReference
         *        a consumer that will call methods on {@link EntityPropertyReference.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #entityPropertyReference(EntityPropertyReference)
         */
        default Builder entityPropertyReference(Consumer<EntityPropertyReference.Builder> entityPropertyReference) {
            return entityPropertyReference(EntityPropertyReference.builder().applyMutation(entityPropertyReference).build());
        }

        /**
         * <p>
         * A list of objects that specify time series property values.
         * </p>
         * 
         * @param propertyValues
         *        A list of objects that specify time series property values.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder propertyValues(Collection<PropertyValue> propertyValues);

        /**
         * <p>
         * A list of objects that specify time series property values.
         * </p>
         * 
         * @param propertyValues
         *        A list of objects that specify time series property values.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder propertyValues(PropertyValue... propertyValues);

        /**
         * <p>
         * A list of objects that specify time series property values.
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link software.amazon.awssdk.services.iottwinmaker.model.PropertyValue.Builder} avoiding the need to create
         * one manually via {@link software.amazon.awssdk.services.iottwinmaker.model.PropertyValue#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes,
         * {@link software.amazon.awssdk.services.iottwinmaker.model.PropertyValue.Builder#build()} is called
         * immediately and its result is passed to {@link #propertyValues(List<PropertyValue>)}.
         * 
         * @param propertyValues
         *        a consumer that will call methods on
         *        {@link software.amazon.awssdk.services.iottwinmaker.model.PropertyValue.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #propertyValues(java.util.Collection<PropertyValue>)
         */
        Builder propertyValues(Consumer<PropertyValue.Builder>... propertyValues);
    }

    static final class BuilderImpl implements Builder {
        private EntityPropertyReference entityPropertyReference;

        private List<PropertyValue> propertyValues = DefaultSdkAutoConstructList.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(PropertyValueEntry model) {
            entityPropertyReference(model.entityPropertyReference);
            propertyValues(model.propertyValues);
        }

        public final EntityPropertyReference.Builder getEntityPropertyReference() {
            return entityPropertyReference != null ? entityPropertyReference.toBuilder() : null;
        }

        public final void setEntityPropertyReference(EntityPropertyReference.BuilderImpl entityPropertyReference) {
            this.entityPropertyReference = entityPropertyReference != null ? entityPropertyReference.build() : null;
        }

        @Override
        public final Builder entityPropertyReference(EntityPropertyReference entityPropertyReference) {
            this.entityPropertyReference = entityPropertyReference;
            return this;
        }

        public final List<PropertyValue.Builder> getPropertyValues() {
            List<PropertyValue.Builder> result = PropertyValuesCopier.copyToBuilder(this.propertyValues);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setPropertyValues(Collection<PropertyValue.BuilderImpl> propertyValues) {
            this.propertyValues = PropertyValuesCopier.copyFromBuilder(propertyValues);
        }

        @Override
        public final Builder propertyValues(Collection<PropertyValue> propertyValues) {
            this.propertyValues = PropertyValuesCopier.copy(propertyValues);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder propertyValues(PropertyValue... propertyValues) {
            propertyValues(Arrays.asList(propertyValues));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder propertyValues(Consumer<PropertyValue.Builder>... propertyValues) {
            propertyValues(Stream.of(propertyValues).map(c -> PropertyValue.builder().applyMutation(c).build())
                    .collect(Collectors.toList()));
            return this;
        }

        @Override
        public PropertyValueEntry build() {
            return new PropertyValueEntry(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
