/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.internetmonitor.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.Mutable;
import software.amazon.awssdk.annotations.NotThreadSafe;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * A filter that you use with the results of a Amazon CloudWatch Internet Monitor query that you created and ran. The
 * query sets up a repository of data that is a subset of your application's Internet Monitor data.
 * <code>FilterParameter</code> is a string that defines how you want to filter the repository of data to return a set
 * of results, based on your criteria.
 * </p>
 * <p>
 * The filter parameters that you can specify depend on the query type that you used to create the repository, since
 * each query type returns a different set of Internet Monitor data.
 * </p>
 * <p>
 * For each filter, you specify a field (such as <code>city</code>), an operator (such as <code>not_equals</code>, and a
 * value or array of values (such as <code>["Seattle", "Redmond"]</code>). Separate values in the array with commas.
 * </p>
 * <p>
 * For more information about specifying filter parameters, see <a
 * href="https://docs.aws.amazon.com/AmazonCloudWatch/latest/monitoring/CloudWatch-IM-view-cw-tools-cwim-query.html"
 * >Using the Amazon CloudWatch Internet Monitor query interface</a> in the Amazon CloudWatch Internet Monitor User
 * Guide.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class FilterParameter implements SdkPojo, Serializable, ToCopyableBuilder<FilterParameter.Builder, FilterParameter> {
    private static final SdkField<String> FIELD_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Field")
            .getter(getter(FilterParameter::field)).setter(setter(Builder::field))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Field").build()).build();

    private static final SdkField<String> OPERATOR_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("Operator").getter(getter(FilterParameter::operatorAsString)).setter(setter(Builder::operator))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Operator").build()).build();

    private static final SdkField<List<String>> VALUES_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("Values")
            .getter(getter(FilterParameter::values))
            .setter(setter(Builder::values))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Values").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(FIELD_FIELD, OPERATOR_FIELD,
            VALUES_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = memberNameToFieldInitializer();

    private static final long serialVersionUID = 1L;

    private final String field;

    private final String operator;

    private final List<String> values;

    private FilterParameter(BuilderImpl builder) {
        this.field = builder.field;
        this.operator = builder.operator;
        this.values = builder.values;
    }

    /**
     * <p>
     * A data field that you want to filter, to further scope your application's Internet Monitor data in a repository
     * that you created by running a query. A field might be <code>city</code>, for example. The field must be one of
     * the fields that was returned by the specific query that you used to create the repository.
     * </p>
     * 
     * @return A data field that you want to filter, to further scope your application's Internet Monitor data in a
     *         repository that you created by running a query. A field might be <code>city</code>, for example. The
     *         field must be one of the fields that was returned by the specific query that you used to create the
     *         repository.
     */
    public final String field() {
        return field;
    }

    /**
     * <p>
     * The operator to use with the filter field and a value, such as <code>not_equals</code>.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #operator} will
     * return {@link Operator#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #operatorAsString}.
     * </p>
     * 
     * @return The operator to use with the filter field and a value, such as <code>not_equals</code>.
     * @see Operator
     */
    public final Operator operator() {
        return Operator.fromValue(operator);
    }

    /**
     * <p>
     * The operator to use with the filter field and a value, such as <code>not_equals</code>.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #operator} will
     * return {@link Operator#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #operatorAsString}.
     * </p>
     * 
     * @return The operator to use with the filter field and a value, such as <code>not_equals</code>.
     * @see Operator
     */
    public final String operatorAsString() {
        return operator;
    }

    /**
     * For responses, this returns true if the service returned a value for the Values property. This DOES NOT check
     * that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is
     * useful because the SDK will never return a null collection or map, but you may need to differentiate between the
     * service returning nothing (or null) and the service returning an empty collection or map. For requests, this
     * returns true if a value for the property was specified in the request builder, and false if a value was not
     * specified.
     */
    public final boolean hasValues() {
        return values != null && !(values instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * One or more values to be used, together with the specified operator, to filter data for a query. For example, you
     * could specify an array of values such as <code>["Seattle", "Redmond"]</code>. Values in the array are separated
     * by commas.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasValues} method.
     * </p>
     * 
     * @return One or more values to be used, together with the specified operator, to filter data for a query. For
     *         example, you could specify an array of values such as <code>["Seattle", "Redmond"]</code>. Values in the
     *         array are separated by commas.
     */
    public final List<String> values() {
        return values;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(field());
        hashCode = 31 * hashCode + Objects.hashCode(operatorAsString());
        hashCode = 31 * hashCode + Objects.hashCode(hasValues() ? values() : null);
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof FilterParameter)) {
            return false;
        }
        FilterParameter other = (FilterParameter) obj;
        return Objects.equals(field(), other.field()) && Objects.equals(operatorAsString(), other.operatorAsString())
                && hasValues() == other.hasValues() && Objects.equals(values(), other.values());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("FilterParameter").add("Field", field()).add("Operator", operatorAsString())
                .add("Values", hasValues() ? values() : null).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "Field":
            return Optional.ofNullable(clazz.cast(field()));
        case "Operator":
            return Optional.ofNullable(clazz.cast(operatorAsString()));
        case "Values":
            return Optional.ofNullable(clazz.cast(values()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static Map<String, SdkField<?>> memberNameToFieldInitializer() {
        Map<String, SdkField<?>> map = new HashMap<>();
        map.put("Field", FIELD_FIELD);
        map.put("Operator", OPERATOR_FIELD);
        map.put("Values", VALUES_FIELD);
        return Collections.unmodifiableMap(map);
    }

    private static <T> Function<Object, T> getter(Function<FilterParameter, T> g) {
        return obj -> g.apply((FilterParameter) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    @Mutable
    @NotThreadSafe
    public interface Builder extends SdkPojo, CopyableBuilder<Builder, FilterParameter> {
        /**
         * <p>
         * A data field that you want to filter, to further scope your application's Internet Monitor data in a
         * repository that you created by running a query. A field might be <code>city</code>, for example. The field
         * must be one of the fields that was returned by the specific query that you used to create the repository.
         * </p>
         * 
         * @param field
         *        A data field that you want to filter, to further scope your application's Internet Monitor data in a
         *        repository that you created by running a query. A field might be <code>city</code>, for example. The
         *        field must be one of the fields that was returned by the specific query that you used to create the
         *        repository.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder field(String field);

        /**
         * <p>
         * The operator to use with the filter field and a value, such as <code>not_equals</code>.
         * </p>
         * 
         * @param operator
         *        The operator to use with the filter field and a value, such as <code>not_equals</code>.
         * @see Operator
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see Operator
         */
        Builder operator(String operator);

        /**
         * <p>
         * The operator to use with the filter field and a value, such as <code>not_equals</code>.
         * </p>
         * 
         * @param operator
         *        The operator to use with the filter field and a value, such as <code>not_equals</code>.
         * @see Operator
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see Operator
         */
        Builder operator(Operator operator);

        /**
         * <p>
         * One or more values to be used, together with the specified operator, to filter data for a query. For example,
         * you could specify an array of values such as <code>["Seattle", "Redmond"]</code>. Values in the array are
         * separated by commas.
         * </p>
         * 
         * @param values
         *        One or more values to be used, together with the specified operator, to filter data for a query. For
         *        example, you could specify an array of values such as <code>["Seattle", "Redmond"]</code>. Values in
         *        the array are separated by commas.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder values(Collection<String> values);

        /**
         * <p>
         * One or more values to be used, together with the specified operator, to filter data for a query. For example,
         * you could specify an array of values such as <code>["Seattle", "Redmond"]</code>. Values in the array are
         * separated by commas.
         * </p>
         * 
         * @param values
         *        One or more values to be used, together with the specified operator, to filter data for a query. For
         *        example, you could specify an array of values such as <code>["Seattle", "Redmond"]</code>. Values in
         *        the array are separated by commas.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder values(String... values);
    }

    static final class BuilderImpl implements Builder {
        private String field;

        private String operator;

        private List<String> values = DefaultSdkAutoConstructList.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(FilterParameter model) {
            field(model.field);
            operator(model.operator);
            values(model.values);
        }

        public final String getField() {
            return field;
        }

        public final void setField(String field) {
            this.field = field;
        }

        @Override
        public final Builder field(String field) {
            this.field = field;
            return this;
        }

        public final String getOperator() {
            return operator;
        }

        public final void setOperator(String operator) {
            this.operator = operator;
        }

        @Override
        public final Builder operator(String operator) {
            this.operator = operator;
            return this;
        }

        @Override
        public final Builder operator(Operator operator) {
            this.operator(operator == null ? null : operator.toString());
            return this;
        }

        public final Collection<String> getValues() {
            if (values instanceof SdkAutoConstructList) {
                return null;
            }
            return values;
        }

        public final void setValues(Collection<String> values) {
            this.values = FilterListCopier.copy(values);
        }

        @Override
        public final Builder values(Collection<String> values) {
            this.values = FilterListCopier.copy(values);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder values(String... values) {
            values(Arrays.asList(values));
            return this;
        }

        @Override
        public FilterParameter build() {
            return new FilterParameter(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
