/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.globalaccelerator.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Returns the ports and associated IP addresses and ports of Amazon EC2 instances in your virtual private cloud (VPC)
 * subnets. Custom routing is a port mapping protocol in AWS Global Accelerator that statically associates port ranges
 * with VPC subnets, which allows Global Accelerator to route to specific instances and ports within one or more
 * subnets.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class PortMapping implements SdkPojo, Serializable, ToCopyableBuilder<PortMapping.Builder, PortMapping> {
    private static final SdkField<Integer> ACCELERATOR_PORT_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER)
            .memberName("AcceleratorPort").getter(getter(PortMapping::acceleratorPort)).setter(setter(Builder::acceleratorPort))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("AcceleratorPort").build()).build();

    private static final SdkField<String> ENDPOINT_GROUP_ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("EndpointGroupArn").getter(getter(PortMapping::endpointGroupArn))
            .setter(setter(Builder::endpointGroupArn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("EndpointGroupArn").build()).build();

    private static final SdkField<String> ENDPOINT_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("EndpointId").getter(getter(PortMapping::endpointId)).setter(setter(Builder::endpointId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("EndpointId").build()).build();

    private static final SdkField<SocketAddress> DESTINATION_SOCKET_ADDRESS_FIELD = SdkField
            .<SocketAddress> builder(MarshallingType.SDK_POJO).memberName("DestinationSocketAddress")
            .getter(getter(PortMapping::destinationSocketAddress)).setter(setter(Builder::destinationSocketAddress))
            .constructor(SocketAddress::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("DestinationSocketAddress").build())
            .build();

    private static final SdkField<List<String>> PROTOCOLS_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("Protocols")
            .getter(getter(PortMapping::protocolsAsStrings))
            .setter(setter(Builder::protocolsWithStrings))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Protocols").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<String> DESTINATION_TRAFFIC_STATE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("DestinationTrafficState").getter(getter(PortMapping::destinationTrafficStateAsString))
            .setter(setter(Builder::destinationTrafficState))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("DestinationTrafficState").build())
            .build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(ACCELERATOR_PORT_FIELD,
            ENDPOINT_GROUP_ARN_FIELD, ENDPOINT_ID_FIELD, DESTINATION_SOCKET_ADDRESS_FIELD, PROTOCOLS_FIELD,
            DESTINATION_TRAFFIC_STATE_FIELD));

    private static final long serialVersionUID = 1L;

    private final Integer acceleratorPort;

    private final String endpointGroupArn;

    private final String endpointId;

    private final SocketAddress destinationSocketAddress;

    private final List<String> protocols;

    private final String destinationTrafficState;

    private PortMapping(BuilderImpl builder) {
        this.acceleratorPort = builder.acceleratorPort;
        this.endpointGroupArn = builder.endpointGroupArn;
        this.endpointId = builder.endpointId;
        this.destinationSocketAddress = builder.destinationSocketAddress;
        this.protocols = builder.protocols;
        this.destinationTrafficState = builder.destinationTrafficState;
    }

    /**
     * <p>
     * The accelerator port.
     * </p>
     * 
     * @return The accelerator port.
     */
    public final Integer acceleratorPort() {
        return acceleratorPort;
    }

    /**
     * <p>
     * The Amazon Resource Name (ARN) of the endpoint group.
     * </p>
     * 
     * @return The Amazon Resource Name (ARN) of the endpoint group.
     */
    public final String endpointGroupArn() {
        return endpointGroupArn;
    }

    /**
     * <p>
     * The IP address of the VPC subnet (the subnet ID).
     * </p>
     * 
     * @return The IP address of the VPC subnet (the subnet ID).
     */
    public final String endpointId() {
        return endpointId;
    }

    /**
     * <p>
     * The EC2 instance IP address and port number in the virtual private cloud (VPC) subnet.
     * </p>
     * 
     * @return The EC2 instance IP address and port number in the virtual private cloud (VPC) subnet.
     */
    public final SocketAddress destinationSocketAddress() {
        return destinationSocketAddress;
    }

    /**
     * <p>
     * The protocols supported by the endpoint group.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasProtocols()} to see if a value was sent in this field.
     * </p>
     * 
     * @return The protocols supported by the endpoint group.
     */
    public final List<CustomRoutingProtocol> protocols() {
        return CustomRoutingProtocolsCopier.copyStringToEnum(protocols);
    }

    /**
     * Returns true if the Protocols property was specified by the sender (it may be empty), or false if the sender did
     * not specify the value (it will be empty). For responses returned by the SDK, the sender is the AWS service.
     */
    public final boolean hasProtocols() {
        return protocols != null && !(protocols instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The protocols supported by the endpoint group.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasProtocols()} to see if a value was sent in this field.
     * </p>
     * 
     * @return The protocols supported by the endpoint group.
     */
    public final List<String> protocolsAsStrings() {
        return protocols;
    }

    /**
     * <p>
     * Indicates whether or not a port mapping destination can receive traffic. The value is either ALLOW, if traffic is
     * allowed to the destination, or DENY, if traffic is not allowed to the destination.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version,
     * {@link #destinationTrafficState} will return {@link CustomRoutingDestinationTrafficState#UNKNOWN_TO_SDK_VERSION}.
     * The raw value returned by the service is available from {@link #destinationTrafficStateAsString}.
     * </p>
     * 
     * @return Indicates whether or not a port mapping destination can receive traffic. The value is either ALLOW, if
     *         traffic is allowed to the destination, or DENY, if traffic is not allowed to the destination.
     * @see CustomRoutingDestinationTrafficState
     */
    public final CustomRoutingDestinationTrafficState destinationTrafficState() {
        return CustomRoutingDestinationTrafficState.fromValue(destinationTrafficState);
    }

    /**
     * <p>
     * Indicates whether or not a port mapping destination can receive traffic. The value is either ALLOW, if traffic is
     * allowed to the destination, or DENY, if traffic is not allowed to the destination.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version,
     * {@link #destinationTrafficState} will return {@link CustomRoutingDestinationTrafficState#UNKNOWN_TO_SDK_VERSION}.
     * The raw value returned by the service is available from {@link #destinationTrafficStateAsString}.
     * </p>
     * 
     * @return Indicates whether or not a port mapping destination can receive traffic. The value is either ALLOW, if
     *         traffic is allowed to the destination, or DENY, if traffic is not allowed to the destination.
     * @see CustomRoutingDestinationTrafficState
     */
    public final String destinationTrafficStateAsString() {
        return destinationTrafficState;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(acceleratorPort());
        hashCode = 31 * hashCode + Objects.hashCode(endpointGroupArn());
        hashCode = 31 * hashCode + Objects.hashCode(endpointId());
        hashCode = 31 * hashCode + Objects.hashCode(destinationSocketAddress());
        hashCode = 31 * hashCode + Objects.hashCode(hasProtocols() ? protocolsAsStrings() : null);
        hashCode = 31 * hashCode + Objects.hashCode(destinationTrafficStateAsString());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof PortMapping)) {
            return false;
        }
        PortMapping other = (PortMapping) obj;
        return Objects.equals(acceleratorPort(), other.acceleratorPort())
                && Objects.equals(endpointGroupArn(), other.endpointGroupArn())
                && Objects.equals(endpointId(), other.endpointId())
                && Objects.equals(destinationSocketAddress(), other.destinationSocketAddress())
                && hasProtocols() == other.hasProtocols() && Objects.equals(protocolsAsStrings(), other.protocolsAsStrings())
                && Objects.equals(destinationTrafficStateAsString(), other.destinationTrafficStateAsString());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("PortMapping").add("AcceleratorPort", acceleratorPort())
                .add("EndpointGroupArn", endpointGroupArn()).add("EndpointId", endpointId())
                .add("DestinationSocketAddress", destinationSocketAddress())
                .add("Protocols", hasProtocols() ? protocolsAsStrings() : null)
                .add("DestinationTrafficState", destinationTrafficStateAsString()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "AcceleratorPort":
            return Optional.ofNullable(clazz.cast(acceleratorPort()));
        case "EndpointGroupArn":
            return Optional.ofNullable(clazz.cast(endpointGroupArn()));
        case "EndpointId":
            return Optional.ofNullable(clazz.cast(endpointId()));
        case "DestinationSocketAddress":
            return Optional.ofNullable(clazz.cast(destinationSocketAddress()));
        case "Protocols":
            return Optional.ofNullable(clazz.cast(protocolsAsStrings()));
        case "DestinationTrafficState":
            return Optional.ofNullable(clazz.cast(destinationTrafficStateAsString()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<PortMapping, T> g) {
        return obj -> g.apply((PortMapping) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, PortMapping> {
        /**
         * <p>
         * The accelerator port.
         * </p>
         * 
         * @param acceleratorPort
         *        The accelerator port.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder acceleratorPort(Integer acceleratorPort);

        /**
         * <p>
         * The Amazon Resource Name (ARN) of the endpoint group.
         * </p>
         * 
         * @param endpointGroupArn
         *        The Amazon Resource Name (ARN) of the endpoint group.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder endpointGroupArn(String endpointGroupArn);

        /**
         * <p>
         * The IP address of the VPC subnet (the subnet ID).
         * </p>
         * 
         * @param endpointId
         *        The IP address of the VPC subnet (the subnet ID).
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder endpointId(String endpointId);

        /**
         * <p>
         * The EC2 instance IP address and port number in the virtual private cloud (VPC) subnet.
         * </p>
         * 
         * @param destinationSocketAddress
         *        The EC2 instance IP address and port number in the virtual private cloud (VPC) subnet.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder destinationSocketAddress(SocketAddress destinationSocketAddress);

        /**
         * <p>
         * The EC2 instance IP address and port number in the virtual private cloud (VPC) subnet.
         * </p>
         * This is a convenience that creates an instance of the {@link SocketAddress.Builder} avoiding the need to
         * create one manually via {@link SocketAddress#builder()}.
         *
         * When the {@link Consumer} completes, {@link SocketAddress.Builder#build()} is called immediately and its
         * result is passed to {@link #destinationSocketAddress(SocketAddress)}.
         * 
         * @param destinationSocketAddress
         *        a consumer that will call methods on {@link SocketAddress.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #destinationSocketAddress(SocketAddress)
         */
        default Builder destinationSocketAddress(Consumer<SocketAddress.Builder> destinationSocketAddress) {
            return destinationSocketAddress(SocketAddress.builder().applyMutation(destinationSocketAddress).build());
        }

        /**
         * <p>
         * The protocols supported by the endpoint group.
         * </p>
         * 
         * @param protocols
         *        The protocols supported by the endpoint group.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder protocolsWithStrings(Collection<String> protocols);

        /**
         * <p>
         * The protocols supported by the endpoint group.
         * </p>
         * 
         * @param protocols
         *        The protocols supported by the endpoint group.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder protocolsWithStrings(String... protocols);

        /**
         * <p>
         * The protocols supported by the endpoint group.
         * </p>
         * 
         * @param protocols
         *        The protocols supported by the endpoint group.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder protocols(Collection<CustomRoutingProtocol> protocols);

        /**
         * <p>
         * The protocols supported by the endpoint group.
         * </p>
         * 
         * @param protocols
         *        The protocols supported by the endpoint group.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder protocols(CustomRoutingProtocol... protocols);

        /**
         * <p>
         * Indicates whether or not a port mapping destination can receive traffic. The value is either ALLOW, if
         * traffic is allowed to the destination, or DENY, if traffic is not allowed to the destination.
         * </p>
         * 
         * @param destinationTrafficState
         *        Indicates whether or not a port mapping destination can receive traffic. The value is either ALLOW, if
         *        traffic is allowed to the destination, or DENY, if traffic is not allowed to the destination.
         * @see CustomRoutingDestinationTrafficState
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see CustomRoutingDestinationTrafficState
         */
        Builder destinationTrafficState(String destinationTrafficState);

        /**
         * <p>
         * Indicates whether or not a port mapping destination can receive traffic. The value is either ALLOW, if
         * traffic is allowed to the destination, or DENY, if traffic is not allowed to the destination.
         * </p>
         * 
         * @param destinationTrafficState
         *        Indicates whether or not a port mapping destination can receive traffic. The value is either ALLOW, if
         *        traffic is allowed to the destination, or DENY, if traffic is not allowed to the destination.
         * @see CustomRoutingDestinationTrafficState
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see CustomRoutingDestinationTrafficState
         */
        Builder destinationTrafficState(CustomRoutingDestinationTrafficState destinationTrafficState);
    }

    static final class BuilderImpl implements Builder {
        private Integer acceleratorPort;

        private String endpointGroupArn;

        private String endpointId;

        private SocketAddress destinationSocketAddress;

        private List<String> protocols = DefaultSdkAutoConstructList.getInstance();

        private String destinationTrafficState;

        private BuilderImpl() {
        }

        private BuilderImpl(PortMapping model) {
            acceleratorPort(model.acceleratorPort);
            endpointGroupArn(model.endpointGroupArn);
            endpointId(model.endpointId);
            destinationSocketAddress(model.destinationSocketAddress);
            protocolsWithStrings(model.protocols);
            destinationTrafficState(model.destinationTrafficState);
        }

        public final Integer getAcceleratorPort() {
            return acceleratorPort;
        }

        @Override
        public final Builder acceleratorPort(Integer acceleratorPort) {
            this.acceleratorPort = acceleratorPort;
            return this;
        }

        public final void setAcceleratorPort(Integer acceleratorPort) {
            this.acceleratorPort = acceleratorPort;
        }

        public final String getEndpointGroupArn() {
            return endpointGroupArn;
        }

        @Override
        public final Builder endpointGroupArn(String endpointGroupArn) {
            this.endpointGroupArn = endpointGroupArn;
            return this;
        }

        public final void setEndpointGroupArn(String endpointGroupArn) {
            this.endpointGroupArn = endpointGroupArn;
        }

        public final String getEndpointId() {
            return endpointId;
        }

        @Override
        public final Builder endpointId(String endpointId) {
            this.endpointId = endpointId;
            return this;
        }

        public final void setEndpointId(String endpointId) {
            this.endpointId = endpointId;
        }

        public final SocketAddress.Builder getDestinationSocketAddress() {
            return destinationSocketAddress != null ? destinationSocketAddress.toBuilder() : null;
        }

        @Override
        public final Builder destinationSocketAddress(SocketAddress destinationSocketAddress) {
            this.destinationSocketAddress = destinationSocketAddress;
            return this;
        }

        public final void setDestinationSocketAddress(SocketAddress.BuilderImpl destinationSocketAddress) {
            this.destinationSocketAddress = destinationSocketAddress != null ? destinationSocketAddress.build() : null;
        }

        public final Collection<String> getProtocols() {
            if (protocols instanceof SdkAutoConstructList) {
                return null;
            }
            return protocols;
        }

        @Override
        public final Builder protocolsWithStrings(Collection<String> protocols) {
            this.protocols = CustomRoutingProtocolsCopier.copy(protocols);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder protocolsWithStrings(String... protocols) {
            protocolsWithStrings(Arrays.asList(protocols));
            return this;
        }

        @Override
        public final Builder protocols(Collection<CustomRoutingProtocol> protocols) {
            this.protocols = CustomRoutingProtocolsCopier.copyEnumToString(protocols);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder protocols(CustomRoutingProtocol... protocols) {
            protocols(Arrays.asList(protocols));
            return this;
        }

        public final void setProtocols(Collection<String> protocols) {
            this.protocols = CustomRoutingProtocolsCopier.copy(protocols);
        }

        public final String getDestinationTrafficState() {
            return destinationTrafficState;
        }

        @Override
        public final Builder destinationTrafficState(String destinationTrafficState) {
            this.destinationTrafficState = destinationTrafficState;
            return this;
        }

        @Override
        public final Builder destinationTrafficState(CustomRoutingDestinationTrafficState destinationTrafficState) {
            this.destinationTrafficState(destinationTrafficState == null ? null : destinationTrafficState.toString());
            return this;
        }

        public final void setDestinationTrafficState(String destinationTrafficState) {
            this.destinationTrafficState = destinationTrafficState;
        }

        @Override
        public PortMapping build() {
            return new PortMapping(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
