/*
 * Copyright 2015-2020 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.globalaccelerator;

import java.util.concurrent.CompletableFuture;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.client.handler.AwsAsyncClientHandler;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.core.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.client.handler.AsyncClientHandler;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.protocols.core.ExceptionMetadata;
import software.amazon.awssdk.protocols.json.AwsJsonProtocol;
import software.amazon.awssdk.protocols.json.AwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.BaseAwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.JsonOperationMetadata;
import software.amazon.awssdk.services.globalaccelerator.model.AcceleratorNotDisabledException;
import software.amazon.awssdk.services.globalaccelerator.model.AcceleratorNotFoundException;
import software.amazon.awssdk.services.globalaccelerator.model.AccessDeniedException;
import software.amazon.awssdk.services.globalaccelerator.model.AssociatedEndpointGroupFoundException;
import software.amazon.awssdk.services.globalaccelerator.model.AssociatedListenerFoundException;
import software.amazon.awssdk.services.globalaccelerator.model.CreateAcceleratorRequest;
import software.amazon.awssdk.services.globalaccelerator.model.CreateAcceleratorResponse;
import software.amazon.awssdk.services.globalaccelerator.model.CreateEndpointGroupRequest;
import software.amazon.awssdk.services.globalaccelerator.model.CreateEndpointGroupResponse;
import software.amazon.awssdk.services.globalaccelerator.model.CreateListenerRequest;
import software.amazon.awssdk.services.globalaccelerator.model.CreateListenerResponse;
import software.amazon.awssdk.services.globalaccelerator.model.DeleteAcceleratorRequest;
import software.amazon.awssdk.services.globalaccelerator.model.DeleteAcceleratorResponse;
import software.amazon.awssdk.services.globalaccelerator.model.DeleteEndpointGroupRequest;
import software.amazon.awssdk.services.globalaccelerator.model.DeleteEndpointGroupResponse;
import software.amazon.awssdk.services.globalaccelerator.model.DeleteListenerRequest;
import software.amazon.awssdk.services.globalaccelerator.model.DeleteListenerResponse;
import software.amazon.awssdk.services.globalaccelerator.model.DescribeAcceleratorAttributesRequest;
import software.amazon.awssdk.services.globalaccelerator.model.DescribeAcceleratorAttributesResponse;
import software.amazon.awssdk.services.globalaccelerator.model.DescribeAcceleratorRequest;
import software.amazon.awssdk.services.globalaccelerator.model.DescribeAcceleratorResponse;
import software.amazon.awssdk.services.globalaccelerator.model.DescribeEndpointGroupRequest;
import software.amazon.awssdk.services.globalaccelerator.model.DescribeEndpointGroupResponse;
import software.amazon.awssdk.services.globalaccelerator.model.DescribeListenerRequest;
import software.amazon.awssdk.services.globalaccelerator.model.DescribeListenerResponse;
import software.amazon.awssdk.services.globalaccelerator.model.EndpointGroupAlreadyExistsException;
import software.amazon.awssdk.services.globalaccelerator.model.EndpointGroupNotFoundException;
import software.amazon.awssdk.services.globalaccelerator.model.GlobalAcceleratorException;
import software.amazon.awssdk.services.globalaccelerator.model.InternalServiceErrorException;
import software.amazon.awssdk.services.globalaccelerator.model.InvalidArgumentException;
import software.amazon.awssdk.services.globalaccelerator.model.InvalidNextTokenException;
import software.amazon.awssdk.services.globalaccelerator.model.InvalidPortRangeException;
import software.amazon.awssdk.services.globalaccelerator.model.LimitExceededException;
import software.amazon.awssdk.services.globalaccelerator.model.ListAcceleratorsRequest;
import software.amazon.awssdk.services.globalaccelerator.model.ListAcceleratorsResponse;
import software.amazon.awssdk.services.globalaccelerator.model.ListEndpointGroupsRequest;
import software.amazon.awssdk.services.globalaccelerator.model.ListEndpointGroupsResponse;
import software.amazon.awssdk.services.globalaccelerator.model.ListListenersRequest;
import software.amazon.awssdk.services.globalaccelerator.model.ListListenersResponse;
import software.amazon.awssdk.services.globalaccelerator.model.ListenerNotFoundException;
import software.amazon.awssdk.services.globalaccelerator.model.UpdateAcceleratorAttributesRequest;
import software.amazon.awssdk.services.globalaccelerator.model.UpdateAcceleratorAttributesResponse;
import software.amazon.awssdk.services.globalaccelerator.model.UpdateAcceleratorRequest;
import software.amazon.awssdk.services.globalaccelerator.model.UpdateAcceleratorResponse;
import software.amazon.awssdk.services.globalaccelerator.model.UpdateEndpointGroupRequest;
import software.amazon.awssdk.services.globalaccelerator.model.UpdateEndpointGroupResponse;
import software.amazon.awssdk.services.globalaccelerator.model.UpdateListenerRequest;
import software.amazon.awssdk.services.globalaccelerator.model.UpdateListenerResponse;
import software.amazon.awssdk.services.globalaccelerator.transform.CreateAcceleratorRequestMarshaller;
import software.amazon.awssdk.services.globalaccelerator.transform.CreateEndpointGroupRequestMarshaller;
import software.amazon.awssdk.services.globalaccelerator.transform.CreateListenerRequestMarshaller;
import software.amazon.awssdk.services.globalaccelerator.transform.DeleteAcceleratorRequestMarshaller;
import software.amazon.awssdk.services.globalaccelerator.transform.DeleteEndpointGroupRequestMarshaller;
import software.amazon.awssdk.services.globalaccelerator.transform.DeleteListenerRequestMarshaller;
import software.amazon.awssdk.services.globalaccelerator.transform.DescribeAcceleratorAttributesRequestMarshaller;
import software.amazon.awssdk.services.globalaccelerator.transform.DescribeAcceleratorRequestMarshaller;
import software.amazon.awssdk.services.globalaccelerator.transform.DescribeEndpointGroupRequestMarshaller;
import software.amazon.awssdk.services.globalaccelerator.transform.DescribeListenerRequestMarshaller;
import software.amazon.awssdk.services.globalaccelerator.transform.ListAcceleratorsRequestMarshaller;
import software.amazon.awssdk.services.globalaccelerator.transform.ListEndpointGroupsRequestMarshaller;
import software.amazon.awssdk.services.globalaccelerator.transform.ListListenersRequestMarshaller;
import software.amazon.awssdk.services.globalaccelerator.transform.UpdateAcceleratorAttributesRequestMarshaller;
import software.amazon.awssdk.services.globalaccelerator.transform.UpdateAcceleratorRequestMarshaller;
import software.amazon.awssdk.services.globalaccelerator.transform.UpdateEndpointGroupRequestMarshaller;
import software.amazon.awssdk.services.globalaccelerator.transform.UpdateListenerRequestMarshaller;
import software.amazon.awssdk.utils.CompletableFutureUtils;

/**
 * Internal implementation of {@link GlobalAcceleratorAsyncClient}.
 *
 * @see GlobalAcceleratorAsyncClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultGlobalAcceleratorAsyncClient implements GlobalAcceleratorAsyncClient {
    private static final Logger log = LoggerFactory.getLogger(DefaultGlobalAcceleratorAsyncClient.class);

    private final AsyncClientHandler clientHandler;

    private final AwsJsonProtocolFactory protocolFactory;

    private final SdkClientConfiguration clientConfiguration;

    protected DefaultGlobalAcceleratorAsyncClient(SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsAsyncClientHandler(clientConfiguration);
        this.clientConfiguration = clientConfiguration;
        this.protocolFactory = init(AwsJsonProtocolFactory.builder()).build();
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    /**
     * <p>
     * Create an accelerator. An accelerator includes one or more listeners that process inbound connections and direct
     * traffic to one or more endpoint groups, each of which includes endpoints, such as Network Load Balancers. To see
     * an AWS CLI example of creating an accelerator, scroll down to <b>Example</b>.
     * </p>
     * <important>
     * <p>
     * You must specify the US-West-2 (Oregon) Region to create or update accelerators.
     * </p>
     * </important>
     *
     * @param createAcceleratorRequest
     * @return A Java Future containing the result of the CreateAccelerator operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServiceErrorException There was an internal error for AWS Global Accelerator.</li>
     *         <li>InvalidArgumentException An argument that you specified is invalid.</li>
     *         <li>LimitExceededException Processing your request would cause you to exceed an AWS Global Accelerator
     *         limit.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>GlobalAcceleratorException Base class for all service exceptions. Unknown exceptions will be thrown
     *         as an instance of this type.</li>
     *         </ul>
     * @sample GlobalAcceleratorAsyncClient.CreateAccelerator
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/globalaccelerator-2018-08-08/CreateAccelerator"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<CreateAcceleratorResponse> createAccelerator(CreateAcceleratorRequest createAcceleratorRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateAcceleratorResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, CreateAcceleratorResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateAcceleratorResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateAcceleratorRequest, CreateAcceleratorResponse>()
                            .withOperationName("CreateAccelerator")
                            .withMarshaller(new CreateAcceleratorRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(createAcceleratorRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Create an endpoint group for the specified listener. An endpoint group is a collection of endpoints in one AWS
     * Region. To see an AWS CLI example of creating an endpoint group, scroll down to <b>Example</b>.
     * </p>
     *
     * @param createEndpointGroupRequest
     * @return A Java Future containing the result of the CreateEndpointGroup operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>AcceleratorNotFoundException The accelerator that you specified doesn't exist.</li>
     *         <li>EndpointGroupAlreadyExistsException The endpoint group that you specified already exists.</li>
     *         <li>ListenerNotFoundException The listener that you specified doesn't exist.</li>
     *         <li>InternalServiceErrorException There was an internal error for AWS Global Accelerator.</li>
     *         <li>InvalidArgumentException An argument that you specified is invalid.</li>
     *         <li>LimitExceededException Processing your request would cause you to exceed an AWS Global Accelerator
     *         limit.</li>
     *         <li>AccessDeniedException You don't have access permission.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>GlobalAcceleratorException Base class for all service exceptions. Unknown exceptions will be thrown
     *         as an instance of this type.</li>
     *         </ul>
     * @sample GlobalAcceleratorAsyncClient.CreateEndpointGroup
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/globalaccelerator-2018-08-08/CreateEndpointGroup"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<CreateEndpointGroupResponse> createEndpointGroup(
            CreateEndpointGroupRequest createEndpointGroupRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateEndpointGroupResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, CreateEndpointGroupResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateEndpointGroupResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateEndpointGroupRequest, CreateEndpointGroupResponse>()
                            .withOperationName("CreateEndpointGroup")
                            .withMarshaller(new CreateEndpointGroupRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(createEndpointGroupRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Create a listener to process inbound connections from clients to an accelerator. Connections arrive to assigned
     * static IP addresses on a port, port range, or list of port ranges that you specify. To see an AWS CLI example of
     * creating a listener, scroll down to <b>Example</b>.
     * </p>
     *
     * @param createListenerRequest
     * @return A Java Future containing the result of the CreateListener operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidArgumentException An argument that you specified is invalid.</li>
     *         <li>AcceleratorNotFoundException The accelerator that you specified doesn't exist.</li>
     *         <li>InvalidPortRangeException The port numbers that you specified are not valid numbers or are not unique
     *         for this accelerator.</li>
     *         <li>InternalServiceErrorException There was an internal error for AWS Global Accelerator.</li>
     *         <li>LimitExceededException Processing your request would cause you to exceed an AWS Global Accelerator
     *         limit.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>GlobalAcceleratorException Base class for all service exceptions. Unknown exceptions will be thrown
     *         as an instance of this type.</li>
     *         </ul>
     * @sample GlobalAcceleratorAsyncClient.CreateListener
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/globalaccelerator-2018-08-08/CreateListener"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<CreateListenerResponse> createListener(CreateListenerRequest createListenerRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateListenerResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, CreateListenerResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateListenerResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateListenerRequest, CreateListenerResponse>()
                            .withOperationName("CreateListener")
                            .withMarshaller(new CreateListenerRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(createListenerRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Delete an accelerator. Note: before you can delete an accelerator, you must disable it and remove all dependent
     * resources (listeners and endpoint groups).
     * </p>
     *
     * @param deleteAcceleratorRequest
     * @return A Java Future containing the result of the DeleteAccelerator operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>AcceleratorNotFoundException The accelerator that you specified doesn't exist.</li>
     *         <li>AcceleratorNotDisabledException The accelerator that you specified could not be disabled.</li>
     *         <li>AssociatedListenerFoundException The accelerator that you specified has a listener associated with
     *         it. You must remove all dependent resources from an accelerator before you can delete it.</li>
     *         <li>InternalServiceErrorException There was an internal error for AWS Global Accelerator.</li>
     *         <li>InvalidArgumentException An argument that you specified is invalid.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>GlobalAcceleratorException Base class for all service exceptions. Unknown exceptions will be thrown
     *         as an instance of this type.</li>
     *         </ul>
     * @sample GlobalAcceleratorAsyncClient.DeleteAccelerator
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/globalaccelerator-2018-08-08/DeleteAccelerator"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteAcceleratorResponse> deleteAccelerator(DeleteAcceleratorRequest deleteAcceleratorRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteAcceleratorResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DeleteAcceleratorResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteAcceleratorResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteAcceleratorRequest, DeleteAcceleratorResponse>()
                            .withOperationName("DeleteAccelerator")
                            .withMarshaller(new DeleteAcceleratorRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(deleteAcceleratorRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Delete an endpoint group from a listener.
     * </p>
     *
     * @param deleteEndpointGroupRequest
     * @return A Java Future containing the result of the DeleteEndpointGroup operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidArgumentException An argument that you specified is invalid.</li>
     *         <li>EndpointGroupNotFoundException The endpoint group that you specified doesn't exist.</li>
     *         <li>InternalServiceErrorException There was an internal error for AWS Global Accelerator.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>GlobalAcceleratorException Base class for all service exceptions. Unknown exceptions will be thrown
     *         as an instance of this type.</li>
     *         </ul>
     * @sample GlobalAcceleratorAsyncClient.DeleteEndpointGroup
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/globalaccelerator-2018-08-08/DeleteEndpointGroup"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteEndpointGroupResponse> deleteEndpointGroup(
            DeleteEndpointGroupRequest deleteEndpointGroupRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteEndpointGroupResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DeleteEndpointGroupResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteEndpointGroupResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteEndpointGroupRequest, DeleteEndpointGroupResponse>()
                            .withOperationName("DeleteEndpointGroup")
                            .withMarshaller(new DeleteEndpointGroupRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(deleteEndpointGroupRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Delete a listener from an accelerator.
     * </p>
     *
     * @param deleteListenerRequest
     * @return A Java Future containing the result of the DeleteListener operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidArgumentException An argument that you specified is invalid.</li>
     *         <li>ListenerNotFoundException The listener that you specified doesn't exist.</li>
     *         <li>AssociatedEndpointGroupFoundException The listener that you specified has an endpoint group
     *         associated with it. You must remove all dependent resources from a listener before you can delete it.</li>
     *         <li>InternalServiceErrorException There was an internal error for AWS Global Accelerator.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>GlobalAcceleratorException Base class for all service exceptions. Unknown exceptions will be thrown
     *         as an instance of this type.</li>
     *         </ul>
     * @sample GlobalAcceleratorAsyncClient.DeleteListener
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/globalaccelerator-2018-08-08/DeleteListener"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteListenerResponse> deleteListener(DeleteListenerRequest deleteListenerRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteListenerResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DeleteListenerResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteListenerResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteListenerRequest, DeleteListenerResponse>()
                            .withOperationName("DeleteListener")
                            .withMarshaller(new DeleteListenerRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(deleteListenerRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Describe an accelerator. To see an AWS CLI example of describing an accelerator, scroll down to <b>Example</b>.
     * </p>
     *
     * @param describeAcceleratorRequest
     * @return A Java Future containing the result of the DescribeAccelerator operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>AcceleratorNotFoundException The accelerator that you specified doesn't exist.</li>
     *         <li>InternalServiceErrorException There was an internal error for AWS Global Accelerator.</li>
     *         <li>InvalidArgumentException An argument that you specified is invalid.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>GlobalAcceleratorException Base class for all service exceptions. Unknown exceptions will be thrown
     *         as an instance of this type.</li>
     *         </ul>
     * @sample GlobalAcceleratorAsyncClient.DescribeAccelerator
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/globalaccelerator-2018-08-08/DescribeAccelerator"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DescribeAcceleratorResponse> describeAccelerator(
            DescribeAcceleratorRequest describeAcceleratorRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DescribeAcceleratorResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DescribeAcceleratorResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DescribeAcceleratorResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribeAcceleratorRequest, DescribeAcceleratorResponse>()
                            .withOperationName("DescribeAccelerator")
                            .withMarshaller(new DescribeAcceleratorRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(describeAcceleratorRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Describe the attributes of an accelerator.
     * </p>
     *
     * @param describeAcceleratorAttributesRequest
     * @return A Java Future containing the result of the DescribeAcceleratorAttributes operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>AcceleratorNotFoundException The accelerator that you specified doesn't exist.</li>
     *         <li>InternalServiceErrorException There was an internal error for AWS Global Accelerator.</li>
     *         <li>InvalidArgumentException An argument that you specified is invalid.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>GlobalAcceleratorException Base class for all service exceptions. Unknown exceptions will be thrown
     *         as an instance of this type.</li>
     *         </ul>
     * @sample GlobalAcceleratorAsyncClient.DescribeAcceleratorAttributes
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/globalaccelerator-2018-08-08/DescribeAcceleratorAttributes"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DescribeAcceleratorAttributesResponse> describeAcceleratorAttributes(
            DescribeAcceleratorAttributesRequest describeAcceleratorAttributesRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DescribeAcceleratorAttributesResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DescribeAcceleratorAttributesResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DescribeAcceleratorAttributesResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribeAcceleratorAttributesRequest, DescribeAcceleratorAttributesResponse>()
                            .withOperationName("DescribeAcceleratorAttributes")
                            .withMarshaller(new DescribeAcceleratorAttributesRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(describeAcceleratorAttributesRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Describe an endpoint group.
     * </p>
     *
     * @param describeEndpointGroupRequest
     * @return A Java Future containing the result of the DescribeEndpointGroup operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidArgumentException An argument that you specified is invalid.</li>
     *         <li>EndpointGroupNotFoundException The endpoint group that you specified doesn't exist.</li>
     *         <li>InternalServiceErrorException There was an internal error for AWS Global Accelerator.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>GlobalAcceleratorException Base class for all service exceptions. Unknown exceptions will be thrown
     *         as an instance of this type.</li>
     *         </ul>
     * @sample GlobalAcceleratorAsyncClient.DescribeEndpointGroup
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/globalaccelerator-2018-08-08/DescribeEndpointGroup"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DescribeEndpointGroupResponse> describeEndpointGroup(
            DescribeEndpointGroupRequest describeEndpointGroupRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DescribeEndpointGroupResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DescribeEndpointGroupResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DescribeEndpointGroupResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribeEndpointGroupRequest, DescribeEndpointGroupResponse>()
                            .withOperationName("DescribeEndpointGroup")
                            .withMarshaller(new DescribeEndpointGroupRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(describeEndpointGroupRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Describe a listener.
     * </p>
     *
     * @param describeListenerRequest
     * @return A Java Future containing the result of the DescribeListener operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidArgumentException An argument that you specified is invalid.</li>
     *         <li>ListenerNotFoundException The listener that you specified doesn't exist.</li>
     *         <li>InternalServiceErrorException There was an internal error for AWS Global Accelerator.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>GlobalAcceleratorException Base class for all service exceptions. Unknown exceptions will be thrown
     *         as an instance of this type.</li>
     *         </ul>
     * @sample GlobalAcceleratorAsyncClient.DescribeListener
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/globalaccelerator-2018-08-08/DescribeListener"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DescribeListenerResponse> describeListener(DescribeListenerRequest describeListenerRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DescribeListenerResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DescribeListenerResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DescribeListenerResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribeListenerRequest, DescribeListenerResponse>()
                            .withOperationName("DescribeListener")
                            .withMarshaller(new DescribeListenerRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(describeListenerRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * List the accelerators for an AWS account.
     * </p>
     *
     * @param listAcceleratorsRequest
     * @return A Java Future containing the result of the ListAccelerators operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidArgumentException An argument that you specified is invalid.</li>
     *         <li>InvalidNextTokenException There isn't another item to return.</li>
     *         <li>InternalServiceErrorException There was an internal error for AWS Global Accelerator.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>GlobalAcceleratorException Base class for all service exceptions. Unknown exceptions will be thrown
     *         as an instance of this type.</li>
     *         </ul>
     * @sample GlobalAcceleratorAsyncClient.ListAccelerators
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/globalaccelerator-2018-08-08/ListAccelerators"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListAcceleratorsResponse> listAccelerators(ListAcceleratorsRequest listAcceleratorsRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListAcceleratorsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListAcceleratorsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListAcceleratorsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListAcceleratorsRequest, ListAcceleratorsResponse>()
                            .withOperationName("ListAccelerators")
                            .withMarshaller(new ListAcceleratorsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(listAcceleratorsRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * List the endpoint groups that are associated with a listener.
     * </p>
     *
     * @param listEndpointGroupsRequest
     * @return A Java Future containing the result of the ListEndpointGroups operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ListenerNotFoundException The listener that you specified doesn't exist.</li>
     *         <li>InvalidNextTokenException There isn't another item to return.</li>
     *         <li>InvalidArgumentException An argument that you specified is invalid.</li>
     *         <li>InternalServiceErrorException There was an internal error for AWS Global Accelerator.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>GlobalAcceleratorException Base class for all service exceptions. Unknown exceptions will be thrown
     *         as an instance of this type.</li>
     *         </ul>
     * @sample GlobalAcceleratorAsyncClient.ListEndpointGroups
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/globalaccelerator-2018-08-08/ListEndpointGroups"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListEndpointGroupsResponse> listEndpointGroups(ListEndpointGroupsRequest listEndpointGroupsRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListEndpointGroupsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListEndpointGroupsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListEndpointGroupsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListEndpointGroupsRequest, ListEndpointGroupsResponse>()
                            .withOperationName("ListEndpointGroups")
                            .withMarshaller(new ListEndpointGroupsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(listEndpointGroupsRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * List the listeners for an accelerator.
     * </p>
     *
     * @param listListenersRequest
     * @return A Java Future containing the result of the ListListeners operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidArgumentException An argument that you specified is invalid.</li>
     *         <li>AcceleratorNotFoundException The accelerator that you specified doesn't exist.</li>
     *         <li>InvalidNextTokenException There isn't another item to return.</li>
     *         <li>InternalServiceErrorException There was an internal error for AWS Global Accelerator.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>GlobalAcceleratorException Base class for all service exceptions. Unknown exceptions will be thrown
     *         as an instance of this type.</li>
     *         </ul>
     * @sample GlobalAcceleratorAsyncClient.ListListeners
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/globalaccelerator-2018-08-08/ListListeners"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListListenersResponse> listListeners(ListListenersRequest listListenersRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListListenersResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    ListListenersResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListListenersResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListListenersRequest, ListListenersResponse>()
                            .withOperationName("ListListeners")
                            .withMarshaller(new ListListenersRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(listListenersRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Update an accelerator. To see an AWS CLI example of updating an accelerator, scroll down to <b>Example</b>.
     * </p>
     * <important>
     * <p>
     * You must specify the US-West-2 (Oregon) Region to create or update accelerators.
     * </p>
     * </important>
     *
     * @param updateAcceleratorRequest
     * @return A Java Future containing the result of the UpdateAccelerator operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>AcceleratorNotFoundException The accelerator that you specified doesn't exist.</li>
     *         <li>InternalServiceErrorException There was an internal error for AWS Global Accelerator.</li>
     *         <li>InvalidArgumentException An argument that you specified is invalid.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>GlobalAcceleratorException Base class for all service exceptions. Unknown exceptions will be thrown
     *         as an instance of this type.</li>
     *         </ul>
     * @sample GlobalAcceleratorAsyncClient.UpdateAccelerator
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/globalaccelerator-2018-08-08/UpdateAccelerator"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateAcceleratorResponse> updateAccelerator(UpdateAcceleratorRequest updateAcceleratorRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateAcceleratorResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, UpdateAcceleratorResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UpdateAcceleratorResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateAcceleratorRequest, UpdateAcceleratorResponse>()
                            .withOperationName("UpdateAccelerator")
                            .withMarshaller(new UpdateAcceleratorRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(updateAcceleratorRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Update the attributes for an accelerator. To see an AWS CLI example of updating an accelerator to enable flow
     * logs, scroll down to <b>Example</b>.
     * </p>
     *
     * @param updateAcceleratorAttributesRequest
     * @return A Java Future containing the result of the UpdateAcceleratorAttributes operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>AcceleratorNotFoundException The accelerator that you specified doesn't exist.</li>
     *         <li>InternalServiceErrorException There was an internal error for AWS Global Accelerator.</li>
     *         <li>InvalidArgumentException An argument that you specified is invalid.</li>
     *         <li>AccessDeniedException You don't have access permission.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>GlobalAcceleratorException Base class for all service exceptions. Unknown exceptions will be thrown
     *         as an instance of this type.</li>
     *         </ul>
     * @sample GlobalAcceleratorAsyncClient.UpdateAcceleratorAttributes
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/globalaccelerator-2018-08-08/UpdateAcceleratorAttributes"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateAcceleratorAttributesResponse> updateAcceleratorAttributes(
            UpdateAcceleratorAttributesRequest updateAcceleratorAttributesRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateAcceleratorAttributesResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, UpdateAcceleratorAttributesResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UpdateAcceleratorAttributesResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateAcceleratorAttributesRequest, UpdateAcceleratorAttributesResponse>()
                            .withOperationName("UpdateAcceleratorAttributes")
                            .withMarshaller(new UpdateAcceleratorAttributesRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(updateAcceleratorAttributesRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Update an endpoint group. To see an AWS CLI example of updating an endpoint group, scroll down to <b>Example</b>.
     * </p>
     *
     * @param updateEndpointGroupRequest
     * @return A Java Future containing the result of the UpdateEndpointGroup operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidArgumentException An argument that you specified is invalid.</li>
     *         <li>EndpointGroupNotFoundException The endpoint group that you specified doesn't exist.</li>
     *         <li>InternalServiceErrorException There was an internal error for AWS Global Accelerator.</li>
     *         <li>LimitExceededException Processing your request would cause you to exceed an AWS Global Accelerator
     *         limit.</li>
     *         <li>AccessDeniedException You don't have access permission.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>GlobalAcceleratorException Base class for all service exceptions. Unknown exceptions will be thrown
     *         as an instance of this type.</li>
     *         </ul>
     * @sample GlobalAcceleratorAsyncClient.UpdateEndpointGroup
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/globalaccelerator-2018-08-08/UpdateEndpointGroup"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateEndpointGroupResponse> updateEndpointGroup(
            UpdateEndpointGroupRequest updateEndpointGroupRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateEndpointGroupResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, UpdateEndpointGroupResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UpdateEndpointGroupResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateEndpointGroupRequest, UpdateEndpointGroupResponse>()
                            .withOperationName("UpdateEndpointGroup")
                            .withMarshaller(new UpdateEndpointGroupRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(updateEndpointGroupRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Update a listener.
     * </p>
     *
     * @param updateListenerRequest
     * @return A Java Future containing the result of the UpdateListener operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidArgumentException An argument that you specified is invalid.</li>
     *         <li>InvalidPortRangeException The port numbers that you specified are not valid numbers or are not unique
     *         for this accelerator.</li>
     *         <li>ListenerNotFoundException The listener that you specified doesn't exist.</li>
     *         <li>InternalServiceErrorException There was an internal error for AWS Global Accelerator.</li>
     *         <li>LimitExceededException Processing your request would cause you to exceed an AWS Global Accelerator
     *         limit.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>GlobalAcceleratorException Base class for all service exceptions. Unknown exceptions will be thrown
     *         as an instance of this type.</li>
     *         </ul>
     * @sample GlobalAcceleratorAsyncClient.UpdateListener
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/globalaccelerator-2018-08-08/UpdateListener"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateListenerResponse> updateListener(UpdateListenerRequest updateListenerRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateListenerResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, UpdateListenerResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UpdateListenerResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateListenerRequest, UpdateListenerResponse>()
                            .withOperationName("UpdateListener")
                            .withMarshaller(new UpdateListenerRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(updateListenerRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    @Override
    public void close() {
        clientHandler.close();
    }

    private <T extends BaseAwsJsonProtocolFactory.Builder<T>> T init(T builder) {
        return builder
                .clientConfiguration(clientConfiguration)
                .defaultServiceExceptionSupplier(GlobalAcceleratorException::builder)
                .protocol(AwsJsonProtocol.AWS_JSON)
                .protocolVersion("1.1")
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("AccessDeniedException")
                                .exceptionBuilderSupplier(AccessDeniedException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidArgumentException")
                                .exceptionBuilderSupplier(InvalidArgumentException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ListenerNotFoundException")
                                .exceptionBuilderSupplier(ListenerNotFoundException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("AcceleratorNotFoundException")
                                .exceptionBuilderSupplier(AcceleratorNotFoundException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("AssociatedListenerFoundException")
                                .exceptionBuilderSupplier(AssociatedListenerFoundException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InternalServiceErrorException")
                                .exceptionBuilderSupplier(InternalServiceErrorException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("AssociatedEndpointGroupFoundException")
                                .exceptionBuilderSupplier(AssociatedEndpointGroupFoundException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("EndpointGroupNotFoundException")
                                .exceptionBuilderSupplier(EndpointGroupNotFoundException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("AcceleratorNotDisabledException")
                                .exceptionBuilderSupplier(AcceleratorNotDisabledException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidNextTokenException")
                                .exceptionBuilderSupplier(InvalidNextTokenException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("EndpointGroupAlreadyExistsException")
                                .exceptionBuilderSupplier(EndpointGroupAlreadyExistsException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidPortRangeException")
                                .exceptionBuilderSupplier(InvalidPortRangeException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("LimitExceededException")
                                .exceptionBuilderSupplier(LimitExceededException::builder).build());
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata) {
        return protocolFactory.createErrorResponseHandler(operationMetadata);
    }
}
