/*
 * Copyright 2015-2020 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.forecastquery;

import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.client.handler.AwsSyncClientHandler;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.core.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.client.handler.SyncClientHandler;
import software.amazon.awssdk.core.exception.SdkClientException;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.protocols.core.ExceptionMetadata;
import software.amazon.awssdk.protocols.json.AwsJsonProtocol;
import software.amazon.awssdk.protocols.json.AwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.BaseAwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.JsonOperationMetadata;
import software.amazon.awssdk.services.forecastquery.model.ForecastqueryException;
import software.amazon.awssdk.services.forecastquery.model.InvalidInputException;
import software.amazon.awssdk.services.forecastquery.model.InvalidNextTokenException;
import software.amazon.awssdk.services.forecastquery.model.LimitExceededException;
import software.amazon.awssdk.services.forecastquery.model.QueryForecastRequest;
import software.amazon.awssdk.services.forecastquery.model.QueryForecastResponse;
import software.amazon.awssdk.services.forecastquery.model.ResourceInUseException;
import software.amazon.awssdk.services.forecastquery.model.ResourceNotFoundException;
import software.amazon.awssdk.services.forecastquery.transform.QueryForecastRequestMarshaller;

/**
 * Internal implementation of {@link ForecastqueryClient}.
 *
 * @see ForecastqueryClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultForecastqueryClient implements ForecastqueryClient {
    private final SyncClientHandler clientHandler;

    private final AwsJsonProtocolFactory protocolFactory;

    private final SdkClientConfiguration clientConfiguration;

    protected DefaultForecastqueryClient(SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsSyncClientHandler(clientConfiguration);
        this.clientConfiguration = clientConfiguration;
        this.protocolFactory = init(AwsJsonProtocolFactory.builder()).build();
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    /**
     * <p>
     * Retrieves a forecast filtered by the supplied criteria.
     * </p>
     * <p>
     * The criteria is a key-value pair. The key is either <code>item_id</code> (or the equivalent non-timestamp,
     * non-target field) from the <code>TARGET_TIME_SERIES</code> dataset, or one of the forecast dimensions specified
     * as part of the <code>FeaturizationConfig</code> object.
     * </p>
     * <p>
     * By default, the complete date range of the filtered forecast is returned. Optionally, you can request a specific
     * date range within the forecast.
     * </p>
     * <note>
     * <p>
     * The forecasts generated by Amazon Forecast are in the same timezone as the dataset that was used to create the
     * predictor.
     * </p>
     * </note>
     *
     * @param queryForecastRequest
     * @return Result of the QueryForecast operation returned by the service.
     * @throws ResourceNotFoundException
     *         We can't find that resource. Check the information that you've provided and try again.
     * @throws ResourceInUseException
     *         The specified resource is in use.
     * @throws InvalidInputException
     *         The value that you provided was invalid or too long.
     * @throws LimitExceededException
     *         The limit on the number of requests per second has been exceeded.
     * @throws InvalidNextTokenException
     *         The token is not valid. Tokens expire after 24 hours.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ForecastqueryException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ForecastqueryClient.QueryForecast
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/forecastquery-2018-06-26/QueryForecast" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public QueryForecastResponse queryForecast(QueryForecastRequest queryForecastRequest) throws ResourceNotFoundException,
            ResourceInUseException, InvalidInputException, LimitExceededException, InvalidNextTokenException,
            AwsServiceException, SdkClientException, ForecastqueryException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<QueryForecastResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                QueryForecastResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<QueryForecastRequest, QueryForecastResponse>()
                .withOperationName("QueryForecast").withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(queryForecastRequest)
                .withMarshaller(new QueryForecastRequestMarshaller(protocolFactory)));
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata) {
        return protocolFactory.createErrorResponseHandler(operationMetadata);
    }

    private <T extends BaseAwsJsonProtocolFactory.Builder<T>> T init(T builder) {
        return builder
                .clientConfiguration(clientConfiguration)
                .defaultServiceExceptionSupplier(ForecastqueryException::builder)
                .protocol(AwsJsonProtocol.AWS_JSON)
                .protocolVersion("1.1")
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidInputException")
                                .exceptionBuilderSupplier(InvalidInputException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidNextTokenException")
                                .exceptionBuilderSupplier(InvalidNextTokenException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ResourceInUseException")
                                .exceptionBuilderSupplier(ResourceInUseException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("LimitExceededException")
                                .exceptionBuilderSupplier(LimitExceededException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ResourceNotFoundException")
                                .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
    }

    @Override
    public void close() {
        clientHandler.close();
    }
}
