/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.ebs;

import java.nio.file.Path;
import java.util.concurrent.CompletableFuture;
import java.util.function.Consumer;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkClient;
import software.amazon.awssdk.core.async.AsyncRequestBody;
import software.amazon.awssdk.core.async.AsyncResponseTransformer;
import software.amazon.awssdk.services.ebs.model.CompleteSnapshotRequest;
import software.amazon.awssdk.services.ebs.model.CompleteSnapshotResponse;
import software.amazon.awssdk.services.ebs.model.GetSnapshotBlockRequest;
import software.amazon.awssdk.services.ebs.model.GetSnapshotBlockResponse;
import software.amazon.awssdk.services.ebs.model.ListChangedBlocksRequest;
import software.amazon.awssdk.services.ebs.model.ListChangedBlocksResponse;
import software.amazon.awssdk.services.ebs.model.ListSnapshotBlocksRequest;
import software.amazon.awssdk.services.ebs.model.ListSnapshotBlocksResponse;
import software.amazon.awssdk.services.ebs.model.PutSnapshotBlockRequest;
import software.amazon.awssdk.services.ebs.model.PutSnapshotBlockResponse;
import software.amazon.awssdk.services.ebs.model.StartSnapshotRequest;
import software.amazon.awssdk.services.ebs.model.StartSnapshotResponse;
import software.amazon.awssdk.services.ebs.paginators.ListChangedBlocksPublisher;
import software.amazon.awssdk.services.ebs.paginators.ListSnapshotBlocksPublisher;

/**
 * Service client for accessing Amazon EBS asynchronously. This can be created using the static {@link #builder()}
 * method.
 *
 * <p>
 * You can use the Amazon Elastic Block Store (Amazon EBS) direct APIs to create EBS snapshots, write data directly to
 * your snapshots, read data on your snapshots, and identify the differences or changes between two snapshots. If you’re
 * an independent software vendor (ISV) who offers backup services for Amazon EBS, the EBS direct APIs make it more
 * efficient and cost-effective to track incremental changes on your EBS volumes through snapshots. This can be done
 * without having to create new volumes from snapshots, and then use Amazon Elastic Compute Cloud (Amazon EC2) instances
 * to compare the differences.
 * </p>
 * <p>
 * You can create incremental snapshots directly from data on-premises into EBS volumes and the cloud to use for quick
 * disaster recovery. With the ability to write and read snapshots, you can write your on-premises data to an EBS
 * snapshot during a disaster. Then after recovery, you can restore it back to AWS or on-premises from the snapshot. You
 * no longer need to build and maintain complex mechanisms to copy data to and from Amazon EBS.
 * </p>
 * <p>
 * This API reference provides detailed information about the actions, data types, parameters, and errors of the EBS
 * direct APIs. For more information about the elements that make up the EBS direct APIs, and examples of how to use
 * them effectively, see <a
 * href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/ebs-accessing-snapshot.html">Accessing the Contents of an
 * EBS Snapshot</a> in the <i>Amazon Elastic Compute Cloud User Guide</i>. For more information about the supported AWS
 * Regions, endpoints, and service quotas for the EBS direct APIs, see <a
 * href="https://docs.aws.amazon.com/general/latest/gr/ebs-service.html">Amazon Elastic Block Store Endpoints and
 * Quotas</a> in the <i>AWS General Reference</i>.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public interface EbsAsyncClient extends SdkClient {
    String SERVICE_NAME = "ebs";

    /**
     * Create a {@link EbsAsyncClient} with the region loaded from the
     * {@link software.amazon.awssdk.regions.providers.DefaultAwsRegionProviderChain} and credentials loaded from the
     * {@link software.amazon.awssdk.auth.credentials.DefaultCredentialsProvider}.
     */
    static EbsAsyncClient create() {
        return builder().build();
    }

    /**
     * Create a builder that can be used to configure and create a {@link EbsAsyncClient}.
     */
    static EbsAsyncClientBuilder builder() {
        return new DefaultEbsAsyncClientBuilder();
    }

    /**
     * <p>
     * Seals and completes the snapshot after all of the required blocks of data have been written to it. Completing the
     * snapshot changes the status to <code>completed</code>. You cannot write new blocks to a snapshot after it has
     * been completed.
     * </p>
     *
     * @param completeSnapshotRequest
     * @return A Java Future containing the result of the CompleteSnapshot operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>ValidationException The input fails to satisfy the constraints of the EBS direct APIs.</li>
     *         <li>ResourceNotFoundException The specified resource does not exist.</li>
     *         <li>RequestThrottledException The number of API requests has exceed the maximum allowed API request
     *         throttling limit.</li>
     *         <li>ServiceQuotaExceededException Your current service quotas do not allow you to perform this action.</li>
     *         <li>InternalServerException An internal error has occurred.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>EbsException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample EbsAsyncClient.CompleteSnapshot
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/ebs-2019-11-02/CompleteSnapshot" target="_top">AWS API
     *      Documentation</a>
     */
    default CompletableFuture<CompleteSnapshotResponse> completeSnapshot(CompleteSnapshotRequest completeSnapshotRequest) {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Seals and completes the snapshot after all of the required blocks of data have been written to it. Completing the
     * snapshot changes the status to <code>completed</code>. You cannot write new blocks to a snapshot after it has
     * been completed.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link CompleteSnapshotRequest.Builder} avoiding the need
     * to create one manually via {@link CompleteSnapshotRequest#builder()}
     * </p>
     *
     * @param completeSnapshotRequest
     *        A {@link Consumer} that will call methods on {@link CompleteSnapshotRequest.Builder} to create a request.
     * @return A Java Future containing the result of the CompleteSnapshot operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>ValidationException The input fails to satisfy the constraints of the EBS direct APIs.</li>
     *         <li>ResourceNotFoundException The specified resource does not exist.</li>
     *         <li>RequestThrottledException The number of API requests has exceed the maximum allowed API request
     *         throttling limit.</li>
     *         <li>ServiceQuotaExceededException Your current service quotas do not allow you to perform this action.</li>
     *         <li>InternalServerException An internal error has occurred.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>EbsException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample EbsAsyncClient.CompleteSnapshot
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/ebs-2019-11-02/CompleteSnapshot" target="_top">AWS API
     *      Documentation</a>
     */
    default CompletableFuture<CompleteSnapshotResponse> completeSnapshot(
            Consumer<CompleteSnapshotRequest.Builder> completeSnapshotRequest) {
        return completeSnapshot(CompleteSnapshotRequest.builder().applyMutation(completeSnapshotRequest).build());
    }

    /**
     * <p>
     * Returns the data in a block in an Amazon Elastic Block Store snapshot.
     * </p>
     *
     * @param getSnapshotBlockRequest
     * @param asyncResponseTransformer
     *        The response transformer for processing the streaming response in a non-blocking manner. See
     *        {@link AsyncResponseTransformer} for details on how this callback should be implemented and for links to
     *        precanned implementations for common scenarios like downloading to a file. The service documentation for
     *        the response content is as follows '
     *        <p>
     *        The data content of the block.
     *        </p>
     *        '.
     * @return A future to the transformed result of the AsyncResponseTransformer.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>ValidationException The input fails to satisfy the constraints of the EBS direct APIs.</li>
     *         <li>ResourceNotFoundException The specified resource does not exist.</li>
     *         <li>RequestThrottledException The number of API requests has exceed the maximum allowed API request
     *         throttling limit.</li>
     *         <li>ServiceQuotaExceededException Your current service quotas do not allow you to perform this action.</li>
     *         <li>InternalServerException An internal error has occurred.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>EbsException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample EbsAsyncClient.GetSnapshotBlock
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/ebs-2019-11-02/GetSnapshotBlock" target="_top">AWS API
     *      Documentation</a>
     */
    default <ReturnT> CompletableFuture<ReturnT> getSnapshotBlock(GetSnapshotBlockRequest getSnapshotBlockRequest,
            AsyncResponseTransformer<GetSnapshotBlockResponse, ReturnT> asyncResponseTransformer) {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Returns the data in a block in an Amazon Elastic Block Store snapshot.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link GetSnapshotBlockRequest.Builder} avoiding the need
     * to create one manually via {@link GetSnapshotBlockRequest#builder()}
     * </p>
     *
     * @param getSnapshotBlockRequest
     *        A {@link Consumer} that will call methods on {@link GetSnapshotBlockRequest.Builder} to create a request.
     * @param asyncResponseTransformer
     *        The response transformer for processing the streaming response in a non-blocking manner. See
     *        {@link AsyncResponseTransformer} for details on how this callback should be implemented and for links to
     *        precanned implementations for common scenarios like downloading to a file. The service documentation for
     *        the response content is as follows '
     *        <p>
     *        The data content of the block.
     *        </p>
     *        '.
     * @return A future to the transformed result of the AsyncResponseTransformer.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>ValidationException The input fails to satisfy the constraints of the EBS direct APIs.</li>
     *         <li>ResourceNotFoundException The specified resource does not exist.</li>
     *         <li>RequestThrottledException The number of API requests has exceed the maximum allowed API request
     *         throttling limit.</li>
     *         <li>ServiceQuotaExceededException Your current service quotas do not allow you to perform this action.</li>
     *         <li>InternalServerException An internal error has occurred.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>EbsException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample EbsAsyncClient.GetSnapshotBlock
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/ebs-2019-11-02/GetSnapshotBlock" target="_top">AWS API
     *      Documentation</a>
     */
    default <ReturnT> CompletableFuture<ReturnT> getSnapshotBlock(
            Consumer<GetSnapshotBlockRequest.Builder> getSnapshotBlockRequest,
            AsyncResponseTransformer<GetSnapshotBlockResponse, ReturnT> asyncResponseTransformer) {
        return getSnapshotBlock(GetSnapshotBlockRequest.builder().applyMutation(getSnapshotBlockRequest).build(),
                asyncResponseTransformer);
    }

    /**
     * <p>
     * Returns the data in a block in an Amazon Elastic Block Store snapshot.
     * </p>
     *
     * @param getSnapshotBlockRequest
     * @param destinationPath
     *        {@link Path} to file that response contents will be written to. The file must not exist or this method
     *        will throw an exception. If the file is not writable by the current user then an exception will be thrown.
     *        The service documentation for the response content is as follows '
     *        <p>
     *        The data content of the block.
     *        </p>
     *        '.
     * @return A future to the transformed result of the AsyncResponseTransformer.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>ValidationException The input fails to satisfy the constraints of the EBS direct APIs.</li>
     *         <li>ResourceNotFoundException The specified resource does not exist.</li>
     *         <li>RequestThrottledException The number of API requests has exceed the maximum allowed API request
     *         throttling limit.</li>
     *         <li>ServiceQuotaExceededException Your current service quotas do not allow you to perform this action.</li>
     *         <li>InternalServerException An internal error has occurred.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>EbsException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample EbsAsyncClient.GetSnapshotBlock
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/ebs-2019-11-02/GetSnapshotBlock" target="_top">AWS API
     *      Documentation</a>
     */
    default CompletableFuture<GetSnapshotBlockResponse> getSnapshotBlock(GetSnapshotBlockRequest getSnapshotBlockRequest,
            Path destinationPath) {
        return getSnapshotBlock(getSnapshotBlockRequest, AsyncResponseTransformer.toFile(destinationPath));
    }

    /**
     * <p>
     * Returns the data in a block in an Amazon Elastic Block Store snapshot.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link GetSnapshotBlockRequest.Builder} avoiding the need
     * to create one manually via {@link GetSnapshotBlockRequest#builder()}
     * </p>
     *
     * @param getSnapshotBlockRequest
     *        A {@link Consumer} that will call methods on {@link GetSnapshotBlockRequest.Builder} to create a request.
     * @param destinationPath
     *        {@link Path} to file that response contents will be written to. The file must not exist or this method
     *        will throw an exception. If the file is not writable by the current user then an exception will be thrown.
     *        The service documentation for the response content is as follows '
     *        <p>
     *        The data content of the block.
     *        </p>
     *        '.
     * @return A future to the transformed result of the AsyncResponseTransformer.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>ValidationException The input fails to satisfy the constraints of the EBS direct APIs.</li>
     *         <li>ResourceNotFoundException The specified resource does not exist.</li>
     *         <li>RequestThrottledException The number of API requests has exceed the maximum allowed API request
     *         throttling limit.</li>
     *         <li>ServiceQuotaExceededException Your current service quotas do not allow you to perform this action.</li>
     *         <li>InternalServerException An internal error has occurred.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>EbsException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample EbsAsyncClient.GetSnapshotBlock
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/ebs-2019-11-02/GetSnapshotBlock" target="_top">AWS API
     *      Documentation</a>
     */
    default CompletableFuture<GetSnapshotBlockResponse> getSnapshotBlock(
            Consumer<GetSnapshotBlockRequest.Builder> getSnapshotBlockRequest, Path destinationPath) {
        return getSnapshotBlock(GetSnapshotBlockRequest.builder().applyMutation(getSnapshotBlockRequest).build(), destinationPath);
    }

    /**
     * <p>
     * Returns information about the blocks that are different between two Amazon Elastic Block Store snapshots of the
     * same volume/snapshot lineage.
     * </p>
     *
     * @param listChangedBlocksRequest
     * @return A Java Future containing the result of the ListChangedBlocks operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>ValidationException The input fails to satisfy the constraints of the EBS direct APIs.</li>
     *         <li>ResourceNotFoundException The specified resource does not exist.</li>
     *         <li>RequestThrottledException The number of API requests has exceed the maximum allowed API request
     *         throttling limit.</li>
     *         <li>ServiceQuotaExceededException Your current service quotas do not allow you to perform this action.</li>
     *         <li>InternalServerException An internal error has occurred.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>EbsException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample EbsAsyncClient.ListChangedBlocks
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/ebs-2019-11-02/ListChangedBlocks" target="_top">AWS API
     *      Documentation</a>
     */
    default CompletableFuture<ListChangedBlocksResponse> listChangedBlocks(ListChangedBlocksRequest listChangedBlocksRequest) {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Returns information about the blocks that are different between two Amazon Elastic Block Store snapshots of the
     * same volume/snapshot lineage.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link ListChangedBlocksRequest.Builder} avoiding the need
     * to create one manually via {@link ListChangedBlocksRequest#builder()}
     * </p>
     *
     * @param listChangedBlocksRequest
     *        A {@link Consumer} that will call methods on {@link ListChangedBlocksRequest.Builder} to create a request.
     * @return A Java Future containing the result of the ListChangedBlocks operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>ValidationException The input fails to satisfy the constraints of the EBS direct APIs.</li>
     *         <li>ResourceNotFoundException The specified resource does not exist.</li>
     *         <li>RequestThrottledException The number of API requests has exceed the maximum allowed API request
     *         throttling limit.</li>
     *         <li>ServiceQuotaExceededException Your current service quotas do not allow you to perform this action.</li>
     *         <li>InternalServerException An internal error has occurred.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>EbsException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample EbsAsyncClient.ListChangedBlocks
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/ebs-2019-11-02/ListChangedBlocks" target="_top">AWS API
     *      Documentation</a>
     */
    default CompletableFuture<ListChangedBlocksResponse> listChangedBlocks(
            Consumer<ListChangedBlocksRequest.Builder> listChangedBlocksRequest) {
        return listChangedBlocks(ListChangedBlocksRequest.builder().applyMutation(listChangedBlocksRequest).build());
    }

    /**
     * <p>
     * Returns information about the blocks that are different between two Amazon Elastic Block Store snapshots of the
     * same volume/snapshot lineage.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #listChangedBlocks(software.amazon.awssdk.services.ebs.model.ListChangedBlocksRequest)} operation. The
     * return type is a custom publisher that can be subscribed to request a stream of response pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.ebs.paginators.ListChangedBlocksPublisher publisher = client.listChangedBlocksPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.ebs.paginators.ListChangedBlocksPublisher publisher = client.listChangedBlocksPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.ebs.model.ListChangedBlocksResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.ebs.model.ListChangedBlocksResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listChangedBlocks(software.amazon.awssdk.services.ebs.model.ListChangedBlocksRequest)} operation.</b>
     * </p>
     *
     * @param listChangedBlocksRequest
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>ValidationException The input fails to satisfy the constraints of the EBS direct APIs.</li>
     *         <li>ResourceNotFoundException The specified resource does not exist.</li>
     *         <li>RequestThrottledException The number of API requests has exceed the maximum allowed API request
     *         throttling limit.</li>
     *         <li>ServiceQuotaExceededException Your current service quotas do not allow you to perform this action.</li>
     *         <li>InternalServerException An internal error has occurred.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>EbsException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample EbsAsyncClient.ListChangedBlocks
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/ebs-2019-11-02/ListChangedBlocks" target="_top">AWS API
     *      Documentation</a>
     */
    default ListChangedBlocksPublisher listChangedBlocksPaginator(ListChangedBlocksRequest listChangedBlocksRequest) {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Returns information about the blocks that are different between two Amazon Elastic Block Store snapshots of the
     * same volume/snapshot lineage.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #listChangedBlocks(software.amazon.awssdk.services.ebs.model.ListChangedBlocksRequest)} operation. The
     * return type is a custom publisher that can be subscribed to request a stream of response pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.ebs.paginators.ListChangedBlocksPublisher publisher = client.listChangedBlocksPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.ebs.paginators.ListChangedBlocksPublisher publisher = client.listChangedBlocksPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.ebs.model.ListChangedBlocksResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.ebs.model.ListChangedBlocksResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listChangedBlocks(software.amazon.awssdk.services.ebs.model.ListChangedBlocksRequest)} operation.</b>
     * </p>
     * <p>
     * This is a convenience which creates an instance of the {@link ListChangedBlocksRequest.Builder} avoiding the need
     * to create one manually via {@link ListChangedBlocksRequest#builder()}
     * </p>
     *
     * @param listChangedBlocksRequest
     *        A {@link Consumer} that will call methods on {@link ListChangedBlocksRequest.Builder} to create a request.
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>ValidationException The input fails to satisfy the constraints of the EBS direct APIs.</li>
     *         <li>ResourceNotFoundException The specified resource does not exist.</li>
     *         <li>RequestThrottledException The number of API requests has exceed the maximum allowed API request
     *         throttling limit.</li>
     *         <li>ServiceQuotaExceededException Your current service quotas do not allow you to perform this action.</li>
     *         <li>InternalServerException An internal error has occurred.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>EbsException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample EbsAsyncClient.ListChangedBlocks
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/ebs-2019-11-02/ListChangedBlocks" target="_top">AWS API
     *      Documentation</a>
     */
    default ListChangedBlocksPublisher listChangedBlocksPaginator(
            Consumer<ListChangedBlocksRequest.Builder> listChangedBlocksRequest) {
        return listChangedBlocksPaginator(ListChangedBlocksRequest.builder().applyMutation(listChangedBlocksRequest).build());
    }

    /**
     * <p>
     * Returns information about the blocks in an Amazon Elastic Block Store snapshot.
     * </p>
     *
     * @param listSnapshotBlocksRequest
     * @return A Java Future containing the result of the ListSnapshotBlocks operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>ValidationException The input fails to satisfy the constraints of the EBS direct APIs.</li>
     *         <li>ResourceNotFoundException The specified resource does not exist.</li>
     *         <li>RequestThrottledException The number of API requests has exceed the maximum allowed API request
     *         throttling limit.</li>
     *         <li>ServiceQuotaExceededException Your current service quotas do not allow you to perform this action.</li>
     *         <li>InternalServerException An internal error has occurred.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>EbsException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample EbsAsyncClient.ListSnapshotBlocks
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/ebs-2019-11-02/ListSnapshotBlocks" target="_top">AWS API
     *      Documentation</a>
     */
    default CompletableFuture<ListSnapshotBlocksResponse> listSnapshotBlocks(ListSnapshotBlocksRequest listSnapshotBlocksRequest) {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Returns information about the blocks in an Amazon Elastic Block Store snapshot.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link ListSnapshotBlocksRequest.Builder} avoiding the
     * need to create one manually via {@link ListSnapshotBlocksRequest#builder()}
     * </p>
     *
     * @param listSnapshotBlocksRequest
     *        A {@link Consumer} that will call methods on {@link ListSnapshotBlocksRequest.Builder} to create a
     *        request.
     * @return A Java Future containing the result of the ListSnapshotBlocks operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>ValidationException The input fails to satisfy the constraints of the EBS direct APIs.</li>
     *         <li>ResourceNotFoundException The specified resource does not exist.</li>
     *         <li>RequestThrottledException The number of API requests has exceed the maximum allowed API request
     *         throttling limit.</li>
     *         <li>ServiceQuotaExceededException Your current service quotas do not allow you to perform this action.</li>
     *         <li>InternalServerException An internal error has occurred.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>EbsException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample EbsAsyncClient.ListSnapshotBlocks
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/ebs-2019-11-02/ListSnapshotBlocks" target="_top">AWS API
     *      Documentation</a>
     */
    default CompletableFuture<ListSnapshotBlocksResponse> listSnapshotBlocks(
            Consumer<ListSnapshotBlocksRequest.Builder> listSnapshotBlocksRequest) {
        return listSnapshotBlocks(ListSnapshotBlocksRequest.builder().applyMutation(listSnapshotBlocksRequest).build());
    }

    /**
     * <p>
     * Returns information about the blocks in an Amazon Elastic Block Store snapshot.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #listSnapshotBlocks(software.amazon.awssdk.services.ebs.model.ListSnapshotBlocksRequest)} operation. The
     * return type is a custom publisher that can be subscribed to request a stream of response pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.ebs.paginators.ListSnapshotBlocksPublisher publisher = client.listSnapshotBlocksPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.ebs.paginators.ListSnapshotBlocksPublisher publisher = client.listSnapshotBlocksPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.ebs.model.ListSnapshotBlocksResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.ebs.model.ListSnapshotBlocksResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listSnapshotBlocks(software.amazon.awssdk.services.ebs.model.ListSnapshotBlocksRequest)} operation.</b>
     * </p>
     *
     * @param listSnapshotBlocksRequest
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>ValidationException The input fails to satisfy the constraints of the EBS direct APIs.</li>
     *         <li>ResourceNotFoundException The specified resource does not exist.</li>
     *         <li>RequestThrottledException The number of API requests has exceed the maximum allowed API request
     *         throttling limit.</li>
     *         <li>ServiceQuotaExceededException Your current service quotas do not allow you to perform this action.</li>
     *         <li>InternalServerException An internal error has occurred.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>EbsException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample EbsAsyncClient.ListSnapshotBlocks
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/ebs-2019-11-02/ListSnapshotBlocks" target="_top">AWS API
     *      Documentation</a>
     */
    default ListSnapshotBlocksPublisher listSnapshotBlocksPaginator(ListSnapshotBlocksRequest listSnapshotBlocksRequest) {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Returns information about the blocks in an Amazon Elastic Block Store snapshot.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #listSnapshotBlocks(software.amazon.awssdk.services.ebs.model.ListSnapshotBlocksRequest)} operation. The
     * return type is a custom publisher that can be subscribed to request a stream of response pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.ebs.paginators.ListSnapshotBlocksPublisher publisher = client.listSnapshotBlocksPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.ebs.paginators.ListSnapshotBlocksPublisher publisher = client.listSnapshotBlocksPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.ebs.model.ListSnapshotBlocksResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.ebs.model.ListSnapshotBlocksResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listSnapshotBlocks(software.amazon.awssdk.services.ebs.model.ListSnapshotBlocksRequest)} operation.</b>
     * </p>
     * <p>
     * This is a convenience which creates an instance of the {@link ListSnapshotBlocksRequest.Builder} avoiding the
     * need to create one manually via {@link ListSnapshotBlocksRequest#builder()}
     * </p>
     *
     * @param listSnapshotBlocksRequest
     *        A {@link Consumer} that will call methods on {@link ListSnapshotBlocksRequest.Builder} to create a
     *        request.
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>ValidationException The input fails to satisfy the constraints of the EBS direct APIs.</li>
     *         <li>ResourceNotFoundException The specified resource does not exist.</li>
     *         <li>RequestThrottledException The number of API requests has exceed the maximum allowed API request
     *         throttling limit.</li>
     *         <li>ServiceQuotaExceededException Your current service quotas do not allow you to perform this action.</li>
     *         <li>InternalServerException An internal error has occurred.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>EbsException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample EbsAsyncClient.ListSnapshotBlocks
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/ebs-2019-11-02/ListSnapshotBlocks" target="_top">AWS API
     *      Documentation</a>
     */
    default ListSnapshotBlocksPublisher listSnapshotBlocksPaginator(
            Consumer<ListSnapshotBlocksRequest.Builder> listSnapshotBlocksRequest) {
        return listSnapshotBlocksPaginator(ListSnapshotBlocksRequest.builder().applyMutation(listSnapshotBlocksRequest).build());
    }

    /**
     * <p>
     * Writes a block of data to a snapshot. If the specified block contains data, the existing data is overwritten. The
     * target snapshot must be in the <code>pending</code> state.
     * </p>
     * <p>
     * Data written to a snapshot must be aligned with 512-byte sectors.
     * </p>
     *
     * @param putSnapshotBlockRequest
     * @param requestBody
     *        Functional interface that can be implemented to produce the request content in a non-blocking manner. The
     *        size of the content is expected to be known up front. See {@link AsyncRequestBody} for specific details on
     *        implementing this interface as well as links to precanned implementations for common scenarios like
     *        uploading from a file. The service documentation for the request content is as follows '
     *        <p>
     *        The data to write to the block.
     *        </p>
     *        <p>
     *        The block data is not signed as part of the Signature Version 4 signing process. As a result, you must
     *        generate and provide a Base64-encoded SHA256 checksum for the block data using the <b>x-amz-Checksum</b>
     *        header. Also, you must specify the checksum algorithm using the <b>x-amz-Checksum-Algorithm</b> header.
     *        The checksum that you provide is part of the Signature Version 4 signing process. It is validated against
     *        a checksum generated by Amazon EBS to ensure the validity and authenticity of the data. If the checksums
     *        do not correspond, the request fails. For more information, see <a href=
     *        "https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/ebs-accessing-snapshot.html#ebsapis-using-checksums">
     *        Using checksums with the EBS direct APIs</a> in the <i>Amazon Elastic Compute Cloud User Guide</i>.
     *        </p>
     *        '
     * @return A Java Future containing the result of the PutSnapshotBlock operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>ValidationException The input fails to satisfy the constraints of the EBS direct APIs.</li>
     *         <li>ResourceNotFoundException The specified resource does not exist.</li>
     *         <li>RequestThrottledException The number of API requests has exceed the maximum allowed API request
     *         throttling limit.</li>
     *         <li>ServiceQuotaExceededException Your current service quotas do not allow you to perform this action.</li>
     *         <li>InternalServerException An internal error has occurred.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>EbsException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample EbsAsyncClient.PutSnapshotBlock
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/ebs-2019-11-02/PutSnapshotBlock" target="_top">AWS API
     *      Documentation</a>
     */
    default CompletableFuture<PutSnapshotBlockResponse> putSnapshotBlock(PutSnapshotBlockRequest putSnapshotBlockRequest,
            AsyncRequestBody requestBody) {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Writes a block of data to a snapshot. If the specified block contains data, the existing data is overwritten. The
     * target snapshot must be in the <code>pending</code> state.
     * </p>
     * <p>
     * Data written to a snapshot must be aligned with 512-byte sectors.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link PutSnapshotBlockRequest.Builder} avoiding the need
     * to create one manually via {@link PutSnapshotBlockRequest#builder()}
     * </p>
     *
     * @param putSnapshotBlockRequest
     *        A {@link Consumer} that will call methods on {@link PutSnapshotBlockRequest.Builder} to create a request.
     * @param requestBody
     *        Functional interface that can be implemented to produce the request content in a non-blocking manner. The
     *        size of the content is expected to be known up front. See {@link AsyncRequestBody} for specific details on
     *        implementing this interface as well as links to precanned implementations for common scenarios like
     *        uploading from a file. The service documentation for the request content is as follows '
     *        <p>
     *        The data to write to the block.
     *        </p>
     *        <p>
     *        The block data is not signed as part of the Signature Version 4 signing process. As a result, you must
     *        generate and provide a Base64-encoded SHA256 checksum for the block data using the <b>x-amz-Checksum</b>
     *        header. Also, you must specify the checksum algorithm using the <b>x-amz-Checksum-Algorithm</b> header.
     *        The checksum that you provide is part of the Signature Version 4 signing process. It is validated against
     *        a checksum generated by Amazon EBS to ensure the validity and authenticity of the data. If the checksums
     *        do not correspond, the request fails. For more information, see <a href=
     *        "https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/ebs-accessing-snapshot.html#ebsapis-using-checksums">
     *        Using checksums with the EBS direct APIs</a> in the <i>Amazon Elastic Compute Cloud User Guide</i>.
     *        </p>
     *        '
     * @return A Java Future containing the result of the PutSnapshotBlock operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>ValidationException The input fails to satisfy the constraints of the EBS direct APIs.</li>
     *         <li>ResourceNotFoundException The specified resource does not exist.</li>
     *         <li>RequestThrottledException The number of API requests has exceed the maximum allowed API request
     *         throttling limit.</li>
     *         <li>ServiceQuotaExceededException Your current service quotas do not allow you to perform this action.</li>
     *         <li>InternalServerException An internal error has occurred.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>EbsException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample EbsAsyncClient.PutSnapshotBlock
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/ebs-2019-11-02/PutSnapshotBlock" target="_top">AWS API
     *      Documentation</a>
     */
    default CompletableFuture<PutSnapshotBlockResponse> putSnapshotBlock(
            Consumer<PutSnapshotBlockRequest.Builder> putSnapshotBlockRequest, AsyncRequestBody requestBody) {
        return putSnapshotBlock(PutSnapshotBlockRequest.builder().applyMutation(putSnapshotBlockRequest).build(), requestBody);
    }

    /**
     * <p>
     * Writes a block of data to a snapshot. If the specified block contains data, the existing data is overwritten. The
     * target snapshot must be in the <code>pending</code> state.
     * </p>
     * <p>
     * Data written to a snapshot must be aligned with 512-byte sectors.
     * </p>
     *
     * @param putSnapshotBlockRequest
     * @param sourcePath
     *        {@link Path} to file containing data to send to the service. File will be read entirely and may be read
     *        multiple times in the event of a retry. If the file does not exist or the current user does not have
     *        access to read it then an exception will be thrown. The service documentation for the request content is
     *        as follows '
     *        <p>
     *        The data to write to the block.
     *        </p>
     *        <p>
     *        The block data is not signed as part of the Signature Version 4 signing process. As a result, you must
     *        generate and provide a Base64-encoded SHA256 checksum for the block data using the <b>x-amz-Checksum</b>
     *        header. Also, you must specify the checksum algorithm using the <b>x-amz-Checksum-Algorithm</b> header.
     *        The checksum that you provide is part of the Signature Version 4 signing process. It is validated against
     *        a checksum generated by Amazon EBS to ensure the validity and authenticity of the data. If the checksums
     *        do not correspond, the request fails. For more information, see <a href=
     *        "https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/ebs-accessing-snapshot.html#ebsapis-using-checksums">
     *        Using checksums with the EBS direct APIs</a> in the <i>Amazon Elastic Compute Cloud User Guide</i>.
     *        </p>
     *        '
     * @return A Java Future containing the result of the PutSnapshotBlock operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>ValidationException The input fails to satisfy the constraints of the EBS direct APIs.</li>
     *         <li>ResourceNotFoundException The specified resource does not exist.</li>
     *         <li>RequestThrottledException The number of API requests has exceed the maximum allowed API request
     *         throttling limit.</li>
     *         <li>ServiceQuotaExceededException Your current service quotas do not allow you to perform this action.</li>
     *         <li>InternalServerException An internal error has occurred.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>EbsException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample EbsAsyncClient.PutSnapshotBlock
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/ebs-2019-11-02/PutSnapshotBlock" target="_top">AWS API
     *      Documentation</a>
     */
    default CompletableFuture<PutSnapshotBlockResponse> putSnapshotBlock(PutSnapshotBlockRequest putSnapshotBlockRequest,
            Path sourcePath) {
        return putSnapshotBlock(putSnapshotBlockRequest, AsyncRequestBody.fromFile(sourcePath));
    }

    /**
     * <p>
     * Writes a block of data to a snapshot. If the specified block contains data, the existing data is overwritten. The
     * target snapshot must be in the <code>pending</code> state.
     * </p>
     * <p>
     * Data written to a snapshot must be aligned with 512-byte sectors.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link PutSnapshotBlockRequest.Builder} avoiding the need
     * to create one manually via {@link PutSnapshotBlockRequest#builder()}
     * </p>
     *
     * @param putSnapshotBlockRequest
     *        A {@link Consumer} that will call methods on {@link PutSnapshotBlockRequest.Builder} to create a request.
     * @param sourcePath
     *        {@link Path} to file containing data to send to the service. File will be read entirely and may be read
     *        multiple times in the event of a retry. If the file does not exist or the current user does not have
     *        access to read it then an exception will be thrown. The service documentation for the request content is
     *        as follows '
     *        <p>
     *        The data to write to the block.
     *        </p>
     *        <p>
     *        The block data is not signed as part of the Signature Version 4 signing process. As a result, you must
     *        generate and provide a Base64-encoded SHA256 checksum for the block data using the <b>x-amz-Checksum</b>
     *        header. Also, you must specify the checksum algorithm using the <b>x-amz-Checksum-Algorithm</b> header.
     *        The checksum that you provide is part of the Signature Version 4 signing process. It is validated against
     *        a checksum generated by Amazon EBS to ensure the validity and authenticity of the data. If the checksums
     *        do not correspond, the request fails. For more information, see <a href=
     *        "https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/ebs-accessing-snapshot.html#ebsapis-using-checksums">
     *        Using checksums with the EBS direct APIs</a> in the <i>Amazon Elastic Compute Cloud User Guide</i>.
     *        </p>
     *        '
     * @return A Java Future containing the result of the PutSnapshotBlock operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>ValidationException The input fails to satisfy the constraints of the EBS direct APIs.</li>
     *         <li>ResourceNotFoundException The specified resource does not exist.</li>
     *         <li>RequestThrottledException The number of API requests has exceed the maximum allowed API request
     *         throttling limit.</li>
     *         <li>ServiceQuotaExceededException Your current service quotas do not allow you to perform this action.</li>
     *         <li>InternalServerException An internal error has occurred.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>EbsException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample EbsAsyncClient.PutSnapshotBlock
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/ebs-2019-11-02/PutSnapshotBlock" target="_top">AWS API
     *      Documentation</a>
     */
    default CompletableFuture<PutSnapshotBlockResponse> putSnapshotBlock(
            Consumer<PutSnapshotBlockRequest.Builder> putSnapshotBlockRequest, Path sourcePath) {
        return putSnapshotBlock(PutSnapshotBlockRequest.builder().applyMutation(putSnapshotBlockRequest).build(), sourcePath);
    }

    /**
     * <p>
     * Creates a new Amazon EBS snapshot. The new snapshot enters the <code>pending</code> state after the request
     * completes.
     * </p>
     * <p>
     * After creating the snapshot, use <a
     * href="https://docs.aws.amazon.com/ebs/latest/APIReference/API_PutSnapshotBlock.html"> PutSnapshotBlock</a> to
     * write blocks of data to the snapshot.
     * </p>
     *
     * @param startSnapshotRequest
     * @return A Java Future containing the result of the StartSnapshot operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>ValidationException The input fails to satisfy the constraints of the EBS direct APIs.</li>
     *         <li>RequestThrottledException The number of API requests has exceed the maximum allowed API request
     *         throttling limit.</li>
     *         <li>ResourceNotFoundException The specified resource does not exist.</li>
     *         <li>ServiceQuotaExceededException Your current service quotas do not allow you to perform this action.</li>
     *         <li>InternalServerException An internal error has occurred.</li>
     *         <li>ConcurrentLimitExceededException You have reached the limit for concurrent API requests. For more
     *         information, see <a href=
     *         "https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/ebs-accessing-snapshot.html#ebsapi-performance"
     *         >Optimizing performance of the EBS direct APIs</a> in the <i>Amazon Elastic Compute Cloud User Guide</i>.
     *         </li>
     *         <li>ConflictException The request uses the same client token as a previous, but non-identical request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>EbsException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample EbsAsyncClient.StartSnapshot
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/ebs-2019-11-02/StartSnapshot" target="_top">AWS API
     *      Documentation</a>
     */
    default CompletableFuture<StartSnapshotResponse> startSnapshot(StartSnapshotRequest startSnapshotRequest) {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Creates a new Amazon EBS snapshot. The new snapshot enters the <code>pending</code> state after the request
     * completes.
     * </p>
     * <p>
     * After creating the snapshot, use <a
     * href="https://docs.aws.amazon.com/ebs/latest/APIReference/API_PutSnapshotBlock.html"> PutSnapshotBlock</a> to
     * write blocks of data to the snapshot.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link StartSnapshotRequest.Builder} avoiding the need to
     * create one manually via {@link StartSnapshotRequest#builder()}
     * </p>
     *
     * @param startSnapshotRequest
     *        A {@link Consumer} that will call methods on {@link StartSnapshotRequest.Builder} to create a request.
     * @return A Java Future containing the result of the StartSnapshot operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>ValidationException The input fails to satisfy the constraints of the EBS direct APIs.</li>
     *         <li>RequestThrottledException The number of API requests has exceed the maximum allowed API request
     *         throttling limit.</li>
     *         <li>ResourceNotFoundException The specified resource does not exist.</li>
     *         <li>ServiceQuotaExceededException Your current service quotas do not allow you to perform this action.</li>
     *         <li>InternalServerException An internal error has occurred.</li>
     *         <li>ConcurrentLimitExceededException You have reached the limit for concurrent API requests. For more
     *         information, see <a href=
     *         "https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/ebs-accessing-snapshot.html#ebsapi-performance"
     *         >Optimizing performance of the EBS direct APIs</a> in the <i>Amazon Elastic Compute Cloud User Guide</i>.
     *         </li>
     *         <li>ConflictException The request uses the same client token as a previous, but non-identical request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>EbsException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample EbsAsyncClient.StartSnapshot
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/ebs-2019-11-02/StartSnapshot" target="_top">AWS API
     *      Documentation</a>
     */
    default CompletableFuture<StartSnapshotResponse> startSnapshot(Consumer<StartSnapshotRequest.Builder> startSnapshotRequest) {
        return startSnapshot(StartSnapshotRequest.builder().applyMutation(startSnapshotRequest).build());
    }
}
