/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.dynamodb.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.Mutable;
import software.amazon.awssdk.annotations.NotThreadSafe;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Represents attributes that are copied (projected) from the table into an index. These are in addition to the primary
 * key attributes and index key attributes, which are automatically projected.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class Projection implements SdkPojo, Serializable, ToCopyableBuilder<Projection.Builder, Projection> {
    private static final SdkField<String> PROJECTION_TYPE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("ProjectionType").getter(getter(Projection::projectionTypeAsString))
            .setter(setter(Builder::projectionType))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ProjectionType").build()).build();

    private static final SdkField<List<String>> NON_KEY_ATTRIBUTES_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("NonKeyAttributes")
            .getter(getter(Projection::nonKeyAttributes))
            .setter(setter(Builder::nonKeyAttributes))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("NonKeyAttributes").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(PROJECTION_TYPE_FIELD,
            NON_KEY_ATTRIBUTES_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = memberNameToFieldInitializer();

    private static final long serialVersionUID = 1L;

    private final String projectionType;

    private final List<String> nonKeyAttributes;

    private Projection(BuilderImpl builder) {
        this.projectionType = builder.projectionType;
        this.nonKeyAttributes = builder.nonKeyAttributes;
    }

    /**
     * <p>
     * The set of attributes that are projected into the index:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>KEYS_ONLY</code> - Only the index and primary keys are projected into the index.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>INCLUDE</code> - In addition to the attributes described in <code>KEYS_ONLY</code>, the secondary index
     * will include other non-key attributes that you specify.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>ALL</code> - All of the table attributes are projected into the index.
     * </p>
     * </li>
     * </ul>
     * <p>
     * When using the DynamoDB console, <code>ALL</code> is selected by default.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #projectionType}
     * will return {@link ProjectionType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #projectionTypeAsString}.
     * </p>
     * 
     * @return The set of attributes that are projected into the index:</p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>KEYS_ONLY</code> - Only the index and primary keys are projected into the index.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>INCLUDE</code> - In addition to the attributes described in <code>KEYS_ONLY</code>, the secondary
     *         index will include other non-key attributes that you specify.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>ALL</code> - All of the table attributes are projected into the index.
     *         </p>
     *         </li>
     *         </ul>
     *         <p>
     *         When using the DynamoDB console, <code>ALL</code> is selected by default.
     * @see ProjectionType
     */
    public final ProjectionType projectionType() {
        return ProjectionType.fromValue(projectionType);
    }

    /**
     * <p>
     * The set of attributes that are projected into the index:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>KEYS_ONLY</code> - Only the index and primary keys are projected into the index.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>INCLUDE</code> - In addition to the attributes described in <code>KEYS_ONLY</code>, the secondary index
     * will include other non-key attributes that you specify.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>ALL</code> - All of the table attributes are projected into the index.
     * </p>
     * </li>
     * </ul>
     * <p>
     * When using the DynamoDB console, <code>ALL</code> is selected by default.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #projectionType}
     * will return {@link ProjectionType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #projectionTypeAsString}.
     * </p>
     * 
     * @return The set of attributes that are projected into the index:</p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>KEYS_ONLY</code> - Only the index and primary keys are projected into the index.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>INCLUDE</code> - In addition to the attributes described in <code>KEYS_ONLY</code>, the secondary
     *         index will include other non-key attributes that you specify.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>ALL</code> - All of the table attributes are projected into the index.
     *         </p>
     *         </li>
     *         </ul>
     *         <p>
     *         When using the DynamoDB console, <code>ALL</code> is selected by default.
     * @see ProjectionType
     */
    public final String projectionTypeAsString() {
        return projectionType;
    }

    /**
     * For responses, this returns true if the service returned a value for the NonKeyAttributes property. This DOES NOT
     * check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasNonKeyAttributes() {
        return nonKeyAttributes != null && !(nonKeyAttributes instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * Represents the non-key attribute names which will be projected into the index.
     * </p>
     * <p>
     * For global and local secondary indexes, the total count of <code>NonKeyAttributes</code> summed across all of the
     * secondary indexes, must not exceed 100. If you project the same attribute into two different indexes, this counts
     * as two distinct attributes when determining the total. This limit only applies when you specify the
     * ProjectionType of <code>INCLUDE</code>. You still can specify the ProjectionType of <code>ALL</code> to project
     * all attributes from the source table, even if the table has more than 100 attributes.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasNonKeyAttributes} method.
     * </p>
     * 
     * @return Represents the non-key attribute names which will be projected into the index.</p>
     *         <p>
     *         For global and local secondary indexes, the total count of <code>NonKeyAttributes</code> summed across
     *         all of the secondary indexes, must not exceed 100. If you project the same attribute into two different
     *         indexes, this counts as two distinct attributes when determining the total. This limit only applies when
     *         you specify the ProjectionType of <code>INCLUDE</code>. You still can specify the ProjectionType of
     *         <code>ALL</code> to project all attributes from the source table, even if the table has more than 100
     *         attributes.
     */
    public final List<String> nonKeyAttributes() {
        return nonKeyAttributes;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(projectionTypeAsString());
        hashCode = 31 * hashCode + Objects.hashCode(hasNonKeyAttributes() ? nonKeyAttributes() : null);
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof Projection)) {
            return false;
        }
        Projection other = (Projection) obj;
        return Objects.equals(projectionTypeAsString(), other.projectionTypeAsString())
                && hasNonKeyAttributes() == other.hasNonKeyAttributes()
                && Objects.equals(nonKeyAttributes(), other.nonKeyAttributes());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("Projection").add("ProjectionType", projectionTypeAsString())
                .add("NonKeyAttributes", hasNonKeyAttributes() ? nonKeyAttributes() : null).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "ProjectionType":
            return Optional.ofNullable(clazz.cast(projectionTypeAsString()));
        case "NonKeyAttributes":
            return Optional.ofNullable(clazz.cast(nonKeyAttributes()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static Map<String, SdkField<?>> memberNameToFieldInitializer() {
        Map<String, SdkField<?>> map = new HashMap<>();
        map.put("ProjectionType", PROJECTION_TYPE_FIELD);
        map.put("NonKeyAttributes", NON_KEY_ATTRIBUTES_FIELD);
        return Collections.unmodifiableMap(map);
    }

    private static <T> Function<Object, T> getter(Function<Projection, T> g) {
        return obj -> g.apply((Projection) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    @Mutable
    @NotThreadSafe
    public interface Builder extends SdkPojo, CopyableBuilder<Builder, Projection> {
        /**
         * <p>
         * The set of attributes that are projected into the index:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>KEYS_ONLY</code> - Only the index and primary keys are projected into the index.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>INCLUDE</code> - In addition to the attributes described in <code>KEYS_ONLY</code>, the secondary index
         * will include other non-key attributes that you specify.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>ALL</code> - All of the table attributes are projected into the index.
         * </p>
         * </li>
         * </ul>
         * <p>
         * When using the DynamoDB console, <code>ALL</code> is selected by default.
         * </p>
         * 
         * @param projectionType
         *        The set of attributes that are projected into the index:</p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>KEYS_ONLY</code> - Only the index and primary keys are projected into the index.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>INCLUDE</code> - In addition to the attributes described in <code>KEYS_ONLY</code>, the
         *        secondary index will include other non-key attributes that you specify.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>ALL</code> - All of the table attributes are projected into the index.
         *        </p>
         *        </li>
         *        </ul>
         *        <p>
         *        When using the DynamoDB console, <code>ALL</code> is selected by default.
         * @see ProjectionType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ProjectionType
         */
        Builder projectionType(String projectionType);

        /**
         * <p>
         * The set of attributes that are projected into the index:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>KEYS_ONLY</code> - Only the index and primary keys are projected into the index.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>INCLUDE</code> - In addition to the attributes described in <code>KEYS_ONLY</code>, the secondary index
         * will include other non-key attributes that you specify.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>ALL</code> - All of the table attributes are projected into the index.
         * </p>
         * </li>
         * </ul>
         * <p>
         * When using the DynamoDB console, <code>ALL</code> is selected by default.
         * </p>
         * 
         * @param projectionType
         *        The set of attributes that are projected into the index:</p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>KEYS_ONLY</code> - Only the index and primary keys are projected into the index.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>INCLUDE</code> - In addition to the attributes described in <code>KEYS_ONLY</code>, the
         *        secondary index will include other non-key attributes that you specify.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>ALL</code> - All of the table attributes are projected into the index.
         *        </p>
         *        </li>
         *        </ul>
         *        <p>
         *        When using the DynamoDB console, <code>ALL</code> is selected by default.
         * @see ProjectionType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ProjectionType
         */
        Builder projectionType(ProjectionType projectionType);

        /**
         * <p>
         * Represents the non-key attribute names which will be projected into the index.
         * </p>
         * <p>
         * For global and local secondary indexes, the total count of <code>NonKeyAttributes</code> summed across all of
         * the secondary indexes, must not exceed 100. If you project the same attribute into two different indexes,
         * this counts as two distinct attributes when determining the total. This limit only applies when you specify
         * the ProjectionType of <code>INCLUDE</code>. You still can specify the ProjectionType of <code>ALL</code> to
         * project all attributes from the source table, even if the table has more than 100 attributes.
         * </p>
         * 
         * @param nonKeyAttributes
         *        Represents the non-key attribute names which will be projected into the index.</p>
         *        <p>
         *        For global and local secondary indexes, the total count of <code>NonKeyAttributes</code> summed across
         *        all of the secondary indexes, must not exceed 100. If you project the same attribute into two
         *        different indexes, this counts as two distinct attributes when determining the total. This limit only
         *        applies when you specify the ProjectionType of <code>INCLUDE</code>. You still can specify the
         *        ProjectionType of <code>ALL</code> to project all attributes from the source table, even if the table
         *        has more than 100 attributes.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder nonKeyAttributes(Collection<String> nonKeyAttributes);

        /**
         * <p>
         * Represents the non-key attribute names which will be projected into the index.
         * </p>
         * <p>
         * For global and local secondary indexes, the total count of <code>NonKeyAttributes</code> summed across all of
         * the secondary indexes, must not exceed 100. If you project the same attribute into two different indexes,
         * this counts as two distinct attributes when determining the total. This limit only applies when you specify
         * the ProjectionType of <code>INCLUDE</code>. You still can specify the ProjectionType of <code>ALL</code> to
         * project all attributes from the source table, even if the table has more than 100 attributes.
         * </p>
         * 
         * @param nonKeyAttributes
         *        Represents the non-key attribute names which will be projected into the index.</p>
         *        <p>
         *        For global and local secondary indexes, the total count of <code>NonKeyAttributes</code> summed across
         *        all of the secondary indexes, must not exceed 100. If you project the same attribute into two
         *        different indexes, this counts as two distinct attributes when determining the total. This limit only
         *        applies when you specify the ProjectionType of <code>INCLUDE</code>. You still can specify the
         *        ProjectionType of <code>ALL</code> to project all attributes from the source table, even if the table
         *        has more than 100 attributes.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder nonKeyAttributes(String... nonKeyAttributes);
    }

    static final class BuilderImpl implements Builder {
        private String projectionType;

        private List<String> nonKeyAttributes = DefaultSdkAutoConstructList.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(Projection model) {
            projectionType(model.projectionType);
            nonKeyAttributes(model.nonKeyAttributes);
        }

        public final String getProjectionType() {
            return projectionType;
        }

        public final void setProjectionType(String projectionType) {
            this.projectionType = projectionType;
        }

        @Override
        public final Builder projectionType(String projectionType) {
            this.projectionType = projectionType;
            return this;
        }

        @Override
        public final Builder projectionType(ProjectionType projectionType) {
            this.projectionType(projectionType == null ? null : projectionType.toString());
            return this;
        }

        public final Collection<String> getNonKeyAttributes() {
            if (nonKeyAttributes instanceof SdkAutoConstructList) {
                return null;
            }
            return nonKeyAttributes;
        }

        public final void setNonKeyAttributes(Collection<String> nonKeyAttributes) {
            this.nonKeyAttributes = NonKeyAttributeNameListCopier.copy(nonKeyAttributes);
        }

        @Override
        public final Builder nonKeyAttributes(Collection<String> nonKeyAttributes) {
            this.nonKeyAttributes = NonKeyAttributeNameListCopier.copy(nonKeyAttributes);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder nonKeyAttributes(String... nonKeyAttributes) {
            nonKeyAttributes(Arrays.asList(nonKeyAttributes));
            return this;
        }

        @Override
        public Projection build() {
            return new Projection(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
