/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.dynamodb.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.traits.MapTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructMap;
import software.amazon.awssdk.core.util.SdkAutoConstructMap;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * An error associated with a statement in a PartiQL batch that was run.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class BatchStatementError implements SdkPojo, Serializable,
        ToCopyableBuilder<BatchStatementError.Builder, BatchStatementError> {
    private static final SdkField<String> CODE_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Code")
            .getter(getter(BatchStatementError::codeAsString)).setter(setter(Builder::code))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Code").build()).build();

    private static final SdkField<String> MESSAGE_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Message")
            .getter(getter(BatchStatementError::message)).setter(setter(Builder::message))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Message").build()).build();

    private static final SdkField<Map<String, AttributeValue>> ITEM_FIELD = SdkField
            .<Map<String, AttributeValue>> builder(MarshallingType.MAP)
            .memberName("Item")
            .getter(getter(BatchStatementError::item))
            .setter(setter(Builder::item))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Item").build(),
                    MapTrait.builder()
                            .keyLocationName("key")
                            .valueLocationName("value")
                            .valueFieldInfo(
                                    SdkField.<AttributeValue> builder(MarshallingType.SDK_POJO)
                                            .constructor(AttributeValue::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("value").build()).build()).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(CODE_FIELD, MESSAGE_FIELD,
            ITEM_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = Collections
            .unmodifiableMap(new HashMap<String, SdkField<?>>() {
                {
                    put("Code", CODE_FIELD);
                    put("Message", MESSAGE_FIELD);
                    put("Item", ITEM_FIELD);
                }
            });

    private static final long serialVersionUID = 1L;

    private final String code;

    private final String message;

    private final Map<String, AttributeValue> item;

    private BatchStatementError(BuilderImpl builder) {
        this.code = builder.code;
        this.message = builder.message;
        this.item = builder.item;
    }

    /**
     * <p>
     * The error code associated with the failed PartiQL batch statement.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #code} will return
     * {@link BatchStatementErrorCodeEnum#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #codeAsString}.
     * </p>
     * 
     * @return The error code associated with the failed PartiQL batch statement.
     * @see BatchStatementErrorCodeEnum
     */
    public final BatchStatementErrorCodeEnum code() {
        return BatchStatementErrorCodeEnum.fromValue(code);
    }

    /**
     * <p>
     * The error code associated with the failed PartiQL batch statement.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #code} will return
     * {@link BatchStatementErrorCodeEnum#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #codeAsString}.
     * </p>
     * 
     * @return The error code associated with the failed PartiQL batch statement.
     * @see BatchStatementErrorCodeEnum
     */
    public final String codeAsString() {
        return code;
    }

    /**
     * <p>
     * The error message associated with the PartiQL batch response.
     * </p>
     * 
     * @return The error message associated with the PartiQL batch response.
     */
    public final String message() {
        return message;
    }

    /**
     * For responses, this returns true if the service returned a value for the Item property. This DOES NOT check that
     * the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is useful
     * because the SDK will never return a null collection or map, but you may need to differentiate between the service
     * returning nothing (or null) and the service returning an empty collection or map. For requests, this returns true
     * if a value for the property was specified in the request builder, and false if a value was not specified.
     */
    public final boolean hasItem() {
        return item != null && !(item instanceof SdkAutoConstructMap);
    }

    /**
     * <p>
     * The item which caused the condition check to fail. This will be set if ReturnValuesOnConditionCheckFailure is
     * specified as <code>ALL_OLD</code>.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasItem} method.
     * </p>
     * 
     * @return The item which caused the condition check to fail. This will be set if
     *         ReturnValuesOnConditionCheckFailure is specified as <code>ALL_OLD</code>.
     */
    public final Map<String, AttributeValue> item() {
        return item;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(codeAsString());
        hashCode = 31 * hashCode + Objects.hashCode(message());
        hashCode = 31 * hashCode + Objects.hashCode(hasItem() ? item() : null);
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof BatchStatementError)) {
            return false;
        }
        BatchStatementError other = (BatchStatementError) obj;
        return Objects.equals(codeAsString(), other.codeAsString()) && Objects.equals(message(), other.message())
                && hasItem() == other.hasItem() && Objects.equals(item(), other.item());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("BatchStatementError").add("Code", codeAsString()).add("Message", message())
                .add("Item", hasItem() ? item() : null).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "Code":
            return Optional.ofNullable(clazz.cast(codeAsString()));
        case "Message":
            return Optional.ofNullable(clazz.cast(message()));
        case "Item":
            return Optional.ofNullable(clazz.cast(item()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static <T> Function<Object, T> getter(Function<BatchStatementError, T> g) {
        return obj -> g.apply((BatchStatementError) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, BatchStatementError> {
        /**
         * <p>
         * The error code associated with the failed PartiQL batch statement.
         * </p>
         * 
         * @param code
         *        The error code associated with the failed PartiQL batch statement.
         * @see BatchStatementErrorCodeEnum
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see BatchStatementErrorCodeEnum
         */
        Builder code(String code);

        /**
         * <p>
         * The error code associated with the failed PartiQL batch statement.
         * </p>
         * 
         * @param code
         *        The error code associated with the failed PartiQL batch statement.
         * @see BatchStatementErrorCodeEnum
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see BatchStatementErrorCodeEnum
         */
        Builder code(BatchStatementErrorCodeEnum code);

        /**
         * <p>
         * The error message associated with the PartiQL batch response.
         * </p>
         * 
         * @param message
         *        The error message associated with the PartiQL batch response.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder message(String message);

        /**
         * <p>
         * The item which caused the condition check to fail. This will be set if ReturnValuesOnConditionCheckFailure is
         * specified as <code>ALL_OLD</code>.
         * </p>
         * 
         * @param item
         *        The item which caused the condition check to fail. This will be set if
         *        ReturnValuesOnConditionCheckFailure is specified as <code>ALL_OLD</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder item(Map<String, AttributeValue> item);
    }

    static final class BuilderImpl implements Builder {
        private String code;

        private String message;

        private Map<String, AttributeValue> item = DefaultSdkAutoConstructMap.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(BatchStatementError model) {
            code(model.code);
            message(model.message);
            item(model.item);
        }

        public final String getCode() {
            return code;
        }

        public final void setCode(String code) {
            this.code = code;
        }

        @Override
        public final Builder code(String code) {
            this.code = code;
            return this;
        }

        @Override
        public final Builder code(BatchStatementErrorCodeEnum code) {
            this.code(code == null ? null : code.toString());
            return this;
        }

        public final String getMessage() {
            return message;
        }

        public final void setMessage(String message) {
            this.message = message;
        }

        @Override
        public final Builder message(String message) {
            this.message = message;
            return this;
        }

        public final Map<String, AttributeValue.Builder> getItem() {
            Map<String, AttributeValue.Builder> result = AttributeMapCopier.copyToBuilder(this.item);
            if (result instanceof SdkAutoConstructMap) {
                return null;
            }
            return result;
        }

        public final void setItem(Map<String, AttributeValue.BuilderImpl> item) {
            this.item = AttributeMapCopier.copyFromBuilder(item);
        }

        @Override
        public final Builder item(Map<String, AttributeValue> item) {
            this.item = AttributeMapCopier.copy(item);
            return this;
        }

        @Override
        public BatchStatementError build() {
            return new BatchStatementError(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
