/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.devopsguru;

import java.util.Collections;
import java.util.List;
import java.util.function.Consumer;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.AwsRequestOverrideConfiguration;
import software.amazon.awssdk.awscore.client.handler.AwsSyncClientHandler;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.core.ApiName;
import software.amazon.awssdk.core.RequestOverrideConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientOption;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.client.handler.SyncClientHandler;
import software.amazon.awssdk.core.exception.SdkClientException;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.core.metrics.CoreMetric;
import software.amazon.awssdk.core.util.VersionInfo;
import software.amazon.awssdk.metrics.MetricCollector;
import software.amazon.awssdk.metrics.MetricPublisher;
import software.amazon.awssdk.metrics.NoOpMetricCollector;
import software.amazon.awssdk.protocols.core.ExceptionMetadata;
import software.amazon.awssdk.protocols.json.AwsJsonProtocol;
import software.amazon.awssdk.protocols.json.AwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.BaseAwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.JsonOperationMetadata;
import software.amazon.awssdk.services.devopsguru.model.AccessDeniedException;
import software.amazon.awssdk.services.devopsguru.model.AddNotificationChannelRequest;
import software.amazon.awssdk.services.devopsguru.model.AddNotificationChannelResponse;
import software.amazon.awssdk.services.devopsguru.model.ConflictException;
import software.amazon.awssdk.services.devopsguru.model.DescribeAccountHealthRequest;
import software.amazon.awssdk.services.devopsguru.model.DescribeAccountHealthResponse;
import software.amazon.awssdk.services.devopsguru.model.DescribeAccountOverviewRequest;
import software.amazon.awssdk.services.devopsguru.model.DescribeAccountOverviewResponse;
import software.amazon.awssdk.services.devopsguru.model.DescribeAnomalyRequest;
import software.amazon.awssdk.services.devopsguru.model.DescribeAnomalyResponse;
import software.amazon.awssdk.services.devopsguru.model.DescribeFeedbackRequest;
import software.amazon.awssdk.services.devopsguru.model.DescribeFeedbackResponse;
import software.amazon.awssdk.services.devopsguru.model.DescribeInsightRequest;
import software.amazon.awssdk.services.devopsguru.model.DescribeInsightResponse;
import software.amazon.awssdk.services.devopsguru.model.DescribeResourceCollectionHealthRequest;
import software.amazon.awssdk.services.devopsguru.model.DescribeResourceCollectionHealthResponse;
import software.amazon.awssdk.services.devopsguru.model.DescribeServiceIntegrationRequest;
import software.amazon.awssdk.services.devopsguru.model.DescribeServiceIntegrationResponse;
import software.amazon.awssdk.services.devopsguru.model.DevOpsGuruException;
import software.amazon.awssdk.services.devopsguru.model.DevOpsGuruRequest;
import software.amazon.awssdk.services.devopsguru.model.GetResourceCollectionRequest;
import software.amazon.awssdk.services.devopsguru.model.GetResourceCollectionResponse;
import software.amazon.awssdk.services.devopsguru.model.InternalServerException;
import software.amazon.awssdk.services.devopsguru.model.ListAnomaliesForInsightRequest;
import software.amazon.awssdk.services.devopsguru.model.ListAnomaliesForInsightResponse;
import software.amazon.awssdk.services.devopsguru.model.ListEventsRequest;
import software.amazon.awssdk.services.devopsguru.model.ListEventsResponse;
import software.amazon.awssdk.services.devopsguru.model.ListInsightsRequest;
import software.amazon.awssdk.services.devopsguru.model.ListInsightsResponse;
import software.amazon.awssdk.services.devopsguru.model.ListNotificationChannelsRequest;
import software.amazon.awssdk.services.devopsguru.model.ListNotificationChannelsResponse;
import software.amazon.awssdk.services.devopsguru.model.ListRecommendationsRequest;
import software.amazon.awssdk.services.devopsguru.model.ListRecommendationsResponse;
import software.amazon.awssdk.services.devopsguru.model.PutFeedbackRequest;
import software.amazon.awssdk.services.devopsguru.model.PutFeedbackResponse;
import software.amazon.awssdk.services.devopsguru.model.RemoveNotificationChannelRequest;
import software.amazon.awssdk.services.devopsguru.model.RemoveNotificationChannelResponse;
import software.amazon.awssdk.services.devopsguru.model.ResourceNotFoundException;
import software.amazon.awssdk.services.devopsguru.model.SearchInsightsRequest;
import software.amazon.awssdk.services.devopsguru.model.SearchInsightsResponse;
import software.amazon.awssdk.services.devopsguru.model.ServiceQuotaExceededException;
import software.amazon.awssdk.services.devopsguru.model.ThrottlingException;
import software.amazon.awssdk.services.devopsguru.model.UpdateResourceCollectionRequest;
import software.amazon.awssdk.services.devopsguru.model.UpdateResourceCollectionResponse;
import software.amazon.awssdk.services.devopsguru.model.UpdateServiceIntegrationRequest;
import software.amazon.awssdk.services.devopsguru.model.UpdateServiceIntegrationResponse;
import software.amazon.awssdk.services.devopsguru.model.ValidationException;
import software.amazon.awssdk.services.devopsguru.paginators.DescribeResourceCollectionHealthIterable;
import software.amazon.awssdk.services.devopsguru.paginators.GetResourceCollectionIterable;
import software.amazon.awssdk.services.devopsguru.paginators.ListAnomaliesForInsightIterable;
import software.amazon.awssdk.services.devopsguru.paginators.ListEventsIterable;
import software.amazon.awssdk.services.devopsguru.paginators.ListInsightsIterable;
import software.amazon.awssdk.services.devopsguru.paginators.ListNotificationChannelsIterable;
import software.amazon.awssdk.services.devopsguru.paginators.ListRecommendationsIterable;
import software.amazon.awssdk.services.devopsguru.paginators.SearchInsightsIterable;
import software.amazon.awssdk.services.devopsguru.transform.AddNotificationChannelRequestMarshaller;
import software.amazon.awssdk.services.devopsguru.transform.DescribeAccountHealthRequestMarshaller;
import software.amazon.awssdk.services.devopsguru.transform.DescribeAccountOverviewRequestMarshaller;
import software.amazon.awssdk.services.devopsguru.transform.DescribeAnomalyRequestMarshaller;
import software.amazon.awssdk.services.devopsguru.transform.DescribeFeedbackRequestMarshaller;
import software.amazon.awssdk.services.devopsguru.transform.DescribeInsightRequestMarshaller;
import software.amazon.awssdk.services.devopsguru.transform.DescribeResourceCollectionHealthRequestMarshaller;
import software.amazon.awssdk.services.devopsguru.transform.DescribeServiceIntegrationRequestMarshaller;
import software.amazon.awssdk.services.devopsguru.transform.GetResourceCollectionRequestMarshaller;
import software.amazon.awssdk.services.devopsguru.transform.ListAnomaliesForInsightRequestMarshaller;
import software.amazon.awssdk.services.devopsguru.transform.ListEventsRequestMarshaller;
import software.amazon.awssdk.services.devopsguru.transform.ListInsightsRequestMarshaller;
import software.amazon.awssdk.services.devopsguru.transform.ListNotificationChannelsRequestMarshaller;
import software.amazon.awssdk.services.devopsguru.transform.ListRecommendationsRequestMarshaller;
import software.amazon.awssdk.services.devopsguru.transform.PutFeedbackRequestMarshaller;
import software.amazon.awssdk.services.devopsguru.transform.RemoveNotificationChannelRequestMarshaller;
import software.amazon.awssdk.services.devopsguru.transform.SearchInsightsRequestMarshaller;
import software.amazon.awssdk.services.devopsguru.transform.UpdateResourceCollectionRequestMarshaller;
import software.amazon.awssdk.services.devopsguru.transform.UpdateServiceIntegrationRequestMarshaller;
import software.amazon.awssdk.utils.Logger;

/**
 * Internal implementation of {@link DevOpsGuruClient}.
 *
 * @see DevOpsGuruClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultDevOpsGuruClient implements DevOpsGuruClient {
    private static final Logger log = Logger.loggerFor(DefaultDevOpsGuruClient.class);

    private final SyncClientHandler clientHandler;

    private final AwsJsonProtocolFactory protocolFactory;

    private final SdkClientConfiguration clientConfiguration;

    protected DefaultDevOpsGuruClient(SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsSyncClientHandler(clientConfiguration);
        this.clientConfiguration = clientConfiguration;
        this.protocolFactory = init(AwsJsonProtocolFactory.builder()).build();
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    /**
     * <p>
     * Adds a notification channel to DevOps Guru. A notification channel is used to notify you about important DevOps
     * Guru events, such as when an insight is generated.
     * </p>
     * <p>
     * If you use an Amazon SNS topic in another account, you must attach a policy to it that grants DevOps Guru
     * permission to it notifications. DevOps Guru adds the required policy on your behalf to send notifications using
     * Amazon SNS in your account. For more information, see <a
     * href="https://docs.aws.amazon.com/devops-guru/latest/userguide/sns-required-permissions.html">Permissions for
     * cross account Amazon SNS topics</a>.
     * </p>
     * <p>
     * If you use an Amazon SNS topic that is encrypted by an AWS Key Management Service customer-managed key (CMK),
     * then you must add permissions to the CMK. For more information, see <a
     * href="https://docs.aws.amazon.com/devops-guru/latest/userguide/sns-kms-permissions.html">Permissions for AWS
     * KMS–encrypted Amazon SNS topics</a>.
     * </p>
     *
     * @param addNotificationChannelRequest
     * @return Result of the AddNotificationChannel operation returned by the service.
     * @throws AccessDeniedException
     *         You don't have permissions to perform the requested operation. The user or role that is making the
     *         request must have at least one IAM permissions policy attached that grants the required permissions. For
     *         more information, see <a href="https://docs.aws.amazon.com/IAM/latest/UserGuide/access.html">Access
     *         Management</a> in the <i>IAM User Guide</i>.
     * @throws ConflictException
     *         An exception that is thrown when a conflict occurs.
     * @throws InternalServerException
     *         An internal failure in an Amazon service occurred.
     * @throws ResourceNotFoundException
     *         A requested resource could not be found
     * @throws ServiceQuotaExceededException
     *         The request contains a value that exceeds a maximum quota.
     * @throws ThrottlingException
     *         The request was denied due to a request throttling.
     * @throws ValidationException
     *         Contains information about data passed in to a field during a request that is not valid.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws DevOpsGuruException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample DevOpsGuruClient.AddNotificationChannel
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/devops-guru-2020-12-01/AddNotificationChannel"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public AddNotificationChannelResponse addNotificationChannel(AddNotificationChannelRequest addNotificationChannelRequest)
            throws AccessDeniedException, ConflictException, InternalServerException, ResourceNotFoundException,
            ServiceQuotaExceededException, ThrottlingException, ValidationException, AwsServiceException, SdkClientException,
            DevOpsGuruException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<AddNotificationChannelResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, AddNotificationChannelResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, addNotificationChannelRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "DevOps Guru");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "AddNotificationChannel");

            return clientHandler
                    .execute(new ClientExecutionParams<AddNotificationChannelRequest, AddNotificationChannelResponse>()
                            .withOperationName("AddNotificationChannel").withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withInput(addNotificationChannelRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new AddNotificationChannelRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Returns the number of open reactive insights, the number of open proactive insights, and the number of metrics
     * analyzed in your AWS account. Use these numbers to gauge the health of operations in your AWS account.
     * </p>
     *
     * @param describeAccountHealthRequest
     * @return Result of the DescribeAccountHealth operation returned by the service.
     * @throws AccessDeniedException
     *         You don't have permissions to perform the requested operation. The user or role that is making the
     *         request must have at least one IAM permissions policy attached that grants the required permissions. For
     *         more information, see <a href="https://docs.aws.amazon.com/IAM/latest/UserGuide/access.html">Access
     *         Management</a> in the <i>IAM User Guide</i>.
     * @throws InternalServerException
     *         An internal failure in an Amazon service occurred.
     * @throws ThrottlingException
     *         The request was denied due to a request throttling.
     * @throws ValidationException
     *         Contains information about data passed in to a field during a request that is not valid.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws DevOpsGuruException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample DevOpsGuruClient.DescribeAccountHealth
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/devops-guru-2020-12-01/DescribeAccountHealth"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DescribeAccountHealthResponse describeAccountHealth(DescribeAccountHealthRequest describeAccountHealthRequest)
            throws AccessDeniedException, InternalServerException, ThrottlingException, ValidationException, AwsServiceException,
            SdkClientException, DevOpsGuruException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DescribeAccountHealthResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, DescribeAccountHealthResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeAccountHealthRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "DevOps Guru");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeAccountHealth");

            return clientHandler.execute(new ClientExecutionParams<DescribeAccountHealthRequest, DescribeAccountHealthResponse>()
                    .withOperationName("DescribeAccountHealth").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(describeAccountHealthRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DescribeAccountHealthRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * For the time range passed in, returns the number of open reactive insight that were created, the number of open
     * proactive insights that were created, and the Mean Time to Recover (MTTR) for all closed reactive insights.
     * </p>
     *
     * @param describeAccountOverviewRequest
     * @return Result of the DescribeAccountOverview operation returned by the service.
     * @throws AccessDeniedException
     *         You don't have permissions to perform the requested operation. The user or role that is making the
     *         request must have at least one IAM permissions policy attached that grants the required permissions. For
     *         more information, see <a href="https://docs.aws.amazon.com/IAM/latest/UserGuide/access.html">Access
     *         Management</a> in the <i>IAM User Guide</i>.
     * @throws InternalServerException
     *         An internal failure in an Amazon service occurred.
     * @throws ThrottlingException
     *         The request was denied due to a request throttling.
     * @throws ValidationException
     *         Contains information about data passed in to a field during a request that is not valid.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws DevOpsGuruException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample DevOpsGuruClient.DescribeAccountOverview
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/devops-guru-2020-12-01/DescribeAccountOverview"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DescribeAccountOverviewResponse describeAccountOverview(DescribeAccountOverviewRequest describeAccountOverviewRequest)
            throws AccessDeniedException, InternalServerException, ThrottlingException, ValidationException, AwsServiceException,
            SdkClientException, DevOpsGuruException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DescribeAccountOverviewResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, DescribeAccountOverviewResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeAccountOverviewRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "DevOps Guru");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeAccountOverview");

            return clientHandler
                    .execute(new ClientExecutionParams<DescribeAccountOverviewRequest, DescribeAccountOverviewResponse>()
                            .withOperationName("DescribeAccountOverview").withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withInput(describeAccountOverviewRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new DescribeAccountOverviewRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Returns details about an anomaly that you specify using its ID.
     * </p>
     *
     * @param describeAnomalyRequest
     * @return Result of the DescribeAnomaly operation returned by the service.
     * @throws AccessDeniedException
     *         You don't have permissions to perform the requested operation. The user or role that is making the
     *         request must have at least one IAM permissions policy attached that grants the required permissions. For
     *         more information, see <a href="https://docs.aws.amazon.com/IAM/latest/UserGuide/access.html">Access
     *         Management</a> in the <i>IAM User Guide</i>.
     * @throws InternalServerException
     *         An internal failure in an Amazon service occurred.
     * @throws ResourceNotFoundException
     *         A requested resource could not be found
     * @throws ThrottlingException
     *         The request was denied due to a request throttling.
     * @throws ValidationException
     *         Contains information about data passed in to a field during a request that is not valid.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws DevOpsGuruException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample DevOpsGuruClient.DescribeAnomaly
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/devops-guru-2020-12-01/DescribeAnomaly" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public DescribeAnomalyResponse describeAnomaly(DescribeAnomalyRequest describeAnomalyRequest) throws AccessDeniedException,
            InternalServerException, ResourceNotFoundException, ThrottlingException, ValidationException, AwsServiceException,
            SdkClientException, DevOpsGuruException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DescribeAnomalyResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                DescribeAnomalyResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeAnomalyRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "DevOps Guru");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeAnomaly");

            return clientHandler.execute(new ClientExecutionParams<DescribeAnomalyRequest, DescribeAnomalyResponse>()
                    .withOperationName("DescribeAnomaly").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(describeAnomalyRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DescribeAnomalyRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Returns the most recent feedback submitted in the current AWS account and Region.
     * </p>
     *
     * @param describeFeedbackRequest
     * @return Result of the DescribeFeedback operation returned by the service.
     * @throws AccessDeniedException
     *         You don't have permissions to perform the requested operation. The user or role that is making the
     *         request must have at least one IAM permissions policy attached that grants the required permissions. For
     *         more information, see <a href="https://docs.aws.amazon.com/IAM/latest/UserGuide/access.html">Access
     *         Management</a> in the <i>IAM User Guide</i>.
     * @throws InternalServerException
     *         An internal failure in an Amazon service occurred.
     * @throws ResourceNotFoundException
     *         A requested resource could not be found
     * @throws ThrottlingException
     *         The request was denied due to a request throttling.
     * @throws ValidationException
     *         Contains information about data passed in to a field during a request that is not valid.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws DevOpsGuruException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample DevOpsGuruClient.DescribeFeedback
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/devops-guru-2020-12-01/DescribeFeedback" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public DescribeFeedbackResponse describeFeedback(DescribeFeedbackRequest describeFeedbackRequest)
            throws AccessDeniedException, InternalServerException, ResourceNotFoundException, ThrottlingException,
            ValidationException, AwsServiceException, SdkClientException, DevOpsGuruException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DescribeFeedbackResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                DescribeFeedbackResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeFeedbackRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "DevOps Guru");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeFeedback");

            return clientHandler.execute(new ClientExecutionParams<DescribeFeedbackRequest, DescribeFeedbackResponse>()
                    .withOperationName("DescribeFeedback").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(describeFeedbackRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DescribeFeedbackRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Returns details about an insight that you specify using its ID.
     * </p>
     *
     * @param describeInsightRequest
     * @return Result of the DescribeInsight operation returned by the service.
     * @throws AccessDeniedException
     *         You don't have permissions to perform the requested operation. The user or role that is making the
     *         request must have at least one IAM permissions policy attached that grants the required permissions. For
     *         more information, see <a href="https://docs.aws.amazon.com/IAM/latest/UserGuide/access.html">Access
     *         Management</a> in the <i>IAM User Guide</i>.
     * @throws InternalServerException
     *         An internal failure in an Amazon service occurred.
     * @throws ResourceNotFoundException
     *         A requested resource could not be found
     * @throws ThrottlingException
     *         The request was denied due to a request throttling.
     * @throws ValidationException
     *         Contains information about data passed in to a field during a request that is not valid.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws DevOpsGuruException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample DevOpsGuruClient.DescribeInsight
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/devops-guru-2020-12-01/DescribeInsight" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public DescribeInsightResponse describeInsight(DescribeInsightRequest describeInsightRequest) throws AccessDeniedException,
            InternalServerException, ResourceNotFoundException, ThrottlingException, ValidationException, AwsServiceException,
            SdkClientException, DevOpsGuruException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DescribeInsightResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                DescribeInsightResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeInsightRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "DevOps Guru");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeInsight");

            return clientHandler.execute(new ClientExecutionParams<DescribeInsightRequest, DescribeInsightResponse>()
                    .withOperationName("DescribeInsight").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(describeInsightRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DescribeInsightRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Returns the number of open proactive insights, open reactive insights, and the Mean Time to Recover (MTTR) for
     * all closed insights in resource collections in your account. You specify the type of AWS resources collection.
     * The one type of AWS resource collection supported is AWS CloudFormation stacks. DevOps Guru can be configured to
     * analyze only the AWS resources that are defined in the stacks.
     * </p>
     *
     * @param describeResourceCollectionHealthRequest
     * @return Result of the DescribeResourceCollectionHealth operation returned by the service.
     * @throws AccessDeniedException
     *         You don't have permissions to perform the requested operation. The user or role that is making the
     *         request must have at least one IAM permissions policy attached that grants the required permissions. For
     *         more information, see <a href="https://docs.aws.amazon.com/IAM/latest/UserGuide/access.html">Access
     *         Management</a> in the <i>IAM User Guide</i>.
     * @throws InternalServerException
     *         An internal failure in an Amazon service occurred.
     * @throws ThrottlingException
     *         The request was denied due to a request throttling.
     * @throws ValidationException
     *         Contains information about data passed in to a field during a request that is not valid.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws DevOpsGuruException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample DevOpsGuruClient.DescribeResourceCollectionHealth
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/devops-guru-2020-12-01/DescribeResourceCollectionHealth"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DescribeResourceCollectionHealthResponse describeResourceCollectionHealth(
            DescribeResourceCollectionHealthRequest describeResourceCollectionHealthRequest) throws AccessDeniedException,
            InternalServerException, ThrottlingException, ValidationException, AwsServiceException, SdkClientException,
            DevOpsGuruException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DescribeResourceCollectionHealthResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, DescribeResourceCollectionHealthResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                describeResourceCollectionHealthRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "DevOps Guru");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeResourceCollectionHealth");

            return clientHandler
                    .execute(new ClientExecutionParams<DescribeResourceCollectionHealthRequest, DescribeResourceCollectionHealthResponse>()
                            .withOperationName("DescribeResourceCollectionHealth").withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withInput(describeResourceCollectionHealthRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new DescribeResourceCollectionHealthRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Returns the number of open proactive insights, open reactive insights, and the Mean Time to Recover (MTTR) for
     * all closed insights in resource collections in your account. You specify the type of AWS resources collection.
     * The one type of AWS resource collection supported is AWS CloudFormation stacks. DevOps Guru can be configured to
     * analyze only the AWS resources that are defined in the stacks.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #describeResourceCollectionHealth(software.amazon.awssdk.services.devopsguru.model.DescribeResourceCollectionHealthRequest)}
     * operation. The return type is a custom iterable that can be used to iterate through all the pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.devopsguru.paginators.DescribeResourceCollectionHealthIterable responses = client.describeResourceCollectionHealthPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.devopsguru.paginators.DescribeResourceCollectionHealthIterable responses = client
     *             .describeResourceCollectionHealthPaginator(request);
     *     for (software.amazon.awssdk.services.devopsguru.model.DescribeResourceCollectionHealthResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.devopsguru.paginators.DescribeResourceCollectionHealthIterable responses = client.describeResourceCollectionHealthPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of null won't limit the number of results you get with the paginator. It
     * only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #describeResourceCollectionHealth(software.amazon.awssdk.services.devopsguru.model.DescribeResourceCollectionHealthRequest)}
     * operation.</b>
     * </p>
     *
     * @param describeResourceCollectionHealthRequest
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws AccessDeniedException
     *         You don't have permissions to perform the requested operation. The user or role that is making the
     *         request must have at least one IAM permissions policy attached that grants the required permissions. For
     *         more information, see <a href="https://docs.aws.amazon.com/IAM/latest/UserGuide/access.html">Access
     *         Management</a> in the <i>IAM User Guide</i>.
     * @throws InternalServerException
     *         An internal failure in an Amazon service occurred.
     * @throws ThrottlingException
     *         The request was denied due to a request throttling.
     * @throws ValidationException
     *         Contains information about data passed in to a field during a request that is not valid.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws DevOpsGuruException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample DevOpsGuruClient.DescribeResourceCollectionHealth
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/devops-guru-2020-12-01/DescribeResourceCollectionHealth"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DescribeResourceCollectionHealthIterable describeResourceCollectionHealthPaginator(
            DescribeResourceCollectionHealthRequest describeResourceCollectionHealthRequest) throws AccessDeniedException,
            InternalServerException, ThrottlingException, ValidationException, AwsServiceException, SdkClientException,
            DevOpsGuruException {
        return new DescribeResourceCollectionHealthIterable(this,
                applyPaginatorUserAgent(describeResourceCollectionHealthRequest));
    }

    /**
     * <p>
     * Returns the integration status of services that are integrated with DevOps Guru. The one service that can be
     * integrated with DevOps Guru is AWS Systems Manager, which can be used to create an OpsItem for each generated
     * insight.
     * </p>
     *
     * @param describeServiceIntegrationRequest
     * @return Result of the DescribeServiceIntegration operation returned by the service.
     * @throws AccessDeniedException
     *         You don't have permissions to perform the requested operation. The user or role that is making the
     *         request must have at least one IAM permissions policy attached that grants the required permissions. For
     *         more information, see <a href="https://docs.aws.amazon.com/IAM/latest/UserGuide/access.html">Access
     *         Management</a> in the <i>IAM User Guide</i>.
     * @throws InternalServerException
     *         An internal failure in an Amazon service occurred.
     * @throws ThrottlingException
     *         The request was denied due to a request throttling.
     * @throws ValidationException
     *         Contains information about data passed in to a field during a request that is not valid.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws DevOpsGuruException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample DevOpsGuruClient.DescribeServiceIntegration
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/devops-guru-2020-12-01/DescribeServiceIntegration"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DescribeServiceIntegrationResponse describeServiceIntegration(
            DescribeServiceIntegrationRequest describeServiceIntegrationRequest) throws AccessDeniedException,
            InternalServerException, ThrottlingException, ValidationException, AwsServiceException, SdkClientException,
            DevOpsGuruException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DescribeServiceIntegrationResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, DescribeServiceIntegrationResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeServiceIntegrationRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "DevOps Guru");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeServiceIntegration");

            return clientHandler
                    .execute(new ClientExecutionParams<DescribeServiceIntegrationRequest, DescribeServiceIntegrationResponse>()
                            .withOperationName("DescribeServiceIntegration").withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withInput(describeServiceIntegrationRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new DescribeServiceIntegrationRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Returns lists AWS resources that are of the specified resource collection type. The one type of AWS resource
     * collection supported is AWS CloudFormation stacks. DevOps Guru can be configured to analyze only the AWS
     * resources that are defined in the stacks.
     * </p>
     *
     * @param getResourceCollectionRequest
     * @return Result of the GetResourceCollection operation returned by the service.
     * @throws AccessDeniedException
     *         You don't have permissions to perform the requested operation. The user or role that is making the
     *         request must have at least one IAM permissions policy attached that grants the required permissions. For
     *         more information, see <a href="https://docs.aws.amazon.com/IAM/latest/UserGuide/access.html">Access
     *         Management</a> in the <i>IAM User Guide</i>.
     * @throws InternalServerException
     *         An internal failure in an Amazon service occurred.
     * @throws ResourceNotFoundException
     *         A requested resource could not be found
     * @throws ThrottlingException
     *         The request was denied due to a request throttling.
     * @throws ValidationException
     *         Contains information about data passed in to a field during a request that is not valid.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws DevOpsGuruException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample DevOpsGuruClient.GetResourceCollection
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/devops-guru-2020-12-01/GetResourceCollection"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public GetResourceCollectionResponse getResourceCollection(GetResourceCollectionRequest getResourceCollectionRequest)
            throws AccessDeniedException, InternalServerException, ResourceNotFoundException, ThrottlingException,
            ValidationException, AwsServiceException, SdkClientException, DevOpsGuruException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetResourceCollectionResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, GetResourceCollectionResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getResourceCollectionRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "DevOps Guru");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetResourceCollection");

            return clientHandler.execute(new ClientExecutionParams<GetResourceCollectionRequest, GetResourceCollectionResponse>()
                    .withOperationName("GetResourceCollection").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(getResourceCollectionRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new GetResourceCollectionRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Returns lists AWS resources that are of the specified resource collection type. The one type of AWS resource
     * collection supported is AWS CloudFormation stacks. DevOps Guru can be configured to analyze only the AWS
     * resources that are defined in the stacks.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #getResourceCollection(software.amazon.awssdk.services.devopsguru.model.GetResourceCollectionRequest)}
     * operation. The return type is a custom iterable that can be used to iterate through all the pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.devopsguru.paginators.GetResourceCollectionIterable responses = client.getResourceCollectionPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.devopsguru.paginators.GetResourceCollectionIterable responses = client
     *             .getResourceCollectionPaginator(request);
     *     for (software.amazon.awssdk.services.devopsguru.model.GetResourceCollectionResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.devopsguru.paginators.GetResourceCollectionIterable responses = client.getResourceCollectionPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of null won't limit the number of results you get with the paginator. It
     * only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #getResourceCollection(software.amazon.awssdk.services.devopsguru.model.GetResourceCollectionRequest)}
     * operation.</b>
     * </p>
     *
     * @param getResourceCollectionRequest
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws AccessDeniedException
     *         You don't have permissions to perform the requested operation. The user or role that is making the
     *         request must have at least one IAM permissions policy attached that grants the required permissions. For
     *         more information, see <a href="https://docs.aws.amazon.com/IAM/latest/UserGuide/access.html">Access
     *         Management</a> in the <i>IAM User Guide</i>.
     * @throws InternalServerException
     *         An internal failure in an Amazon service occurred.
     * @throws ResourceNotFoundException
     *         A requested resource could not be found
     * @throws ThrottlingException
     *         The request was denied due to a request throttling.
     * @throws ValidationException
     *         Contains information about data passed in to a field during a request that is not valid.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws DevOpsGuruException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample DevOpsGuruClient.GetResourceCollection
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/devops-guru-2020-12-01/GetResourceCollection"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public GetResourceCollectionIterable getResourceCollectionPaginator(GetResourceCollectionRequest getResourceCollectionRequest)
            throws AccessDeniedException, InternalServerException, ResourceNotFoundException, ThrottlingException,
            ValidationException, AwsServiceException, SdkClientException, DevOpsGuruException {
        return new GetResourceCollectionIterable(this, applyPaginatorUserAgent(getResourceCollectionRequest));
    }

    /**
     * <p>
     * Returns a list of the anomalies that belong to an insight that you specify using its ID.
     * </p>
     *
     * @param listAnomaliesForInsightRequest
     * @return Result of the ListAnomaliesForInsight operation returned by the service.
     * @throws AccessDeniedException
     *         You don't have permissions to perform the requested operation. The user or role that is making the
     *         request must have at least one IAM permissions policy attached that grants the required permissions. For
     *         more information, see <a href="https://docs.aws.amazon.com/IAM/latest/UserGuide/access.html">Access
     *         Management</a> in the <i>IAM User Guide</i>.
     * @throws InternalServerException
     *         An internal failure in an Amazon service occurred.
     * @throws ResourceNotFoundException
     *         A requested resource could not be found
     * @throws ThrottlingException
     *         The request was denied due to a request throttling.
     * @throws ValidationException
     *         Contains information about data passed in to a field during a request that is not valid.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws DevOpsGuruException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample DevOpsGuruClient.ListAnomaliesForInsight
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/devops-guru-2020-12-01/ListAnomaliesForInsight"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListAnomaliesForInsightResponse listAnomaliesForInsight(ListAnomaliesForInsightRequest listAnomaliesForInsightRequest)
            throws AccessDeniedException, InternalServerException, ResourceNotFoundException, ThrottlingException,
            ValidationException, AwsServiceException, SdkClientException, DevOpsGuruException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListAnomaliesForInsightResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, ListAnomaliesForInsightResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listAnomaliesForInsightRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "DevOps Guru");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListAnomaliesForInsight");

            return clientHandler
                    .execute(new ClientExecutionParams<ListAnomaliesForInsightRequest, ListAnomaliesForInsightResponse>()
                            .withOperationName("ListAnomaliesForInsight").withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withInput(listAnomaliesForInsightRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new ListAnomaliesForInsightRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Returns a list of the anomalies that belong to an insight that you specify using its ID.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #listAnomaliesForInsight(software.amazon.awssdk.services.devopsguru.model.ListAnomaliesForInsightRequest)}
     * operation. The return type is a custom iterable that can be used to iterate through all the pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.devopsguru.paginators.ListAnomaliesForInsightIterable responses = client.listAnomaliesForInsightPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.devopsguru.paginators.ListAnomaliesForInsightIterable responses = client
     *             .listAnomaliesForInsightPaginator(request);
     *     for (software.amazon.awssdk.services.devopsguru.model.ListAnomaliesForInsightResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.devopsguru.paginators.ListAnomaliesForInsightIterable responses = client.listAnomaliesForInsightPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listAnomaliesForInsight(software.amazon.awssdk.services.devopsguru.model.ListAnomaliesForInsightRequest)}
     * operation.</b>
     * </p>
     *
     * @param listAnomaliesForInsightRequest
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws AccessDeniedException
     *         You don't have permissions to perform the requested operation. The user or role that is making the
     *         request must have at least one IAM permissions policy attached that grants the required permissions. For
     *         more information, see <a href="https://docs.aws.amazon.com/IAM/latest/UserGuide/access.html">Access
     *         Management</a> in the <i>IAM User Guide</i>.
     * @throws InternalServerException
     *         An internal failure in an Amazon service occurred.
     * @throws ResourceNotFoundException
     *         A requested resource could not be found
     * @throws ThrottlingException
     *         The request was denied due to a request throttling.
     * @throws ValidationException
     *         Contains information about data passed in to a field during a request that is not valid.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws DevOpsGuruException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample DevOpsGuruClient.ListAnomaliesForInsight
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/devops-guru-2020-12-01/ListAnomaliesForInsight"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListAnomaliesForInsightIterable listAnomaliesForInsightPaginator(
            ListAnomaliesForInsightRequest listAnomaliesForInsightRequest) throws AccessDeniedException, InternalServerException,
            ResourceNotFoundException, ThrottlingException, ValidationException, AwsServiceException, SdkClientException,
            DevOpsGuruException {
        return new ListAnomaliesForInsightIterable(this, applyPaginatorUserAgent(listAnomaliesForInsightRequest));
    }

    /**
     * <p>
     * Returns a list of the events emitted by the resources that are evaluated by DevOps Guru. You can use filters to
     * specify which events are returned.
     * </p>
     *
     * @param listEventsRequest
     * @return Result of the ListEvents operation returned by the service.
     * @throws AccessDeniedException
     *         You don't have permissions to perform the requested operation. The user or role that is making the
     *         request must have at least one IAM permissions policy attached that grants the required permissions. For
     *         more information, see <a href="https://docs.aws.amazon.com/IAM/latest/UserGuide/access.html">Access
     *         Management</a> in the <i>IAM User Guide</i>.
     * @throws InternalServerException
     *         An internal failure in an Amazon service occurred.
     * @throws ResourceNotFoundException
     *         A requested resource could not be found
     * @throws ThrottlingException
     *         The request was denied due to a request throttling.
     * @throws ValidationException
     *         Contains information about data passed in to a field during a request that is not valid.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws DevOpsGuruException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample DevOpsGuruClient.ListEvents
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/devops-guru-2020-12-01/ListEvents" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public ListEventsResponse listEvents(ListEventsRequest listEventsRequest) throws AccessDeniedException,
            InternalServerException, ResourceNotFoundException, ThrottlingException, ValidationException, AwsServiceException,
            SdkClientException, DevOpsGuruException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListEventsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                ListEventsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listEventsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "DevOps Guru");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListEvents");

            return clientHandler
                    .execute(new ClientExecutionParams<ListEventsRequest, ListEventsResponse>().withOperationName("ListEvents")
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(listEventsRequest).withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new ListEventsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Returns a list of the events emitted by the resources that are evaluated by DevOps Guru. You can use filters to
     * specify which events are returned.
     * </p>
     * <br/>
     * <p>
     * This is a variant of {@link #listEvents(software.amazon.awssdk.services.devopsguru.model.ListEventsRequest)}
     * operation. The return type is a custom iterable that can be used to iterate through all the pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.devopsguru.paginators.ListEventsIterable responses = client.listEventsPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.devopsguru.paginators.ListEventsIterable responses = client.listEventsPaginator(request);
     *     for (software.amazon.awssdk.services.devopsguru.model.ListEventsResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.devopsguru.paginators.ListEventsIterable responses = client.listEventsPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listEvents(software.amazon.awssdk.services.devopsguru.model.ListEventsRequest)} operation.</b>
     * </p>
     *
     * @param listEventsRequest
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws AccessDeniedException
     *         You don't have permissions to perform the requested operation. The user or role that is making the
     *         request must have at least one IAM permissions policy attached that grants the required permissions. For
     *         more information, see <a href="https://docs.aws.amazon.com/IAM/latest/UserGuide/access.html">Access
     *         Management</a> in the <i>IAM User Guide</i>.
     * @throws InternalServerException
     *         An internal failure in an Amazon service occurred.
     * @throws ResourceNotFoundException
     *         A requested resource could not be found
     * @throws ThrottlingException
     *         The request was denied due to a request throttling.
     * @throws ValidationException
     *         Contains information about data passed in to a field during a request that is not valid.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws DevOpsGuruException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample DevOpsGuruClient.ListEvents
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/devops-guru-2020-12-01/ListEvents" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public ListEventsIterable listEventsPaginator(ListEventsRequest listEventsRequest) throws AccessDeniedException,
            InternalServerException, ResourceNotFoundException, ThrottlingException, ValidationException, AwsServiceException,
            SdkClientException, DevOpsGuruException {
        return new ListEventsIterable(this, applyPaginatorUserAgent(listEventsRequest));
    }

    /**
     * <p>
     * Returns a list of insights in your AWS account. You can specify which insights are returned by their start time
     * and status (<code>ONGOING</code>, <code>CLOSED</code>, or <code>ANY</code>).
     * </p>
     *
     * @param listInsightsRequest
     * @return Result of the ListInsights operation returned by the service.
     * @throws AccessDeniedException
     *         You don't have permissions to perform the requested operation. The user or role that is making the
     *         request must have at least one IAM permissions policy attached that grants the required permissions. For
     *         more information, see <a href="https://docs.aws.amazon.com/IAM/latest/UserGuide/access.html">Access
     *         Management</a> in the <i>IAM User Guide</i>.
     * @throws InternalServerException
     *         An internal failure in an Amazon service occurred.
     * @throws ThrottlingException
     *         The request was denied due to a request throttling.
     * @throws ValidationException
     *         Contains information about data passed in to a field during a request that is not valid.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws DevOpsGuruException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample DevOpsGuruClient.ListInsights
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/devops-guru-2020-12-01/ListInsights" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public ListInsightsResponse listInsights(ListInsightsRequest listInsightsRequest) throws AccessDeniedException,
            InternalServerException, ThrottlingException, ValidationException, AwsServiceException, SdkClientException,
            DevOpsGuruException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListInsightsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                ListInsightsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listInsightsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "DevOps Guru");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListInsights");

            return clientHandler.execute(new ClientExecutionParams<ListInsightsRequest, ListInsightsResponse>()
                    .withOperationName("ListInsights").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(listInsightsRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListInsightsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Returns a list of insights in your AWS account. You can specify which insights are returned by their start time
     * and status (<code>ONGOING</code>, <code>CLOSED</code>, or <code>ANY</code>).
     * </p>
     * <br/>
     * <p>
     * This is a variant of {@link #listInsights(software.amazon.awssdk.services.devopsguru.model.ListInsightsRequest)}
     * operation. The return type is a custom iterable that can be used to iterate through all the pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.devopsguru.paginators.ListInsightsIterable responses = client.listInsightsPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.devopsguru.paginators.ListInsightsIterable responses = client.listInsightsPaginator(request);
     *     for (software.amazon.awssdk.services.devopsguru.model.ListInsightsResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.devopsguru.paginators.ListInsightsIterable responses = client.listInsightsPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listInsights(software.amazon.awssdk.services.devopsguru.model.ListInsightsRequest)} operation.</b>
     * </p>
     *
     * @param listInsightsRequest
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws AccessDeniedException
     *         You don't have permissions to perform the requested operation. The user or role that is making the
     *         request must have at least one IAM permissions policy attached that grants the required permissions. For
     *         more information, see <a href="https://docs.aws.amazon.com/IAM/latest/UserGuide/access.html">Access
     *         Management</a> in the <i>IAM User Guide</i>.
     * @throws InternalServerException
     *         An internal failure in an Amazon service occurred.
     * @throws ThrottlingException
     *         The request was denied due to a request throttling.
     * @throws ValidationException
     *         Contains information about data passed in to a field during a request that is not valid.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws DevOpsGuruException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample DevOpsGuruClient.ListInsights
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/devops-guru-2020-12-01/ListInsights" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public ListInsightsIterable listInsightsPaginator(ListInsightsRequest listInsightsRequest) throws AccessDeniedException,
            InternalServerException, ThrottlingException, ValidationException, AwsServiceException, SdkClientException,
            DevOpsGuruException {
        return new ListInsightsIterable(this, applyPaginatorUserAgent(listInsightsRequest));
    }

    /**
     * <p>
     * Returns a list of notification channels configured for DevOps Guru. Each notification channel is used to notify
     * you when DevOps Guru generates an insight that contains information about how to improve your operations. The one
     * supported notification channel is Amazon Simple Notification Service (Amazon SNS).
     * </p>
     *
     * @param listNotificationChannelsRequest
     * @return Result of the ListNotificationChannels operation returned by the service.
     * @throws AccessDeniedException
     *         You don't have permissions to perform the requested operation. The user or role that is making the
     *         request must have at least one IAM permissions policy attached that grants the required permissions. For
     *         more information, see <a href="https://docs.aws.amazon.com/IAM/latest/UserGuide/access.html">Access
     *         Management</a> in the <i>IAM User Guide</i>.
     * @throws InternalServerException
     *         An internal failure in an Amazon service occurred.
     * @throws ThrottlingException
     *         The request was denied due to a request throttling.
     * @throws ValidationException
     *         Contains information about data passed in to a field during a request that is not valid.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws DevOpsGuruException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample DevOpsGuruClient.ListNotificationChannels
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/devops-guru-2020-12-01/ListNotificationChannels"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListNotificationChannelsResponse listNotificationChannels(
            ListNotificationChannelsRequest listNotificationChannelsRequest) throws AccessDeniedException,
            InternalServerException, ThrottlingException, ValidationException, AwsServiceException, SdkClientException,
            DevOpsGuruException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListNotificationChannelsResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, ListNotificationChannelsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listNotificationChannelsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "DevOps Guru");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListNotificationChannels");

            return clientHandler
                    .execute(new ClientExecutionParams<ListNotificationChannelsRequest, ListNotificationChannelsResponse>()
                            .withOperationName("ListNotificationChannels").withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withInput(listNotificationChannelsRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new ListNotificationChannelsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Returns a list of notification channels configured for DevOps Guru. Each notification channel is used to notify
     * you when DevOps Guru generates an insight that contains information about how to improve your operations. The one
     * supported notification channel is Amazon Simple Notification Service (Amazon SNS).
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #listNotificationChannels(software.amazon.awssdk.services.devopsguru.model.ListNotificationChannelsRequest)}
     * operation. The return type is a custom iterable that can be used to iterate through all the pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.devopsguru.paginators.ListNotificationChannelsIterable responses = client.listNotificationChannelsPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.devopsguru.paginators.ListNotificationChannelsIterable responses = client
     *             .listNotificationChannelsPaginator(request);
     *     for (software.amazon.awssdk.services.devopsguru.model.ListNotificationChannelsResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.devopsguru.paginators.ListNotificationChannelsIterable responses = client.listNotificationChannelsPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of null won't limit the number of results you get with the paginator. It
     * only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listNotificationChannels(software.amazon.awssdk.services.devopsguru.model.ListNotificationChannelsRequest)}
     * operation.</b>
     * </p>
     *
     * @param listNotificationChannelsRequest
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws AccessDeniedException
     *         You don't have permissions to perform the requested operation. The user or role that is making the
     *         request must have at least one IAM permissions policy attached that grants the required permissions. For
     *         more information, see <a href="https://docs.aws.amazon.com/IAM/latest/UserGuide/access.html">Access
     *         Management</a> in the <i>IAM User Guide</i>.
     * @throws InternalServerException
     *         An internal failure in an Amazon service occurred.
     * @throws ThrottlingException
     *         The request was denied due to a request throttling.
     * @throws ValidationException
     *         Contains information about data passed in to a field during a request that is not valid.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws DevOpsGuruException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample DevOpsGuruClient.ListNotificationChannels
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/devops-guru-2020-12-01/ListNotificationChannels"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListNotificationChannelsIterable listNotificationChannelsPaginator(
            ListNotificationChannelsRequest listNotificationChannelsRequest) throws AccessDeniedException,
            InternalServerException, ThrottlingException, ValidationException, AwsServiceException, SdkClientException,
            DevOpsGuruException {
        return new ListNotificationChannelsIterable(this, applyPaginatorUserAgent(listNotificationChannelsRequest));
    }

    /**
     * <p>
     * Returns a list of a specified insight's recommendations. Each recommendation includes a list of related metrics
     * and a list of related events.
     * </p>
     *
     * @param listRecommendationsRequest
     * @return Result of the ListRecommendations operation returned by the service.
     * @throws AccessDeniedException
     *         You don't have permissions to perform the requested operation. The user or role that is making the
     *         request must have at least one IAM permissions policy attached that grants the required permissions. For
     *         more information, see <a href="https://docs.aws.amazon.com/IAM/latest/UserGuide/access.html">Access
     *         Management</a> in the <i>IAM User Guide</i>.
     * @throws InternalServerException
     *         An internal failure in an Amazon service occurred.
     * @throws ResourceNotFoundException
     *         A requested resource could not be found
     * @throws ThrottlingException
     *         The request was denied due to a request throttling.
     * @throws ValidationException
     *         Contains information about data passed in to a field during a request that is not valid.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws DevOpsGuruException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample DevOpsGuruClient.ListRecommendations
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/devops-guru-2020-12-01/ListRecommendations"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListRecommendationsResponse listRecommendations(ListRecommendationsRequest listRecommendationsRequest)
            throws AccessDeniedException, InternalServerException, ResourceNotFoundException, ThrottlingException,
            ValidationException, AwsServiceException, SdkClientException, DevOpsGuruException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListRecommendationsResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, ListRecommendationsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listRecommendationsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "DevOps Guru");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListRecommendations");

            return clientHandler.execute(new ClientExecutionParams<ListRecommendationsRequest, ListRecommendationsResponse>()
                    .withOperationName("ListRecommendations").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(listRecommendationsRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListRecommendationsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Returns a list of a specified insight's recommendations. Each recommendation includes a list of related metrics
     * and a list of related events.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #listRecommendations(software.amazon.awssdk.services.devopsguru.model.ListRecommendationsRequest)}
     * operation. The return type is a custom iterable that can be used to iterate through all the pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.devopsguru.paginators.ListRecommendationsIterable responses = client.listRecommendationsPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.devopsguru.paginators.ListRecommendationsIterable responses = client
     *             .listRecommendationsPaginator(request);
     *     for (software.amazon.awssdk.services.devopsguru.model.ListRecommendationsResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.devopsguru.paginators.ListRecommendationsIterable responses = client.listRecommendationsPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of null won't limit the number of results you get with the paginator. It
     * only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listRecommendations(software.amazon.awssdk.services.devopsguru.model.ListRecommendationsRequest)}
     * operation.</b>
     * </p>
     *
     * @param listRecommendationsRequest
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws AccessDeniedException
     *         You don't have permissions to perform the requested operation. The user or role that is making the
     *         request must have at least one IAM permissions policy attached that grants the required permissions. For
     *         more information, see <a href="https://docs.aws.amazon.com/IAM/latest/UserGuide/access.html">Access
     *         Management</a> in the <i>IAM User Guide</i>.
     * @throws InternalServerException
     *         An internal failure in an Amazon service occurred.
     * @throws ResourceNotFoundException
     *         A requested resource could not be found
     * @throws ThrottlingException
     *         The request was denied due to a request throttling.
     * @throws ValidationException
     *         Contains information about data passed in to a field during a request that is not valid.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws DevOpsGuruException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample DevOpsGuruClient.ListRecommendations
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/devops-guru-2020-12-01/ListRecommendations"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListRecommendationsIterable listRecommendationsPaginator(ListRecommendationsRequest listRecommendationsRequest)
            throws AccessDeniedException, InternalServerException, ResourceNotFoundException, ThrottlingException,
            ValidationException, AwsServiceException, SdkClientException, DevOpsGuruException {
        return new ListRecommendationsIterable(this, applyPaginatorUserAgent(listRecommendationsRequest));
    }

    /**
     * <p>
     * Collects customer feedback about the specified insight.
     * </p>
     *
     * @param putFeedbackRequest
     * @return Result of the PutFeedback operation returned by the service.
     * @throws AccessDeniedException
     *         You don't have permissions to perform the requested operation. The user or role that is making the
     *         request must have at least one IAM permissions policy attached that grants the required permissions. For
     *         more information, see <a href="https://docs.aws.amazon.com/IAM/latest/UserGuide/access.html">Access
     *         Management</a> in the <i>IAM User Guide</i>.
     * @throws ConflictException
     *         An exception that is thrown when a conflict occurs.
     * @throws InternalServerException
     *         An internal failure in an Amazon service occurred.
     * @throws ResourceNotFoundException
     *         A requested resource could not be found
     * @throws ThrottlingException
     *         The request was denied due to a request throttling.
     * @throws ValidationException
     *         Contains information about data passed in to a field during a request that is not valid.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws DevOpsGuruException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample DevOpsGuruClient.PutFeedback
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/devops-guru-2020-12-01/PutFeedback" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public PutFeedbackResponse putFeedback(PutFeedbackRequest putFeedbackRequest) throws AccessDeniedException,
            ConflictException, InternalServerException, ResourceNotFoundException, ThrottlingException, ValidationException,
            AwsServiceException, SdkClientException, DevOpsGuruException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<PutFeedbackResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                PutFeedbackResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, putFeedbackRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "DevOps Guru");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "PutFeedback");

            return clientHandler.execute(new ClientExecutionParams<PutFeedbackRequest, PutFeedbackResponse>()
                    .withOperationName("PutFeedback").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(putFeedbackRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new PutFeedbackRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Removes a notification channel from DevOps Guru. A notification channel is used to notify you when DevOps Guru
     * generates an insight that contains information about how to improve your operations.
     * </p>
     *
     * @param removeNotificationChannelRequest
     * @return Result of the RemoveNotificationChannel operation returned by the service.
     * @throws AccessDeniedException
     *         You don't have permissions to perform the requested operation. The user or role that is making the
     *         request must have at least one IAM permissions policy attached that grants the required permissions. For
     *         more information, see <a href="https://docs.aws.amazon.com/IAM/latest/UserGuide/access.html">Access
     *         Management</a> in the <i>IAM User Guide</i>.
     * @throws ConflictException
     *         An exception that is thrown when a conflict occurs.
     * @throws InternalServerException
     *         An internal failure in an Amazon service occurred.
     * @throws ResourceNotFoundException
     *         A requested resource could not be found
     * @throws ThrottlingException
     *         The request was denied due to a request throttling.
     * @throws ValidationException
     *         Contains information about data passed in to a field during a request that is not valid.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws DevOpsGuruException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample DevOpsGuruClient.RemoveNotificationChannel
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/devops-guru-2020-12-01/RemoveNotificationChannel"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public RemoveNotificationChannelResponse removeNotificationChannel(
            RemoveNotificationChannelRequest removeNotificationChannelRequest) throws AccessDeniedException, ConflictException,
            InternalServerException, ResourceNotFoundException, ThrottlingException, ValidationException, AwsServiceException,
            SdkClientException, DevOpsGuruException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<RemoveNotificationChannelResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, RemoveNotificationChannelResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, removeNotificationChannelRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "DevOps Guru");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "RemoveNotificationChannel");

            return clientHandler
                    .execute(new ClientExecutionParams<RemoveNotificationChannelRequest, RemoveNotificationChannelResponse>()
                            .withOperationName("RemoveNotificationChannel").withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withInput(removeNotificationChannelRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new RemoveNotificationChannelRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Returns a list of insights in your AWS account. You can specify which insights are returned by their start time,
     * one or more statuses (<code>ONGOING</code>, <code>CLOSED</code>, and <code>CLOSED</code>), one or more severities
     * (<code>LOW</code>, <code>MEDIUM</code>, and <code>HIGH</code>), and type (<code>REACTIVE</code> or
     * <code>PROACTIVE</code>).
     * </p>
     * <p>
     * Use the <code>Filters</code> parameter to specify status and severity search parameters. Use the
     * <code>Type</code> parameter to specify <code>REACTIVE</code> or <code>PROACTIVE</code> in your search.
     * </p>
     *
     * @param searchInsightsRequest
     * @return Result of the SearchInsights operation returned by the service.
     * @throws AccessDeniedException
     *         You don't have permissions to perform the requested operation. The user or role that is making the
     *         request must have at least one IAM permissions policy attached that grants the required permissions. For
     *         more information, see <a href="https://docs.aws.amazon.com/IAM/latest/UserGuide/access.html">Access
     *         Management</a> in the <i>IAM User Guide</i>.
     * @throws InternalServerException
     *         An internal failure in an Amazon service occurred.
     * @throws ThrottlingException
     *         The request was denied due to a request throttling.
     * @throws ValidationException
     *         Contains information about data passed in to a field during a request that is not valid.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws DevOpsGuruException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample DevOpsGuruClient.SearchInsights
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/devops-guru-2020-12-01/SearchInsights" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public SearchInsightsResponse searchInsights(SearchInsightsRequest searchInsightsRequest) throws AccessDeniedException,
            InternalServerException, ThrottlingException, ValidationException, AwsServiceException, SdkClientException,
            DevOpsGuruException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<SearchInsightsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                SearchInsightsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, searchInsightsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "DevOps Guru");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "SearchInsights");

            return clientHandler.execute(new ClientExecutionParams<SearchInsightsRequest, SearchInsightsResponse>()
                    .withOperationName("SearchInsights").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(searchInsightsRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new SearchInsightsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Returns a list of insights in your AWS account. You can specify which insights are returned by their start time,
     * one or more statuses (<code>ONGOING</code>, <code>CLOSED</code>, and <code>CLOSED</code>), one or more severities
     * (<code>LOW</code>, <code>MEDIUM</code>, and <code>HIGH</code>), and type (<code>REACTIVE</code> or
     * <code>PROACTIVE</code>).
     * </p>
     * <p>
     * Use the <code>Filters</code> parameter to specify status and severity search parameters. Use the
     * <code>Type</code> parameter to specify <code>REACTIVE</code> or <code>PROACTIVE</code> in your search.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #searchInsights(software.amazon.awssdk.services.devopsguru.model.SearchInsightsRequest)} operation. The
     * return type is a custom iterable that can be used to iterate through all the pages. SDK will internally handle
     * making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.devopsguru.paginators.SearchInsightsIterable responses = client.searchInsightsPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.devopsguru.paginators.SearchInsightsIterable responses = client
     *             .searchInsightsPaginator(request);
     *     for (software.amazon.awssdk.services.devopsguru.model.SearchInsightsResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.devopsguru.paginators.SearchInsightsIterable responses = client.searchInsightsPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #searchInsights(software.amazon.awssdk.services.devopsguru.model.SearchInsightsRequest)} operation.</b>
     * </p>
     *
     * @param searchInsightsRequest
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws AccessDeniedException
     *         You don't have permissions to perform the requested operation. The user or role that is making the
     *         request must have at least one IAM permissions policy attached that grants the required permissions. For
     *         more information, see <a href="https://docs.aws.amazon.com/IAM/latest/UserGuide/access.html">Access
     *         Management</a> in the <i>IAM User Guide</i>.
     * @throws InternalServerException
     *         An internal failure in an Amazon service occurred.
     * @throws ThrottlingException
     *         The request was denied due to a request throttling.
     * @throws ValidationException
     *         Contains information about data passed in to a field during a request that is not valid.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws DevOpsGuruException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample DevOpsGuruClient.SearchInsights
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/devops-guru-2020-12-01/SearchInsights" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public SearchInsightsIterable searchInsightsPaginator(SearchInsightsRequest searchInsightsRequest)
            throws AccessDeniedException, InternalServerException, ThrottlingException, ValidationException, AwsServiceException,
            SdkClientException, DevOpsGuruException {
        return new SearchInsightsIterable(this, applyPaginatorUserAgent(searchInsightsRequest));
    }

    /**
     * <p>
     * Updates the collection of resources that DevOps Guru analyzes. The one type of AWS resource collection supported
     * is AWS CloudFormation stacks. DevOps Guru can be configured to analyze only the AWS resources that are defined in
     * the stacks. This method also creates the IAM role required for you to use DevOps Guru.
     * </p>
     *
     * @param updateResourceCollectionRequest
     * @return Result of the UpdateResourceCollection operation returned by the service.
     * @throws AccessDeniedException
     *         You don't have permissions to perform the requested operation. The user or role that is making the
     *         request must have at least one IAM permissions policy attached that grants the required permissions. For
     *         more information, see <a href="https://docs.aws.amazon.com/IAM/latest/UserGuide/access.html">Access
     *         Management</a> in the <i>IAM User Guide</i>.
     * @throws ConflictException
     *         An exception that is thrown when a conflict occurs.
     * @throws InternalServerException
     *         An internal failure in an Amazon service occurred.
     * @throws ThrottlingException
     *         The request was denied due to a request throttling.
     * @throws ValidationException
     *         Contains information about data passed in to a field during a request that is not valid.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws DevOpsGuruException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample DevOpsGuruClient.UpdateResourceCollection
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/devops-guru-2020-12-01/UpdateResourceCollection"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public UpdateResourceCollectionResponse updateResourceCollection(
            UpdateResourceCollectionRequest updateResourceCollectionRequest) throws AccessDeniedException, ConflictException,
            InternalServerException, ThrottlingException, ValidationException, AwsServiceException, SdkClientException,
            DevOpsGuruException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UpdateResourceCollectionResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, UpdateResourceCollectionResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateResourceCollectionRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "DevOps Guru");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateResourceCollection");

            return clientHandler
                    .execute(new ClientExecutionParams<UpdateResourceCollectionRequest, UpdateResourceCollectionResponse>()
                            .withOperationName("UpdateResourceCollection").withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withInput(updateResourceCollectionRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new UpdateResourceCollectionRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Enables or disables integration with a service that can be integrated with DevOps Guru. The one service that can
     * be integrated with DevOps Guru is AWS Systems Manager, which can be used to create an OpsItem for each generated
     * insight.
     * </p>
     *
     * @param updateServiceIntegrationRequest
     * @return Result of the UpdateServiceIntegration operation returned by the service.
     * @throws AccessDeniedException
     *         You don't have permissions to perform the requested operation. The user or role that is making the
     *         request must have at least one IAM permissions policy attached that grants the required permissions. For
     *         more information, see <a href="https://docs.aws.amazon.com/IAM/latest/UserGuide/access.html">Access
     *         Management</a> in the <i>IAM User Guide</i>.
     * @throws ConflictException
     *         An exception that is thrown when a conflict occurs.
     * @throws InternalServerException
     *         An internal failure in an Amazon service occurred.
     * @throws ThrottlingException
     *         The request was denied due to a request throttling.
     * @throws ValidationException
     *         Contains information about data passed in to a field during a request that is not valid.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws DevOpsGuruException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample DevOpsGuruClient.UpdateServiceIntegration
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/devops-guru-2020-12-01/UpdateServiceIntegration"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public UpdateServiceIntegrationResponse updateServiceIntegration(
            UpdateServiceIntegrationRequest updateServiceIntegrationRequest) throws AccessDeniedException, ConflictException,
            InternalServerException, ThrottlingException, ValidationException, AwsServiceException, SdkClientException,
            DevOpsGuruException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UpdateServiceIntegrationResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, UpdateServiceIntegrationResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateServiceIntegrationRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "DevOps Guru");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateServiceIntegration");

            return clientHandler
                    .execute(new ClientExecutionParams<UpdateServiceIntegrationRequest, UpdateServiceIntegrationResponse>()
                            .withOperationName("UpdateServiceIntegration").withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withInput(updateServiceIntegrationRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new UpdateServiceIntegrationRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    private static List<MetricPublisher> resolveMetricPublishers(SdkClientConfiguration clientConfiguration,
            RequestOverrideConfiguration requestOverrideConfiguration) {
        List<MetricPublisher> publishers = null;
        if (requestOverrideConfiguration != null) {
            publishers = requestOverrideConfiguration.metricPublishers();
        }
        if (publishers == null || publishers.isEmpty()) {
            publishers = clientConfiguration.option(SdkClientOption.METRIC_PUBLISHERS);
        }
        if (publishers == null) {
            publishers = Collections.emptyList();
        }
        return publishers;
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata) {
        return protocolFactory.createErrorResponseHandler(operationMetadata);
    }

    private <T extends BaseAwsJsonProtocolFactory.Builder<T>> T init(T builder) {
        return builder
                .clientConfiguration(clientConfiguration)
                .defaultServiceExceptionSupplier(DevOpsGuruException::builder)
                .protocol(AwsJsonProtocol.REST_JSON)
                .protocolVersion("1.1")
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ThrottlingException")
                                .exceptionBuilderSupplier(ThrottlingException::builder).httpStatusCode(429).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException")
                                .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).httpStatusCode(402).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InternalServerException")
                                .exceptionBuilderSupplier(InternalServerException::builder).httpStatusCode(500).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("AccessDeniedException")
                                .exceptionBuilderSupplier(AccessDeniedException::builder).httpStatusCode(403).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ConflictException")
                                .exceptionBuilderSupplier(ConflictException::builder).httpStatusCode(409).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ResourceNotFoundException")
                                .exceptionBuilderSupplier(ResourceNotFoundException::builder).httpStatusCode(404).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ValidationException")
                                .exceptionBuilderSupplier(ValidationException::builder).httpStatusCode(400).build());
    }

    @Override
    public void close() {
        clientHandler.close();
    }

    private <T extends DevOpsGuruRequest> T applyPaginatorUserAgent(T request) {
        Consumer<AwsRequestOverrideConfiguration.Builder> userAgentApplier = b -> b.addApiName(ApiName.builder()
                .version(VersionInfo.SDK_VERSION).name("PAGINATED").build());
        AwsRequestOverrideConfiguration overrideConfiguration = request.overrideConfiguration()
                .map(c -> c.toBuilder().applyMutation(userAgentApplier).build())
                .orElse((AwsRequestOverrideConfiguration.builder().applyMutation(userAgentApplier).build()));
        return (T) request.toBuilder().overrideConfiguration(overrideConfiguration).build();
    }
}
