/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.datasync.model;

import java.time.Instant;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 */
@Generated("software.amazon.awssdk:codegen")
public final class DescribeStorageSystemResponse extends DataSyncResponse implements
        ToCopyableBuilder<DescribeStorageSystemResponse.Builder, DescribeStorageSystemResponse> {
    private static final SdkField<String> STORAGE_SYSTEM_ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("StorageSystemArn").getter(getter(DescribeStorageSystemResponse::storageSystemArn))
            .setter(setter(Builder::storageSystemArn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("StorageSystemArn").build()).build();

    private static final SdkField<DiscoveryServerConfiguration> SERVER_CONFIGURATION_FIELD = SdkField
            .<DiscoveryServerConfiguration> builder(MarshallingType.SDK_POJO).memberName("ServerConfiguration")
            .getter(getter(DescribeStorageSystemResponse::serverConfiguration)).setter(setter(Builder::serverConfiguration))
            .constructor(DiscoveryServerConfiguration::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ServerConfiguration").build())
            .build();

    private static final SdkField<String> SYSTEM_TYPE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("SystemType").getter(getter(DescribeStorageSystemResponse::systemTypeAsString))
            .setter(setter(Builder::systemType))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("SystemType").build()).build();

    private static final SdkField<List<String>> AGENT_ARNS_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("AgentArns")
            .getter(getter(DescribeStorageSystemResponse::agentArns))
            .setter(setter(Builder::agentArns))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("AgentArns").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<String> NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Name")
            .getter(getter(DescribeStorageSystemResponse::name)).setter(setter(Builder::name))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Name").build()).build();

    private static final SdkField<String> ERROR_MESSAGE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("ErrorMessage").getter(getter(DescribeStorageSystemResponse::errorMessage))
            .setter(setter(Builder::errorMessage))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ErrorMessage").build()).build();

    private static final SdkField<String> CONNECTIVITY_STATUS_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("ConnectivityStatus").getter(getter(DescribeStorageSystemResponse::connectivityStatusAsString))
            .setter(setter(Builder::connectivityStatus))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ConnectivityStatus").build())
            .build();

    private static final SdkField<String> CLOUD_WATCH_LOG_GROUP_ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("CloudWatchLogGroupArn").getter(getter(DescribeStorageSystemResponse::cloudWatchLogGroupArn))
            .setter(setter(Builder::cloudWatchLogGroupArn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("CloudWatchLogGroupArn").build())
            .build();

    private static final SdkField<Instant> CREATION_TIME_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .memberName("CreationTime").getter(getter(DescribeStorageSystemResponse::creationTime))
            .setter(setter(Builder::creationTime))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("CreationTime").build()).build();

    private static final SdkField<String> SECRETS_MANAGER_ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("SecretsManagerArn").getter(getter(DescribeStorageSystemResponse::secretsManagerArn))
            .setter(setter(Builder::secretsManagerArn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("SecretsManagerArn").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(STORAGE_SYSTEM_ARN_FIELD,
            SERVER_CONFIGURATION_FIELD, SYSTEM_TYPE_FIELD, AGENT_ARNS_FIELD, NAME_FIELD, ERROR_MESSAGE_FIELD,
            CONNECTIVITY_STATUS_FIELD, CLOUD_WATCH_LOG_GROUP_ARN_FIELD, CREATION_TIME_FIELD, SECRETS_MANAGER_ARN_FIELD));

    private final String storageSystemArn;

    private final DiscoveryServerConfiguration serverConfiguration;

    private final String systemType;

    private final List<String> agentArns;

    private final String name;

    private final String errorMessage;

    private final String connectivityStatus;

    private final String cloudWatchLogGroupArn;

    private final Instant creationTime;

    private final String secretsManagerArn;

    private DescribeStorageSystemResponse(BuilderImpl builder) {
        super(builder);
        this.storageSystemArn = builder.storageSystemArn;
        this.serverConfiguration = builder.serverConfiguration;
        this.systemType = builder.systemType;
        this.agentArns = builder.agentArns;
        this.name = builder.name;
        this.errorMessage = builder.errorMessage;
        this.connectivityStatus = builder.connectivityStatus;
        this.cloudWatchLogGroupArn = builder.cloudWatchLogGroupArn;
        this.creationTime = builder.creationTime;
        this.secretsManagerArn = builder.secretsManagerArn;
    }

    /**
     * <p>
     * The ARN of the on-premises storage system that the discovery job looked at.
     * </p>
     * 
     * @return The ARN of the on-premises storage system that the discovery job looked at.
     */
    public final String storageSystemArn() {
        return storageSystemArn;
    }

    /**
     * <p>
     * The server name and network port required to connect with your on-premises storage system's management interface.
     * </p>
     * 
     * @return The server name and network port required to connect with your on-premises storage system's management
     *         interface.
     */
    public final DiscoveryServerConfiguration serverConfiguration() {
        return serverConfiguration;
    }

    /**
     * <p>
     * The type of on-premises storage system.
     * </p>
     * <note>
     * <p>
     * DataSync Discovery currently only supports NetApp Fabric-Attached Storage (FAS) and All Flash FAS (AFF) systems
     * running ONTAP 9.7 or later.
     * </p>
     * </note>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #systemType} will
     * return {@link DiscoverySystemType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #systemTypeAsString}.
     * </p>
     * 
     * @return The type of on-premises storage system.</p> <note>
     *         <p>
     *         DataSync Discovery currently only supports NetApp Fabric-Attached Storage (FAS) and All Flash FAS (AFF)
     *         systems running ONTAP 9.7 or later.
     *         </p>
     * @see DiscoverySystemType
     */
    public final DiscoverySystemType systemType() {
        return DiscoverySystemType.fromValue(systemType);
    }

    /**
     * <p>
     * The type of on-premises storage system.
     * </p>
     * <note>
     * <p>
     * DataSync Discovery currently only supports NetApp Fabric-Attached Storage (FAS) and All Flash FAS (AFF) systems
     * running ONTAP 9.7 or later.
     * </p>
     * </note>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #systemType} will
     * return {@link DiscoverySystemType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #systemTypeAsString}.
     * </p>
     * 
     * @return The type of on-premises storage system.</p> <note>
     *         <p>
     *         DataSync Discovery currently only supports NetApp Fabric-Attached Storage (FAS) and All Flash FAS (AFF)
     *         systems running ONTAP 9.7 or later.
     *         </p>
     * @see DiscoverySystemType
     */
    public final String systemTypeAsString() {
        return systemType;
    }

    /**
     * For responses, this returns true if the service returned a value for the AgentArns property. This DOES NOT check
     * that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is
     * useful because the SDK will never return a null collection or map, but you may need to differentiate between the
     * service returning nothing (or null) and the service returning an empty collection or map. For requests, this
     * returns true if a value for the property was specified in the request builder, and false if a value was not
     * specified.
     */
    public final boolean hasAgentArns() {
        return agentArns != null && !(agentArns instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The ARN of the DataSync agent that connects to and reads from your on-premises storage system.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasAgentArns} method.
     * </p>
     * 
     * @return The ARN of the DataSync agent that connects to and reads from your on-premises storage system.
     */
    public final List<String> agentArns() {
        return agentArns;
    }

    /**
     * <p>
     * The name that you gave your on-premises storage system when adding it to DataSync Discovery.
     * </p>
     * 
     * @return The name that you gave your on-premises storage system when adding it to DataSync Discovery.
     */
    public final String name() {
        return name;
    }

    /**
     * <p>
     * Describes the connectivity error that the DataSync agent is encountering with your on-premises storage system.
     * </p>
     * 
     * @return Describes the connectivity error that the DataSync agent is encountering with your on-premises storage
     *         system.
     */
    public final String errorMessage() {
        return errorMessage;
    }

    /**
     * <p>
     * Indicates whether your DataSync agent can connect to your on-premises storage system.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version,
     * {@link #connectivityStatus} will return {@link StorageSystemConnectivityStatus#UNKNOWN_TO_SDK_VERSION}. The raw
     * value returned by the service is available from {@link #connectivityStatusAsString}.
     * </p>
     * 
     * @return Indicates whether your DataSync agent can connect to your on-premises storage system.
     * @see StorageSystemConnectivityStatus
     */
    public final StorageSystemConnectivityStatus connectivityStatus() {
        return StorageSystemConnectivityStatus.fromValue(connectivityStatus);
    }

    /**
     * <p>
     * Indicates whether your DataSync agent can connect to your on-premises storage system.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version,
     * {@link #connectivityStatus} will return {@link StorageSystemConnectivityStatus#UNKNOWN_TO_SDK_VERSION}. The raw
     * value returned by the service is available from {@link #connectivityStatusAsString}.
     * </p>
     * 
     * @return Indicates whether your DataSync agent can connect to your on-premises storage system.
     * @see StorageSystemConnectivityStatus
     */
    public final String connectivityStatusAsString() {
        return connectivityStatus;
    }

    /**
     * <p>
     * The ARN of the Amazon CloudWatch log group that's used to monitor and log discovery job events.
     * </p>
     * 
     * @return The ARN of the Amazon CloudWatch log group that's used to monitor and log discovery job events.
     */
    public final String cloudWatchLogGroupArn() {
        return cloudWatchLogGroupArn;
    }

    /**
     * <p>
     * The time when you added the on-premises storage system to DataSync Discovery.
     * </p>
     * 
     * @return The time when you added the on-premises storage system to DataSync Discovery.
     */
    public final Instant creationTime() {
        return creationTime;
    }

    /**
     * <p>
     * The ARN of the secret that stores your on-premises storage system's credentials. DataSync Discovery stores these
     * credentials in <a href=
     * "https://docs.aws.amazon.com/datasync/latest/userguide/discovery-configure-storage.html#discovery-add-storage"
     * >Secrets Manager</a>.
     * </p>
     * 
     * @return The ARN of the secret that stores your on-premises storage system's credentials. DataSync Discovery
     *         stores these credentials in <a href=
     *         "https://docs.aws.amazon.com/datasync/latest/userguide/discovery-configure-storage.html#discovery-add-storage"
     *         >Secrets Manager</a>.
     */
    public final String secretsManagerArn() {
        return secretsManagerArn;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + super.hashCode();
        hashCode = 31 * hashCode + Objects.hashCode(storageSystemArn());
        hashCode = 31 * hashCode + Objects.hashCode(serverConfiguration());
        hashCode = 31 * hashCode + Objects.hashCode(systemTypeAsString());
        hashCode = 31 * hashCode + Objects.hashCode(hasAgentArns() ? agentArns() : null);
        hashCode = 31 * hashCode + Objects.hashCode(name());
        hashCode = 31 * hashCode + Objects.hashCode(errorMessage());
        hashCode = 31 * hashCode + Objects.hashCode(connectivityStatusAsString());
        hashCode = 31 * hashCode + Objects.hashCode(cloudWatchLogGroupArn());
        hashCode = 31 * hashCode + Objects.hashCode(creationTime());
        hashCode = 31 * hashCode + Objects.hashCode(secretsManagerArn());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return super.equals(obj) && equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof DescribeStorageSystemResponse)) {
            return false;
        }
        DescribeStorageSystemResponse other = (DescribeStorageSystemResponse) obj;
        return Objects.equals(storageSystemArn(), other.storageSystemArn())
                && Objects.equals(serverConfiguration(), other.serverConfiguration())
                && Objects.equals(systemTypeAsString(), other.systemTypeAsString()) && hasAgentArns() == other.hasAgentArns()
                && Objects.equals(agentArns(), other.agentArns()) && Objects.equals(name(), other.name())
                && Objects.equals(errorMessage(), other.errorMessage())
                && Objects.equals(connectivityStatusAsString(), other.connectivityStatusAsString())
                && Objects.equals(cloudWatchLogGroupArn(), other.cloudWatchLogGroupArn())
                && Objects.equals(creationTime(), other.creationTime())
                && Objects.equals(secretsManagerArn(), other.secretsManagerArn());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("DescribeStorageSystemResponse").add("StorageSystemArn", storageSystemArn())
                .add("ServerConfiguration", serverConfiguration()).add("SystemType", systemTypeAsString())
                .add("AgentArns", hasAgentArns() ? agentArns() : null).add("Name", name()).add("ErrorMessage", errorMessage())
                .add("ConnectivityStatus", connectivityStatusAsString()).add("CloudWatchLogGroupArn", cloudWatchLogGroupArn())
                .add("CreationTime", creationTime()).add("SecretsManagerArn", secretsManagerArn()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "StorageSystemArn":
            return Optional.ofNullable(clazz.cast(storageSystemArn()));
        case "ServerConfiguration":
            return Optional.ofNullable(clazz.cast(serverConfiguration()));
        case "SystemType":
            return Optional.ofNullable(clazz.cast(systemTypeAsString()));
        case "AgentArns":
            return Optional.ofNullable(clazz.cast(agentArns()));
        case "Name":
            return Optional.ofNullable(clazz.cast(name()));
        case "ErrorMessage":
            return Optional.ofNullable(clazz.cast(errorMessage()));
        case "ConnectivityStatus":
            return Optional.ofNullable(clazz.cast(connectivityStatusAsString()));
        case "CloudWatchLogGroupArn":
            return Optional.ofNullable(clazz.cast(cloudWatchLogGroupArn()));
        case "CreationTime":
            return Optional.ofNullable(clazz.cast(creationTime()));
        case "SecretsManagerArn":
            return Optional.ofNullable(clazz.cast(secretsManagerArn()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<DescribeStorageSystemResponse, T> g) {
        return obj -> g.apply((DescribeStorageSystemResponse) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends DataSyncResponse.Builder, SdkPojo, CopyableBuilder<Builder, DescribeStorageSystemResponse> {
        /**
         * <p>
         * The ARN of the on-premises storage system that the discovery job looked at.
         * </p>
         * 
         * @param storageSystemArn
         *        The ARN of the on-premises storage system that the discovery job looked at.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder storageSystemArn(String storageSystemArn);

        /**
         * <p>
         * The server name and network port required to connect with your on-premises storage system's management
         * interface.
         * </p>
         * 
         * @param serverConfiguration
         *        The server name and network port required to connect with your on-premises storage system's management
         *        interface.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder serverConfiguration(DiscoveryServerConfiguration serverConfiguration);

        /**
         * <p>
         * The server name and network port required to connect with your on-premises storage system's management
         * interface.
         * </p>
         * This is a convenience method that creates an instance of the {@link DiscoveryServerConfiguration.Builder}
         * avoiding the need to create one manually via {@link DiscoveryServerConfiguration#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link DiscoveryServerConfiguration.Builder#build()} is called
         * immediately and its result is passed to {@link #serverConfiguration(DiscoveryServerConfiguration)}.
         * 
         * @param serverConfiguration
         *        a consumer that will call methods on {@link DiscoveryServerConfiguration.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #serverConfiguration(DiscoveryServerConfiguration)
         */
        default Builder serverConfiguration(Consumer<DiscoveryServerConfiguration.Builder> serverConfiguration) {
            return serverConfiguration(DiscoveryServerConfiguration.builder().applyMutation(serverConfiguration).build());
        }

        /**
         * <p>
         * The type of on-premises storage system.
         * </p>
         * <note>
         * <p>
         * DataSync Discovery currently only supports NetApp Fabric-Attached Storage (FAS) and All Flash FAS (AFF)
         * systems running ONTAP 9.7 or later.
         * </p>
         * </note>
         * 
         * @param systemType
         *        The type of on-premises storage system.</p> <note>
         *        <p>
         *        DataSync Discovery currently only supports NetApp Fabric-Attached Storage (FAS) and All Flash FAS
         *        (AFF) systems running ONTAP 9.7 or later.
         *        </p>
         * @see DiscoverySystemType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see DiscoverySystemType
         */
        Builder systemType(String systemType);

        /**
         * <p>
         * The type of on-premises storage system.
         * </p>
         * <note>
         * <p>
         * DataSync Discovery currently only supports NetApp Fabric-Attached Storage (FAS) and All Flash FAS (AFF)
         * systems running ONTAP 9.7 or later.
         * </p>
         * </note>
         * 
         * @param systemType
         *        The type of on-premises storage system.</p> <note>
         *        <p>
         *        DataSync Discovery currently only supports NetApp Fabric-Attached Storage (FAS) and All Flash FAS
         *        (AFF) systems running ONTAP 9.7 or later.
         *        </p>
         * @see DiscoverySystemType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see DiscoverySystemType
         */
        Builder systemType(DiscoverySystemType systemType);

        /**
         * <p>
         * The ARN of the DataSync agent that connects to and reads from your on-premises storage system.
         * </p>
         * 
         * @param agentArns
         *        The ARN of the DataSync agent that connects to and reads from your on-premises storage system.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder agentArns(Collection<String> agentArns);

        /**
         * <p>
         * The ARN of the DataSync agent that connects to and reads from your on-premises storage system.
         * </p>
         * 
         * @param agentArns
         *        The ARN of the DataSync agent that connects to and reads from your on-premises storage system.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder agentArns(String... agentArns);

        /**
         * <p>
         * The name that you gave your on-premises storage system when adding it to DataSync Discovery.
         * </p>
         * 
         * @param name
         *        The name that you gave your on-premises storage system when adding it to DataSync Discovery.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder name(String name);

        /**
         * <p>
         * Describes the connectivity error that the DataSync agent is encountering with your on-premises storage
         * system.
         * </p>
         * 
         * @param errorMessage
         *        Describes the connectivity error that the DataSync agent is encountering with your on-premises storage
         *        system.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder errorMessage(String errorMessage);

        /**
         * <p>
         * Indicates whether your DataSync agent can connect to your on-premises storage system.
         * </p>
         * 
         * @param connectivityStatus
         *        Indicates whether your DataSync agent can connect to your on-premises storage system.
         * @see StorageSystemConnectivityStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see StorageSystemConnectivityStatus
         */
        Builder connectivityStatus(String connectivityStatus);

        /**
         * <p>
         * Indicates whether your DataSync agent can connect to your on-premises storage system.
         * </p>
         * 
         * @param connectivityStatus
         *        Indicates whether your DataSync agent can connect to your on-premises storage system.
         * @see StorageSystemConnectivityStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see StorageSystemConnectivityStatus
         */
        Builder connectivityStatus(StorageSystemConnectivityStatus connectivityStatus);

        /**
         * <p>
         * The ARN of the Amazon CloudWatch log group that's used to monitor and log discovery job events.
         * </p>
         * 
         * @param cloudWatchLogGroupArn
         *        The ARN of the Amazon CloudWatch log group that's used to monitor and log discovery job events.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder cloudWatchLogGroupArn(String cloudWatchLogGroupArn);

        /**
         * <p>
         * The time when you added the on-premises storage system to DataSync Discovery.
         * </p>
         * 
         * @param creationTime
         *        The time when you added the on-premises storage system to DataSync Discovery.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder creationTime(Instant creationTime);

        /**
         * <p>
         * The ARN of the secret that stores your on-premises storage system's credentials. DataSync Discovery stores
         * these credentials in <a href=
         * "https://docs.aws.amazon.com/datasync/latest/userguide/discovery-configure-storage.html#discovery-add-storage"
         * >Secrets Manager</a>.
         * </p>
         * 
         * @param secretsManagerArn
         *        The ARN of the secret that stores your on-premises storage system's credentials. DataSync Discovery
         *        stores these credentials in <a href=
         *        "https://docs.aws.amazon.com/datasync/latest/userguide/discovery-configure-storage.html#discovery-add-storage"
         *        >Secrets Manager</a>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder secretsManagerArn(String secretsManagerArn);
    }

    static final class BuilderImpl extends DataSyncResponse.BuilderImpl implements Builder {
        private String storageSystemArn;

        private DiscoveryServerConfiguration serverConfiguration;

        private String systemType;

        private List<String> agentArns = DefaultSdkAutoConstructList.getInstance();

        private String name;

        private String errorMessage;

        private String connectivityStatus;

        private String cloudWatchLogGroupArn;

        private Instant creationTime;

        private String secretsManagerArn;

        private BuilderImpl() {
        }

        private BuilderImpl(DescribeStorageSystemResponse model) {
            super(model);
            storageSystemArn(model.storageSystemArn);
            serverConfiguration(model.serverConfiguration);
            systemType(model.systemType);
            agentArns(model.agentArns);
            name(model.name);
            errorMessage(model.errorMessage);
            connectivityStatus(model.connectivityStatus);
            cloudWatchLogGroupArn(model.cloudWatchLogGroupArn);
            creationTime(model.creationTime);
            secretsManagerArn(model.secretsManagerArn);
        }

        public final String getStorageSystemArn() {
            return storageSystemArn;
        }

        public final void setStorageSystemArn(String storageSystemArn) {
            this.storageSystemArn = storageSystemArn;
        }

        @Override
        public final Builder storageSystemArn(String storageSystemArn) {
            this.storageSystemArn = storageSystemArn;
            return this;
        }

        public final DiscoveryServerConfiguration.Builder getServerConfiguration() {
            return serverConfiguration != null ? serverConfiguration.toBuilder() : null;
        }

        public final void setServerConfiguration(DiscoveryServerConfiguration.BuilderImpl serverConfiguration) {
            this.serverConfiguration = serverConfiguration != null ? serverConfiguration.build() : null;
        }

        @Override
        public final Builder serverConfiguration(DiscoveryServerConfiguration serverConfiguration) {
            this.serverConfiguration = serverConfiguration;
            return this;
        }

        public final String getSystemType() {
            return systemType;
        }

        public final void setSystemType(String systemType) {
            this.systemType = systemType;
        }

        @Override
        public final Builder systemType(String systemType) {
            this.systemType = systemType;
            return this;
        }

        @Override
        public final Builder systemType(DiscoverySystemType systemType) {
            this.systemType(systemType == null ? null : systemType.toString());
            return this;
        }

        public final Collection<String> getAgentArns() {
            if (agentArns instanceof SdkAutoConstructList) {
                return null;
            }
            return agentArns;
        }

        public final void setAgentArns(Collection<String> agentArns) {
            this.agentArns = DiscoveryAgentArnListCopier.copy(agentArns);
        }

        @Override
        public final Builder agentArns(Collection<String> agentArns) {
            this.agentArns = DiscoveryAgentArnListCopier.copy(agentArns);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder agentArns(String... agentArns) {
            agentArns(Arrays.asList(agentArns));
            return this;
        }

        public final String getName() {
            return name;
        }

        public final void setName(String name) {
            this.name = name;
        }

        @Override
        public final Builder name(String name) {
            this.name = name;
            return this;
        }

        public final String getErrorMessage() {
            return errorMessage;
        }

        public final void setErrorMessage(String errorMessage) {
            this.errorMessage = errorMessage;
        }

        @Override
        public final Builder errorMessage(String errorMessage) {
            this.errorMessage = errorMessage;
            return this;
        }

        public final String getConnectivityStatus() {
            return connectivityStatus;
        }

        public final void setConnectivityStatus(String connectivityStatus) {
            this.connectivityStatus = connectivityStatus;
        }

        @Override
        public final Builder connectivityStatus(String connectivityStatus) {
            this.connectivityStatus = connectivityStatus;
            return this;
        }

        @Override
        public final Builder connectivityStatus(StorageSystemConnectivityStatus connectivityStatus) {
            this.connectivityStatus(connectivityStatus == null ? null : connectivityStatus.toString());
            return this;
        }

        public final String getCloudWatchLogGroupArn() {
            return cloudWatchLogGroupArn;
        }

        public final void setCloudWatchLogGroupArn(String cloudWatchLogGroupArn) {
            this.cloudWatchLogGroupArn = cloudWatchLogGroupArn;
        }

        @Override
        public final Builder cloudWatchLogGroupArn(String cloudWatchLogGroupArn) {
            this.cloudWatchLogGroupArn = cloudWatchLogGroupArn;
            return this;
        }

        public final Instant getCreationTime() {
            return creationTime;
        }

        public final void setCreationTime(Instant creationTime) {
            this.creationTime = creationTime;
        }

        @Override
        public final Builder creationTime(Instant creationTime) {
            this.creationTime = creationTime;
            return this;
        }

        public final String getSecretsManagerArn() {
            return secretsManagerArn;
        }

        public final void setSecretsManagerArn(String secretsManagerArn) {
            this.secretsManagerArn = secretsManagerArn;
        }

        @Override
        public final Builder secretsManagerArn(String secretsManagerArn) {
            this.secretsManagerArn = secretsManagerArn;
            return this;
        }

        @Override
        public DescribeStorageSystemResponse build() {
            return new DescribeStorageSystemResponse(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
