/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.connect.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.Mutable;
import software.amazon.awssdk.annotations.NotThreadSafe;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.traits.MapTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructMap;
import software.amazon.awssdk.core.util.SdkAutoConstructMap;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Information about a phone number that has been claimed to your Amazon Connect instance or traffic distribution group.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class ClaimedPhoneNumberSummary implements SdkPojo, Serializable,
        ToCopyableBuilder<ClaimedPhoneNumberSummary.Builder, ClaimedPhoneNumberSummary> {
    private static final SdkField<String> PHONE_NUMBER_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("PhoneNumberId").getter(getter(ClaimedPhoneNumberSummary::phoneNumberId))
            .setter(setter(Builder::phoneNumberId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("PhoneNumberId").build()).build();

    private static final SdkField<String> PHONE_NUMBER_ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("PhoneNumberArn").getter(getter(ClaimedPhoneNumberSummary::phoneNumberArn))
            .setter(setter(Builder::phoneNumberArn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("PhoneNumberArn").build()).build();

    private static final SdkField<String> PHONE_NUMBER_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("PhoneNumber").getter(getter(ClaimedPhoneNumberSummary::phoneNumber))
            .setter(setter(Builder::phoneNumber))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("PhoneNumber").build()).build();

    private static final SdkField<String> PHONE_NUMBER_COUNTRY_CODE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("PhoneNumberCountryCode").getter(getter(ClaimedPhoneNumberSummary::phoneNumberCountryCodeAsString))
            .setter(setter(Builder::phoneNumberCountryCode))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("PhoneNumberCountryCode").build())
            .build();

    private static final SdkField<String> PHONE_NUMBER_TYPE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("PhoneNumberType").getter(getter(ClaimedPhoneNumberSummary::phoneNumberTypeAsString))
            .setter(setter(Builder::phoneNumberType))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("PhoneNumberType").build()).build();

    private static final SdkField<String> PHONE_NUMBER_DESCRIPTION_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("PhoneNumberDescription").getter(getter(ClaimedPhoneNumberSummary::phoneNumberDescription))
            .setter(setter(Builder::phoneNumberDescription))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("PhoneNumberDescription").build())
            .build();

    private static final SdkField<String> TARGET_ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("TargetArn").getter(getter(ClaimedPhoneNumberSummary::targetArn)).setter(setter(Builder::targetArn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("TargetArn").build()).build();

    private static final SdkField<String> INSTANCE_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("InstanceId").getter(getter(ClaimedPhoneNumberSummary::instanceId)).setter(setter(Builder::instanceId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("InstanceId").build()).build();

    private static final SdkField<Map<String, String>> TAGS_FIELD = SdkField
            .<Map<String, String>> builder(MarshallingType.MAP)
            .memberName("Tags")
            .getter(getter(ClaimedPhoneNumberSummary::tags))
            .setter(setter(Builder::tags))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Tags").build(),
                    MapTrait.builder()
                            .keyLocationName("key")
                            .valueLocationName("value")
                            .valueFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("value").build()).build()).build()).build();

    private static final SdkField<PhoneNumberStatus> PHONE_NUMBER_STATUS_FIELD = SdkField
            .<PhoneNumberStatus> builder(MarshallingType.SDK_POJO).memberName("PhoneNumberStatus")
            .getter(getter(ClaimedPhoneNumberSummary::phoneNumberStatus)).setter(setter(Builder::phoneNumberStatus))
            .constructor(PhoneNumberStatus::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("PhoneNumberStatus").build()).build();

    private static final SdkField<String> SOURCE_PHONE_NUMBER_ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("SourcePhoneNumberArn").getter(getter(ClaimedPhoneNumberSummary::sourcePhoneNumberArn))
            .setter(setter(Builder::sourcePhoneNumberArn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("SourcePhoneNumberArn").build())
            .build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(PHONE_NUMBER_ID_FIELD,
            PHONE_NUMBER_ARN_FIELD, PHONE_NUMBER_FIELD, PHONE_NUMBER_COUNTRY_CODE_FIELD, PHONE_NUMBER_TYPE_FIELD,
            PHONE_NUMBER_DESCRIPTION_FIELD, TARGET_ARN_FIELD, INSTANCE_ID_FIELD, TAGS_FIELD, PHONE_NUMBER_STATUS_FIELD,
            SOURCE_PHONE_NUMBER_ARN_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = memberNameToFieldInitializer();

    private static final long serialVersionUID = 1L;

    private final String phoneNumberId;

    private final String phoneNumberArn;

    private final String phoneNumber;

    private final String phoneNumberCountryCode;

    private final String phoneNumberType;

    private final String phoneNumberDescription;

    private final String targetArn;

    private final String instanceId;

    private final Map<String, String> tags;

    private final PhoneNumberStatus phoneNumberStatus;

    private final String sourcePhoneNumberArn;

    private ClaimedPhoneNumberSummary(BuilderImpl builder) {
        this.phoneNumberId = builder.phoneNumberId;
        this.phoneNumberArn = builder.phoneNumberArn;
        this.phoneNumber = builder.phoneNumber;
        this.phoneNumberCountryCode = builder.phoneNumberCountryCode;
        this.phoneNumberType = builder.phoneNumberType;
        this.phoneNumberDescription = builder.phoneNumberDescription;
        this.targetArn = builder.targetArn;
        this.instanceId = builder.instanceId;
        this.tags = builder.tags;
        this.phoneNumberStatus = builder.phoneNumberStatus;
        this.sourcePhoneNumberArn = builder.sourcePhoneNumberArn;
    }

    /**
     * <p>
     * A unique identifier for the phone number.
     * </p>
     * 
     * @return A unique identifier for the phone number.
     */
    public final String phoneNumberId() {
        return phoneNumberId;
    }

    /**
     * <p>
     * The Amazon Resource Name (ARN) of the phone number.
     * </p>
     * 
     * @return The Amazon Resource Name (ARN) of the phone number.
     */
    public final String phoneNumberArn() {
        return phoneNumberArn;
    }

    /**
     * <p>
     * The phone number. Phone numbers are formatted
     * <code>[+] [country code] [subscriber number including area code]</code>.
     * </p>
     * 
     * @return The phone number. Phone numbers are formatted
     *         <code>[+] [country code] [subscriber number including area code]</code>.
     */
    public final String phoneNumber() {
        return phoneNumber;
    }

    /**
     * <p>
     * The ISO country code.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version,
     * {@link #phoneNumberCountryCode} will return {@link PhoneNumberCountryCode#UNKNOWN_TO_SDK_VERSION}. The raw value
     * returned by the service is available from {@link #phoneNumberCountryCodeAsString}.
     * </p>
     * 
     * @return The ISO country code.
     * @see PhoneNumberCountryCode
     */
    public final PhoneNumberCountryCode phoneNumberCountryCode() {
        return PhoneNumberCountryCode.fromValue(phoneNumberCountryCode);
    }

    /**
     * <p>
     * The ISO country code.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version,
     * {@link #phoneNumberCountryCode} will return {@link PhoneNumberCountryCode#UNKNOWN_TO_SDK_VERSION}. The raw value
     * returned by the service is available from {@link #phoneNumberCountryCodeAsString}.
     * </p>
     * 
     * @return The ISO country code.
     * @see PhoneNumberCountryCode
     */
    public final String phoneNumberCountryCodeAsString() {
        return phoneNumberCountryCode;
    }

    /**
     * <p>
     * The type of phone number.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #phoneNumberType}
     * will return {@link PhoneNumberType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #phoneNumberTypeAsString}.
     * </p>
     * 
     * @return The type of phone number.
     * @see PhoneNumberType
     */
    public final PhoneNumberType phoneNumberType() {
        return PhoneNumberType.fromValue(phoneNumberType);
    }

    /**
     * <p>
     * The type of phone number.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #phoneNumberType}
     * will return {@link PhoneNumberType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #phoneNumberTypeAsString}.
     * </p>
     * 
     * @return The type of phone number.
     * @see PhoneNumberType
     */
    public final String phoneNumberTypeAsString() {
        return phoneNumberType;
    }

    /**
     * <p>
     * The description of the phone number.
     * </p>
     * 
     * @return The description of the phone number.
     */
    public final String phoneNumberDescription() {
        return phoneNumberDescription;
    }

    /**
     * <p>
     * The Amazon Resource Name (ARN) for Amazon Connect instances or traffic distribution groups that phone number
     * inbound traffic is routed through.
     * </p>
     * 
     * @return The Amazon Resource Name (ARN) for Amazon Connect instances or traffic distribution groups that phone
     *         number inbound traffic is routed through.
     */
    public final String targetArn() {
        return targetArn;
    }

    /**
     * <p>
     * The identifier of the Amazon Connect instance that phone numbers are claimed to. You can <a
     * href="https://docs.aws.amazon.com/connect/latest/adminguide/find-instance-arn.html">find the instance ID</a> in
     * the Amazon Resource Name (ARN) of the instance.
     * </p>
     * 
     * @return The identifier of the Amazon Connect instance that phone numbers are claimed to. You can <a
     *         href="https://docs.aws.amazon.com/connect/latest/adminguide/find-instance-arn.html">find the instance
     *         ID</a> in the Amazon Resource Name (ARN) of the instance.
     */
    public final String instanceId() {
        return instanceId;
    }

    /**
     * For responses, this returns true if the service returned a value for the Tags property. This DOES NOT check that
     * the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is useful
     * because the SDK will never return a null collection or map, but you may need to differentiate between the service
     * returning nothing (or null) and the service returning an empty collection or map. For requests, this returns true
     * if a value for the property was specified in the request builder, and false if a value was not specified.
     */
    public final boolean hasTags() {
        return tags != null && !(tags instanceof SdkAutoConstructMap);
    }

    /**
     * <p>
     * The tags used to organize, track, or control access for this resource. For example, { "Tags": {"key1":"value1",
     * "key2":"value2"} }.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasTags} method.
     * </p>
     * 
     * @return The tags used to organize, track, or control access for this resource. For example, { "Tags":
     *         {"key1":"value1", "key2":"value2"} }.
     */
    public final Map<String, String> tags() {
        return tags;
    }

    /**
     * <p>
     * The status of the phone number.
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>CLAIMED</code> means the previous <a
     * href="https://docs.aws.amazon.com/connect/latest/APIReference/API_ClaimPhoneNumber.html">ClaimPhoneNumber</a> or
     * <a
     * href="https://docs.aws.amazon.com/connect/latest/APIReference/API_UpdatePhoneNumber.html">UpdatePhoneNumber</a>
     * operation succeeded.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>IN_PROGRESS</code> means a <a
     * href="https://docs.aws.amazon.com/connect/latest/APIReference/API_ClaimPhoneNumber.html">ClaimPhoneNumber</a>, <a
     * href="https://docs.aws.amazon.com/connect/latest/APIReference/API_UpdatePhoneNumber.html">UpdatePhoneNumber</a>,
     * or <a href="https://docs.aws.amazon.com/connect/latest/APIReference/API_UpdatePhoneNumberMetadata.html">
     * UpdatePhoneNumberMetadata</a> operation is still in progress and has not yet completed. You can call <a
     * href="https://docs.aws.amazon.com/connect/latest/APIReference/API_DescribePhoneNumber.html"
     * >DescribePhoneNumber</a> at a later time to verify if the previous operation has completed.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>FAILED</code> indicates that the previous <a
     * href="https://docs.aws.amazon.com/connect/latest/APIReference/API_ClaimPhoneNumber.html">ClaimPhoneNumber</a> or
     * <a
     * href="https://docs.aws.amazon.com/connect/latest/APIReference/API_UpdatePhoneNumber.html">UpdatePhoneNumber</a>
     * operation has failed. It will include a message indicating the failure reason. A common reason for a failure may
     * be that the <code>TargetArn</code> value you are claiming or updating a phone number to has reached its limit of
     * total claimed numbers. If you received a <code>FAILED</code> status from a <code>ClaimPhoneNumber</code> API
     * call, you have one day to retry claiming the phone number before the number is released back to the inventory for
     * other customers to claim.
     * </p>
     * </li>
     * </ul>
     * <note>
     * <p>
     * You will not be billed for the phone number during the 1-day period if number claiming fails.
     * </p>
     * </note>
     * 
     * @return The status of the phone number.</p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>CLAIMED</code> means the previous <a
     *         href="https://docs.aws.amazon.com/connect/latest/APIReference/API_ClaimPhoneNumber.html"
     *         >ClaimPhoneNumber</a> or <a
     *         href="https://docs.aws.amazon.com/connect/latest/APIReference/API_UpdatePhoneNumber.html"
     *         >UpdatePhoneNumber</a> operation succeeded.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>IN_PROGRESS</code> means a <a
     *         href="https://docs.aws.amazon.com/connect/latest/APIReference/API_ClaimPhoneNumber.html"
     *         >ClaimPhoneNumber</a>, <a
     *         href="https://docs.aws.amazon.com/connect/latest/APIReference/API_UpdatePhoneNumber.html"
     *         >UpdatePhoneNumber</a>, or <a
     *         href="https://docs.aws.amazon.com/connect/latest/APIReference/API_UpdatePhoneNumberMetadata.html"
     *         >UpdatePhoneNumberMetadata</a> operation is still in progress and has not yet completed. You can call <a
     *         href="https://docs.aws.amazon.com/connect/latest/APIReference/API_DescribePhoneNumber.html">
     *         DescribePhoneNumber</a> at a later time to verify if the previous operation has completed.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>FAILED</code> indicates that the previous <a
     *         href="https://docs.aws.amazon.com/connect/latest/APIReference/API_ClaimPhoneNumber.html"
     *         >ClaimPhoneNumber</a> or <a
     *         href="https://docs.aws.amazon.com/connect/latest/APIReference/API_UpdatePhoneNumber.html"
     *         >UpdatePhoneNumber</a> operation has failed. It will include a message indicating the failure reason. A
     *         common reason for a failure may be that the <code>TargetArn</code> value you are claiming or updating a
     *         phone number to has reached its limit of total claimed numbers. If you received a <code>FAILED</code>
     *         status from a <code>ClaimPhoneNumber</code> API call, you have one day to retry claiming the phone number
     *         before the number is released back to the inventory for other customers to claim.
     *         </p>
     *         </li>
     *         </ul>
     *         <note>
     *         <p>
     *         You will not be billed for the phone number during the 1-day period if number claiming fails.
     *         </p>
     */
    public final PhoneNumberStatus phoneNumberStatus() {
        return phoneNumberStatus;
    }

    /**
     * <p>
     * The claimed phone number ARN that was previously imported from the external service, such as Amazon Web Services
     * End User Messaging. If it is from Amazon Web Services End User Messaging, it looks like the ARN of the phone
     * number that was imported from Amazon Web Services End User Messaging.
     * </p>
     * 
     * @return The claimed phone number ARN that was previously imported from the external service, such as Amazon Web
     *         Services End User Messaging. If it is from Amazon Web Services End User Messaging, it looks like the ARN
     *         of the phone number that was imported from Amazon Web Services End User Messaging.
     */
    public final String sourcePhoneNumberArn() {
        return sourcePhoneNumberArn;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(phoneNumberId());
        hashCode = 31 * hashCode + Objects.hashCode(phoneNumberArn());
        hashCode = 31 * hashCode + Objects.hashCode(phoneNumber());
        hashCode = 31 * hashCode + Objects.hashCode(phoneNumberCountryCodeAsString());
        hashCode = 31 * hashCode + Objects.hashCode(phoneNumberTypeAsString());
        hashCode = 31 * hashCode + Objects.hashCode(phoneNumberDescription());
        hashCode = 31 * hashCode + Objects.hashCode(targetArn());
        hashCode = 31 * hashCode + Objects.hashCode(instanceId());
        hashCode = 31 * hashCode + Objects.hashCode(hasTags() ? tags() : null);
        hashCode = 31 * hashCode + Objects.hashCode(phoneNumberStatus());
        hashCode = 31 * hashCode + Objects.hashCode(sourcePhoneNumberArn());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof ClaimedPhoneNumberSummary)) {
            return false;
        }
        ClaimedPhoneNumberSummary other = (ClaimedPhoneNumberSummary) obj;
        return Objects.equals(phoneNumberId(), other.phoneNumberId()) && Objects.equals(phoneNumberArn(), other.phoneNumberArn())
                && Objects.equals(phoneNumber(), other.phoneNumber())
                && Objects.equals(phoneNumberCountryCodeAsString(), other.phoneNumberCountryCodeAsString())
                && Objects.equals(phoneNumberTypeAsString(), other.phoneNumberTypeAsString())
                && Objects.equals(phoneNumberDescription(), other.phoneNumberDescription())
                && Objects.equals(targetArn(), other.targetArn()) && Objects.equals(instanceId(), other.instanceId())
                && hasTags() == other.hasTags() && Objects.equals(tags(), other.tags())
                && Objects.equals(phoneNumberStatus(), other.phoneNumberStatus())
                && Objects.equals(sourcePhoneNumberArn(), other.sourcePhoneNumberArn());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("ClaimedPhoneNumberSummary").add("PhoneNumberId", phoneNumberId())
                .add("PhoneNumberArn", phoneNumberArn()).add("PhoneNumber", phoneNumber())
                .add("PhoneNumberCountryCode", phoneNumberCountryCodeAsString())
                .add("PhoneNumberType", phoneNumberTypeAsString()).add("PhoneNumberDescription", phoneNumberDescription())
                .add("TargetArn", targetArn()).add("InstanceId", instanceId()).add("Tags", hasTags() ? tags() : null)
                .add("PhoneNumberStatus", phoneNumberStatus()).add("SourcePhoneNumberArn", sourcePhoneNumberArn()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "PhoneNumberId":
            return Optional.ofNullable(clazz.cast(phoneNumberId()));
        case "PhoneNumberArn":
            return Optional.ofNullable(clazz.cast(phoneNumberArn()));
        case "PhoneNumber":
            return Optional.ofNullable(clazz.cast(phoneNumber()));
        case "PhoneNumberCountryCode":
            return Optional.ofNullable(clazz.cast(phoneNumberCountryCodeAsString()));
        case "PhoneNumberType":
            return Optional.ofNullable(clazz.cast(phoneNumberTypeAsString()));
        case "PhoneNumberDescription":
            return Optional.ofNullable(clazz.cast(phoneNumberDescription()));
        case "TargetArn":
            return Optional.ofNullable(clazz.cast(targetArn()));
        case "InstanceId":
            return Optional.ofNullable(clazz.cast(instanceId()));
        case "Tags":
            return Optional.ofNullable(clazz.cast(tags()));
        case "PhoneNumberStatus":
            return Optional.ofNullable(clazz.cast(phoneNumberStatus()));
        case "SourcePhoneNumberArn":
            return Optional.ofNullable(clazz.cast(sourcePhoneNumberArn()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static Map<String, SdkField<?>> memberNameToFieldInitializer() {
        Map<String, SdkField<?>> map = new HashMap<>();
        map.put("PhoneNumberId", PHONE_NUMBER_ID_FIELD);
        map.put("PhoneNumberArn", PHONE_NUMBER_ARN_FIELD);
        map.put("PhoneNumber", PHONE_NUMBER_FIELD);
        map.put("PhoneNumberCountryCode", PHONE_NUMBER_COUNTRY_CODE_FIELD);
        map.put("PhoneNumberType", PHONE_NUMBER_TYPE_FIELD);
        map.put("PhoneNumberDescription", PHONE_NUMBER_DESCRIPTION_FIELD);
        map.put("TargetArn", TARGET_ARN_FIELD);
        map.put("InstanceId", INSTANCE_ID_FIELD);
        map.put("Tags", TAGS_FIELD);
        map.put("PhoneNumberStatus", PHONE_NUMBER_STATUS_FIELD);
        map.put("SourcePhoneNumberArn", SOURCE_PHONE_NUMBER_ARN_FIELD);
        return Collections.unmodifiableMap(map);
    }

    private static <T> Function<Object, T> getter(Function<ClaimedPhoneNumberSummary, T> g) {
        return obj -> g.apply((ClaimedPhoneNumberSummary) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    @Mutable
    @NotThreadSafe
    public interface Builder extends SdkPojo, CopyableBuilder<Builder, ClaimedPhoneNumberSummary> {
        /**
         * <p>
         * A unique identifier for the phone number.
         * </p>
         * 
         * @param phoneNumberId
         *        A unique identifier for the phone number.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder phoneNumberId(String phoneNumberId);

        /**
         * <p>
         * The Amazon Resource Name (ARN) of the phone number.
         * </p>
         * 
         * @param phoneNumberArn
         *        The Amazon Resource Name (ARN) of the phone number.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder phoneNumberArn(String phoneNumberArn);

        /**
         * <p>
         * The phone number. Phone numbers are formatted
         * <code>[+] [country code] [subscriber number including area code]</code>.
         * </p>
         * 
         * @param phoneNumber
         *        The phone number. Phone numbers are formatted
         *        <code>[+] [country code] [subscriber number including area code]</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder phoneNumber(String phoneNumber);

        /**
         * <p>
         * The ISO country code.
         * </p>
         * 
         * @param phoneNumberCountryCode
         *        The ISO country code.
         * @see PhoneNumberCountryCode
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see PhoneNumberCountryCode
         */
        Builder phoneNumberCountryCode(String phoneNumberCountryCode);

        /**
         * <p>
         * The ISO country code.
         * </p>
         * 
         * @param phoneNumberCountryCode
         *        The ISO country code.
         * @see PhoneNumberCountryCode
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see PhoneNumberCountryCode
         */
        Builder phoneNumberCountryCode(PhoneNumberCountryCode phoneNumberCountryCode);

        /**
         * <p>
         * The type of phone number.
         * </p>
         * 
         * @param phoneNumberType
         *        The type of phone number.
         * @see PhoneNumberType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see PhoneNumberType
         */
        Builder phoneNumberType(String phoneNumberType);

        /**
         * <p>
         * The type of phone number.
         * </p>
         * 
         * @param phoneNumberType
         *        The type of phone number.
         * @see PhoneNumberType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see PhoneNumberType
         */
        Builder phoneNumberType(PhoneNumberType phoneNumberType);

        /**
         * <p>
         * The description of the phone number.
         * </p>
         * 
         * @param phoneNumberDescription
         *        The description of the phone number.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder phoneNumberDescription(String phoneNumberDescription);

        /**
         * <p>
         * The Amazon Resource Name (ARN) for Amazon Connect instances or traffic distribution groups that phone number
         * inbound traffic is routed through.
         * </p>
         * 
         * @param targetArn
         *        The Amazon Resource Name (ARN) for Amazon Connect instances or traffic distribution groups that phone
         *        number inbound traffic is routed through.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder targetArn(String targetArn);

        /**
         * <p>
         * The identifier of the Amazon Connect instance that phone numbers are claimed to. You can <a
         * href="https://docs.aws.amazon.com/connect/latest/adminguide/find-instance-arn.html">find the instance ID</a>
         * in the Amazon Resource Name (ARN) of the instance.
         * </p>
         * 
         * @param instanceId
         *        The identifier of the Amazon Connect instance that phone numbers are claimed to. You can <a
         *        href="https://docs.aws.amazon.com/connect/latest/adminguide/find-instance-arn.html">find the instance
         *        ID</a> in the Amazon Resource Name (ARN) of the instance.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder instanceId(String instanceId);

        /**
         * <p>
         * The tags used to organize, track, or control access for this resource. For example, { "Tags":
         * {"key1":"value1", "key2":"value2"} }.
         * </p>
         * 
         * @param tags
         *        The tags used to organize, track, or control access for this resource. For example, { "Tags":
         *        {"key1":"value1", "key2":"value2"} }.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder tags(Map<String, String> tags);

        /**
         * <p>
         * The status of the phone number.
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>CLAIMED</code> means the previous <a
         * href="https://docs.aws.amazon.com/connect/latest/APIReference/API_ClaimPhoneNumber.html">ClaimPhoneNumber</a>
         * or <a
         * href="https://docs.aws.amazon.com/connect/latest/APIReference/API_UpdatePhoneNumber.html">UpdatePhoneNumber
         * </a> operation succeeded.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>IN_PROGRESS</code> means a <a
         * href="https://docs.aws.amazon.com/connect/latest/APIReference/API_ClaimPhoneNumber.html"
         * >ClaimPhoneNumber</a>, <a
         * href="https://docs.aws.amazon.com/connect/latest/APIReference/API_UpdatePhoneNumber.html"
         * >UpdatePhoneNumber</a>, or <a
         * href="https://docs.aws.amazon.com/connect/latest/APIReference/API_UpdatePhoneNumberMetadata.html"
         * >UpdatePhoneNumberMetadata</a> operation is still in progress and has not yet completed. You can call <a
         * href=
         * "https://docs.aws.amazon.com/connect/latest/APIReference/API_DescribePhoneNumber.html">DescribePhoneNumber
         * </a> at a later time to verify if the previous operation has completed.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>FAILED</code> indicates that the previous <a
         * href="https://docs.aws.amazon.com/connect/latest/APIReference/API_ClaimPhoneNumber.html">ClaimPhoneNumber</a>
         * or <a
         * href="https://docs.aws.amazon.com/connect/latest/APIReference/API_UpdatePhoneNumber.html">UpdatePhoneNumber
         * </a> operation has failed. It will include a message indicating the failure reason. A common reason for a
         * failure may be that the <code>TargetArn</code> value you are claiming or updating a phone number to has
         * reached its limit of total claimed numbers. If you received a <code>FAILED</code> status from a
         * <code>ClaimPhoneNumber</code> API call, you have one day to retry claiming the phone number before the number
         * is released back to the inventory for other customers to claim.
         * </p>
         * </li>
         * </ul>
         * <note>
         * <p>
         * You will not be billed for the phone number during the 1-day period if number claiming fails.
         * </p>
         * </note>
         * 
         * @param phoneNumberStatus
         *        The status of the phone number.</p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>CLAIMED</code> means the previous <a
         *        href="https://docs.aws.amazon.com/connect/latest/APIReference/API_ClaimPhoneNumber.html"
         *        >ClaimPhoneNumber</a> or <a
         *        href="https://docs.aws.amazon.com/connect/latest/APIReference/API_UpdatePhoneNumber.html"
         *        >UpdatePhoneNumber</a> operation succeeded.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>IN_PROGRESS</code> means a <a
         *        href="https://docs.aws.amazon.com/connect/latest/APIReference/API_ClaimPhoneNumber.html"
         *        >ClaimPhoneNumber</a>, <a
         *        href="https://docs.aws.amazon.com/connect/latest/APIReference/API_UpdatePhoneNumber.html"
         *        >UpdatePhoneNumber</a>, or <a
         *        href="https://docs.aws.amazon.com/connect/latest/APIReference/API_UpdatePhoneNumberMetadata.html"
         *        >UpdatePhoneNumberMetadata</a> operation is still in progress and has not yet completed. You can call
         *        <a href="https://docs.aws.amazon.com/connect/latest/APIReference/API_DescribePhoneNumber.html">
         *        DescribePhoneNumber</a> at a later time to verify if the previous operation has completed.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>FAILED</code> indicates that the previous <a
         *        href="https://docs.aws.amazon.com/connect/latest/APIReference/API_ClaimPhoneNumber.html"
         *        >ClaimPhoneNumber</a> or <a
         *        href="https://docs.aws.amazon.com/connect/latest/APIReference/API_UpdatePhoneNumber.html"
         *        >UpdatePhoneNumber</a> operation has failed. It will include a message indicating the failure reason.
         *        A common reason for a failure may be that the <code>TargetArn</code> value you are claiming or
         *        updating a phone number to has reached its limit of total claimed numbers. If you received a
         *        <code>FAILED</code> status from a <code>ClaimPhoneNumber</code> API call, you have one day to retry
         *        claiming the phone number before the number is released back to the inventory for other customers to
         *        claim.
         *        </p>
         *        </li>
         *        </ul>
         *        <note>
         *        <p>
         *        You will not be billed for the phone number during the 1-day period if number claiming fails.
         *        </p>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder phoneNumberStatus(PhoneNumberStatus phoneNumberStatus);

        /**
         * <p>
         * The status of the phone number.
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>CLAIMED</code> means the previous <a
         * href="https://docs.aws.amazon.com/connect/latest/APIReference/API_ClaimPhoneNumber.html">ClaimPhoneNumber</a>
         * or <a
         * href="https://docs.aws.amazon.com/connect/latest/APIReference/API_UpdatePhoneNumber.html">UpdatePhoneNumber
         * </a> operation succeeded.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>IN_PROGRESS</code> means a <a
         * href="https://docs.aws.amazon.com/connect/latest/APIReference/API_ClaimPhoneNumber.html"
         * >ClaimPhoneNumber</a>, <a
         * href="https://docs.aws.amazon.com/connect/latest/APIReference/API_UpdatePhoneNumber.html"
         * >UpdatePhoneNumber</a>, or <a
         * href="https://docs.aws.amazon.com/connect/latest/APIReference/API_UpdatePhoneNumberMetadata.html"
         * >UpdatePhoneNumberMetadata</a> operation is still in progress and has not yet completed. You can call <a
         * href=
         * "https://docs.aws.amazon.com/connect/latest/APIReference/API_DescribePhoneNumber.html">DescribePhoneNumber
         * </a> at a later time to verify if the previous operation has completed.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>FAILED</code> indicates that the previous <a
         * href="https://docs.aws.amazon.com/connect/latest/APIReference/API_ClaimPhoneNumber.html">ClaimPhoneNumber</a>
         * or <a
         * href="https://docs.aws.amazon.com/connect/latest/APIReference/API_UpdatePhoneNumber.html">UpdatePhoneNumber
         * </a> operation has failed. It will include a message indicating the failure reason. A common reason for a
         * failure may be that the <code>TargetArn</code> value you are claiming or updating a phone number to has
         * reached its limit of total claimed numbers. If you received a <code>FAILED</code> status from a
         * <code>ClaimPhoneNumber</code> API call, you have one day to retry claiming the phone number before the number
         * is released back to the inventory for other customers to claim.
         * </p>
         * </li>
         * </ul>
         * <note>
         * <p>
         * You will not be billed for the phone number during the 1-day period if number claiming fails.
         * </p>
         * </note> This is a convenience method that creates an instance of the {@link PhoneNumberStatus.Builder}
         * avoiding the need to create one manually via {@link PhoneNumberStatus#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link PhoneNumberStatus.Builder#build()} is called immediately and its
         * result is passed to {@link #phoneNumberStatus(PhoneNumberStatus)}.
         * 
         * @param phoneNumberStatus
         *        a consumer that will call methods on {@link PhoneNumberStatus.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #phoneNumberStatus(PhoneNumberStatus)
         */
        default Builder phoneNumberStatus(Consumer<PhoneNumberStatus.Builder> phoneNumberStatus) {
            return phoneNumberStatus(PhoneNumberStatus.builder().applyMutation(phoneNumberStatus).build());
        }

        /**
         * <p>
         * The claimed phone number ARN that was previously imported from the external service, such as Amazon Web
         * Services End User Messaging. If it is from Amazon Web Services End User Messaging, it looks like the ARN of
         * the phone number that was imported from Amazon Web Services End User Messaging.
         * </p>
         * 
         * @param sourcePhoneNumberArn
         *        The claimed phone number ARN that was previously imported from the external service, such as Amazon
         *        Web Services End User Messaging. If it is from Amazon Web Services End User Messaging, it looks like
         *        the ARN of the phone number that was imported from Amazon Web Services End User Messaging.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder sourcePhoneNumberArn(String sourcePhoneNumberArn);
    }

    static final class BuilderImpl implements Builder {
        private String phoneNumberId;

        private String phoneNumberArn;

        private String phoneNumber;

        private String phoneNumberCountryCode;

        private String phoneNumberType;

        private String phoneNumberDescription;

        private String targetArn;

        private String instanceId;

        private Map<String, String> tags = DefaultSdkAutoConstructMap.getInstance();

        private PhoneNumberStatus phoneNumberStatus;

        private String sourcePhoneNumberArn;

        private BuilderImpl() {
        }

        private BuilderImpl(ClaimedPhoneNumberSummary model) {
            phoneNumberId(model.phoneNumberId);
            phoneNumberArn(model.phoneNumberArn);
            phoneNumber(model.phoneNumber);
            phoneNumberCountryCode(model.phoneNumberCountryCode);
            phoneNumberType(model.phoneNumberType);
            phoneNumberDescription(model.phoneNumberDescription);
            targetArn(model.targetArn);
            instanceId(model.instanceId);
            tags(model.tags);
            phoneNumberStatus(model.phoneNumberStatus);
            sourcePhoneNumberArn(model.sourcePhoneNumberArn);
        }

        public final String getPhoneNumberId() {
            return phoneNumberId;
        }

        public final void setPhoneNumberId(String phoneNumberId) {
            this.phoneNumberId = phoneNumberId;
        }

        @Override
        public final Builder phoneNumberId(String phoneNumberId) {
            this.phoneNumberId = phoneNumberId;
            return this;
        }

        public final String getPhoneNumberArn() {
            return phoneNumberArn;
        }

        public final void setPhoneNumberArn(String phoneNumberArn) {
            this.phoneNumberArn = phoneNumberArn;
        }

        @Override
        public final Builder phoneNumberArn(String phoneNumberArn) {
            this.phoneNumberArn = phoneNumberArn;
            return this;
        }

        public final String getPhoneNumber() {
            return phoneNumber;
        }

        public final void setPhoneNumber(String phoneNumber) {
            this.phoneNumber = phoneNumber;
        }

        @Override
        public final Builder phoneNumber(String phoneNumber) {
            this.phoneNumber = phoneNumber;
            return this;
        }

        public final String getPhoneNumberCountryCode() {
            return phoneNumberCountryCode;
        }

        public final void setPhoneNumberCountryCode(String phoneNumberCountryCode) {
            this.phoneNumberCountryCode = phoneNumberCountryCode;
        }

        @Override
        public final Builder phoneNumberCountryCode(String phoneNumberCountryCode) {
            this.phoneNumberCountryCode = phoneNumberCountryCode;
            return this;
        }

        @Override
        public final Builder phoneNumberCountryCode(PhoneNumberCountryCode phoneNumberCountryCode) {
            this.phoneNumberCountryCode(phoneNumberCountryCode == null ? null : phoneNumberCountryCode.toString());
            return this;
        }

        public final String getPhoneNumberType() {
            return phoneNumberType;
        }

        public final void setPhoneNumberType(String phoneNumberType) {
            this.phoneNumberType = phoneNumberType;
        }

        @Override
        public final Builder phoneNumberType(String phoneNumberType) {
            this.phoneNumberType = phoneNumberType;
            return this;
        }

        @Override
        public final Builder phoneNumberType(PhoneNumberType phoneNumberType) {
            this.phoneNumberType(phoneNumberType == null ? null : phoneNumberType.toString());
            return this;
        }

        public final String getPhoneNumberDescription() {
            return phoneNumberDescription;
        }

        public final void setPhoneNumberDescription(String phoneNumberDescription) {
            this.phoneNumberDescription = phoneNumberDescription;
        }

        @Override
        public final Builder phoneNumberDescription(String phoneNumberDescription) {
            this.phoneNumberDescription = phoneNumberDescription;
            return this;
        }

        public final String getTargetArn() {
            return targetArn;
        }

        public final void setTargetArn(String targetArn) {
            this.targetArn = targetArn;
        }

        @Override
        public final Builder targetArn(String targetArn) {
            this.targetArn = targetArn;
            return this;
        }

        public final String getInstanceId() {
            return instanceId;
        }

        public final void setInstanceId(String instanceId) {
            this.instanceId = instanceId;
        }

        @Override
        public final Builder instanceId(String instanceId) {
            this.instanceId = instanceId;
            return this;
        }

        public final Map<String, String> getTags() {
            if (tags instanceof SdkAutoConstructMap) {
                return null;
            }
            return tags;
        }

        public final void setTags(Map<String, String> tags) {
            this.tags = TagMapCopier.copy(tags);
        }

        @Override
        public final Builder tags(Map<String, String> tags) {
            this.tags = TagMapCopier.copy(tags);
            return this;
        }

        public final PhoneNumberStatus.Builder getPhoneNumberStatus() {
            return phoneNumberStatus != null ? phoneNumberStatus.toBuilder() : null;
        }

        public final void setPhoneNumberStatus(PhoneNumberStatus.BuilderImpl phoneNumberStatus) {
            this.phoneNumberStatus = phoneNumberStatus != null ? phoneNumberStatus.build() : null;
        }

        @Override
        public final Builder phoneNumberStatus(PhoneNumberStatus phoneNumberStatus) {
            this.phoneNumberStatus = phoneNumberStatus;
            return this;
        }

        public final String getSourcePhoneNumberArn() {
            return sourcePhoneNumberArn;
        }

        public final void setSourcePhoneNumberArn(String sourcePhoneNumberArn) {
            this.sourcePhoneNumberArn = sourcePhoneNumberArn;
        }

        @Override
        public final Builder sourcePhoneNumberArn(String sourcePhoneNumberArn) {
            this.sourcePhoneNumberArn = sourcePhoneNumberArn;
            return this;
        }

        @Override
        public ClaimedPhoneNumberSummary build() {
            return new ClaimedPhoneNumberSummary(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
