/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.computeoptimizer.model;

import java.io.Serializable;
import java.time.Instant;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.Mutable;
import software.amazon.awssdk.annotations.NotThreadSafe;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Describes a license recommendation for an EC2 instance.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class LicenseRecommendation implements SdkPojo, Serializable,
        ToCopyableBuilder<LicenseRecommendation.Builder, LicenseRecommendation> {
    private static final SdkField<String> RESOURCE_ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("resourceArn").getter(getter(LicenseRecommendation::resourceArn)).setter(setter(Builder::resourceArn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("resourceArn").build()).build();

    private static final SdkField<String> ACCOUNT_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("accountId").getter(getter(LicenseRecommendation::accountId)).setter(setter(Builder::accountId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("accountId").build()).build();

    private static final SdkField<LicenseConfiguration> CURRENT_LICENSE_CONFIGURATION_FIELD = SdkField
            .<LicenseConfiguration> builder(MarshallingType.SDK_POJO)
            .memberName("currentLicenseConfiguration")
            .getter(getter(LicenseRecommendation::currentLicenseConfiguration))
            .setter(setter(Builder::currentLicenseConfiguration))
            .constructor(LicenseConfiguration::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("currentLicenseConfiguration")
                    .build()).build();

    private static final SdkField<Double> LOOKBACK_PERIOD_IN_DAYS_FIELD = SdkField.<Double> builder(MarshallingType.DOUBLE)
            .memberName("lookbackPeriodInDays").getter(getter(LicenseRecommendation::lookbackPeriodInDays))
            .setter(setter(Builder::lookbackPeriodInDays))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("lookbackPeriodInDays").build())
            .build();

    private static final SdkField<Instant> LAST_REFRESH_TIMESTAMP_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .memberName("lastRefreshTimestamp").getter(getter(LicenseRecommendation::lastRefreshTimestamp))
            .setter(setter(Builder::lastRefreshTimestamp))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("lastRefreshTimestamp").build())
            .build();

    private static final SdkField<String> FINDING_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("finding")
            .getter(getter(LicenseRecommendation::findingAsString)).setter(setter(Builder::finding))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("finding").build()).build();

    private static final SdkField<List<String>> FINDING_REASON_CODES_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("findingReasonCodes")
            .getter(getter(LicenseRecommendation::findingReasonCodesAsStrings))
            .setter(setter(Builder::findingReasonCodesWithStrings))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("findingReasonCodes").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<List<LicenseRecommendationOption>> LICENSE_RECOMMENDATION_OPTIONS_FIELD = SdkField
            .<List<LicenseRecommendationOption>> builder(MarshallingType.LIST)
            .memberName("licenseRecommendationOptions")
            .getter(getter(LicenseRecommendation::licenseRecommendationOptions))
            .setter(setter(Builder::licenseRecommendationOptions))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("licenseRecommendationOptions")
                    .build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<LicenseRecommendationOption> builder(MarshallingType.SDK_POJO)
                                            .constructor(LicenseRecommendationOption::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<List<Tag>> TAGS_FIELD = SdkField
            .<List<Tag>> builder(MarshallingType.LIST)
            .memberName("tags")
            .getter(getter(LicenseRecommendation::tags))
            .setter(setter(Builder::tags))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("tags").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<Tag> builder(MarshallingType.SDK_POJO)
                                            .constructor(Tag::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(RESOURCE_ARN_FIELD,
            ACCOUNT_ID_FIELD, CURRENT_LICENSE_CONFIGURATION_FIELD, LOOKBACK_PERIOD_IN_DAYS_FIELD, LAST_REFRESH_TIMESTAMP_FIELD,
            FINDING_FIELD, FINDING_REASON_CODES_FIELD, LICENSE_RECOMMENDATION_OPTIONS_FIELD, TAGS_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = memberNameToFieldInitializer();

    private static final long serialVersionUID = 1L;

    private final String resourceArn;

    private final String accountId;

    private final LicenseConfiguration currentLicenseConfiguration;

    private final Double lookbackPeriodInDays;

    private final Instant lastRefreshTimestamp;

    private final String finding;

    private final List<String> findingReasonCodes;

    private final List<LicenseRecommendationOption> licenseRecommendationOptions;

    private final List<Tag> tags;

    private LicenseRecommendation(BuilderImpl builder) {
        this.resourceArn = builder.resourceArn;
        this.accountId = builder.accountId;
        this.currentLicenseConfiguration = builder.currentLicenseConfiguration;
        this.lookbackPeriodInDays = builder.lookbackPeriodInDays;
        this.lastRefreshTimestamp = builder.lastRefreshTimestamp;
        this.finding = builder.finding;
        this.findingReasonCodes = builder.findingReasonCodes;
        this.licenseRecommendationOptions = builder.licenseRecommendationOptions;
        this.tags = builder.tags;
    }

    /**
     * <p>
     * The ARN that identifies the Amazon EC2 instance.
     * </p>
     * 
     * @return The ARN that identifies the Amazon EC2 instance.
     */
    public final String resourceArn() {
        return resourceArn;
    }

    /**
     * <p>
     * The Amazon Web Services account ID of the license.
     * </p>
     * 
     * @return The Amazon Web Services account ID of the license.
     */
    public final String accountId() {
        return accountId;
    }

    /**
     * <p>
     * An object that describes the current configuration of an instance that runs on a license.
     * </p>
     * 
     * @return An object that describes the current configuration of an instance that runs on a license.
     */
    public final LicenseConfiguration currentLicenseConfiguration() {
        return currentLicenseConfiguration;
    }

    /**
     * <p>
     * The number of days for which utilization metrics were analyzed for an instance that runs on a license.
     * </p>
     * 
     * @return The number of days for which utilization metrics were analyzed for an instance that runs on a license.
     */
    public final Double lookbackPeriodInDays() {
        return lookbackPeriodInDays;
    }

    /**
     * <p>
     * The timestamp of when the license recommendation was last generated.
     * </p>
     * 
     * @return The timestamp of when the license recommendation was last generated.
     */
    public final Instant lastRefreshTimestamp() {
        return lastRefreshTimestamp;
    }

    /**
     * <p>
     * The finding classification for an instance that runs on a license.
     * </p>
     * <p>
     * Findings include:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>InsufficentMetrics</code> — When Compute Optimizer detects that your CloudWatch Application Insights isn't
     * enabled or is enabled with insufficient permissions.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>NotOptimized</code> — When Compute Optimizer detects that your EC2 infrastructure isn't using any of the
     * SQL server license features you're paying for, a license is considered not optimized.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>Optimized</code> — When Compute Optimizer detects that all specifications of your license meet the
     * performance requirements of your workload.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #finding} will
     * return {@link LicenseFinding#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #findingAsString}.
     * </p>
     * 
     * @return The finding classification for an instance that runs on a license. </p>
     *         <p>
     *         Findings include:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>InsufficentMetrics</code> — When Compute Optimizer detects that your CloudWatch Application
     *         Insights isn't enabled or is enabled with insufficient permissions.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>NotOptimized</code> — When Compute Optimizer detects that your EC2 infrastructure isn't using any
     *         of the SQL server license features you're paying for, a license is considered not optimized.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>Optimized</code> — When Compute Optimizer detects that all specifications of your license meet the
     *         performance requirements of your workload.
     *         </p>
     *         </li>
     * @see LicenseFinding
     */
    public final LicenseFinding finding() {
        return LicenseFinding.fromValue(finding);
    }

    /**
     * <p>
     * The finding classification for an instance that runs on a license.
     * </p>
     * <p>
     * Findings include:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>InsufficentMetrics</code> — When Compute Optimizer detects that your CloudWatch Application Insights isn't
     * enabled or is enabled with insufficient permissions.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>NotOptimized</code> — When Compute Optimizer detects that your EC2 infrastructure isn't using any of the
     * SQL server license features you're paying for, a license is considered not optimized.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>Optimized</code> — When Compute Optimizer detects that all specifications of your license meet the
     * performance requirements of your workload.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #finding} will
     * return {@link LicenseFinding#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #findingAsString}.
     * </p>
     * 
     * @return The finding classification for an instance that runs on a license. </p>
     *         <p>
     *         Findings include:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>InsufficentMetrics</code> — When Compute Optimizer detects that your CloudWatch Application
     *         Insights isn't enabled or is enabled with insufficient permissions.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>NotOptimized</code> — When Compute Optimizer detects that your EC2 infrastructure isn't using any
     *         of the SQL server license features you're paying for, a license is considered not optimized.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>Optimized</code> — When Compute Optimizer detects that all specifications of your license meet the
     *         performance requirements of your workload.
     *         </p>
     *         </li>
     * @see LicenseFinding
     */
    public final String findingAsString() {
        return finding;
    }

    /**
     * <p>
     * The reason for the finding classification for an instance that runs on a license.
     * </p>
     * <p>
     * Finding reason codes include:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>Optimized</code> — All specifications of your license meet the performance requirements of your workload.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>LicenseOverprovisioned</code> — A license is considered over-provisioned when your license can be
     * downgraded while still meeting the performance requirements of your workload.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>InvalidCloudwatchApplicationInsights</code> — CloudWatch Application Insights isn't configured properly.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>CloudwatchApplicationInsightsError</code> — There is a CloudWatch Application Insights error.
     * </p>
     * </li>
     * </ul>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasFindingReasonCodes} method.
     * </p>
     * 
     * @return The reason for the finding classification for an instance that runs on a license. </p>
     *         <p>
     *         Finding reason codes include:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>Optimized</code> — All specifications of your license meet the performance requirements of your
     *         workload.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>LicenseOverprovisioned</code> — A license is considered over-provisioned when your license can be
     *         downgraded while still meeting the performance requirements of your workload.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>InvalidCloudwatchApplicationInsights</code> — CloudWatch Application Insights isn't configured
     *         properly.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>CloudwatchApplicationInsightsError</code> — There is a CloudWatch Application Insights error.
     *         </p>
     *         </li>
     */
    public final List<LicenseFindingReasonCode> findingReasonCodes() {
        return LicenseFindingReasonCodesCopier.copyStringToEnum(findingReasonCodes);
    }

    /**
     * For responses, this returns true if the service returned a value for the FindingReasonCodes property. This DOES
     * NOT check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasFindingReasonCodes() {
        return findingReasonCodes != null && !(findingReasonCodes instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The reason for the finding classification for an instance that runs on a license.
     * </p>
     * <p>
     * Finding reason codes include:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>Optimized</code> — All specifications of your license meet the performance requirements of your workload.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>LicenseOverprovisioned</code> — A license is considered over-provisioned when your license can be
     * downgraded while still meeting the performance requirements of your workload.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>InvalidCloudwatchApplicationInsights</code> — CloudWatch Application Insights isn't configured properly.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>CloudwatchApplicationInsightsError</code> — There is a CloudWatch Application Insights error.
     * </p>
     * </li>
     * </ul>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasFindingReasonCodes} method.
     * </p>
     * 
     * @return The reason for the finding classification for an instance that runs on a license. </p>
     *         <p>
     *         Finding reason codes include:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>Optimized</code> — All specifications of your license meet the performance requirements of your
     *         workload.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>LicenseOverprovisioned</code> — A license is considered over-provisioned when your license can be
     *         downgraded while still meeting the performance requirements of your workload.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>InvalidCloudwatchApplicationInsights</code> — CloudWatch Application Insights isn't configured
     *         properly.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>CloudwatchApplicationInsightsError</code> — There is a CloudWatch Application Insights error.
     *         </p>
     *         </li>
     */
    public final List<String> findingReasonCodesAsStrings() {
        return findingReasonCodes;
    }

    /**
     * For responses, this returns true if the service returned a value for the LicenseRecommendationOptions property.
     * This DOES NOT check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the
     * property). This is useful because the SDK will never return a null collection or map, but you may need to
     * differentiate between the service returning nothing (or null) and the service returning an empty collection or
     * map. For requests, this returns true if a value for the property was specified in the request builder, and false
     * if a value was not specified.
     */
    public final boolean hasLicenseRecommendationOptions() {
        return licenseRecommendationOptions != null && !(licenseRecommendationOptions instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * An array of objects that describe the license recommendation options.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasLicenseRecommendationOptions} method.
     * </p>
     * 
     * @return An array of objects that describe the license recommendation options.
     */
    public final List<LicenseRecommendationOption> licenseRecommendationOptions() {
        return licenseRecommendationOptions;
    }

    /**
     * For responses, this returns true if the service returned a value for the Tags property. This DOES NOT check that
     * the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is useful
     * because the SDK will never return a null collection or map, but you may need to differentiate between the service
     * returning nothing (or null) and the service returning an empty collection or map. For requests, this returns true
     * if a value for the property was specified in the request builder, and false if a value was not specified.
     */
    public final boolean hasTags() {
        return tags != null && !(tags instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * A list of tags assigned to an EC2 instance.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasTags} method.
     * </p>
     * 
     * @return A list of tags assigned to an EC2 instance.
     */
    public final List<Tag> tags() {
        return tags;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(resourceArn());
        hashCode = 31 * hashCode + Objects.hashCode(accountId());
        hashCode = 31 * hashCode + Objects.hashCode(currentLicenseConfiguration());
        hashCode = 31 * hashCode + Objects.hashCode(lookbackPeriodInDays());
        hashCode = 31 * hashCode + Objects.hashCode(lastRefreshTimestamp());
        hashCode = 31 * hashCode + Objects.hashCode(findingAsString());
        hashCode = 31 * hashCode + Objects.hashCode(hasFindingReasonCodes() ? findingReasonCodesAsStrings() : null);
        hashCode = 31 * hashCode + Objects.hashCode(hasLicenseRecommendationOptions() ? licenseRecommendationOptions() : null);
        hashCode = 31 * hashCode + Objects.hashCode(hasTags() ? tags() : null);
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof LicenseRecommendation)) {
            return false;
        }
        LicenseRecommendation other = (LicenseRecommendation) obj;
        return Objects.equals(resourceArn(), other.resourceArn()) && Objects.equals(accountId(), other.accountId())
                && Objects.equals(currentLicenseConfiguration(), other.currentLicenseConfiguration())
                && Objects.equals(lookbackPeriodInDays(), other.lookbackPeriodInDays())
                && Objects.equals(lastRefreshTimestamp(), other.lastRefreshTimestamp())
                && Objects.equals(findingAsString(), other.findingAsString())
                && hasFindingReasonCodes() == other.hasFindingReasonCodes()
                && Objects.equals(findingReasonCodesAsStrings(), other.findingReasonCodesAsStrings())
                && hasLicenseRecommendationOptions() == other.hasLicenseRecommendationOptions()
                && Objects.equals(licenseRecommendationOptions(), other.licenseRecommendationOptions())
                && hasTags() == other.hasTags() && Objects.equals(tags(), other.tags());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("LicenseRecommendation").add("ResourceArn", resourceArn()).add("AccountId", accountId())
                .add("CurrentLicenseConfiguration", currentLicenseConfiguration())
                .add("LookbackPeriodInDays", lookbackPeriodInDays()).add("LastRefreshTimestamp", lastRefreshTimestamp())
                .add("Finding", findingAsString())
                .add("FindingReasonCodes", hasFindingReasonCodes() ? findingReasonCodesAsStrings() : null)
                .add("LicenseRecommendationOptions", hasLicenseRecommendationOptions() ? licenseRecommendationOptions() : null)
                .add("Tags", hasTags() ? tags() : null).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "resourceArn":
            return Optional.ofNullable(clazz.cast(resourceArn()));
        case "accountId":
            return Optional.ofNullable(clazz.cast(accountId()));
        case "currentLicenseConfiguration":
            return Optional.ofNullable(clazz.cast(currentLicenseConfiguration()));
        case "lookbackPeriodInDays":
            return Optional.ofNullable(clazz.cast(lookbackPeriodInDays()));
        case "lastRefreshTimestamp":
            return Optional.ofNullable(clazz.cast(lastRefreshTimestamp()));
        case "finding":
            return Optional.ofNullable(clazz.cast(findingAsString()));
        case "findingReasonCodes":
            return Optional.ofNullable(clazz.cast(findingReasonCodesAsStrings()));
        case "licenseRecommendationOptions":
            return Optional.ofNullable(clazz.cast(licenseRecommendationOptions()));
        case "tags":
            return Optional.ofNullable(clazz.cast(tags()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static Map<String, SdkField<?>> memberNameToFieldInitializer() {
        Map<String, SdkField<?>> map = new HashMap<>();
        map.put("resourceArn", RESOURCE_ARN_FIELD);
        map.put("accountId", ACCOUNT_ID_FIELD);
        map.put("currentLicenseConfiguration", CURRENT_LICENSE_CONFIGURATION_FIELD);
        map.put("lookbackPeriodInDays", LOOKBACK_PERIOD_IN_DAYS_FIELD);
        map.put("lastRefreshTimestamp", LAST_REFRESH_TIMESTAMP_FIELD);
        map.put("finding", FINDING_FIELD);
        map.put("findingReasonCodes", FINDING_REASON_CODES_FIELD);
        map.put("licenseRecommendationOptions", LICENSE_RECOMMENDATION_OPTIONS_FIELD);
        map.put("tags", TAGS_FIELD);
        return Collections.unmodifiableMap(map);
    }

    private static <T> Function<Object, T> getter(Function<LicenseRecommendation, T> g) {
        return obj -> g.apply((LicenseRecommendation) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    @Mutable
    @NotThreadSafe
    public interface Builder extends SdkPojo, CopyableBuilder<Builder, LicenseRecommendation> {
        /**
         * <p>
         * The ARN that identifies the Amazon EC2 instance.
         * </p>
         * 
         * @param resourceArn
         *        The ARN that identifies the Amazon EC2 instance.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder resourceArn(String resourceArn);

        /**
         * <p>
         * The Amazon Web Services account ID of the license.
         * </p>
         * 
         * @param accountId
         *        The Amazon Web Services account ID of the license.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder accountId(String accountId);

        /**
         * <p>
         * An object that describes the current configuration of an instance that runs on a license.
         * </p>
         * 
         * @param currentLicenseConfiguration
         *        An object that describes the current configuration of an instance that runs on a license.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder currentLicenseConfiguration(LicenseConfiguration currentLicenseConfiguration);

        /**
         * <p>
         * An object that describes the current configuration of an instance that runs on a license.
         * </p>
         * This is a convenience method that creates an instance of the {@link LicenseConfiguration.Builder} avoiding
         * the need to create one manually via {@link LicenseConfiguration#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link LicenseConfiguration.Builder#build()} is called immediately and
         * its result is passed to {@link #currentLicenseConfiguration(LicenseConfiguration)}.
         * 
         * @param currentLicenseConfiguration
         *        a consumer that will call methods on {@link LicenseConfiguration.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #currentLicenseConfiguration(LicenseConfiguration)
         */
        default Builder currentLicenseConfiguration(Consumer<LicenseConfiguration.Builder> currentLicenseConfiguration) {
            return currentLicenseConfiguration(LicenseConfiguration.builder().applyMutation(currentLicenseConfiguration).build());
        }

        /**
         * <p>
         * The number of days for which utilization metrics were analyzed for an instance that runs on a license.
         * </p>
         * 
         * @param lookbackPeriodInDays
         *        The number of days for which utilization metrics were analyzed for an instance that runs on a license.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder lookbackPeriodInDays(Double lookbackPeriodInDays);

        /**
         * <p>
         * The timestamp of when the license recommendation was last generated.
         * </p>
         * 
         * @param lastRefreshTimestamp
         *        The timestamp of when the license recommendation was last generated.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder lastRefreshTimestamp(Instant lastRefreshTimestamp);

        /**
         * <p>
         * The finding classification for an instance that runs on a license.
         * </p>
         * <p>
         * Findings include:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>InsufficentMetrics</code> — When Compute Optimizer detects that your CloudWatch Application Insights
         * isn't enabled or is enabled with insufficient permissions.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>NotOptimized</code> — When Compute Optimizer detects that your EC2 infrastructure isn't using any of
         * the SQL server license features you're paying for, a license is considered not optimized.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>Optimized</code> — When Compute Optimizer detects that all specifications of your license meet the
         * performance requirements of your workload.
         * </p>
         * </li>
         * </ul>
         * 
         * @param finding
         *        The finding classification for an instance that runs on a license. </p>
         *        <p>
         *        Findings include:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>InsufficentMetrics</code> — When Compute Optimizer detects that your CloudWatch Application
         *        Insights isn't enabled or is enabled with insufficient permissions.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>NotOptimized</code> — When Compute Optimizer detects that your EC2 infrastructure isn't using
         *        any of the SQL server license features you're paying for, a license is considered not optimized.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>Optimized</code> — When Compute Optimizer detects that all specifications of your license meet
         *        the performance requirements of your workload.
         *        </p>
         *        </li>
         * @see LicenseFinding
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see LicenseFinding
         */
        Builder finding(String finding);

        /**
         * <p>
         * The finding classification for an instance that runs on a license.
         * </p>
         * <p>
         * Findings include:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>InsufficentMetrics</code> — When Compute Optimizer detects that your CloudWatch Application Insights
         * isn't enabled or is enabled with insufficient permissions.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>NotOptimized</code> — When Compute Optimizer detects that your EC2 infrastructure isn't using any of
         * the SQL server license features you're paying for, a license is considered not optimized.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>Optimized</code> — When Compute Optimizer detects that all specifications of your license meet the
         * performance requirements of your workload.
         * </p>
         * </li>
         * </ul>
         * 
         * @param finding
         *        The finding classification for an instance that runs on a license. </p>
         *        <p>
         *        Findings include:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>InsufficentMetrics</code> — When Compute Optimizer detects that your CloudWatch Application
         *        Insights isn't enabled or is enabled with insufficient permissions.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>NotOptimized</code> — When Compute Optimizer detects that your EC2 infrastructure isn't using
         *        any of the SQL server license features you're paying for, a license is considered not optimized.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>Optimized</code> — When Compute Optimizer detects that all specifications of your license meet
         *        the performance requirements of your workload.
         *        </p>
         *        </li>
         * @see LicenseFinding
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see LicenseFinding
         */
        Builder finding(LicenseFinding finding);

        /**
         * <p>
         * The reason for the finding classification for an instance that runs on a license.
         * </p>
         * <p>
         * Finding reason codes include:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>Optimized</code> — All specifications of your license meet the performance requirements of your
         * workload.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>LicenseOverprovisioned</code> — A license is considered over-provisioned when your license can be
         * downgraded while still meeting the performance requirements of your workload.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>InvalidCloudwatchApplicationInsights</code> — CloudWatch Application Insights isn't configured
         * properly.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>CloudwatchApplicationInsightsError</code> — There is a CloudWatch Application Insights error.
         * </p>
         * </li>
         * </ul>
         * 
         * @param findingReasonCodes
         *        The reason for the finding classification for an instance that runs on a license. </p>
         *        <p>
         *        Finding reason codes include:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>Optimized</code> — All specifications of your license meet the performance requirements of your
         *        workload.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>LicenseOverprovisioned</code> — A license is considered over-provisioned when your license can
         *        be downgraded while still meeting the performance requirements of your workload.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>InvalidCloudwatchApplicationInsights</code> — CloudWatch Application Insights isn't configured
         *        properly.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>CloudwatchApplicationInsightsError</code> — There is a CloudWatch Application Insights error.
         *        </p>
         *        </li>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder findingReasonCodesWithStrings(Collection<String> findingReasonCodes);

        /**
         * <p>
         * The reason for the finding classification for an instance that runs on a license.
         * </p>
         * <p>
         * Finding reason codes include:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>Optimized</code> — All specifications of your license meet the performance requirements of your
         * workload.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>LicenseOverprovisioned</code> — A license is considered over-provisioned when your license can be
         * downgraded while still meeting the performance requirements of your workload.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>InvalidCloudwatchApplicationInsights</code> — CloudWatch Application Insights isn't configured
         * properly.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>CloudwatchApplicationInsightsError</code> — There is a CloudWatch Application Insights error.
         * </p>
         * </li>
         * </ul>
         * 
         * @param findingReasonCodes
         *        The reason for the finding classification for an instance that runs on a license. </p>
         *        <p>
         *        Finding reason codes include:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>Optimized</code> — All specifications of your license meet the performance requirements of your
         *        workload.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>LicenseOverprovisioned</code> — A license is considered over-provisioned when your license can
         *        be downgraded while still meeting the performance requirements of your workload.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>InvalidCloudwatchApplicationInsights</code> — CloudWatch Application Insights isn't configured
         *        properly.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>CloudwatchApplicationInsightsError</code> — There is a CloudWatch Application Insights error.
         *        </p>
         *        </li>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder findingReasonCodesWithStrings(String... findingReasonCodes);

        /**
         * <p>
         * The reason for the finding classification for an instance that runs on a license.
         * </p>
         * <p>
         * Finding reason codes include:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>Optimized</code> — All specifications of your license meet the performance requirements of your
         * workload.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>LicenseOverprovisioned</code> — A license is considered over-provisioned when your license can be
         * downgraded while still meeting the performance requirements of your workload.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>InvalidCloudwatchApplicationInsights</code> — CloudWatch Application Insights isn't configured
         * properly.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>CloudwatchApplicationInsightsError</code> — There is a CloudWatch Application Insights error.
         * </p>
         * </li>
         * </ul>
         * 
         * @param findingReasonCodes
         *        The reason for the finding classification for an instance that runs on a license. </p>
         *        <p>
         *        Finding reason codes include:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>Optimized</code> — All specifications of your license meet the performance requirements of your
         *        workload.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>LicenseOverprovisioned</code> — A license is considered over-provisioned when your license can
         *        be downgraded while still meeting the performance requirements of your workload.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>InvalidCloudwatchApplicationInsights</code> — CloudWatch Application Insights isn't configured
         *        properly.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>CloudwatchApplicationInsightsError</code> — There is a CloudWatch Application Insights error.
         *        </p>
         *        </li>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder findingReasonCodes(Collection<LicenseFindingReasonCode> findingReasonCodes);

        /**
         * <p>
         * The reason for the finding classification for an instance that runs on a license.
         * </p>
         * <p>
         * Finding reason codes include:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>Optimized</code> — All specifications of your license meet the performance requirements of your
         * workload.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>LicenseOverprovisioned</code> — A license is considered over-provisioned when your license can be
         * downgraded while still meeting the performance requirements of your workload.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>InvalidCloudwatchApplicationInsights</code> — CloudWatch Application Insights isn't configured
         * properly.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>CloudwatchApplicationInsightsError</code> — There is a CloudWatch Application Insights error.
         * </p>
         * </li>
         * </ul>
         * 
         * @param findingReasonCodes
         *        The reason for the finding classification for an instance that runs on a license. </p>
         *        <p>
         *        Finding reason codes include:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>Optimized</code> — All specifications of your license meet the performance requirements of your
         *        workload.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>LicenseOverprovisioned</code> — A license is considered over-provisioned when your license can
         *        be downgraded while still meeting the performance requirements of your workload.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>InvalidCloudwatchApplicationInsights</code> — CloudWatch Application Insights isn't configured
         *        properly.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>CloudwatchApplicationInsightsError</code> — There is a CloudWatch Application Insights error.
         *        </p>
         *        </li>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder findingReasonCodes(LicenseFindingReasonCode... findingReasonCodes);

        /**
         * <p>
         * An array of objects that describe the license recommendation options.
         * </p>
         * 
         * @param licenseRecommendationOptions
         *        An array of objects that describe the license recommendation options.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder licenseRecommendationOptions(Collection<LicenseRecommendationOption> licenseRecommendationOptions);

        /**
         * <p>
         * An array of objects that describe the license recommendation options.
         * </p>
         * 
         * @param licenseRecommendationOptions
         *        An array of objects that describe the license recommendation options.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder licenseRecommendationOptions(LicenseRecommendationOption... licenseRecommendationOptions);

        /**
         * <p>
         * An array of objects that describe the license recommendation options.
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link software.amazon.awssdk.services.computeoptimizer.model.LicenseRecommendationOption.Builder} avoiding
         * the need to create one manually via
         * {@link software.amazon.awssdk.services.computeoptimizer.model.LicenseRecommendationOption#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes,
         * {@link software.amazon.awssdk.services.computeoptimizer.model.LicenseRecommendationOption.Builder#build()} is
         * called immediately and its result is passed to {@link
         * #licenseRecommendationOptions(List<LicenseRecommendationOption>)}.
         * 
         * @param licenseRecommendationOptions
         *        a consumer that will call methods on
         *        {@link software.amazon.awssdk.services.computeoptimizer.model.LicenseRecommendationOption.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #licenseRecommendationOptions(java.util.Collection<LicenseRecommendationOption>)
         */
        Builder licenseRecommendationOptions(Consumer<LicenseRecommendationOption.Builder>... licenseRecommendationOptions);

        /**
         * <p>
         * A list of tags assigned to an EC2 instance.
         * </p>
         * 
         * @param tags
         *        A list of tags assigned to an EC2 instance.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder tags(Collection<Tag> tags);

        /**
         * <p>
         * A list of tags assigned to an EC2 instance.
         * </p>
         * 
         * @param tags
         *        A list of tags assigned to an EC2 instance.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder tags(Tag... tags);

        /**
         * <p>
         * A list of tags assigned to an EC2 instance.
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link software.amazon.awssdk.services.computeoptimizer.model.Tag.Builder} avoiding the need to create one
         * manually via {@link software.amazon.awssdk.services.computeoptimizer.model.Tag#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes,
         * {@link software.amazon.awssdk.services.computeoptimizer.model.Tag.Builder#build()} is called immediately and
         * its result is passed to {@link #tags(List<Tag>)}.
         * 
         * @param tags
         *        a consumer that will call methods on
         *        {@link software.amazon.awssdk.services.computeoptimizer.model.Tag.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #tags(java.util.Collection<Tag>)
         */
        Builder tags(Consumer<Tag.Builder>... tags);
    }

    static final class BuilderImpl implements Builder {
        private String resourceArn;

        private String accountId;

        private LicenseConfiguration currentLicenseConfiguration;

        private Double lookbackPeriodInDays;

        private Instant lastRefreshTimestamp;

        private String finding;

        private List<String> findingReasonCodes = DefaultSdkAutoConstructList.getInstance();

        private List<LicenseRecommendationOption> licenseRecommendationOptions = DefaultSdkAutoConstructList.getInstance();

        private List<Tag> tags = DefaultSdkAutoConstructList.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(LicenseRecommendation model) {
            resourceArn(model.resourceArn);
            accountId(model.accountId);
            currentLicenseConfiguration(model.currentLicenseConfiguration);
            lookbackPeriodInDays(model.lookbackPeriodInDays);
            lastRefreshTimestamp(model.lastRefreshTimestamp);
            finding(model.finding);
            findingReasonCodesWithStrings(model.findingReasonCodes);
            licenseRecommendationOptions(model.licenseRecommendationOptions);
            tags(model.tags);
        }

        public final String getResourceArn() {
            return resourceArn;
        }

        public final void setResourceArn(String resourceArn) {
            this.resourceArn = resourceArn;
        }

        @Override
        public final Builder resourceArn(String resourceArn) {
            this.resourceArn = resourceArn;
            return this;
        }

        public final String getAccountId() {
            return accountId;
        }

        public final void setAccountId(String accountId) {
            this.accountId = accountId;
        }

        @Override
        public final Builder accountId(String accountId) {
            this.accountId = accountId;
            return this;
        }

        public final LicenseConfiguration.Builder getCurrentLicenseConfiguration() {
            return currentLicenseConfiguration != null ? currentLicenseConfiguration.toBuilder() : null;
        }

        public final void setCurrentLicenseConfiguration(LicenseConfiguration.BuilderImpl currentLicenseConfiguration) {
            this.currentLicenseConfiguration = currentLicenseConfiguration != null ? currentLicenseConfiguration.build() : null;
        }

        @Override
        public final Builder currentLicenseConfiguration(LicenseConfiguration currentLicenseConfiguration) {
            this.currentLicenseConfiguration = currentLicenseConfiguration;
            return this;
        }

        public final Double getLookbackPeriodInDays() {
            return lookbackPeriodInDays;
        }

        public final void setLookbackPeriodInDays(Double lookbackPeriodInDays) {
            this.lookbackPeriodInDays = lookbackPeriodInDays;
        }

        @Override
        public final Builder lookbackPeriodInDays(Double lookbackPeriodInDays) {
            this.lookbackPeriodInDays = lookbackPeriodInDays;
            return this;
        }

        public final Instant getLastRefreshTimestamp() {
            return lastRefreshTimestamp;
        }

        public final void setLastRefreshTimestamp(Instant lastRefreshTimestamp) {
            this.lastRefreshTimestamp = lastRefreshTimestamp;
        }

        @Override
        public final Builder lastRefreshTimestamp(Instant lastRefreshTimestamp) {
            this.lastRefreshTimestamp = lastRefreshTimestamp;
            return this;
        }

        public final String getFinding() {
            return finding;
        }

        public final void setFinding(String finding) {
            this.finding = finding;
        }

        @Override
        public final Builder finding(String finding) {
            this.finding = finding;
            return this;
        }

        @Override
        public final Builder finding(LicenseFinding finding) {
            this.finding(finding == null ? null : finding.toString());
            return this;
        }

        public final Collection<String> getFindingReasonCodes() {
            if (findingReasonCodes instanceof SdkAutoConstructList) {
                return null;
            }
            return findingReasonCodes;
        }

        public final void setFindingReasonCodes(Collection<String> findingReasonCodes) {
            this.findingReasonCodes = LicenseFindingReasonCodesCopier.copy(findingReasonCodes);
        }

        @Override
        public final Builder findingReasonCodesWithStrings(Collection<String> findingReasonCodes) {
            this.findingReasonCodes = LicenseFindingReasonCodesCopier.copy(findingReasonCodes);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder findingReasonCodesWithStrings(String... findingReasonCodes) {
            findingReasonCodesWithStrings(Arrays.asList(findingReasonCodes));
            return this;
        }

        @Override
        public final Builder findingReasonCodes(Collection<LicenseFindingReasonCode> findingReasonCodes) {
            this.findingReasonCodes = LicenseFindingReasonCodesCopier.copyEnumToString(findingReasonCodes);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder findingReasonCodes(LicenseFindingReasonCode... findingReasonCodes) {
            findingReasonCodes(Arrays.asList(findingReasonCodes));
            return this;
        }

        public final List<LicenseRecommendationOption.Builder> getLicenseRecommendationOptions() {
            List<LicenseRecommendationOption.Builder> result = LicenseRecommendationOptionsCopier
                    .copyToBuilder(this.licenseRecommendationOptions);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setLicenseRecommendationOptions(
                Collection<LicenseRecommendationOption.BuilderImpl> licenseRecommendationOptions) {
            this.licenseRecommendationOptions = LicenseRecommendationOptionsCopier.copyFromBuilder(licenseRecommendationOptions);
        }

        @Override
        public final Builder licenseRecommendationOptions(Collection<LicenseRecommendationOption> licenseRecommendationOptions) {
            this.licenseRecommendationOptions = LicenseRecommendationOptionsCopier.copy(licenseRecommendationOptions);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder licenseRecommendationOptions(LicenseRecommendationOption... licenseRecommendationOptions) {
            licenseRecommendationOptions(Arrays.asList(licenseRecommendationOptions));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder licenseRecommendationOptions(
                Consumer<LicenseRecommendationOption.Builder>... licenseRecommendationOptions) {
            licenseRecommendationOptions(Stream.of(licenseRecommendationOptions)
                    .map(c -> LicenseRecommendationOption.builder().applyMutation(c).build()).collect(Collectors.toList()));
            return this;
        }

        public final List<Tag.Builder> getTags() {
            List<Tag.Builder> result = TagsCopier.copyToBuilder(this.tags);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setTags(Collection<Tag.BuilderImpl> tags) {
            this.tags = TagsCopier.copyFromBuilder(tags);
        }

        @Override
        public final Builder tags(Collection<Tag> tags) {
            this.tags = TagsCopier.copy(tags);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder tags(Tag... tags) {
            tags(Arrays.asList(tags));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder tags(Consumer<Tag.Builder>... tags) {
            tags(Stream.of(tags).map(c -> Tag.builder().applyMutation(c).build()).collect(Collectors.toList()));
            return this;
        }

        @Override
        public LicenseRecommendation build() {
            return new LicenseRecommendation(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
