/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.comprehend.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.Mutable;
import software.amazon.awssdk.annotations.NotThreadSafe;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Information about one mention of an entity. The mention information includes the location of the mention in the text
 * and the sentiment of the mention.
 * </p>
 * <p>
 * For more information about targeted sentiment, see <a
 * href="https://docs.aws.amazon.com/comprehend/latest/dg/how-targeted-sentiment.html">Targeted sentiment</a> in the
 * <i>Amazon Comprehend Developer Guide</i>.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class TargetedSentimentMention implements SdkPojo, Serializable,
        ToCopyableBuilder<TargetedSentimentMention.Builder, TargetedSentimentMention> {
    private static final SdkField<Float> SCORE_FIELD = SdkField.<Float> builder(MarshallingType.FLOAT).memberName("Score")
            .getter(getter(TargetedSentimentMention::score)).setter(setter(Builder::score))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Score").build()).build();

    private static final SdkField<Float> GROUP_SCORE_FIELD = SdkField.<Float> builder(MarshallingType.FLOAT)
            .memberName("GroupScore").getter(getter(TargetedSentimentMention::groupScore)).setter(setter(Builder::groupScore))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("GroupScore").build()).build();

    private static final SdkField<String> TEXT_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Text")
            .getter(getter(TargetedSentimentMention::text)).setter(setter(Builder::text))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Text").build()).build();

    private static final SdkField<String> TYPE_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Type")
            .getter(getter(TargetedSentimentMention::typeAsString)).setter(setter(Builder::type))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Type").build()).build();

    private static final SdkField<MentionSentiment> MENTION_SENTIMENT_FIELD = SdkField
            .<MentionSentiment> builder(MarshallingType.SDK_POJO).memberName("MentionSentiment")
            .getter(getter(TargetedSentimentMention::mentionSentiment)).setter(setter(Builder::mentionSentiment))
            .constructor(MentionSentiment::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("MentionSentiment").build()).build();

    private static final SdkField<Integer> BEGIN_OFFSET_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER)
            .memberName("BeginOffset").getter(getter(TargetedSentimentMention::beginOffset)).setter(setter(Builder::beginOffset))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("BeginOffset").build()).build();

    private static final SdkField<Integer> END_OFFSET_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER)
            .memberName("EndOffset").getter(getter(TargetedSentimentMention::endOffset)).setter(setter(Builder::endOffset))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("EndOffset").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(SCORE_FIELD,
            GROUP_SCORE_FIELD, TEXT_FIELD, TYPE_FIELD, MENTION_SENTIMENT_FIELD, BEGIN_OFFSET_FIELD, END_OFFSET_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = memberNameToFieldInitializer();

    private static final long serialVersionUID = 1L;

    private final Float score;

    private final Float groupScore;

    private final String text;

    private final String type;

    private final MentionSentiment mentionSentiment;

    private final Integer beginOffset;

    private final Integer endOffset;

    private TargetedSentimentMention(BuilderImpl builder) {
        this.score = builder.score;
        this.groupScore = builder.groupScore;
        this.text = builder.text;
        this.type = builder.type;
        this.mentionSentiment = builder.mentionSentiment;
        this.beginOffset = builder.beginOffset;
        this.endOffset = builder.endOffset;
    }

    /**
     * <p>
     * Model confidence that the entity is relevant. Value range is zero to one, where one is highest confidence.
     * </p>
     * 
     * @return Model confidence that the entity is relevant. Value range is zero to one, where one is highest
     *         confidence.
     */
    public final Float score() {
        return score;
    }

    /**
     * <p>
     * The confidence that all the entities mentioned in the group relate to the same entity.
     * </p>
     * 
     * @return The confidence that all the entities mentioned in the group relate to the same entity.
     */
    public final Float groupScore() {
        return groupScore;
    }

    /**
     * <p>
     * The text in the document that identifies the entity.
     * </p>
     * 
     * @return The text in the document that identifies the entity.
     */
    public final String text() {
        return text;
    }

    /**
     * <p>
     * The type of the entity. Amazon Comprehend supports a variety of <a href=
     * "https://docs.aws.amazon.com/comprehend/latest/dg/how-targeted-sentiment.html#how-targeted-sentiment-entities"
     * >entity types</a>.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #type} will return
     * {@link TargetedSentimentEntityType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #typeAsString}.
     * </p>
     * 
     * @return The type of the entity. Amazon Comprehend supports a variety of <a href=
     *         "https://docs.aws.amazon.com/comprehend/latest/dg/how-targeted-sentiment.html#how-targeted-sentiment-entities"
     *         >entity types</a>.
     * @see TargetedSentimentEntityType
     */
    public final TargetedSentimentEntityType type() {
        return TargetedSentimentEntityType.fromValue(type);
    }

    /**
     * <p>
     * The type of the entity. Amazon Comprehend supports a variety of <a href=
     * "https://docs.aws.amazon.com/comprehend/latest/dg/how-targeted-sentiment.html#how-targeted-sentiment-entities"
     * >entity types</a>.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #type} will return
     * {@link TargetedSentimentEntityType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #typeAsString}.
     * </p>
     * 
     * @return The type of the entity. Amazon Comprehend supports a variety of <a href=
     *         "https://docs.aws.amazon.com/comprehend/latest/dg/how-targeted-sentiment.html#how-targeted-sentiment-entities"
     *         >entity types</a>.
     * @see TargetedSentimentEntityType
     */
    public final String typeAsString() {
        return type;
    }

    /**
     * <p>
     * Contains the sentiment and sentiment score for the mention.
     * </p>
     * 
     * @return Contains the sentiment and sentiment score for the mention.
     */
    public final MentionSentiment mentionSentiment() {
        return mentionSentiment;
    }

    /**
     * <p>
     * The offset into the document text where the mention begins.
     * </p>
     * 
     * @return The offset into the document text where the mention begins.
     */
    public final Integer beginOffset() {
        return beginOffset;
    }

    /**
     * <p>
     * The offset into the document text where the mention ends.
     * </p>
     * 
     * @return The offset into the document text where the mention ends.
     */
    public final Integer endOffset() {
        return endOffset;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(score());
        hashCode = 31 * hashCode + Objects.hashCode(groupScore());
        hashCode = 31 * hashCode + Objects.hashCode(text());
        hashCode = 31 * hashCode + Objects.hashCode(typeAsString());
        hashCode = 31 * hashCode + Objects.hashCode(mentionSentiment());
        hashCode = 31 * hashCode + Objects.hashCode(beginOffset());
        hashCode = 31 * hashCode + Objects.hashCode(endOffset());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof TargetedSentimentMention)) {
            return false;
        }
        TargetedSentimentMention other = (TargetedSentimentMention) obj;
        return Objects.equals(score(), other.score()) && Objects.equals(groupScore(), other.groupScore())
                && Objects.equals(text(), other.text()) && Objects.equals(typeAsString(), other.typeAsString())
                && Objects.equals(mentionSentiment(), other.mentionSentiment())
                && Objects.equals(beginOffset(), other.beginOffset()) && Objects.equals(endOffset(), other.endOffset());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("TargetedSentimentMention").add("Score", score()).add("GroupScore", groupScore())
                .add("Text", text()).add("Type", typeAsString()).add("MentionSentiment", mentionSentiment())
                .add("BeginOffset", beginOffset()).add("EndOffset", endOffset()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "Score":
            return Optional.ofNullable(clazz.cast(score()));
        case "GroupScore":
            return Optional.ofNullable(clazz.cast(groupScore()));
        case "Text":
            return Optional.ofNullable(clazz.cast(text()));
        case "Type":
            return Optional.ofNullable(clazz.cast(typeAsString()));
        case "MentionSentiment":
            return Optional.ofNullable(clazz.cast(mentionSentiment()));
        case "BeginOffset":
            return Optional.ofNullable(clazz.cast(beginOffset()));
        case "EndOffset":
            return Optional.ofNullable(clazz.cast(endOffset()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static Map<String, SdkField<?>> memberNameToFieldInitializer() {
        Map<String, SdkField<?>> map = new HashMap<>();
        map.put("Score", SCORE_FIELD);
        map.put("GroupScore", GROUP_SCORE_FIELD);
        map.put("Text", TEXT_FIELD);
        map.put("Type", TYPE_FIELD);
        map.put("MentionSentiment", MENTION_SENTIMENT_FIELD);
        map.put("BeginOffset", BEGIN_OFFSET_FIELD);
        map.put("EndOffset", END_OFFSET_FIELD);
        return Collections.unmodifiableMap(map);
    }

    private static <T> Function<Object, T> getter(Function<TargetedSentimentMention, T> g) {
        return obj -> g.apply((TargetedSentimentMention) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    @Mutable
    @NotThreadSafe
    public interface Builder extends SdkPojo, CopyableBuilder<Builder, TargetedSentimentMention> {
        /**
         * <p>
         * Model confidence that the entity is relevant. Value range is zero to one, where one is highest confidence.
         * </p>
         * 
         * @param score
         *        Model confidence that the entity is relevant. Value range is zero to one, where one is highest
         *        confidence.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder score(Float score);

        /**
         * <p>
         * The confidence that all the entities mentioned in the group relate to the same entity.
         * </p>
         * 
         * @param groupScore
         *        The confidence that all the entities mentioned in the group relate to the same entity.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder groupScore(Float groupScore);

        /**
         * <p>
         * The text in the document that identifies the entity.
         * </p>
         * 
         * @param text
         *        The text in the document that identifies the entity.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder text(String text);

        /**
         * <p>
         * The type of the entity. Amazon Comprehend supports a variety of <a href=
         * "https://docs.aws.amazon.com/comprehend/latest/dg/how-targeted-sentiment.html#how-targeted-sentiment-entities"
         * >entity types</a>.
         * </p>
         * 
         * @param type
         *        The type of the entity. Amazon Comprehend supports a variety of <a href=
         *        "https://docs.aws.amazon.com/comprehend/latest/dg/how-targeted-sentiment.html#how-targeted-sentiment-entities"
         *        >entity types</a>.
         * @see TargetedSentimentEntityType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see TargetedSentimentEntityType
         */
        Builder type(String type);

        /**
         * <p>
         * The type of the entity. Amazon Comprehend supports a variety of <a href=
         * "https://docs.aws.amazon.com/comprehend/latest/dg/how-targeted-sentiment.html#how-targeted-sentiment-entities"
         * >entity types</a>.
         * </p>
         * 
         * @param type
         *        The type of the entity. Amazon Comprehend supports a variety of <a href=
         *        "https://docs.aws.amazon.com/comprehend/latest/dg/how-targeted-sentiment.html#how-targeted-sentiment-entities"
         *        >entity types</a>.
         * @see TargetedSentimentEntityType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see TargetedSentimentEntityType
         */
        Builder type(TargetedSentimentEntityType type);

        /**
         * <p>
         * Contains the sentiment and sentiment score for the mention.
         * </p>
         * 
         * @param mentionSentiment
         *        Contains the sentiment and sentiment score for the mention.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder mentionSentiment(MentionSentiment mentionSentiment);

        /**
         * <p>
         * Contains the sentiment and sentiment score for the mention.
         * </p>
         * This is a convenience method that creates an instance of the {@link MentionSentiment.Builder} avoiding the
         * need to create one manually via {@link MentionSentiment#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link MentionSentiment.Builder#build()} is called immediately and its
         * result is passed to {@link #mentionSentiment(MentionSentiment)}.
         * 
         * @param mentionSentiment
         *        a consumer that will call methods on {@link MentionSentiment.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #mentionSentiment(MentionSentiment)
         */
        default Builder mentionSentiment(Consumer<MentionSentiment.Builder> mentionSentiment) {
            return mentionSentiment(MentionSentiment.builder().applyMutation(mentionSentiment).build());
        }

        /**
         * <p>
         * The offset into the document text where the mention begins.
         * </p>
         * 
         * @param beginOffset
         *        The offset into the document text where the mention begins.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder beginOffset(Integer beginOffset);

        /**
         * <p>
         * The offset into the document text where the mention ends.
         * </p>
         * 
         * @param endOffset
         *        The offset into the document text where the mention ends.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder endOffset(Integer endOffset);
    }

    static final class BuilderImpl implements Builder {
        private Float score;

        private Float groupScore;

        private String text;

        private String type;

        private MentionSentiment mentionSentiment;

        private Integer beginOffset;

        private Integer endOffset;

        private BuilderImpl() {
        }

        private BuilderImpl(TargetedSentimentMention model) {
            score(model.score);
            groupScore(model.groupScore);
            text(model.text);
            type(model.type);
            mentionSentiment(model.mentionSentiment);
            beginOffset(model.beginOffset);
            endOffset(model.endOffset);
        }

        public final Float getScore() {
            return score;
        }

        public final void setScore(Float score) {
            this.score = score;
        }

        @Override
        public final Builder score(Float score) {
            this.score = score;
            return this;
        }

        public final Float getGroupScore() {
            return groupScore;
        }

        public final void setGroupScore(Float groupScore) {
            this.groupScore = groupScore;
        }

        @Override
        public final Builder groupScore(Float groupScore) {
            this.groupScore = groupScore;
            return this;
        }

        public final String getText() {
            return text;
        }

        public final void setText(String text) {
            this.text = text;
        }

        @Override
        public final Builder text(String text) {
            this.text = text;
            return this;
        }

        public final String getType() {
            return type;
        }

        public final void setType(String type) {
            this.type = type;
        }

        @Override
        public final Builder type(String type) {
            this.type = type;
            return this;
        }

        @Override
        public final Builder type(TargetedSentimentEntityType type) {
            this.type(type == null ? null : type.toString());
            return this;
        }

        public final MentionSentiment.Builder getMentionSentiment() {
            return mentionSentiment != null ? mentionSentiment.toBuilder() : null;
        }

        public final void setMentionSentiment(MentionSentiment.BuilderImpl mentionSentiment) {
            this.mentionSentiment = mentionSentiment != null ? mentionSentiment.build() : null;
        }

        @Override
        public final Builder mentionSentiment(MentionSentiment mentionSentiment) {
            this.mentionSentiment = mentionSentiment;
            return this;
        }

        public final Integer getBeginOffset() {
            return beginOffset;
        }

        public final void setBeginOffset(Integer beginOffset) {
            this.beginOffset = beginOffset;
        }

        @Override
        public final Builder beginOffset(Integer beginOffset) {
            this.beginOffset = beginOffset;
            return this;
        }

        public final Integer getEndOffset() {
            return endOffset;
        }

        public final void setEndOffset(Integer endOffset) {
            this.endOffset = endOffset;
        }

        @Override
        public final Builder endOffset(Integer endOffset) {
            this.endOffset = endOffset;
            return this;
        }

        @Override
        public TargetedSentimentMention build() {
            return new TargetedSentimentMention(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
