/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.comprehend.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Provides configuration parameters to override the default actions for extracting text from PDF documents and image
 * files.
 * </p>
 * <p>
 * By default, Amazon Comprehend performs the following actions to extract text from files, based on the input file
 * type:
 * </p>
 * <ul>
 * <li>
 * <p>
 * <b>Word files</b> - Amazon Comprehend parser extracts the text.
 * </p>
 * </li>
 * <li>
 * <p>
 * <b>Digital PDF files</b> - Amazon Comprehend parser extracts the text.
 * </p>
 * </li>
 * <li>
 * <p>
 * <b>Image files and scanned PDF files</b> - Amazon Comprehend uses the Amazon Textract <code>DetectDocumentText</code>
 * API to extract the text.
 * </p>
 * </li>
 * </ul>
 * <p>
 * <code>DocumentReaderConfig</code> does not apply to plain text files or Word files.
 * </p>
 * <p>
 * For image files and PDF documents, you can override these default actions using the fields listed below. For more
 * information, see <a href="https://docs.aws.amazon.com/comprehend/latest/dg/idp-set-textract-options.html"> Setting
 * text extraction options</a> in the Comprehend Developer Guide.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class DocumentReaderConfig implements SdkPojo, Serializable,
        ToCopyableBuilder<DocumentReaderConfig.Builder, DocumentReaderConfig> {
    private static final SdkField<String> DOCUMENT_READ_ACTION_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("DocumentReadAction").getter(getter(DocumentReaderConfig::documentReadActionAsString))
            .setter(setter(Builder::documentReadAction))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("DocumentReadAction").build())
            .build();

    private static final SdkField<String> DOCUMENT_READ_MODE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("DocumentReadMode").getter(getter(DocumentReaderConfig::documentReadModeAsString))
            .setter(setter(Builder::documentReadMode))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("DocumentReadMode").build()).build();

    private static final SdkField<List<String>> FEATURE_TYPES_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("FeatureTypes")
            .getter(getter(DocumentReaderConfig::featureTypesAsStrings))
            .setter(setter(Builder::featureTypesWithStrings))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("FeatureTypes").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(DOCUMENT_READ_ACTION_FIELD,
            DOCUMENT_READ_MODE_FIELD, FEATURE_TYPES_FIELD));

    private static final long serialVersionUID = 1L;

    private final String documentReadAction;

    private final String documentReadMode;

    private final List<String> featureTypes;

    private DocumentReaderConfig(BuilderImpl builder) {
        this.documentReadAction = builder.documentReadAction;
        this.documentReadMode = builder.documentReadMode;
        this.featureTypes = builder.featureTypes;
    }

    /**
     * <p>
     * This field defines the Amazon Textract API operation that Amazon Comprehend uses to extract text from PDF files
     * and image files. Enter one of the following values:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>TEXTRACT_DETECT_DOCUMENT_TEXT</code> - The Amazon Comprehend service uses the
     * <code>DetectDocumentText</code> API operation.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>TEXTRACT_ANALYZE_DOCUMENT</code> - The Amazon Comprehend service uses the <code>AnalyzeDocument</code> API
     * operation.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version,
     * {@link #documentReadAction} will return {@link DocumentReadAction#UNKNOWN_TO_SDK_VERSION}. The raw value returned
     * by the service is available from {@link #documentReadActionAsString}.
     * </p>
     * 
     * @return This field defines the Amazon Textract API operation that Amazon Comprehend uses to extract text from PDF
     *         files and image files. Enter one of the following values:</p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>TEXTRACT_DETECT_DOCUMENT_TEXT</code> - The Amazon Comprehend service uses the
     *         <code>DetectDocumentText</code> API operation.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>TEXTRACT_ANALYZE_DOCUMENT</code> - The Amazon Comprehend service uses the
     *         <code>AnalyzeDocument</code> API operation.
     *         </p>
     *         </li>
     * @see DocumentReadAction
     */
    public final DocumentReadAction documentReadAction() {
        return DocumentReadAction.fromValue(documentReadAction);
    }

    /**
     * <p>
     * This field defines the Amazon Textract API operation that Amazon Comprehend uses to extract text from PDF files
     * and image files. Enter one of the following values:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>TEXTRACT_DETECT_DOCUMENT_TEXT</code> - The Amazon Comprehend service uses the
     * <code>DetectDocumentText</code> API operation.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>TEXTRACT_ANALYZE_DOCUMENT</code> - The Amazon Comprehend service uses the <code>AnalyzeDocument</code> API
     * operation.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version,
     * {@link #documentReadAction} will return {@link DocumentReadAction#UNKNOWN_TO_SDK_VERSION}. The raw value returned
     * by the service is available from {@link #documentReadActionAsString}.
     * </p>
     * 
     * @return This field defines the Amazon Textract API operation that Amazon Comprehend uses to extract text from PDF
     *         files and image files. Enter one of the following values:</p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>TEXTRACT_DETECT_DOCUMENT_TEXT</code> - The Amazon Comprehend service uses the
     *         <code>DetectDocumentText</code> API operation.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>TEXTRACT_ANALYZE_DOCUMENT</code> - The Amazon Comprehend service uses the
     *         <code>AnalyzeDocument</code> API operation.
     *         </p>
     *         </li>
     * @see DocumentReadAction
     */
    public final String documentReadActionAsString() {
        return documentReadAction;
    }

    /**
     * <p>
     * Determines the text extraction actions for PDF files. Enter one of the following values:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>SERVICE_DEFAULT</code> - use the Amazon Comprehend service defaults for PDF files.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>FORCE_DOCUMENT_READ_ACTION</code> - Amazon Comprehend uses the Textract API specified by DocumentReadAction
     * for all PDF files, including digital PDF files.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #documentReadMode}
     * will return {@link DocumentReadMode#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #documentReadModeAsString}.
     * </p>
     * 
     * @return Determines the text extraction actions for PDF files. Enter one of the following values:</p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>SERVICE_DEFAULT</code> - use the Amazon Comprehend service defaults for PDF files.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>FORCE_DOCUMENT_READ_ACTION</code> - Amazon Comprehend uses the Textract API specified by
     *         DocumentReadAction for all PDF files, including digital PDF files.
     *         </p>
     *         </li>
     * @see DocumentReadMode
     */
    public final DocumentReadMode documentReadMode() {
        return DocumentReadMode.fromValue(documentReadMode);
    }

    /**
     * <p>
     * Determines the text extraction actions for PDF files. Enter one of the following values:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>SERVICE_DEFAULT</code> - use the Amazon Comprehend service defaults for PDF files.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>FORCE_DOCUMENT_READ_ACTION</code> - Amazon Comprehend uses the Textract API specified by DocumentReadAction
     * for all PDF files, including digital PDF files.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #documentReadMode}
     * will return {@link DocumentReadMode#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #documentReadModeAsString}.
     * </p>
     * 
     * @return Determines the text extraction actions for PDF files. Enter one of the following values:</p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>SERVICE_DEFAULT</code> - use the Amazon Comprehend service defaults for PDF files.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>FORCE_DOCUMENT_READ_ACTION</code> - Amazon Comprehend uses the Textract API specified by
     *         DocumentReadAction for all PDF files, including digital PDF files.
     *         </p>
     *         </li>
     * @see DocumentReadMode
     */
    public final String documentReadModeAsString() {
        return documentReadMode;
    }

    /**
     * <p>
     * Specifies the type of Amazon Textract features to apply. If you chose <code>TEXTRACT_ANALYZE_DOCUMENT</code> as
     * the read action, you must specify one or both of the following values:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>TABLES</code> - Returns information about any tables that are detected in the input document.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>FORMS</code> - Returns information and the data from any forms that are detected in the input document.
     * </p>
     * </li>
     * </ul>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasFeatureTypes} method.
     * </p>
     * 
     * @return Specifies the type of Amazon Textract features to apply. If you chose
     *         <code>TEXTRACT_ANALYZE_DOCUMENT</code> as the read action, you must specify one or both of the following
     *         values:</p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>TABLES</code> - Returns information about any tables that are detected in the input document.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>FORMS</code> - Returns information and the data from any forms that are detected in the input
     *         document.
     *         </p>
     *         </li>
     */
    public final List<DocumentReadFeatureTypes> featureTypes() {
        return ListOfDocumentReadFeatureTypesCopier.copyStringToEnum(featureTypes);
    }

    /**
     * For responses, this returns true if the service returned a value for the FeatureTypes property. This DOES NOT
     * check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasFeatureTypes() {
        return featureTypes != null && !(featureTypes instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * Specifies the type of Amazon Textract features to apply. If you chose <code>TEXTRACT_ANALYZE_DOCUMENT</code> as
     * the read action, you must specify one or both of the following values:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>TABLES</code> - Returns information about any tables that are detected in the input document.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>FORMS</code> - Returns information and the data from any forms that are detected in the input document.
     * </p>
     * </li>
     * </ul>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasFeatureTypes} method.
     * </p>
     * 
     * @return Specifies the type of Amazon Textract features to apply. If you chose
     *         <code>TEXTRACT_ANALYZE_DOCUMENT</code> as the read action, you must specify one or both of the following
     *         values:</p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>TABLES</code> - Returns information about any tables that are detected in the input document.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>FORMS</code> - Returns information and the data from any forms that are detected in the input
     *         document.
     *         </p>
     *         </li>
     */
    public final List<String> featureTypesAsStrings() {
        return featureTypes;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(documentReadActionAsString());
        hashCode = 31 * hashCode + Objects.hashCode(documentReadModeAsString());
        hashCode = 31 * hashCode + Objects.hashCode(hasFeatureTypes() ? featureTypesAsStrings() : null);
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof DocumentReaderConfig)) {
            return false;
        }
        DocumentReaderConfig other = (DocumentReaderConfig) obj;
        return Objects.equals(documentReadActionAsString(), other.documentReadActionAsString())
                && Objects.equals(documentReadModeAsString(), other.documentReadModeAsString())
                && hasFeatureTypes() == other.hasFeatureTypes()
                && Objects.equals(featureTypesAsStrings(), other.featureTypesAsStrings());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("DocumentReaderConfig").add("DocumentReadAction", documentReadActionAsString())
                .add("DocumentReadMode", documentReadModeAsString())
                .add("FeatureTypes", hasFeatureTypes() ? featureTypesAsStrings() : null).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "DocumentReadAction":
            return Optional.ofNullable(clazz.cast(documentReadActionAsString()));
        case "DocumentReadMode":
            return Optional.ofNullable(clazz.cast(documentReadModeAsString()));
        case "FeatureTypes":
            return Optional.ofNullable(clazz.cast(featureTypesAsStrings()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<DocumentReaderConfig, T> g) {
        return obj -> g.apply((DocumentReaderConfig) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, DocumentReaderConfig> {
        /**
         * <p>
         * This field defines the Amazon Textract API operation that Amazon Comprehend uses to extract text from PDF
         * files and image files. Enter one of the following values:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>TEXTRACT_DETECT_DOCUMENT_TEXT</code> - The Amazon Comprehend service uses the
         * <code>DetectDocumentText</code> API operation.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>TEXTRACT_ANALYZE_DOCUMENT</code> - The Amazon Comprehend service uses the <code>AnalyzeDocument</code>
         * API operation.
         * </p>
         * </li>
         * </ul>
         * 
         * @param documentReadAction
         *        This field defines the Amazon Textract API operation that Amazon Comprehend uses to extract text from
         *        PDF files and image files. Enter one of the following values:</p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>TEXTRACT_DETECT_DOCUMENT_TEXT</code> - The Amazon Comprehend service uses the
         *        <code>DetectDocumentText</code> API operation.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>TEXTRACT_ANALYZE_DOCUMENT</code> - The Amazon Comprehend service uses the
         *        <code>AnalyzeDocument</code> API operation.
         *        </p>
         *        </li>
         * @see DocumentReadAction
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see DocumentReadAction
         */
        Builder documentReadAction(String documentReadAction);

        /**
         * <p>
         * This field defines the Amazon Textract API operation that Amazon Comprehend uses to extract text from PDF
         * files and image files. Enter one of the following values:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>TEXTRACT_DETECT_DOCUMENT_TEXT</code> - The Amazon Comprehend service uses the
         * <code>DetectDocumentText</code> API operation.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>TEXTRACT_ANALYZE_DOCUMENT</code> - The Amazon Comprehend service uses the <code>AnalyzeDocument</code>
         * API operation.
         * </p>
         * </li>
         * </ul>
         * 
         * @param documentReadAction
         *        This field defines the Amazon Textract API operation that Amazon Comprehend uses to extract text from
         *        PDF files and image files. Enter one of the following values:</p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>TEXTRACT_DETECT_DOCUMENT_TEXT</code> - The Amazon Comprehend service uses the
         *        <code>DetectDocumentText</code> API operation.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>TEXTRACT_ANALYZE_DOCUMENT</code> - The Amazon Comprehend service uses the
         *        <code>AnalyzeDocument</code> API operation.
         *        </p>
         *        </li>
         * @see DocumentReadAction
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see DocumentReadAction
         */
        Builder documentReadAction(DocumentReadAction documentReadAction);

        /**
         * <p>
         * Determines the text extraction actions for PDF files. Enter one of the following values:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>SERVICE_DEFAULT</code> - use the Amazon Comprehend service defaults for PDF files.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>FORCE_DOCUMENT_READ_ACTION</code> - Amazon Comprehend uses the Textract API specified by
         * DocumentReadAction for all PDF files, including digital PDF files.
         * </p>
         * </li>
         * </ul>
         * 
         * @param documentReadMode
         *        Determines the text extraction actions for PDF files. Enter one of the following values:</p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>SERVICE_DEFAULT</code> - use the Amazon Comprehend service defaults for PDF files.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>FORCE_DOCUMENT_READ_ACTION</code> - Amazon Comprehend uses the Textract API specified by
         *        DocumentReadAction for all PDF files, including digital PDF files.
         *        </p>
         *        </li>
         * @see DocumentReadMode
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see DocumentReadMode
         */
        Builder documentReadMode(String documentReadMode);

        /**
         * <p>
         * Determines the text extraction actions for PDF files. Enter one of the following values:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>SERVICE_DEFAULT</code> - use the Amazon Comprehend service defaults for PDF files.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>FORCE_DOCUMENT_READ_ACTION</code> - Amazon Comprehend uses the Textract API specified by
         * DocumentReadAction for all PDF files, including digital PDF files.
         * </p>
         * </li>
         * </ul>
         * 
         * @param documentReadMode
         *        Determines the text extraction actions for PDF files. Enter one of the following values:</p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>SERVICE_DEFAULT</code> - use the Amazon Comprehend service defaults for PDF files.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>FORCE_DOCUMENT_READ_ACTION</code> - Amazon Comprehend uses the Textract API specified by
         *        DocumentReadAction for all PDF files, including digital PDF files.
         *        </p>
         *        </li>
         * @see DocumentReadMode
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see DocumentReadMode
         */
        Builder documentReadMode(DocumentReadMode documentReadMode);

        /**
         * <p>
         * Specifies the type of Amazon Textract features to apply. If you chose <code>TEXTRACT_ANALYZE_DOCUMENT</code>
         * as the read action, you must specify one or both of the following values:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>TABLES</code> - Returns information about any tables that are detected in the input document.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>FORMS</code> - Returns information and the data from any forms that are detected in the input document.
         * </p>
         * </li>
         * </ul>
         * 
         * @param featureTypes
         *        Specifies the type of Amazon Textract features to apply. If you chose
         *        <code>TEXTRACT_ANALYZE_DOCUMENT</code> as the read action, you must specify one or both of the
         *        following values:</p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>TABLES</code> - Returns information about any tables that are detected in the input document.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>FORMS</code> - Returns information and the data from any forms that are detected in the input
         *        document.
         *        </p>
         *        </li>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder featureTypesWithStrings(Collection<String> featureTypes);

        /**
         * <p>
         * Specifies the type of Amazon Textract features to apply. If you chose <code>TEXTRACT_ANALYZE_DOCUMENT</code>
         * as the read action, you must specify one or both of the following values:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>TABLES</code> - Returns information about any tables that are detected in the input document.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>FORMS</code> - Returns information and the data from any forms that are detected in the input document.
         * </p>
         * </li>
         * </ul>
         * 
         * @param featureTypes
         *        Specifies the type of Amazon Textract features to apply. If you chose
         *        <code>TEXTRACT_ANALYZE_DOCUMENT</code> as the read action, you must specify one or both of the
         *        following values:</p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>TABLES</code> - Returns information about any tables that are detected in the input document.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>FORMS</code> - Returns information and the data from any forms that are detected in the input
         *        document.
         *        </p>
         *        </li>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder featureTypesWithStrings(String... featureTypes);

        /**
         * <p>
         * Specifies the type of Amazon Textract features to apply. If you chose <code>TEXTRACT_ANALYZE_DOCUMENT</code>
         * as the read action, you must specify one or both of the following values:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>TABLES</code> - Returns information about any tables that are detected in the input document.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>FORMS</code> - Returns information and the data from any forms that are detected in the input document.
         * </p>
         * </li>
         * </ul>
         * 
         * @param featureTypes
         *        Specifies the type of Amazon Textract features to apply. If you chose
         *        <code>TEXTRACT_ANALYZE_DOCUMENT</code> as the read action, you must specify one or both of the
         *        following values:</p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>TABLES</code> - Returns information about any tables that are detected in the input document.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>FORMS</code> - Returns information and the data from any forms that are detected in the input
         *        document.
         *        </p>
         *        </li>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder featureTypes(Collection<DocumentReadFeatureTypes> featureTypes);

        /**
         * <p>
         * Specifies the type of Amazon Textract features to apply. If you chose <code>TEXTRACT_ANALYZE_DOCUMENT</code>
         * as the read action, you must specify one or both of the following values:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>TABLES</code> - Returns information about any tables that are detected in the input document.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>FORMS</code> - Returns information and the data from any forms that are detected in the input document.
         * </p>
         * </li>
         * </ul>
         * 
         * @param featureTypes
         *        Specifies the type of Amazon Textract features to apply. If you chose
         *        <code>TEXTRACT_ANALYZE_DOCUMENT</code> as the read action, you must specify one or both of the
         *        following values:</p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>TABLES</code> - Returns information about any tables that are detected in the input document.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>FORMS</code> - Returns information and the data from any forms that are detected in the input
         *        document.
         *        </p>
         *        </li>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder featureTypes(DocumentReadFeatureTypes... featureTypes);
    }

    static final class BuilderImpl implements Builder {
        private String documentReadAction;

        private String documentReadMode;

        private List<String> featureTypes = DefaultSdkAutoConstructList.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(DocumentReaderConfig model) {
            documentReadAction(model.documentReadAction);
            documentReadMode(model.documentReadMode);
            featureTypesWithStrings(model.featureTypes);
        }

        public final String getDocumentReadAction() {
            return documentReadAction;
        }

        public final void setDocumentReadAction(String documentReadAction) {
            this.documentReadAction = documentReadAction;
        }

        @Override
        public final Builder documentReadAction(String documentReadAction) {
            this.documentReadAction = documentReadAction;
            return this;
        }

        @Override
        public final Builder documentReadAction(DocumentReadAction documentReadAction) {
            this.documentReadAction(documentReadAction == null ? null : documentReadAction.toString());
            return this;
        }

        public final String getDocumentReadMode() {
            return documentReadMode;
        }

        public final void setDocumentReadMode(String documentReadMode) {
            this.documentReadMode = documentReadMode;
        }

        @Override
        public final Builder documentReadMode(String documentReadMode) {
            this.documentReadMode = documentReadMode;
            return this;
        }

        @Override
        public final Builder documentReadMode(DocumentReadMode documentReadMode) {
            this.documentReadMode(documentReadMode == null ? null : documentReadMode.toString());
            return this;
        }

        public final Collection<String> getFeatureTypes() {
            if (featureTypes instanceof SdkAutoConstructList) {
                return null;
            }
            return featureTypes;
        }

        public final void setFeatureTypes(Collection<String> featureTypes) {
            this.featureTypes = ListOfDocumentReadFeatureTypesCopier.copy(featureTypes);
        }

        @Override
        public final Builder featureTypesWithStrings(Collection<String> featureTypes) {
            this.featureTypes = ListOfDocumentReadFeatureTypesCopier.copy(featureTypes);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder featureTypesWithStrings(String... featureTypes) {
            featureTypesWithStrings(Arrays.asList(featureTypes));
            return this;
        }

        @Override
        public final Builder featureTypes(Collection<DocumentReadFeatureTypes> featureTypes) {
            this.featureTypes = ListOfDocumentReadFeatureTypesCopier.copyEnumToString(featureTypes);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder featureTypes(DocumentReadFeatureTypes... featureTypes) {
            featureTypes(Arrays.asList(featureTypes));
            return this;
        }

        @Override
        public DocumentReaderConfig build() {
            return new DocumentReaderConfig(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
