/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.comprehend.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Information about each word or line of text in the input document.
 * </p>
 * <p>
 * For additional information, see <a href="https://docs.aws.amazon.com/textract/latest/dg/API_Block.html">Block</a> in
 * the Amazon Textract API reference.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class Block implements SdkPojo, Serializable, ToCopyableBuilder<Block.Builder, Block> {
    private static final SdkField<String> ID_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Id")
            .getter(getter(Block::id)).setter(setter(Builder::id))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Id").build()).build();

    private static final SdkField<String> BLOCK_TYPE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("BlockType").getter(getter(Block::blockTypeAsString)).setter(setter(Builder::blockType))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("BlockType").build()).build();

    private static final SdkField<String> TEXT_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Text")
            .getter(getter(Block::text)).setter(setter(Builder::text))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Text").build()).build();

    private static final SdkField<Integer> PAGE_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER).memberName("Page")
            .getter(getter(Block::page)).setter(setter(Builder::page))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Page").build()).build();

    private static final SdkField<Geometry> GEOMETRY_FIELD = SdkField.<Geometry> builder(MarshallingType.SDK_POJO)
            .memberName("Geometry").getter(getter(Block::geometry)).setter(setter(Builder::geometry))
            .constructor(Geometry::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Geometry").build()).build();

    private static final SdkField<List<RelationshipsListItem>> RELATIONSHIPS_FIELD = SdkField
            .<List<RelationshipsListItem>> builder(MarshallingType.LIST)
            .memberName("Relationships")
            .getter(getter(Block::relationships))
            .setter(setter(Builder::relationships))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Relationships").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<RelationshipsListItem> builder(MarshallingType.SDK_POJO)
                                            .constructor(RelationshipsListItem::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(ID_FIELD, BLOCK_TYPE_FIELD,
            TEXT_FIELD, PAGE_FIELD, GEOMETRY_FIELD, RELATIONSHIPS_FIELD));

    private static final long serialVersionUID = 1L;

    private final String id;

    private final String blockType;

    private final String text;

    private final Integer page;

    private final Geometry geometry;

    private final List<RelationshipsListItem> relationships;

    private Block(BuilderImpl builder) {
        this.id = builder.id;
        this.blockType = builder.blockType;
        this.text = builder.text;
        this.page = builder.page;
        this.geometry = builder.geometry;
        this.relationships = builder.relationships;
    }

    /**
     * <p>
     * Unique identifier for the block.
     * </p>
     * 
     * @return Unique identifier for the block.
     */
    public final String id() {
        return id;
    }

    /**
     * <p>
     * The block represents a line of text or one word of text.
     * </p>
     * <ul>
     * <li>
     * <p>
     * WORD - A word that's detected on a document page. A word is one or more ISO basic Latin script characters that
     * aren't separated by spaces.
     * </p>
     * </li>
     * <li>
     * <p>
     * LINE - A string of tab-delimited, contiguous words that are detected on a document page
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #blockType} will
     * return {@link BlockType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #blockTypeAsString}.
     * </p>
     * 
     * @return The block represents a line of text or one word of text.</p>
     *         <ul>
     *         <li>
     *         <p>
     *         WORD - A word that's detected on a document page. A word is one or more ISO basic Latin script characters
     *         that aren't separated by spaces.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         LINE - A string of tab-delimited, contiguous words that are detected on a document page
     *         </p>
     *         </li>
     * @see BlockType
     */
    public final BlockType blockType() {
        return BlockType.fromValue(blockType);
    }

    /**
     * <p>
     * The block represents a line of text or one word of text.
     * </p>
     * <ul>
     * <li>
     * <p>
     * WORD - A word that's detected on a document page. A word is one or more ISO basic Latin script characters that
     * aren't separated by spaces.
     * </p>
     * </li>
     * <li>
     * <p>
     * LINE - A string of tab-delimited, contiguous words that are detected on a document page
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #blockType} will
     * return {@link BlockType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #blockTypeAsString}.
     * </p>
     * 
     * @return The block represents a line of text or one word of text.</p>
     *         <ul>
     *         <li>
     *         <p>
     *         WORD - A word that's detected on a document page. A word is one or more ISO basic Latin script characters
     *         that aren't separated by spaces.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         LINE - A string of tab-delimited, contiguous words that are detected on a document page
     *         </p>
     *         </li>
     * @see BlockType
     */
    public final String blockTypeAsString() {
        return blockType;
    }

    /**
     * <p>
     * The word or line of text extracted from the block.
     * </p>
     * 
     * @return The word or line of text extracted from the block.
     */
    public final String text() {
        return text;
    }

    /**
     * <p>
     * Page number where the block appears.
     * </p>
     * 
     * @return Page number where the block appears.
     */
    public final Integer page() {
        return page;
    }

    /**
     * <p>
     * Co-ordinates of the rectangle or polygon that contains the text.
     * </p>
     * 
     * @return Co-ordinates of the rectangle or polygon that contains the text.
     */
    public final Geometry geometry() {
        return geometry;
    }

    /**
     * For responses, this returns true if the service returned a value for the Relationships property. This DOES NOT
     * check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasRelationships() {
        return relationships != null && !(relationships instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * A list of child blocks of the current block. For example, a LINE object has child blocks for each WORD block
     * that's part of the line of text.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasRelationships} method.
     * </p>
     * 
     * @return A list of child blocks of the current block. For example, a LINE object has child blocks for each WORD
     *         block that's part of the line of text.
     */
    public final List<RelationshipsListItem> relationships() {
        return relationships;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(id());
        hashCode = 31 * hashCode + Objects.hashCode(blockTypeAsString());
        hashCode = 31 * hashCode + Objects.hashCode(text());
        hashCode = 31 * hashCode + Objects.hashCode(page());
        hashCode = 31 * hashCode + Objects.hashCode(geometry());
        hashCode = 31 * hashCode + Objects.hashCode(hasRelationships() ? relationships() : null);
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof Block)) {
            return false;
        }
        Block other = (Block) obj;
        return Objects.equals(id(), other.id()) && Objects.equals(blockTypeAsString(), other.blockTypeAsString())
                && Objects.equals(text(), other.text()) && Objects.equals(page(), other.page())
                && Objects.equals(geometry(), other.geometry()) && hasRelationships() == other.hasRelationships()
                && Objects.equals(relationships(), other.relationships());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("Block").add("Id", id()).add("BlockType", blockTypeAsString()).add("Text", text())
                .add("Page", page()).add("Geometry", geometry())
                .add("Relationships", hasRelationships() ? relationships() : null).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "Id":
            return Optional.ofNullable(clazz.cast(id()));
        case "BlockType":
            return Optional.ofNullable(clazz.cast(blockTypeAsString()));
        case "Text":
            return Optional.ofNullable(clazz.cast(text()));
        case "Page":
            return Optional.ofNullable(clazz.cast(page()));
        case "Geometry":
            return Optional.ofNullable(clazz.cast(geometry()));
        case "Relationships":
            return Optional.ofNullable(clazz.cast(relationships()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<Block, T> g) {
        return obj -> g.apply((Block) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, Block> {
        /**
         * <p>
         * Unique identifier for the block.
         * </p>
         * 
         * @param id
         *        Unique identifier for the block.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder id(String id);

        /**
         * <p>
         * The block represents a line of text or one word of text.
         * </p>
         * <ul>
         * <li>
         * <p>
         * WORD - A word that's detected on a document page. A word is one or more ISO basic Latin script characters
         * that aren't separated by spaces.
         * </p>
         * </li>
         * <li>
         * <p>
         * LINE - A string of tab-delimited, contiguous words that are detected on a document page
         * </p>
         * </li>
         * </ul>
         * 
         * @param blockType
         *        The block represents a line of text or one word of text.</p>
         *        <ul>
         *        <li>
         *        <p>
         *        WORD - A word that's detected on a document page. A word is one or more ISO basic Latin script
         *        characters that aren't separated by spaces.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        LINE - A string of tab-delimited, contiguous words that are detected on a document page
         *        </p>
         *        </li>
         * @see BlockType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see BlockType
         */
        Builder blockType(String blockType);

        /**
         * <p>
         * The block represents a line of text or one word of text.
         * </p>
         * <ul>
         * <li>
         * <p>
         * WORD - A word that's detected on a document page. A word is one or more ISO basic Latin script characters
         * that aren't separated by spaces.
         * </p>
         * </li>
         * <li>
         * <p>
         * LINE - A string of tab-delimited, contiguous words that are detected on a document page
         * </p>
         * </li>
         * </ul>
         * 
         * @param blockType
         *        The block represents a line of text or one word of text.</p>
         *        <ul>
         *        <li>
         *        <p>
         *        WORD - A word that's detected on a document page. A word is one or more ISO basic Latin script
         *        characters that aren't separated by spaces.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        LINE - A string of tab-delimited, contiguous words that are detected on a document page
         *        </p>
         *        </li>
         * @see BlockType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see BlockType
         */
        Builder blockType(BlockType blockType);

        /**
         * <p>
         * The word or line of text extracted from the block.
         * </p>
         * 
         * @param text
         *        The word or line of text extracted from the block.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder text(String text);

        /**
         * <p>
         * Page number where the block appears.
         * </p>
         * 
         * @param page
         *        Page number where the block appears.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder page(Integer page);

        /**
         * <p>
         * Co-ordinates of the rectangle or polygon that contains the text.
         * </p>
         * 
         * @param geometry
         *        Co-ordinates of the rectangle or polygon that contains the text.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder geometry(Geometry geometry);

        /**
         * <p>
         * Co-ordinates of the rectangle or polygon that contains the text.
         * </p>
         * This is a convenience method that creates an instance of the {@link Geometry.Builder} avoiding the need to
         * create one manually via {@link Geometry#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link Geometry.Builder#build()} is called immediately and its result is
         * passed to {@link #geometry(Geometry)}.
         * 
         * @param geometry
         *        a consumer that will call methods on {@link Geometry.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #geometry(Geometry)
         */
        default Builder geometry(Consumer<Geometry.Builder> geometry) {
            return geometry(Geometry.builder().applyMutation(geometry).build());
        }

        /**
         * <p>
         * A list of child blocks of the current block. For example, a LINE object has child blocks for each WORD block
         * that's part of the line of text.
         * </p>
         * 
         * @param relationships
         *        A list of child blocks of the current block. For example, a LINE object has child blocks for each WORD
         *        block that's part of the line of text.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder relationships(Collection<RelationshipsListItem> relationships);

        /**
         * <p>
         * A list of child blocks of the current block. For example, a LINE object has child blocks for each WORD block
         * that's part of the line of text.
         * </p>
         * 
         * @param relationships
         *        A list of child blocks of the current block. For example, a LINE object has child blocks for each WORD
         *        block that's part of the line of text.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder relationships(RelationshipsListItem... relationships);

        /**
         * <p>
         * A list of child blocks of the current block. For example, a LINE object has child blocks for each WORD block
         * that's part of the line of text.
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link software.amazon.awssdk.services.comprehend.model.RelationshipsListItem.Builder} avoiding the need to
         * create one manually via
         * {@link software.amazon.awssdk.services.comprehend.model.RelationshipsListItem#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes,
         * {@link software.amazon.awssdk.services.comprehend.model.RelationshipsListItem.Builder#build()} is called
         * immediately and its result is passed to {@link #relationships(List<RelationshipsListItem>)}.
         * 
         * @param relationships
         *        a consumer that will call methods on
         *        {@link software.amazon.awssdk.services.comprehend.model.RelationshipsListItem.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #relationships(java.util.Collection<RelationshipsListItem>)
         */
        Builder relationships(Consumer<RelationshipsListItem.Builder>... relationships);
    }

    static final class BuilderImpl implements Builder {
        private String id;

        private String blockType;

        private String text;

        private Integer page;

        private Geometry geometry;

        private List<RelationshipsListItem> relationships = DefaultSdkAutoConstructList.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(Block model) {
            id(model.id);
            blockType(model.blockType);
            text(model.text);
            page(model.page);
            geometry(model.geometry);
            relationships(model.relationships);
        }

        public final String getId() {
            return id;
        }

        public final void setId(String id) {
            this.id = id;
        }

        @Override
        public final Builder id(String id) {
            this.id = id;
            return this;
        }

        public final String getBlockType() {
            return blockType;
        }

        public final void setBlockType(String blockType) {
            this.blockType = blockType;
        }

        @Override
        public final Builder blockType(String blockType) {
            this.blockType = blockType;
            return this;
        }

        @Override
        public final Builder blockType(BlockType blockType) {
            this.blockType(blockType == null ? null : blockType.toString());
            return this;
        }

        public final String getText() {
            return text;
        }

        public final void setText(String text) {
            this.text = text;
        }

        @Override
        public final Builder text(String text) {
            this.text = text;
            return this;
        }

        public final Integer getPage() {
            return page;
        }

        public final void setPage(Integer page) {
            this.page = page;
        }

        @Override
        public final Builder page(Integer page) {
            this.page = page;
            return this;
        }

        public final Geometry.Builder getGeometry() {
            return geometry != null ? geometry.toBuilder() : null;
        }

        public final void setGeometry(Geometry.BuilderImpl geometry) {
            this.geometry = geometry != null ? geometry.build() : null;
        }

        @Override
        public final Builder geometry(Geometry geometry) {
            this.geometry = geometry;
            return this;
        }

        public final List<RelationshipsListItem.Builder> getRelationships() {
            List<RelationshipsListItem.Builder> result = ListOfRelationshipsCopier.copyToBuilder(this.relationships);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setRelationships(Collection<RelationshipsListItem.BuilderImpl> relationships) {
            this.relationships = ListOfRelationshipsCopier.copyFromBuilder(relationships);
        }

        @Override
        public final Builder relationships(Collection<RelationshipsListItem> relationships) {
            this.relationships = ListOfRelationshipsCopier.copy(relationships);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder relationships(RelationshipsListItem... relationships) {
            relationships(Arrays.asList(relationships));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder relationships(Consumer<RelationshipsListItem.Builder>... relationships) {
            relationships(Stream.of(relationships).map(c -> RelationshipsListItem.builder().applyMutation(c).build())
                    .collect(Collectors.toList()));
            return this;
        }

        @Override
        public Block build() {
            return new Block(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
