/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.comprehend.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * An augmented manifest file that provides training data for your custom model. An augmented manifest file is a labeled
 * dataset that is produced by Amazon SageMaker Ground Truth.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class AugmentedManifestsListItem implements SdkPojo, Serializable,
        ToCopyableBuilder<AugmentedManifestsListItem.Builder, AugmentedManifestsListItem> {
    private static final SdkField<String> S3_URI_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("S3Uri")
            .getter(getter(AugmentedManifestsListItem::s3Uri)).setter(setter(Builder::s3Uri))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("S3Uri").build()).build();

    private static final SdkField<List<String>> ATTRIBUTE_NAMES_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("AttributeNames")
            .getter(getter(AugmentedManifestsListItem::attributeNames))
            .setter(setter(Builder::attributeNames))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("AttributeNames").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(S3_URI_FIELD,
            ATTRIBUTE_NAMES_FIELD));

    private static final long serialVersionUID = 1L;

    private final String s3Uri;

    private final List<String> attributeNames;

    private AugmentedManifestsListItem(BuilderImpl builder) {
        this.s3Uri = builder.s3Uri;
        this.attributeNames = builder.attributeNames;
    }

    /**
     * <p>
     * The Amazon S3 location of the augmented manifest file.
     * </p>
     * 
     * @return The Amazon S3 location of the augmented manifest file.
     */
    public final String s3Uri() {
        return s3Uri;
    }

    /**
     * Returns true if the AttributeNames property was specified by the sender (it may be empty), or false if the sender
     * did not specify the value (it will be empty). For responses returned by the SDK, the sender is the AWS service.
     */
    public final boolean hasAttributeNames() {
        return attributeNames != null && !(attributeNames instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The JSON attribute that contains the annotations for your training documents. The number of attribute names that
     * you specify depends on whether your augmented manifest file is the output of a single labeling job or a chained
     * labeling job.
     * </p>
     * <p>
     * If your file is the output of a single labeling job, specify the LabelAttributeName key that was used when the
     * job was created in Ground Truth.
     * </p>
     * <p>
     * If your file is the output of a chained labeling job, specify the LabelAttributeName key for one or more jobs in
     * the chain. Each LabelAttributeName key provides the annotations from an individual job.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasAttributeNames()} to see if a value was sent in this field.
     * </p>
     * 
     * @return The JSON attribute that contains the annotations for your training documents. The number of attribute
     *         names that you specify depends on whether your augmented manifest file is the output of a single labeling
     *         job or a chained labeling job.</p>
     *         <p>
     *         If your file is the output of a single labeling job, specify the LabelAttributeName key that was used
     *         when the job was created in Ground Truth.
     *         </p>
     *         <p>
     *         If your file is the output of a chained labeling job, specify the LabelAttributeName key for one or more
     *         jobs in the chain. Each LabelAttributeName key provides the annotations from an individual job.
     */
    public final List<String> attributeNames() {
        return attributeNames;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(s3Uri());
        hashCode = 31 * hashCode + Objects.hashCode(hasAttributeNames() ? attributeNames() : null);
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof AugmentedManifestsListItem)) {
            return false;
        }
        AugmentedManifestsListItem other = (AugmentedManifestsListItem) obj;
        return Objects.equals(s3Uri(), other.s3Uri()) && hasAttributeNames() == other.hasAttributeNames()
                && Objects.equals(attributeNames(), other.attributeNames());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("AugmentedManifestsListItem").add("S3Uri", s3Uri())
                .add("AttributeNames", hasAttributeNames() ? attributeNames() : null).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "S3Uri":
            return Optional.ofNullable(clazz.cast(s3Uri()));
        case "AttributeNames":
            return Optional.ofNullable(clazz.cast(attributeNames()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<AugmentedManifestsListItem, T> g) {
        return obj -> g.apply((AugmentedManifestsListItem) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, AugmentedManifestsListItem> {
        /**
         * <p>
         * The Amazon S3 location of the augmented manifest file.
         * </p>
         * 
         * @param s3Uri
         *        The Amazon S3 location of the augmented manifest file.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder s3Uri(String s3Uri);

        /**
         * <p>
         * The JSON attribute that contains the annotations for your training documents. The number of attribute names
         * that you specify depends on whether your augmented manifest file is the output of a single labeling job or a
         * chained labeling job.
         * </p>
         * <p>
         * If your file is the output of a single labeling job, specify the LabelAttributeName key that was used when
         * the job was created in Ground Truth.
         * </p>
         * <p>
         * If your file is the output of a chained labeling job, specify the LabelAttributeName key for one or more jobs
         * in the chain. Each LabelAttributeName key provides the annotations from an individual job.
         * </p>
         * 
         * @param attributeNames
         *        The JSON attribute that contains the annotations for your training documents. The number of attribute
         *        names that you specify depends on whether your augmented manifest file is the output of a single
         *        labeling job or a chained labeling job.</p>
         *        <p>
         *        If your file is the output of a single labeling job, specify the LabelAttributeName key that was used
         *        when the job was created in Ground Truth.
         *        </p>
         *        <p>
         *        If your file is the output of a chained labeling job, specify the LabelAttributeName key for one or
         *        more jobs in the chain. Each LabelAttributeName key provides the annotations from an individual job.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder attributeNames(Collection<String> attributeNames);

        /**
         * <p>
         * The JSON attribute that contains the annotations for your training documents. The number of attribute names
         * that you specify depends on whether your augmented manifest file is the output of a single labeling job or a
         * chained labeling job.
         * </p>
         * <p>
         * If your file is the output of a single labeling job, specify the LabelAttributeName key that was used when
         * the job was created in Ground Truth.
         * </p>
         * <p>
         * If your file is the output of a chained labeling job, specify the LabelAttributeName key for one or more jobs
         * in the chain. Each LabelAttributeName key provides the annotations from an individual job.
         * </p>
         * 
         * @param attributeNames
         *        The JSON attribute that contains the annotations for your training documents. The number of attribute
         *        names that you specify depends on whether your augmented manifest file is the output of a single
         *        labeling job or a chained labeling job.</p>
         *        <p>
         *        If your file is the output of a single labeling job, specify the LabelAttributeName key that was used
         *        when the job was created in Ground Truth.
         *        </p>
         *        <p>
         *        If your file is the output of a chained labeling job, specify the LabelAttributeName key for one or
         *        more jobs in the chain. Each LabelAttributeName key provides the annotations from an individual job.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder attributeNames(String... attributeNames);
    }

    static final class BuilderImpl implements Builder {
        private String s3Uri;

        private List<String> attributeNames = DefaultSdkAutoConstructList.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(AugmentedManifestsListItem model) {
            s3Uri(model.s3Uri);
            attributeNames(model.attributeNames);
        }

        public final String getS3Uri() {
            return s3Uri;
        }

        @Override
        public final Builder s3Uri(String s3Uri) {
            this.s3Uri = s3Uri;
            return this;
        }

        public final void setS3Uri(String s3Uri) {
            this.s3Uri = s3Uri;
        }

        public final Collection<String> getAttributeNames() {
            if (attributeNames instanceof SdkAutoConstructList) {
                return null;
            }
            return attributeNames;
        }

        @Override
        public final Builder attributeNames(Collection<String> attributeNames) {
            this.attributeNames = AttributeNamesListCopier.copy(attributeNames);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder attributeNames(String... attributeNames) {
            attributeNames(Arrays.asList(attributeNames));
            return this;
        }

        public final void setAttributeNames(Collection<String> attributeNames) {
            this.attributeNames = AttributeNamesListCopier.copy(attributeNames);
        }

        @Override
        public AugmentedManifestsListItem build() {
            return new AugmentedManifestsListItem(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
