/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.cognitosync;

import java.util.Collections;
import java.util.List;
import java.util.concurrent.CompletableFuture;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.AwsRequestOverrideConfiguration;
import software.amazon.awssdk.awscore.client.handler.AwsAsyncClientHandler;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.core.RequestOverrideConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientOption;
import software.amazon.awssdk.core.client.handler.AsyncClientHandler;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.core.metrics.CoreMetric;
import software.amazon.awssdk.metrics.MetricCollector;
import software.amazon.awssdk.metrics.MetricPublisher;
import software.amazon.awssdk.metrics.NoOpMetricCollector;
import software.amazon.awssdk.protocols.core.ExceptionMetadata;
import software.amazon.awssdk.protocols.json.AwsJsonProtocol;
import software.amazon.awssdk.protocols.json.AwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.BaseAwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.JsonOperationMetadata;
import software.amazon.awssdk.services.cognitosync.model.AlreadyStreamedException;
import software.amazon.awssdk.services.cognitosync.model.BulkPublishRequest;
import software.amazon.awssdk.services.cognitosync.model.BulkPublishResponse;
import software.amazon.awssdk.services.cognitosync.model.CognitoSyncException;
import software.amazon.awssdk.services.cognitosync.model.ConcurrentModificationException;
import software.amazon.awssdk.services.cognitosync.model.DeleteDatasetRequest;
import software.amazon.awssdk.services.cognitosync.model.DeleteDatasetResponse;
import software.amazon.awssdk.services.cognitosync.model.DescribeDatasetRequest;
import software.amazon.awssdk.services.cognitosync.model.DescribeDatasetResponse;
import software.amazon.awssdk.services.cognitosync.model.DescribeIdentityPoolUsageRequest;
import software.amazon.awssdk.services.cognitosync.model.DescribeIdentityPoolUsageResponse;
import software.amazon.awssdk.services.cognitosync.model.DescribeIdentityUsageRequest;
import software.amazon.awssdk.services.cognitosync.model.DescribeIdentityUsageResponse;
import software.amazon.awssdk.services.cognitosync.model.DuplicateRequestException;
import software.amazon.awssdk.services.cognitosync.model.GetBulkPublishDetailsRequest;
import software.amazon.awssdk.services.cognitosync.model.GetBulkPublishDetailsResponse;
import software.amazon.awssdk.services.cognitosync.model.GetCognitoEventsRequest;
import software.amazon.awssdk.services.cognitosync.model.GetCognitoEventsResponse;
import software.amazon.awssdk.services.cognitosync.model.GetIdentityPoolConfigurationRequest;
import software.amazon.awssdk.services.cognitosync.model.GetIdentityPoolConfigurationResponse;
import software.amazon.awssdk.services.cognitosync.model.InternalErrorException;
import software.amazon.awssdk.services.cognitosync.model.InvalidConfigurationException;
import software.amazon.awssdk.services.cognitosync.model.InvalidLambdaFunctionOutputException;
import software.amazon.awssdk.services.cognitosync.model.InvalidParameterException;
import software.amazon.awssdk.services.cognitosync.model.LambdaThrottledException;
import software.amazon.awssdk.services.cognitosync.model.LimitExceededException;
import software.amazon.awssdk.services.cognitosync.model.ListDatasetsRequest;
import software.amazon.awssdk.services.cognitosync.model.ListDatasetsResponse;
import software.amazon.awssdk.services.cognitosync.model.ListIdentityPoolUsageRequest;
import software.amazon.awssdk.services.cognitosync.model.ListIdentityPoolUsageResponse;
import software.amazon.awssdk.services.cognitosync.model.ListRecordsRequest;
import software.amazon.awssdk.services.cognitosync.model.ListRecordsResponse;
import software.amazon.awssdk.services.cognitosync.model.NotAuthorizedException;
import software.amazon.awssdk.services.cognitosync.model.RegisterDeviceRequest;
import software.amazon.awssdk.services.cognitosync.model.RegisterDeviceResponse;
import software.amazon.awssdk.services.cognitosync.model.ResourceConflictException;
import software.amazon.awssdk.services.cognitosync.model.ResourceNotFoundException;
import software.amazon.awssdk.services.cognitosync.model.SetCognitoEventsRequest;
import software.amazon.awssdk.services.cognitosync.model.SetCognitoEventsResponse;
import software.amazon.awssdk.services.cognitosync.model.SetIdentityPoolConfigurationRequest;
import software.amazon.awssdk.services.cognitosync.model.SetIdentityPoolConfigurationResponse;
import software.amazon.awssdk.services.cognitosync.model.SubscribeToDatasetRequest;
import software.amazon.awssdk.services.cognitosync.model.SubscribeToDatasetResponse;
import software.amazon.awssdk.services.cognitosync.model.TooManyRequestsException;
import software.amazon.awssdk.services.cognitosync.model.UnsubscribeFromDatasetRequest;
import software.amazon.awssdk.services.cognitosync.model.UnsubscribeFromDatasetResponse;
import software.amazon.awssdk.services.cognitosync.model.UpdateRecordsRequest;
import software.amazon.awssdk.services.cognitosync.model.UpdateRecordsResponse;
import software.amazon.awssdk.services.cognitosync.transform.BulkPublishRequestMarshaller;
import software.amazon.awssdk.services.cognitosync.transform.DeleteDatasetRequestMarshaller;
import software.amazon.awssdk.services.cognitosync.transform.DescribeDatasetRequestMarshaller;
import software.amazon.awssdk.services.cognitosync.transform.DescribeIdentityPoolUsageRequestMarshaller;
import software.amazon.awssdk.services.cognitosync.transform.DescribeIdentityUsageRequestMarshaller;
import software.amazon.awssdk.services.cognitosync.transform.GetBulkPublishDetailsRequestMarshaller;
import software.amazon.awssdk.services.cognitosync.transform.GetCognitoEventsRequestMarshaller;
import software.amazon.awssdk.services.cognitosync.transform.GetIdentityPoolConfigurationRequestMarshaller;
import software.amazon.awssdk.services.cognitosync.transform.ListDatasetsRequestMarshaller;
import software.amazon.awssdk.services.cognitosync.transform.ListIdentityPoolUsageRequestMarshaller;
import software.amazon.awssdk.services.cognitosync.transform.ListRecordsRequestMarshaller;
import software.amazon.awssdk.services.cognitosync.transform.RegisterDeviceRequestMarshaller;
import software.amazon.awssdk.services.cognitosync.transform.SetCognitoEventsRequestMarshaller;
import software.amazon.awssdk.services.cognitosync.transform.SetIdentityPoolConfigurationRequestMarshaller;
import software.amazon.awssdk.services.cognitosync.transform.SubscribeToDatasetRequestMarshaller;
import software.amazon.awssdk.services.cognitosync.transform.UnsubscribeFromDatasetRequestMarshaller;
import software.amazon.awssdk.services.cognitosync.transform.UpdateRecordsRequestMarshaller;
import software.amazon.awssdk.utils.CompletableFutureUtils;

/**
 * Internal implementation of {@link CognitoSyncAsyncClient}.
 *
 * @see CognitoSyncAsyncClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultCognitoSyncAsyncClient implements CognitoSyncAsyncClient {
    private static final Logger log = LoggerFactory.getLogger(DefaultCognitoSyncAsyncClient.class);

    private final AsyncClientHandler clientHandler;

    private final AwsJsonProtocolFactory protocolFactory;

    private final SdkClientConfiguration clientConfiguration;

    protected DefaultCognitoSyncAsyncClient(SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsAsyncClientHandler(clientConfiguration);
        this.clientConfiguration = clientConfiguration;
        this.protocolFactory = init(AwsJsonProtocolFactory.builder()).build();
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    /**
     * <p>
     * Initiates a bulk publish of all existing datasets for an Identity Pool to the configured stream. Customers are
     * limited to one successful bulk publish per 24 hours. Bulk publish is an asynchronous request, customers can see
     * the status of the request via the GetBulkPublishDetails operation.
     * </p>
     * <p>
     * This API can only be called with developer credentials. You cannot call this API with the temporary user
     * credentials provided by Cognito Identity.
     * </p>
     *
     * @param bulkPublishRequest
     *        The input for the BulkPublish operation.
     * @return A Java Future containing the result of the BulkPublish operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>NotAuthorizedException Thrown when a user is not authorized to access the requested resource.</li>
     *         <li>InvalidParameterException Thrown when a request parameter does not comply with the associated
     *         constraints.</li>
     *         <li>ResourceNotFoundException Thrown if the resource doesn't exist.</li>
     *         <li>InternalErrorException Indicates an internal service error.</li>
     *         <li>DuplicateRequestException An exception thrown when there is an IN_PROGRESS bulk publish operation for
     *         the given identity pool.</li>
     *         <li>AlreadyStreamedException An exception thrown when a bulk publish operation is requested less than 24
     *         hours after a previous bulk publish operation completed successfully.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CognitoSyncException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample CognitoSyncAsyncClient.BulkPublish
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/cognito-sync-2014-06-30/BulkPublish" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<BulkPublishResponse> bulkPublish(BulkPublishRequest bulkPublishRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, bulkPublishRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Cognito Sync");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "BulkPublish");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<BulkPublishResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    BulkPublishResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<BulkPublishResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<BulkPublishRequest, BulkPublishResponse>()
                            .withOperationName("BulkPublish").withMarshaller(new BulkPublishRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(bulkPublishRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = bulkPublishRequest.overrideConfiguration().orElse(null);
            CompletableFuture<BulkPublishResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes the specific dataset. The dataset will be deleted permanently, and the action can't be undone. Datasets
     * that this dataset was merged with will no longer report the merge. Any subsequent operation on this dataset will
     * result in a ResourceNotFoundException.
     * </p>
     * <p>
     * This API can be called with temporary user credentials provided by Cognito Identity or with developer
     * credentials.
     * </p>
     *
     * @param deleteDatasetRequest
     *        A request to delete the specific dataset.
     * @return A Java Future containing the result of the DeleteDataset operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>NotAuthorizedException Thrown when a user is not authorized to access the requested resource.</li>
     *         <li>InvalidParameterException Thrown when a request parameter does not comply with the associated
     *         constraints.</li>
     *         <li>ResourceNotFoundException Thrown if the resource doesn't exist.</li>
     *         <li>InternalErrorException Indicates an internal service error.</li>
     *         <li>TooManyRequestsException Thrown if the request is throttled.</li>
     *         <li>ResourceConflictException Thrown if an update can't be applied because the resource was changed by
     *         another call and this would result in a conflict.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CognitoSyncException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample CognitoSyncAsyncClient.DeleteDataset
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/cognito-sync-2014-06-30/DeleteDataset" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteDatasetResponse> deleteDataset(DeleteDatasetRequest deleteDatasetRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteDatasetRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Cognito Sync");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteDataset");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteDatasetResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    DeleteDatasetResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteDatasetResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteDatasetRequest, DeleteDatasetResponse>()
                            .withOperationName("DeleteDataset")
                            .withMarshaller(new DeleteDatasetRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(deleteDatasetRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = deleteDatasetRequest.overrideConfiguration().orElse(null);
            CompletableFuture<DeleteDatasetResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Gets meta data about a dataset by identity and dataset name. With Amazon Cognito Sync, each identity has access
     * only to its own data. Thus, the credentials used to make this API call need to have access to the identity data.
     * </p>
     * <p>
     * This API can be called with temporary user credentials provided by Cognito Identity or with developer
     * credentials. You should use Cognito Identity credentials to make this API call.
     * </p>
     *
     * @param describeDatasetRequest
     *        A request for meta data about a dataset (creation date, number of records, size) by owner and dataset
     *        name.
     * @return A Java Future containing the result of the DescribeDataset operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>NotAuthorizedException Thrown when a user is not authorized to access the requested resource.</li>
     *         <li>InvalidParameterException Thrown when a request parameter does not comply with the associated
     *         constraints.</li>
     *         <li>ResourceNotFoundException Thrown if the resource doesn't exist.</li>
     *         <li>InternalErrorException Indicates an internal service error.</li>
     *         <li>TooManyRequestsException Thrown if the request is throttled.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CognitoSyncException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample CognitoSyncAsyncClient.DescribeDataset
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/cognito-sync-2014-06-30/DescribeDataset" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<DescribeDatasetResponse> describeDataset(DescribeDatasetRequest describeDatasetRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeDatasetRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Cognito Sync");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeDataset");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DescribeDatasetResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DescribeDatasetResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DescribeDatasetResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribeDatasetRequest, DescribeDatasetResponse>()
                            .withOperationName("DescribeDataset")
                            .withMarshaller(new DescribeDatasetRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(describeDatasetRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = describeDatasetRequest.overrideConfiguration().orElse(null);
            CompletableFuture<DescribeDatasetResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Gets usage details (for example, data storage) about a particular identity pool.
     * </p>
     * <p>
     * This API can only be called with developer credentials. You cannot call this API with the temporary user
     * credentials provided by Cognito Identity.
     * </p>
     *
     * @param describeIdentityPoolUsageRequest
     *        A request for usage information about the identity pool.
     * @return A Java Future containing the result of the DescribeIdentityPoolUsage operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>NotAuthorizedException Thrown when a user is not authorized to access the requested resource.</li>
     *         <li>InvalidParameterException Thrown when a request parameter does not comply with the associated
     *         constraints.</li>
     *         <li>ResourceNotFoundException Thrown if the resource doesn't exist.</li>
     *         <li>InternalErrorException Indicates an internal service error.</li>
     *         <li>TooManyRequestsException Thrown if the request is throttled.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CognitoSyncException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample CognitoSyncAsyncClient.DescribeIdentityPoolUsage
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/cognito-sync-2014-06-30/DescribeIdentityPoolUsage"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DescribeIdentityPoolUsageResponse> describeIdentityPoolUsage(
            DescribeIdentityPoolUsageRequest describeIdentityPoolUsageRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeIdentityPoolUsageRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Cognito Sync");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeIdentityPoolUsage");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DescribeIdentityPoolUsageResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DescribeIdentityPoolUsageResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DescribeIdentityPoolUsageResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribeIdentityPoolUsageRequest, DescribeIdentityPoolUsageResponse>()
                            .withOperationName("DescribeIdentityPoolUsage")
                            .withMarshaller(new DescribeIdentityPoolUsageRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(describeIdentityPoolUsageRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = describeIdentityPoolUsageRequest.overrideConfiguration()
                    .orElse(null);
            CompletableFuture<DescribeIdentityPoolUsageResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Gets usage information for an identity, including number of datasets and data usage.
     * </p>
     * <p>
     * This API can be called with temporary user credentials provided by Cognito Identity or with developer
     * credentials.
     * </p>
     *
     * @param describeIdentityUsageRequest
     *        A request for information about the usage of an identity pool.
     * @return A Java Future containing the result of the DescribeIdentityUsage operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>NotAuthorizedException Thrown when a user is not authorized to access the requested resource.</li>
     *         <li>InvalidParameterException Thrown when a request parameter does not comply with the associated
     *         constraints.</li>
     *         <li>ResourceNotFoundException Thrown if the resource doesn't exist.</li>
     *         <li>InternalErrorException Indicates an internal service error.</li>
     *         <li>TooManyRequestsException Thrown if the request is throttled.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CognitoSyncException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample CognitoSyncAsyncClient.DescribeIdentityUsage
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/cognito-sync-2014-06-30/DescribeIdentityUsage"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DescribeIdentityUsageResponse> describeIdentityUsage(
            DescribeIdentityUsageRequest describeIdentityUsageRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeIdentityUsageRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Cognito Sync");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeIdentityUsage");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DescribeIdentityUsageResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DescribeIdentityUsageResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DescribeIdentityUsageResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribeIdentityUsageRequest, DescribeIdentityUsageResponse>()
                            .withOperationName("DescribeIdentityUsage")
                            .withMarshaller(new DescribeIdentityUsageRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(describeIdentityUsageRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = describeIdentityUsageRequest.overrideConfiguration().orElse(
                    null);
            CompletableFuture<DescribeIdentityUsageResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Get the status of the last BulkPublish operation for an identity pool.
     * </p>
     * <p>
     * This API can only be called with developer credentials. You cannot call this API with the temporary user
     * credentials provided by Cognito Identity.
     * </p>
     *
     * @param getBulkPublishDetailsRequest
     *        The input for the GetBulkPublishDetails operation.
     * @return A Java Future containing the result of the GetBulkPublishDetails operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>NotAuthorizedException Thrown when a user is not authorized to access the requested resource.</li>
     *         <li>InvalidParameterException Thrown when a request parameter does not comply with the associated
     *         constraints.</li>
     *         <li>ResourceNotFoundException Thrown if the resource doesn't exist.</li>
     *         <li>InternalErrorException Indicates an internal service error.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CognitoSyncException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample CognitoSyncAsyncClient.GetBulkPublishDetails
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/cognito-sync-2014-06-30/GetBulkPublishDetails"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<GetBulkPublishDetailsResponse> getBulkPublishDetails(
            GetBulkPublishDetailsRequest getBulkPublishDetailsRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getBulkPublishDetailsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Cognito Sync");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetBulkPublishDetails");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetBulkPublishDetailsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GetBulkPublishDetailsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetBulkPublishDetailsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetBulkPublishDetailsRequest, GetBulkPublishDetailsResponse>()
                            .withOperationName("GetBulkPublishDetails")
                            .withMarshaller(new GetBulkPublishDetailsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(getBulkPublishDetailsRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = getBulkPublishDetailsRequest.overrideConfiguration().orElse(
                    null);
            CompletableFuture<GetBulkPublishDetailsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Gets the events and the corresponding Lambda functions associated with an identity pool.
     * </p>
     * <p>
     * This API can only be called with developer credentials. You cannot call this API with the temporary user
     * credentials provided by Cognito Identity.
     * </p>
     *
     * @param getCognitoEventsRequest
     *        A request for a list of the configured Cognito Events
     * @return A Java Future containing the result of the GetCognitoEvents operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidParameterException Thrown when a request parameter does not comply with the associated
     *         constraints.</li>
     *         <li>ResourceNotFoundException Thrown if the resource doesn't exist.</li>
     *         <li>NotAuthorizedException Thrown when a user is not authorized to access the requested resource.</li>
     *         <li>InternalErrorException Indicates an internal service error.</li>
     *         <li>TooManyRequestsException Thrown if the request is throttled.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CognitoSyncException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample CognitoSyncAsyncClient.GetCognitoEvents
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/cognito-sync-2014-06-30/GetCognitoEvents" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<GetCognitoEventsResponse> getCognitoEvents(GetCognitoEventsRequest getCognitoEventsRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getCognitoEventsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Cognito Sync");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetCognitoEvents");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetCognitoEventsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GetCognitoEventsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetCognitoEventsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetCognitoEventsRequest, GetCognitoEventsResponse>()
                            .withOperationName("GetCognitoEvents")
                            .withMarshaller(new GetCognitoEventsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(getCognitoEventsRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = getCognitoEventsRequest.overrideConfiguration().orElse(null);
            CompletableFuture<GetCognitoEventsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Gets the configuration settings of an identity pool.
     * </p>
     * <p>
     * This API can only be called with developer credentials. You cannot call this API with the temporary user
     * credentials provided by Cognito Identity.
     * </p>
     *
     * @param getIdentityPoolConfigurationRequest
     *        The input for the GetIdentityPoolConfiguration operation.
     * @return A Java Future containing the result of the GetIdentityPoolConfiguration operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>NotAuthorizedException Thrown when a user is not authorized to access the requested resource.</li>
     *         <li>InvalidParameterException Thrown when a request parameter does not comply with the associated
     *         constraints.</li>
     *         <li>ResourceNotFoundException Thrown if the resource doesn't exist.</li>
     *         <li>InternalErrorException Indicates an internal service error.</li>
     *         <li>TooManyRequestsException Thrown if the request is throttled.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CognitoSyncException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample CognitoSyncAsyncClient.GetIdentityPoolConfiguration
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/cognito-sync-2014-06-30/GetIdentityPoolConfiguration"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<GetIdentityPoolConfigurationResponse> getIdentityPoolConfiguration(
            GetIdentityPoolConfigurationRequest getIdentityPoolConfigurationRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getIdentityPoolConfigurationRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Cognito Sync");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetIdentityPoolConfiguration");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetIdentityPoolConfigurationResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GetIdentityPoolConfigurationResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetIdentityPoolConfigurationResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetIdentityPoolConfigurationRequest, GetIdentityPoolConfigurationResponse>()
                            .withOperationName("GetIdentityPoolConfiguration")
                            .withMarshaller(new GetIdentityPoolConfigurationRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(getIdentityPoolConfigurationRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = getIdentityPoolConfigurationRequest.overrideConfiguration()
                    .orElse(null);
            CompletableFuture<GetIdentityPoolConfigurationResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists datasets for an identity. With Amazon Cognito Sync, each identity has access only to its own data. Thus,
     * the credentials used to make this API call need to have access to the identity data.
     * </p>
     * <p>
     * ListDatasets can be called with temporary user credentials provided by Cognito Identity or with developer
     * credentials. You should use the Cognito Identity credentials to make this API call.
     * </p>
     *
     * @param listDatasetsRequest
     *        Request for a list of datasets for an identity.
     * @return A Java Future containing the result of the ListDatasets operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>NotAuthorizedException Thrown when a user is not authorized to access the requested resource.</li>
     *         <li>InvalidParameterException Thrown when a request parameter does not comply with the associated
     *         constraints.</li>
     *         <li>InternalErrorException Indicates an internal service error.</li>
     *         <li>TooManyRequestsException Thrown if the request is throttled.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CognitoSyncException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample CognitoSyncAsyncClient.ListDatasets
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/cognito-sync-2014-06-30/ListDatasets" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<ListDatasetsResponse> listDatasets(ListDatasetsRequest listDatasetsRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listDatasetsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Cognito Sync");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListDatasets");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListDatasetsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    ListDatasetsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListDatasetsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListDatasetsRequest, ListDatasetsResponse>()
                            .withOperationName("ListDatasets").withMarshaller(new ListDatasetsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(listDatasetsRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = listDatasetsRequest.overrideConfiguration().orElse(null);
            CompletableFuture<ListDatasetsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Gets a list of identity pools registered with Cognito.
     * </p>
     * <p>
     * ListIdentityPoolUsage can only be called with developer credentials. You cannot make this API call with the
     * temporary user credentials provided by Cognito Identity.
     * </p>
     *
     * @param listIdentityPoolUsageRequest
     *        A request for usage information on an identity pool.
     * @return A Java Future containing the result of the ListIdentityPoolUsage operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>NotAuthorizedException Thrown when a user is not authorized to access the requested resource.</li>
     *         <li>InvalidParameterException Thrown when a request parameter does not comply with the associated
     *         constraints.</li>
     *         <li>InternalErrorException Indicates an internal service error.</li>
     *         <li>TooManyRequestsException Thrown if the request is throttled.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CognitoSyncException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample CognitoSyncAsyncClient.ListIdentityPoolUsage
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/cognito-sync-2014-06-30/ListIdentityPoolUsage"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListIdentityPoolUsageResponse> listIdentityPoolUsage(
            ListIdentityPoolUsageRequest listIdentityPoolUsageRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listIdentityPoolUsageRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Cognito Sync");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListIdentityPoolUsage");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListIdentityPoolUsageResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListIdentityPoolUsageResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListIdentityPoolUsageResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListIdentityPoolUsageRequest, ListIdentityPoolUsageResponse>()
                            .withOperationName("ListIdentityPoolUsage")
                            .withMarshaller(new ListIdentityPoolUsageRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(listIdentityPoolUsageRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = listIdentityPoolUsageRequest.overrideConfiguration().orElse(
                    null);
            CompletableFuture<ListIdentityPoolUsageResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Gets paginated records, optionally changed after a particular sync count for a dataset and identity. With Amazon
     * Cognito Sync, each identity has access only to its own data. Thus, the credentials used to make this API call
     * need to have access to the identity data.
     * </p>
     * <p>
     * ListRecords can be called with temporary user credentials provided by Cognito Identity or with developer
     * credentials. You should use Cognito Identity credentials to make this API call.
     * </p>
     *
     * @param listRecordsRequest
     *        A request for a list of records.
     * @return A Java Future containing the result of the ListRecords operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidParameterException Thrown when a request parameter does not comply with the associated
     *         constraints.</li>
     *         <li>NotAuthorizedException Thrown when a user is not authorized to access the requested resource.</li>
     *         <li>TooManyRequestsException Thrown if the request is throttled.</li>
     *         <li>InternalErrorException Indicates an internal service error.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CognitoSyncException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample CognitoSyncAsyncClient.ListRecords
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/cognito-sync-2014-06-30/ListRecords" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<ListRecordsResponse> listRecords(ListRecordsRequest listRecordsRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listRecordsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Cognito Sync");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListRecords");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListRecordsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    ListRecordsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListRecordsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListRecordsRequest, ListRecordsResponse>()
                            .withOperationName("ListRecords").withMarshaller(new ListRecordsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(listRecordsRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = listRecordsRequest.overrideConfiguration().orElse(null);
            CompletableFuture<ListRecordsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Registers a device to receive push sync notifications.
     * </p>
     * <p>
     * This API can only be called with temporary credentials provided by Cognito Identity. You cannot call this API
     * with developer credentials.
     * </p>
     *
     * @param registerDeviceRequest
     *        A request to RegisterDevice.
     * @return A Java Future containing the result of the RegisterDevice operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>NotAuthorizedException Thrown when a user is not authorized to access the requested resource.</li>
     *         <li>InvalidParameterException Thrown when a request parameter does not comply with the associated
     *         constraints.</li>
     *         <li>ResourceNotFoundException Thrown if the resource doesn't exist.</li>
     *         <li>InternalErrorException Indicates an internal service error.</li>
     *         <li>InvalidConfigurationException</li>
     *         <li>TooManyRequestsException Thrown if the request is throttled.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CognitoSyncException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample CognitoSyncAsyncClient.RegisterDevice
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/cognito-sync-2014-06-30/RegisterDevice" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<RegisterDeviceResponse> registerDevice(RegisterDeviceRequest registerDeviceRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, registerDeviceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Cognito Sync");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "RegisterDevice");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<RegisterDeviceResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, RegisterDeviceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<RegisterDeviceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<RegisterDeviceRequest, RegisterDeviceResponse>()
                            .withOperationName("RegisterDevice")
                            .withMarshaller(new RegisterDeviceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(registerDeviceRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = registerDeviceRequest.overrideConfiguration().orElse(null);
            CompletableFuture<RegisterDeviceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Sets the AWS Lambda function for a given event type for an identity pool. This request only updates the key/value
     * pair specified. Other key/values pairs are not updated. To remove a key value pair, pass a empty value for the
     * particular key.
     * </p>
     * <p>
     * This API can only be called with developer credentials. You cannot call this API with the temporary user
     * credentials provided by Cognito Identity.
     * </p>
     *
     * @param setCognitoEventsRequest
     *        A request to configure Cognito Events"
     * @return A Java Future containing the result of the SetCognitoEvents operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidParameterException Thrown when a request parameter does not comply with the associated
     *         constraints.</li>
     *         <li>ResourceNotFoundException Thrown if the resource doesn't exist.</li>
     *         <li>NotAuthorizedException Thrown when a user is not authorized to access the requested resource.</li>
     *         <li>InternalErrorException Indicates an internal service error.</li>
     *         <li>TooManyRequestsException Thrown if the request is throttled.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CognitoSyncException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample CognitoSyncAsyncClient.SetCognitoEvents
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/cognito-sync-2014-06-30/SetCognitoEvents" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<SetCognitoEventsResponse> setCognitoEvents(SetCognitoEventsRequest setCognitoEventsRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, setCognitoEventsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Cognito Sync");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "SetCognitoEvents");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<SetCognitoEventsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, SetCognitoEventsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<SetCognitoEventsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<SetCognitoEventsRequest, SetCognitoEventsResponse>()
                            .withOperationName("SetCognitoEvents")
                            .withMarshaller(new SetCognitoEventsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(setCognitoEventsRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = setCognitoEventsRequest.overrideConfiguration().orElse(null);
            CompletableFuture<SetCognitoEventsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Sets the necessary configuration for push sync.
     * </p>
     * <p>
     * This API can only be called with developer credentials. You cannot call this API with the temporary user
     * credentials provided by Cognito Identity.
     * </p>
     *
     * @param setIdentityPoolConfigurationRequest
     *        The input for the SetIdentityPoolConfiguration operation.
     * @return A Java Future containing the result of the SetIdentityPoolConfiguration operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>NotAuthorizedException Thrown when a user is not authorized to access the requested resource.</li>
     *         <li>InvalidParameterException Thrown when a request parameter does not comply with the associated
     *         constraints.</li>
     *         <li>ResourceNotFoundException Thrown if the resource doesn't exist.</li>
     *         <li>InternalErrorException Indicates an internal service error.</li>
     *         <li>TooManyRequestsException Thrown if the request is throttled.</li>
     *         <li>ConcurrentModificationException Thrown if there are parallel requests to modify a resource.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CognitoSyncException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample CognitoSyncAsyncClient.SetIdentityPoolConfiguration
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/cognito-sync-2014-06-30/SetIdentityPoolConfiguration"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<SetIdentityPoolConfigurationResponse> setIdentityPoolConfiguration(
            SetIdentityPoolConfigurationRequest setIdentityPoolConfigurationRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, setIdentityPoolConfigurationRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Cognito Sync");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "SetIdentityPoolConfiguration");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<SetIdentityPoolConfigurationResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, SetIdentityPoolConfigurationResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<SetIdentityPoolConfigurationResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<SetIdentityPoolConfigurationRequest, SetIdentityPoolConfigurationResponse>()
                            .withOperationName("SetIdentityPoolConfiguration")
                            .withMarshaller(new SetIdentityPoolConfigurationRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(setIdentityPoolConfigurationRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = setIdentityPoolConfigurationRequest.overrideConfiguration()
                    .orElse(null);
            CompletableFuture<SetIdentityPoolConfigurationResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Subscribes to receive notifications when a dataset is modified by another device.
     * </p>
     * <p>
     * This API can only be called with temporary credentials provided by Cognito Identity. You cannot call this API
     * with developer credentials.
     * </p>
     *
     * @param subscribeToDatasetRequest
     *        A request to SubscribeToDatasetRequest.
     * @return A Java Future containing the result of the SubscribeToDataset operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>NotAuthorizedException Thrown when a user is not authorized to access the requested resource.</li>
     *         <li>InvalidParameterException Thrown when a request parameter does not comply with the associated
     *         constraints.</li>
     *         <li>ResourceNotFoundException Thrown if the resource doesn't exist.</li>
     *         <li>InternalErrorException Indicates an internal service error.</li>
     *         <li>InvalidConfigurationException</li>
     *         <li>TooManyRequestsException Thrown if the request is throttled.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CognitoSyncException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample CognitoSyncAsyncClient.SubscribeToDataset
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/cognito-sync-2014-06-30/SubscribeToDataset"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<SubscribeToDatasetResponse> subscribeToDataset(SubscribeToDatasetRequest subscribeToDatasetRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, subscribeToDatasetRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Cognito Sync");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "SubscribeToDataset");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<SubscribeToDatasetResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, SubscribeToDatasetResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<SubscribeToDatasetResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<SubscribeToDatasetRequest, SubscribeToDatasetResponse>()
                            .withOperationName("SubscribeToDataset")
                            .withMarshaller(new SubscribeToDatasetRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(subscribeToDatasetRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = subscribeToDatasetRequest.overrideConfiguration()
                    .orElse(null);
            CompletableFuture<SubscribeToDatasetResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Unsubscribes from receiving notifications when a dataset is modified by another device.
     * </p>
     * <p>
     * This API can only be called with temporary credentials provided by Cognito Identity. You cannot call this API
     * with developer credentials.
     * </p>
     *
     * @param unsubscribeFromDatasetRequest
     *        A request to UnsubscribeFromDataset.
     * @return A Java Future containing the result of the UnsubscribeFromDataset operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>NotAuthorizedException Thrown when a user is not authorized to access the requested resource.</li>
     *         <li>InvalidParameterException Thrown when a request parameter does not comply with the associated
     *         constraints.</li>
     *         <li>ResourceNotFoundException Thrown if the resource doesn't exist.</li>
     *         <li>InternalErrorException Indicates an internal service error.</li>
     *         <li>InvalidConfigurationException</li>
     *         <li>TooManyRequestsException Thrown if the request is throttled.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CognitoSyncException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample CognitoSyncAsyncClient.UnsubscribeFromDataset
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/cognito-sync-2014-06-30/UnsubscribeFromDataset"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<UnsubscribeFromDatasetResponse> unsubscribeFromDataset(
            UnsubscribeFromDatasetRequest unsubscribeFromDatasetRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, unsubscribeFromDatasetRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Cognito Sync");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UnsubscribeFromDataset");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UnsubscribeFromDatasetResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, UnsubscribeFromDatasetResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UnsubscribeFromDatasetResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UnsubscribeFromDatasetRequest, UnsubscribeFromDatasetResponse>()
                            .withOperationName("UnsubscribeFromDataset")
                            .withMarshaller(new UnsubscribeFromDatasetRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(unsubscribeFromDatasetRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = unsubscribeFromDatasetRequest.overrideConfiguration().orElse(
                    null);
            CompletableFuture<UnsubscribeFromDatasetResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Posts updates to records and adds and deletes records for a dataset and user.
     * </p>
     * <p>
     * The sync count in the record patch is your last known sync count for that record. The server will reject an
     * UpdateRecords request with a ResourceConflictException if you try to patch a record with a new value but a stale
     * sync count.
     * </p>
     * <p>
     * For example, if the sync count on the server is 5 for a key called highScore and you try and submit a new
     * highScore with sync count of 4, the request will be rejected. To obtain the current sync count for a record, call
     * ListRecords. On a successful update of the record, the response returns the new sync count for that record. You
     * should present that sync count the next time you try to update that same record. When the record does not exist,
     * specify the sync count as 0.
     * </p>
     * <p>
     * This API can be called with temporary user credentials provided by Cognito Identity or with developer
     * credentials.
     * </p>
     *
     * @param updateRecordsRequest
     *        A request to post updates to records or add and delete records for a dataset and user.
     * @return A Java Future containing the result of the UpdateRecords operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidParameterException Thrown when a request parameter does not comply with the associated
     *         constraints.</li>
     *         <li>LimitExceededException Thrown when the limit on the number of objects or operations has been
     *         exceeded.</li>
     *         <li>NotAuthorizedException Thrown when a user is not authorized to access the requested resource.</li>
     *         <li>ResourceNotFoundException Thrown if the resource doesn't exist.</li>
     *         <li>ResourceConflictException Thrown if an update can't be applied because the resource was changed by
     *         another call and this would result in a conflict.</li>
     *         <li>InvalidLambdaFunctionOutputException The AWS Lambda function returned invalid output or an exception.
     *         </li>
     *         <li>LambdaThrottledException AWS Lambda throttled your account, please contact AWS Support</li>
     *         <li>TooManyRequestsException Thrown if the request is throttled.</li>
     *         <li>InternalErrorException Indicates an internal service error.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CognitoSyncException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample CognitoSyncAsyncClient.UpdateRecords
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/cognito-sync-2014-06-30/UpdateRecords" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateRecordsResponse> updateRecords(UpdateRecordsRequest updateRecordsRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateRecordsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Cognito Sync");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateRecords");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateRecordsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    UpdateRecordsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UpdateRecordsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateRecordsRequest, UpdateRecordsResponse>()
                            .withOperationName("UpdateRecords")
                            .withMarshaller(new UpdateRecordsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(updateRecordsRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = updateRecordsRequest.overrideConfiguration().orElse(null);
            CompletableFuture<UpdateRecordsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    @Override
    public void close() {
        clientHandler.close();
    }

    private <T extends BaseAwsJsonProtocolFactory.Builder<T>> T init(T builder) {
        return builder
                .clientConfiguration(clientConfiguration)
                .defaultServiceExceptionSupplier(CognitoSyncException::builder)
                .protocol(AwsJsonProtocol.REST_JSON)
                .protocolVersion("1.1")
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ConcurrentModification")
                                .exceptionBuilderSupplier(ConcurrentModificationException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidConfiguration")
                                .exceptionBuilderSupplier(InvalidConfigurationException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidParameter")
                                .exceptionBuilderSupplier(InvalidParameterException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ResourceNotFound")
                                .exceptionBuilderSupplier(ResourceNotFoundException::builder).httpStatusCode(404).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("DuplicateRequest")
                                .exceptionBuilderSupplier(DuplicateRequestException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ResourceConflict")
                                .exceptionBuilderSupplier(ResourceConflictException::builder).httpStatusCode(409).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("LambdaThrottled")
                                .exceptionBuilderSupplier(LambdaThrottledException::builder).httpStatusCode(429).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("NotAuthorizedError")
                                .exceptionBuilderSupplier(NotAuthorizedException::builder).httpStatusCode(403).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InternalError")
                                .exceptionBuilderSupplier(InternalErrorException::builder).httpStatusCode(500).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("TooManyRequests")
                                .exceptionBuilderSupplier(TooManyRequestsException::builder).httpStatusCode(429).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidLambdaFunctionOutput")
                                .exceptionBuilderSupplier(InvalidLambdaFunctionOutputException::builder).httpStatusCode(400)
                                .build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("LimitExceeded")
                                .exceptionBuilderSupplier(LimitExceededException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("AlreadyStreamed")
                                .exceptionBuilderSupplier(AlreadyStreamedException::builder).httpStatusCode(400).build());
    }

    private static List<MetricPublisher> resolveMetricPublishers(SdkClientConfiguration clientConfiguration,
            RequestOverrideConfiguration requestOverrideConfiguration) {
        List<MetricPublisher> publishers = null;
        if (requestOverrideConfiguration != null) {
            publishers = requestOverrideConfiguration.metricPublishers();
        }
        if (publishers == null || publishers.isEmpty()) {
            publishers = clientConfiguration.option(SdkClientOption.METRIC_PUBLISHERS);
        }
        if (publishers == null) {
            publishers = Collections.emptyList();
        }
        return publishers;
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata) {
        return protocolFactory.createErrorResponseHandler(operationMetadata);
    }
}
