/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.codestarnotifications;

import java.util.function.Consumer;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkPublicApi;
import software.amazon.awssdk.annotations.ThreadSafe;
import software.amazon.awssdk.awscore.AwsClient;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.core.exception.SdkClientException;
import software.amazon.awssdk.regions.ServiceMetadata;
import software.amazon.awssdk.services.codestarnotifications.model.AccessDeniedException;
import software.amazon.awssdk.services.codestarnotifications.model.CodestarNotificationsException;
import software.amazon.awssdk.services.codestarnotifications.model.ConcurrentModificationException;
import software.amazon.awssdk.services.codestarnotifications.model.ConfigurationException;
import software.amazon.awssdk.services.codestarnotifications.model.CreateNotificationRuleRequest;
import software.amazon.awssdk.services.codestarnotifications.model.CreateNotificationRuleResponse;
import software.amazon.awssdk.services.codestarnotifications.model.DeleteNotificationRuleRequest;
import software.amazon.awssdk.services.codestarnotifications.model.DeleteNotificationRuleResponse;
import software.amazon.awssdk.services.codestarnotifications.model.DeleteTargetRequest;
import software.amazon.awssdk.services.codestarnotifications.model.DeleteTargetResponse;
import software.amazon.awssdk.services.codestarnotifications.model.DescribeNotificationRuleRequest;
import software.amazon.awssdk.services.codestarnotifications.model.DescribeNotificationRuleResponse;
import software.amazon.awssdk.services.codestarnotifications.model.InvalidNextTokenException;
import software.amazon.awssdk.services.codestarnotifications.model.LimitExceededException;
import software.amazon.awssdk.services.codestarnotifications.model.ListEventTypesRequest;
import software.amazon.awssdk.services.codestarnotifications.model.ListEventTypesResponse;
import software.amazon.awssdk.services.codestarnotifications.model.ListNotificationRulesRequest;
import software.amazon.awssdk.services.codestarnotifications.model.ListNotificationRulesResponse;
import software.amazon.awssdk.services.codestarnotifications.model.ListTagsForResourceRequest;
import software.amazon.awssdk.services.codestarnotifications.model.ListTagsForResourceResponse;
import software.amazon.awssdk.services.codestarnotifications.model.ListTargetsRequest;
import software.amazon.awssdk.services.codestarnotifications.model.ListTargetsResponse;
import software.amazon.awssdk.services.codestarnotifications.model.ResourceAlreadyExistsException;
import software.amazon.awssdk.services.codestarnotifications.model.ResourceNotFoundException;
import software.amazon.awssdk.services.codestarnotifications.model.SubscribeRequest;
import software.amazon.awssdk.services.codestarnotifications.model.SubscribeResponse;
import software.amazon.awssdk.services.codestarnotifications.model.TagResourceRequest;
import software.amazon.awssdk.services.codestarnotifications.model.TagResourceResponse;
import software.amazon.awssdk.services.codestarnotifications.model.UnsubscribeRequest;
import software.amazon.awssdk.services.codestarnotifications.model.UnsubscribeResponse;
import software.amazon.awssdk.services.codestarnotifications.model.UntagResourceRequest;
import software.amazon.awssdk.services.codestarnotifications.model.UntagResourceResponse;
import software.amazon.awssdk.services.codestarnotifications.model.UpdateNotificationRuleRequest;
import software.amazon.awssdk.services.codestarnotifications.model.UpdateNotificationRuleResponse;
import software.amazon.awssdk.services.codestarnotifications.model.ValidationException;
import software.amazon.awssdk.services.codestarnotifications.paginators.ListEventTypesIterable;
import software.amazon.awssdk.services.codestarnotifications.paginators.ListNotificationRulesIterable;
import software.amazon.awssdk.services.codestarnotifications.paginators.ListTargetsIterable;

/**
 * Service client for accessing AWS CodeStar Notifications. This can be created using the static {@link #builder()}
 * method.
 *
 * <p>
 * This CodeStar Notifications API Reference provides descriptions and usage examples of the operations and data types
 * for the CodeStar Notifications API. You can use the CodeStar Notifications API to work with the following objects:
 * </p>
 * <p>
 * Notification rules, by calling the following:
 * </p>
 * <ul>
 * <li>
 * <p>
 * <a>CreateNotificationRule</a>, which creates a notification rule for a resource in your account.
 * </p>
 * </li>
 * <li>
 * <p>
 * <a>DeleteNotificationRule</a>, which deletes a notification rule.
 * </p>
 * </li>
 * <li>
 * <p>
 * <a>DescribeNotificationRule</a>, which provides information about a notification rule.
 * </p>
 * </li>
 * <li>
 * <p>
 * <a>ListNotificationRules</a>, which lists the notification rules associated with your account.
 * </p>
 * </li>
 * <li>
 * <p>
 * <a>UpdateNotificationRule</a>, which changes the name, events, or targets associated with a notification rule.
 * </p>
 * </li>
 * <li>
 * <p>
 * <a>Subscribe</a>, which subscribes a target to a notification rule.
 * </p>
 * </li>
 * <li>
 * <p>
 * <a>Unsubscribe</a>, which removes a target from a notification rule.
 * </p>
 * </li>
 * </ul>
 * <p>
 * Targets, by calling the following:
 * </p>
 * <ul>
 * <li>
 * <p>
 * <a>DeleteTarget</a>, which removes a notification rule target from a notification rule.
 * </p>
 * </li>
 * <li>
 * <p>
 * <a>ListTargets</a>, which lists the targets associated with a notification rule.
 * </p>
 * </li>
 * </ul>
 * <p>
 * Events, by calling the following:
 * </p>
 * <ul>
 * <li>
 * <p>
 * <a>ListEventTypes</a>, which lists the event types you can include in a notification rule.
 * </p>
 * </li>
 * </ul>
 * <p>
 * Tags, by calling the following:
 * </p>
 * <ul>
 * <li>
 * <p>
 * <a>ListTagsForResource</a>, which lists the tags already associated with a notification rule in your account.
 * </p>
 * </li>
 * <li>
 * <p>
 * <a>TagResource</a>, which associates a tag you provide with a notification rule in your account.
 * </p>
 * </li>
 * <li>
 * <p>
 * <a>UntagResource</a>, which removes a tag from a notification rule in your account.
 * </p>
 * </li>
 * </ul>
 * <p>
 * For information about how to use CodeStar Notifications, see the <a
 * href="https://docs.aws.amazon.com/dtconsole/latest/userguide/what-is-dtconsole.html">Amazon Web Services Developer
 * Tools Console User Guide</a>.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
@SdkPublicApi
@ThreadSafe
public interface CodestarNotificationsClient extends AwsClient {
    String SERVICE_NAME = "codestar-notifications";

    /**
     * Value for looking up the service's metadata from the
     * {@link software.amazon.awssdk.regions.ServiceMetadataProvider}.
     */
    String SERVICE_METADATA_ID = "codestar-notifications";

    /**
     * <p>
     * Creates a notification rule for a resource. The rule specifies the events you want notifications about and the
     * targets (such as Amazon Q Developer in chat applications topics or Amazon Q Developer in chat applications
     * clients configured for Slack) where you want to receive them.
     * </p>
     *
     * @param createNotificationRuleRequest
     * @return Result of the CreateNotificationRule operation returned by the service.
     * @throws ResourceAlreadyExistsException
     *         A resource with the same name or ID already exists. Notification rule names must be unique in your Amazon
     *         Web Services account.
     * @throws ValidationException
     *         One or more parameter values are not valid.
     * @throws LimitExceededException
     *         One of the CodeStar Notifications limits has been exceeded. Limits apply to accounts, notification rules,
     *         notifications, resources, and targets. For more information, see Limits.
     * @throws ConfigurationException
     *         Some or all of the configuration is incomplete, missing, or not valid.
     * @throws ConcurrentModificationException
     *         CodeStar Notifications can't complete the request because the resource is being modified by another
     *         process. Wait a few minutes and try again.
     * @throws AccessDeniedException
     *         CodeStar Notifications can't create the notification rule because you do not have sufficient permissions.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws CodestarNotificationsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample CodestarNotificationsClient.CreateNotificationRule
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/codestar-notifications-2019-10-15/CreateNotificationRule"
     *      target="_top">AWS API Documentation</a>
     */
    default CreateNotificationRuleResponse createNotificationRule(CreateNotificationRuleRequest createNotificationRuleRequest)
            throws ResourceAlreadyExistsException, ValidationException, LimitExceededException, ConfigurationException,
            ConcurrentModificationException, AccessDeniedException, AwsServiceException, SdkClientException,
            CodestarNotificationsException {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Creates a notification rule for a resource. The rule specifies the events you want notifications about and the
     * targets (such as Amazon Q Developer in chat applications topics or Amazon Q Developer in chat applications
     * clients configured for Slack) where you want to receive them.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link CreateNotificationRuleRequest.Builder} avoiding the
     * need to create one manually via {@link CreateNotificationRuleRequest#builder()}
     * </p>
     *
     * @param createNotificationRuleRequest
     *        A {@link Consumer} that will call methods on
     *        {@link software.amazon.awssdk.services.codestarnotifications.model.CreateNotificationRuleRequest.Builder}
     *        to create a request.
     * @return Result of the CreateNotificationRule operation returned by the service.
     * @throws ResourceAlreadyExistsException
     *         A resource with the same name or ID already exists. Notification rule names must be unique in your Amazon
     *         Web Services account.
     * @throws ValidationException
     *         One or more parameter values are not valid.
     * @throws LimitExceededException
     *         One of the CodeStar Notifications limits has been exceeded. Limits apply to accounts, notification rules,
     *         notifications, resources, and targets. For more information, see Limits.
     * @throws ConfigurationException
     *         Some or all of the configuration is incomplete, missing, or not valid.
     * @throws ConcurrentModificationException
     *         CodeStar Notifications can't complete the request because the resource is being modified by another
     *         process. Wait a few minutes and try again.
     * @throws AccessDeniedException
     *         CodeStar Notifications can't create the notification rule because you do not have sufficient permissions.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws CodestarNotificationsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample CodestarNotificationsClient.CreateNotificationRule
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/codestar-notifications-2019-10-15/CreateNotificationRule"
     *      target="_top">AWS API Documentation</a>
     */
    default CreateNotificationRuleResponse createNotificationRule(
            Consumer<CreateNotificationRuleRequest.Builder> createNotificationRuleRequest) throws ResourceAlreadyExistsException,
            ValidationException, LimitExceededException, ConfigurationException, ConcurrentModificationException,
            AccessDeniedException, AwsServiceException, SdkClientException, CodestarNotificationsException {
        return createNotificationRule(CreateNotificationRuleRequest.builder().applyMutation(createNotificationRuleRequest)
                .build());
    }

    /**
     * <p>
     * Deletes a notification rule for a resource.
     * </p>
     *
     * @param deleteNotificationRuleRequest
     * @return Result of the DeleteNotificationRule operation returned by the service.
     * @throws ValidationException
     *         One or more parameter values are not valid.
     * @throws LimitExceededException
     *         One of the CodeStar Notifications limits has been exceeded. Limits apply to accounts, notification rules,
     *         notifications, resources, and targets. For more information, see Limits.
     * @throws ConcurrentModificationException
     *         CodeStar Notifications can't complete the request because the resource is being modified by another
     *         process. Wait a few minutes and try again.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws CodestarNotificationsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample CodestarNotificationsClient.DeleteNotificationRule
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/codestar-notifications-2019-10-15/DeleteNotificationRule"
     *      target="_top">AWS API Documentation</a>
     */
    default DeleteNotificationRuleResponse deleteNotificationRule(DeleteNotificationRuleRequest deleteNotificationRuleRequest)
            throws ValidationException, LimitExceededException, ConcurrentModificationException, AwsServiceException,
            SdkClientException, CodestarNotificationsException {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Deletes a notification rule for a resource.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link DeleteNotificationRuleRequest.Builder} avoiding the
     * need to create one manually via {@link DeleteNotificationRuleRequest#builder()}
     * </p>
     *
     * @param deleteNotificationRuleRequest
     *        A {@link Consumer} that will call methods on
     *        {@link software.amazon.awssdk.services.codestarnotifications.model.DeleteNotificationRuleRequest.Builder}
     *        to create a request.
     * @return Result of the DeleteNotificationRule operation returned by the service.
     * @throws ValidationException
     *         One or more parameter values are not valid.
     * @throws LimitExceededException
     *         One of the CodeStar Notifications limits has been exceeded. Limits apply to accounts, notification rules,
     *         notifications, resources, and targets. For more information, see Limits.
     * @throws ConcurrentModificationException
     *         CodeStar Notifications can't complete the request because the resource is being modified by another
     *         process. Wait a few minutes and try again.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws CodestarNotificationsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample CodestarNotificationsClient.DeleteNotificationRule
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/codestar-notifications-2019-10-15/DeleteNotificationRule"
     *      target="_top">AWS API Documentation</a>
     */
    default DeleteNotificationRuleResponse deleteNotificationRule(
            Consumer<DeleteNotificationRuleRequest.Builder> deleteNotificationRuleRequest) throws ValidationException,
            LimitExceededException, ConcurrentModificationException, AwsServiceException, SdkClientException,
            CodestarNotificationsException {
        return deleteNotificationRule(DeleteNotificationRuleRequest.builder().applyMutation(deleteNotificationRuleRequest)
                .build());
    }

    /**
     * <p>
     * Deletes a specified target for notifications.
     * </p>
     *
     * @param deleteTargetRequest
     * @return Result of the DeleteTarget operation returned by the service.
     * @throws ValidationException
     *         One or more parameter values are not valid.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws CodestarNotificationsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample CodestarNotificationsClient.DeleteTarget
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/codestar-notifications-2019-10-15/DeleteTarget"
     *      target="_top">AWS API Documentation</a>
     */
    default DeleteTargetResponse deleteTarget(DeleteTargetRequest deleteTargetRequest) throws ValidationException,
            AwsServiceException, SdkClientException, CodestarNotificationsException {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Deletes a specified target for notifications.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link DeleteTargetRequest.Builder} avoiding the need to
     * create one manually via {@link DeleteTargetRequest#builder()}
     * </p>
     *
     * @param deleteTargetRequest
     *        A {@link Consumer} that will call methods on
     *        {@link software.amazon.awssdk.services.codestarnotifications.model.DeleteTargetRequest.Builder} to create
     *        a request.
     * @return Result of the DeleteTarget operation returned by the service.
     * @throws ValidationException
     *         One or more parameter values are not valid.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws CodestarNotificationsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample CodestarNotificationsClient.DeleteTarget
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/codestar-notifications-2019-10-15/DeleteTarget"
     *      target="_top">AWS API Documentation</a>
     */
    default DeleteTargetResponse deleteTarget(Consumer<DeleteTargetRequest.Builder> deleteTargetRequest)
            throws ValidationException, AwsServiceException, SdkClientException, CodestarNotificationsException {
        return deleteTarget(DeleteTargetRequest.builder().applyMutation(deleteTargetRequest).build());
    }

    /**
     * <p>
     * Returns information about a specified notification rule.
     * </p>
     *
     * @param describeNotificationRuleRequest
     * @return Result of the DescribeNotificationRule operation returned by the service.
     * @throws ResourceNotFoundException
     *         CodeStar Notifications can't find a resource that matches the provided ARN.
     * @throws ValidationException
     *         One or more parameter values are not valid.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws CodestarNotificationsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample CodestarNotificationsClient.DescribeNotificationRule
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/codestar-notifications-2019-10-15/DescribeNotificationRule"
     *      target="_top">AWS API Documentation</a>
     */
    default DescribeNotificationRuleResponse describeNotificationRule(
            DescribeNotificationRuleRequest describeNotificationRuleRequest) throws ResourceNotFoundException,
            ValidationException, AwsServiceException, SdkClientException, CodestarNotificationsException {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Returns information about a specified notification rule.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link DescribeNotificationRuleRequest.Builder} avoiding
     * the need to create one manually via {@link DescribeNotificationRuleRequest#builder()}
     * </p>
     *
     * @param describeNotificationRuleRequest
     *        A {@link Consumer} that will call methods on
     *        {@link software.amazon.awssdk.services.codestarnotifications.model.DescribeNotificationRuleRequest.Builder}
     *        to create a request.
     * @return Result of the DescribeNotificationRule operation returned by the service.
     * @throws ResourceNotFoundException
     *         CodeStar Notifications can't find a resource that matches the provided ARN.
     * @throws ValidationException
     *         One or more parameter values are not valid.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws CodestarNotificationsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample CodestarNotificationsClient.DescribeNotificationRule
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/codestar-notifications-2019-10-15/DescribeNotificationRule"
     *      target="_top">AWS API Documentation</a>
     */
    default DescribeNotificationRuleResponse describeNotificationRule(
            Consumer<DescribeNotificationRuleRequest.Builder> describeNotificationRuleRequest) throws ResourceNotFoundException,
            ValidationException, AwsServiceException, SdkClientException, CodestarNotificationsException {
        return describeNotificationRule(DescribeNotificationRuleRequest.builder().applyMutation(describeNotificationRuleRequest)
                .build());
    }

    /**
     * <p>
     * Returns information about the event types available for configuring notifications.
     * </p>
     *
     * @param listEventTypesRequest
     * @return Result of the ListEventTypes operation returned by the service.
     * @throws InvalidNextTokenException
     *         The value for the enumeration token used in the request to return the next batch of the results is not
     *         valid.
     * @throws ValidationException
     *         One or more parameter values are not valid.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws CodestarNotificationsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample CodestarNotificationsClient.ListEventTypes
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/codestar-notifications-2019-10-15/ListEventTypes"
     *      target="_top">AWS API Documentation</a>
     */
    default ListEventTypesResponse listEventTypes(ListEventTypesRequest listEventTypesRequest) throws InvalidNextTokenException,
            ValidationException, AwsServiceException, SdkClientException, CodestarNotificationsException {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Returns information about the event types available for configuring notifications.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link ListEventTypesRequest.Builder} avoiding the need to
     * create one manually via {@link ListEventTypesRequest#builder()}
     * </p>
     *
     * @param listEventTypesRequest
     *        A {@link Consumer} that will call methods on
     *        {@link software.amazon.awssdk.services.codestarnotifications.model.ListEventTypesRequest.Builder} to
     *        create a request.
     * @return Result of the ListEventTypes operation returned by the service.
     * @throws InvalidNextTokenException
     *         The value for the enumeration token used in the request to return the next batch of the results is not
     *         valid.
     * @throws ValidationException
     *         One or more parameter values are not valid.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws CodestarNotificationsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample CodestarNotificationsClient.ListEventTypes
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/codestar-notifications-2019-10-15/ListEventTypes"
     *      target="_top">AWS API Documentation</a>
     */
    default ListEventTypesResponse listEventTypes(Consumer<ListEventTypesRequest.Builder> listEventTypesRequest)
            throws InvalidNextTokenException, ValidationException, AwsServiceException, SdkClientException,
            CodestarNotificationsException {
        return listEventTypes(ListEventTypesRequest.builder().applyMutation(listEventTypesRequest).build());
    }

    /**
     * <p>
     * This is a variant of
     * {@link #listEventTypes(software.amazon.awssdk.services.codestarnotifications.model.ListEventTypesRequest)}
     * operation. The return type is a custom iterable that can be used to iterate through all the pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.codestarnotifications.paginators.ListEventTypesIterable responses = client.listEventTypesPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.codestarnotifications.paginators.ListEventTypesIterable responses = client
     *             .listEventTypesPaginator(request);
     *     for (software.amazon.awssdk.services.codestarnotifications.model.ListEventTypesResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.codestarnotifications.paginators.ListEventTypesIterable responses = client.listEventTypesPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listEventTypes(software.amazon.awssdk.services.codestarnotifications.model.ListEventTypesRequest)}
     * operation.</b>
     * </p>
     *
     * @param listEventTypesRequest
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws InvalidNextTokenException
     *         The value for the enumeration token used in the request to return the next batch of the results is not
     *         valid.
     * @throws ValidationException
     *         One or more parameter values are not valid.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws CodestarNotificationsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample CodestarNotificationsClient.ListEventTypes
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/codestar-notifications-2019-10-15/ListEventTypes"
     *      target="_top">AWS API Documentation</a>
     */
    default ListEventTypesIterable listEventTypesPaginator(ListEventTypesRequest listEventTypesRequest)
            throws InvalidNextTokenException, ValidationException, AwsServiceException, SdkClientException,
            CodestarNotificationsException {
        return new ListEventTypesIterable(this, listEventTypesRequest);
    }

    /**
     * <p>
     * This is a variant of
     * {@link #listEventTypes(software.amazon.awssdk.services.codestarnotifications.model.ListEventTypesRequest)}
     * operation. The return type is a custom iterable that can be used to iterate through all the pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.codestarnotifications.paginators.ListEventTypesIterable responses = client.listEventTypesPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.codestarnotifications.paginators.ListEventTypesIterable responses = client
     *             .listEventTypesPaginator(request);
     *     for (software.amazon.awssdk.services.codestarnotifications.model.ListEventTypesResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.codestarnotifications.paginators.ListEventTypesIterable responses = client.listEventTypesPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listEventTypes(software.amazon.awssdk.services.codestarnotifications.model.ListEventTypesRequest)}
     * operation.</b>
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link ListEventTypesRequest.Builder} avoiding the need to
     * create one manually via {@link ListEventTypesRequest#builder()}
     * </p>
     *
     * @param listEventTypesRequest
     *        A {@link Consumer} that will call methods on
     *        {@link software.amazon.awssdk.services.codestarnotifications.model.ListEventTypesRequest.Builder} to
     *        create a request.
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws InvalidNextTokenException
     *         The value for the enumeration token used in the request to return the next batch of the results is not
     *         valid.
     * @throws ValidationException
     *         One or more parameter values are not valid.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws CodestarNotificationsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample CodestarNotificationsClient.ListEventTypes
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/codestar-notifications-2019-10-15/ListEventTypes"
     *      target="_top">AWS API Documentation</a>
     */
    default ListEventTypesIterable listEventTypesPaginator(Consumer<ListEventTypesRequest.Builder> listEventTypesRequest)
            throws InvalidNextTokenException, ValidationException, AwsServiceException, SdkClientException,
            CodestarNotificationsException {
        return listEventTypesPaginator(ListEventTypesRequest.builder().applyMutation(listEventTypesRequest).build());
    }

    /**
     * <p>
     * Returns a list of the notification rules for an Amazon Web Services account.
     * </p>
     *
     * @param listNotificationRulesRequest
     * @return Result of the ListNotificationRules operation returned by the service.
     * @throws InvalidNextTokenException
     *         The value for the enumeration token used in the request to return the next batch of the results is not
     *         valid.
     * @throws ValidationException
     *         One or more parameter values are not valid.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws CodestarNotificationsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample CodestarNotificationsClient.ListNotificationRules
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/codestar-notifications-2019-10-15/ListNotificationRules"
     *      target="_top">AWS API Documentation</a>
     */
    default ListNotificationRulesResponse listNotificationRules(ListNotificationRulesRequest listNotificationRulesRequest)
            throws InvalidNextTokenException, ValidationException, AwsServiceException, SdkClientException,
            CodestarNotificationsException {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Returns a list of the notification rules for an Amazon Web Services account.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link ListNotificationRulesRequest.Builder} avoiding the
     * need to create one manually via {@link ListNotificationRulesRequest#builder()}
     * </p>
     *
     * @param listNotificationRulesRequest
     *        A {@link Consumer} that will call methods on
     *        {@link software.amazon.awssdk.services.codestarnotifications.model.ListNotificationRulesRequest.Builder}
     *        to create a request.
     * @return Result of the ListNotificationRules operation returned by the service.
     * @throws InvalidNextTokenException
     *         The value for the enumeration token used in the request to return the next batch of the results is not
     *         valid.
     * @throws ValidationException
     *         One or more parameter values are not valid.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws CodestarNotificationsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample CodestarNotificationsClient.ListNotificationRules
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/codestar-notifications-2019-10-15/ListNotificationRules"
     *      target="_top">AWS API Documentation</a>
     */
    default ListNotificationRulesResponse listNotificationRules(
            Consumer<ListNotificationRulesRequest.Builder> listNotificationRulesRequest) throws InvalidNextTokenException,
            ValidationException, AwsServiceException, SdkClientException, CodestarNotificationsException {
        return listNotificationRules(ListNotificationRulesRequest.builder().applyMutation(listNotificationRulesRequest).build());
    }

    /**
     * <p>
     * This is a variant of
     * {@link #listNotificationRules(software.amazon.awssdk.services.codestarnotifications.model.ListNotificationRulesRequest)}
     * operation. The return type is a custom iterable that can be used to iterate through all the pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.codestarnotifications.paginators.ListNotificationRulesIterable responses = client.listNotificationRulesPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.codestarnotifications.paginators.ListNotificationRulesIterable responses = client
     *             .listNotificationRulesPaginator(request);
     *     for (software.amazon.awssdk.services.codestarnotifications.model.ListNotificationRulesResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.codestarnotifications.paginators.ListNotificationRulesIterable responses = client.listNotificationRulesPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listNotificationRules(software.amazon.awssdk.services.codestarnotifications.model.ListNotificationRulesRequest)}
     * operation.</b>
     * </p>
     *
     * @param listNotificationRulesRequest
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws InvalidNextTokenException
     *         The value for the enumeration token used in the request to return the next batch of the results is not
     *         valid.
     * @throws ValidationException
     *         One or more parameter values are not valid.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws CodestarNotificationsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample CodestarNotificationsClient.ListNotificationRules
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/codestar-notifications-2019-10-15/ListNotificationRules"
     *      target="_top">AWS API Documentation</a>
     */
    default ListNotificationRulesIterable listNotificationRulesPaginator(ListNotificationRulesRequest listNotificationRulesRequest)
            throws InvalidNextTokenException, ValidationException, AwsServiceException, SdkClientException,
            CodestarNotificationsException {
        return new ListNotificationRulesIterable(this, listNotificationRulesRequest);
    }

    /**
     * <p>
     * This is a variant of
     * {@link #listNotificationRules(software.amazon.awssdk.services.codestarnotifications.model.ListNotificationRulesRequest)}
     * operation. The return type is a custom iterable that can be used to iterate through all the pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.codestarnotifications.paginators.ListNotificationRulesIterable responses = client.listNotificationRulesPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.codestarnotifications.paginators.ListNotificationRulesIterable responses = client
     *             .listNotificationRulesPaginator(request);
     *     for (software.amazon.awssdk.services.codestarnotifications.model.ListNotificationRulesResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.codestarnotifications.paginators.ListNotificationRulesIterable responses = client.listNotificationRulesPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listNotificationRules(software.amazon.awssdk.services.codestarnotifications.model.ListNotificationRulesRequest)}
     * operation.</b>
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link ListNotificationRulesRequest.Builder} avoiding the
     * need to create one manually via {@link ListNotificationRulesRequest#builder()}
     * </p>
     *
     * @param listNotificationRulesRequest
     *        A {@link Consumer} that will call methods on
     *        {@link software.amazon.awssdk.services.codestarnotifications.model.ListNotificationRulesRequest.Builder}
     *        to create a request.
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws InvalidNextTokenException
     *         The value for the enumeration token used in the request to return the next batch of the results is not
     *         valid.
     * @throws ValidationException
     *         One or more parameter values are not valid.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws CodestarNotificationsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample CodestarNotificationsClient.ListNotificationRules
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/codestar-notifications-2019-10-15/ListNotificationRules"
     *      target="_top">AWS API Documentation</a>
     */
    default ListNotificationRulesIterable listNotificationRulesPaginator(
            Consumer<ListNotificationRulesRequest.Builder> listNotificationRulesRequest) throws InvalidNextTokenException,
            ValidationException, AwsServiceException, SdkClientException, CodestarNotificationsException {
        return listNotificationRulesPaginator(ListNotificationRulesRequest.builder().applyMutation(listNotificationRulesRequest)
                .build());
    }

    /**
     * <p>
     * Returns a list of the tags associated with a notification rule.
     * </p>
     *
     * @param listTagsForResourceRequest
     * @return Result of the ListTagsForResource operation returned by the service.
     * @throws ResourceNotFoundException
     *         CodeStar Notifications can't find a resource that matches the provided ARN.
     * @throws ValidationException
     *         One or more parameter values are not valid.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws CodestarNotificationsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample CodestarNotificationsClient.ListTagsForResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/codestar-notifications-2019-10-15/ListTagsForResource"
     *      target="_top">AWS API Documentation</a>
     */
    default ListTagsForResourceResponse listTagsForResource(ListTagsForResourceRequest listTagsForResourceRequest)
            throws ResourceNotFoundException, ValidationException, AwsServiceException, SdkClientException,
            CodestarNotificationsException {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Returns a list of the tags associated with a notification rule.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link ListTagsForResourceRequest.Builder} avoiding the
     * need to create one manually via {@link ListTagsForResourceRequest#builder()}
     * </p>
     *
     * @param listTagsForResourceRequest
     *        A {@link Consumer} that will call methods on
     *        {@link software.amazon.awssdk.services.codestarnotifications.model.ListTagsForResourceRequest.Builder} to
     *        create a request.
     * @return Result of the ListTagsForResource operation returned by the service.
     * @throws ResourceNotFoundException
     *         CodeStar Notifications can't find a resource that matches the provided ARN.
     * @throws ValidationException
     *         One or more parameter values are not valid.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws CodestarNotificationsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample CodestarNotificationsClient.ListTagsForResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/codestar-notifications-2019-10-15/ListTagsForResource"
     *      target="_top">AWS API Documentation</a>
     */
    default ListTagsForResourceResponse listTagsForResource(
            Consumer<ListTagsForResourceRequest.Builder> listTagsForResourceRequest) throws ResourceNotFoundException,
            ValidationException, AwsServiceException, SdkClientException, CodestarNotificationsException {
        return listTagsForResource(ListTagsForResourceRequest.builder().applyMutation(listTagsForResourceRequest).build());
    }

    /**
     * <p>
     * Returns a list of the notification rule targets for an Amazon Web Services account.
     * </p>
     *
     * @param listTargetsRequest
     * @return Result of the ListTargets operation returned by the service.
     * @throws InvalidNextTokenException
     *         The value for the enumeration token used in the request to return the next batch of the results is not
     *         valid.
     * @throws ValidationException
     *         One or more parameter values are not valid.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws CodestarNotificationsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample CodestarNotificationsClient.ListTargets
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/codestar-notifications-2019-10-15/ListTargets"
     *      target="_top">AWS API Documentation</a>
     */
    default ListTargetsResponse listTargets(ListTargetsRequest listTargetsRequest) throws InvalidNextTokenException,
            ValidationException, AwsServiceException, SdkClientException, CodestarNotificationsException {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Returns a list of the notification rule targets for an Amazon Web Services account.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link ListTargetsRequest.Builder} avoiding the need to
     * create one manually via {@link ListTargetsRequest#builder()}
     * </p>
     *
     * @param listTargetsRequest
     *        A {@link Consumer} that will call methods on
     *        {@link software.amazon.awssdk.services.codestarnotifications.model.ListTargetsRequest.Builder} to create a
     *        request.
     * @return Result of the ListTargets operation returned by the service.
     * @throws InvalidNextTokenException
     *         The value for the enumeration token used in the request to return the next batch of the results is not
     *         valid.
     * @throws ValidationException
     *         One or more parameter values are not valid.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws CodestarNotificationsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample CodestarNotificationsClient.ListTargets
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/codestar-notifications-2019-10-15/ListTargets"
     *      target="_top">AWS API Documentation</a>
     */
    default ListTargetsResponse listTargets(Consumer<ListTargetsRequest.Builder> listTargetsRequest)
            throws InvalidNextTokenException, ValidationException, AwsServiceException, SdkClientException,
            CodestarNotificationsException {
        return listTargets(ListTargetsRequest.builder().applyMutation(listTargetsRequest).build());
    }

    /**
     * <p>
     * This is a variant of
     * {@link #listTargets(software.amazon.awssdk.services.codestarnotifications.model.ListTargetsRequest)} operation.
     * The return type is a custom iterable that can be used to iterate through all the pages. SDK will internally
     * handle making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.codestarnotifications.paginators.ListTargetsIterable responses = client.listTargetsPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.codestarnotifications.paginators.ListTargetsIterable responses = client
     *             .listTargetsPaginator(request);
     *     for (software.amazon.awssdk.services.codestarnotifications.model.ListTargetsResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.codestarnotifications.paginators.ListTargetsIterable responses = client.listTargetsPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listTargets(software.amazon.awssdk.services.codestarnotifications.model.ListTargetsRequest)}
     * operation.</b>
     * </p>
     *
     * @param listTargetsRequest
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws InvalidNextTokenException
     *         The value for the enumeration token used in the request to return the next batch of the results is not
     *         valid.
     * @throws ValidationException
     *         One or more parameter values are not valid.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws CodestarNotificationsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample CodestarNotificationsClient.ListTargets
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/codestar-notifications-2019-10-15/ListTargets"
     *      target="_top">AWS API Documentation</a>
     */
    default ListTargetsIterable listTargetsPaginator(ListTargetsRequest listTargetsRequest) throws InvalidNextTokenException,
            ValidationException, AwsServiceException, SdkClientException, CodestarNotificationsException {
        return new ListTargetsIterable(this, listTargetsRequest);
    }

    /**
     * <p>
     * This is a variant of
     * {@link #listTargets(software.amazon.awssdk.services.codestarnotifications.model.ListTargetsRequest)} operation.
     * The return type is a custom iterable that can be used to iterate through all the pages. SDK will internally
     * handle making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.codestarnotifications.paginators.ListTargetsIterable responses = client.listTargetsPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.codestarnotifications.paginators.ListTargetsIterable responses = client
     *             .listTargetsPaginator(request);
     *     for (software.amazon.awssdk.services.codestarnotifications.model.ListTargetsResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.codestarnotifications.paginators.ListTargetsIterable responses = client.listTargetsPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listTargets(software.amazon.awssdk.services.codestarnotifications.model.ListTargetsRequest)}
     * operation.</b>
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link ListTargetsRequest.Builder} avoiding the need to
     * create one manually via {@link ListTargetsRequest#builder()}
     * </p>
     *
     * @param listTargetsRequest
     *        A {@link Consumer} that will call methods on
     *        {@link software.amazon.awssdk.services.codestarnotifications.model.ListTargetsRequest.Builder} to create a
     *        request.
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws InvalidNextTokenException
     *         The value for the enumeration token used in the request to return the next batch of the results is not
     *         valid.
     * @throws ValidationException
     *         One or more parameter values are not valid.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws CodestarNotificationsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample CodestarNotificationsClient.ListTargets
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/codestar-notifications-2019-10-15/ListTargets"
     *      target="_top">AWS API Documentation</a>
     */
    default ListTargetsIterable listTargetsPaginator(Consumer<ListTargetsRequest.Builder> listTargetsRequest)
            throws InvalidNextTokenException, ValidationException, AwsServiceException, SdkClientException,
            CodestarNotificationsException {
        return listTargetsPaginator(ListTargetsRequest.builder().applyMutation(listTargetsRequest).build());
    }

    /**
     * <p>
     * Creates an association between a notification rule and an Amazon Q Developer in chat applications topic or Amazon
     * Q Developer in chat applications client so that the associated target can receive notifications when the events
     * described in the rule are triggered.
     * </p>
     *
     * @param subscribeRequest
     * @return Result of the Subscribe operation returned by the service.
     * @throws ValidationException
     *         One or more parameter values are not valid.
     * @throws ResourceNotFoundException
     *         CodeStar Notifications can't find a resource that matches the provided ARN.
     * @throws ConfigurationException
     *         Some or all of the configuration is incomplete, missing, or not valid.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws CodestarNotificationsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample CodestarNotificationsClient.Subscribe
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/codestar-notifications-2019-10-15/Subscribe"
     *      target="_top">AWS API Documentation</a>
     */
    default SubscribeResponse subscribe(SubscribeRequest subscribeRequest) throws ValidationException, ResourceNotFoundException,
            ConfigurationException, AwsServiceException, SdkClientException, CodestarNotificationsException {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Creates an association between a notification rule and an Amazon Q Developer in chat applications topic or Amazon
     * Q Developer in chat applications client so that the associated target can receive notifications when the events
     * described in the rule are triggered.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link SubscribeRequest.Builder} avoiding the need to
     * create one manually via {@link SubscribeRequest#builder()}
     * </p>
     *
     * @param subscribeRequest
     *        A {@link Consumer} that will call methods on
     *        {@link software.amazon.awssdk.services.codestarnotifications.model.SubscribeRequest.Builder} to create a
     *        request.
     * @return Result of the Subscribe operation returned by the service.
     * @throws ValidationException
     *         One or more parameter values are not valid.
     * @throws ResourceNotFoundException
     *         CodeStar Notifications can't find a resource that matches the provided ARN.
     * @throws ConfigurationException
     *         Some or all of the configuration is incomplete, missing, or not valid.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws CodestarNotificationsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample CodestarNotificationsClient.Subscribe
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/codestar-notifications-2019-10-15/Subscribe"
     *      target="_top">AWS API Documentation</a>
     */
    default SubscribeResponse subscribe(Consumer<SubscribeRequest.Builder> subscribeRequest) throws ValidationException,
            ResourceNotFoundException, ConfigurationException, AwsServiceException, SdkClientException,
            CodestarNotificationsException {
        return subscribe(SubscribeRequest.builder().applyMutation(subscribeRequest).build());
    }

    /**
     * <p>
     * Associates a set of provided tags with a notification rule.
     * </p>
     *
     * @param tagResourceRequest
     * @return Result of the TagResource operation returned by the service.
     * @throws ResourceNotFoundException
     *         CodeStar Notifications can't find a resource that matches the provided ARN.
     * @throws LimitExceededException
     *         One of the CodeStar Notifications limits has been exceeded. Limits apply to accounts, notification rules,
     *         notifications, resources, and targets. For more information, see Limits.
     * @throws ValidationException
     *         One or more parameter values are not valid.
     * @throws ConcurrentModificationException
     *         CodeStar Notifications can't complete the request because the resource is being modified by another
     *         process. Wait a few minutes and try again.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws CodestarNotificationsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample CodestarNotificationsClient.TagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/codestar-notifications-2019-10-15/TagResource"
     *      target="_top">AWS API Documentation</a>
     */
    default TagResourceResponse tagResource(TagResourceRequest tagResourceRequest) throws ResourceNotFoundException,
            LimitExceededException, ValidationException, ConcurrentModificationException, AwsServiceException,
            SdkClientException, CodestarNotificationsException {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Associates a set of provided tags with a notification rule.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link TagResourceRequest.Builder} avoiding the need to
     * create one manually via {@link TagResourceRequest#builder()}
     * </p>
     *
     * @param tagResourceRequest
     *        A {@link Consumer} that will call methods on
     *        {@link software.amazon.awssdk.services.codestarnotifications.model.TagResourceRequest.Builder} to create a
     *        request.
     * @return Result of the TagResource operation returned by the service.
     * @throws ResourceNotFoundException
     *         CodeStar Notifications can't find a resource that matches the provided ARN.
     * @throws LimitExceededException
     *         One of the CodeStar Notifications limits has been exceeded. Limits apply to accounts, notification rules,
     *         notifications, resources, and targets. For more information, see Limits.
     * @throws ValidationException
     *         One or more parameter values are not valid.
     * @throws ConcurrentModificationException
     *         CodeStar Notifications can't complete the request because the resource is being modified by another
     *         process. Wait a few minutes and try again.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws CodestarNotificationsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample CodestarNotificationsClient.TagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/codestar-notifications-2019-10-15/TagResource"
     *      target="_top">AWS API Documentation</a>
     */
    default TagResourceResponse tagResource(Consumer<TagResourceRequest.Builder> tagResourceRequest)
            throws ResourceNotFoundException, LimitExceededException, ValidationException, ConcurrentModificationException,
            AwsServiceException, SdkClientException, CodestarNotificationsException {
        return tagResource(TagResourceRequest.builder().applyMutation(tagResourceRequest).build());
    }

    /**
     * <p>
     * Removes an association between a notification rule and an Amazon Q Developer in chat applications topic so that
     * subscribers to that topic stop receiving notifications when the events described in the rule are triggered.
     * </p>
     *
     * @param unsubscribeRequest
     * @return Result of the Unsubscribe operation returned by the service.
     * @throws ValidationException
     *         One or more parameter values are not valid.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws CodestarNotificationsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample CodestarNotificationsClient.Unsubscribe
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/codestar-notifications-2019-10-15/Unsubscribe"
     *      target="_top">AWS API Documentation</a>
     */
    default UnsubscribeResponse unsubscribe(UnsubscribeRequest unsubscribeRequest) throws ValidationException,
            AwsServiceException, SdkClientException, CodestarNotificationsException {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Removes an association between a notification rule and an Amazon Q Developer in chat applications topic so that
     * subscribers to that topic stop receiving notifications when the events described in the rule are triggered.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link UnsubscribeRequest.Builder} avoiding the need to
     * create one manually via {@link UnsubscribeRequest#builder()}
     * </p>
     *
     * @param unsubscribeRequest
     *        A {@link Consumer} that will call methods on
     *        {@link software.amazon.awssdk.services.codestarnotifications.model.UnsubscribeRequest.Builder} to create a
     *        request.
     * @return Result of the Unsubscribe operation returned by the service.
     * @throws ValidationException
     *         One or more parameter values are not valid.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws CodestarNotificationsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample CodestarNotificationsClient.Unsubscribe
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/codestar-notifications-2019-10-15/Unsubscribe"
     *      target="_top">AWS API Documentation</a>
     */
    default UnsubscribeResponse unsubscribe(Consumer<UnsubscribeRequest.Builder> unsubscribeRequest) throws ValidationException,
            AwsServiceException, SdkClientException, CodestarNotificationsException {
        return unsubscribe(UnsubscribeRequest.builder().applyMutation(unsubscribeRequest).build());
    }

    /**
     * <p>
     * Removes the association between one or more provided tags and a notification rule.
     * </p>
     *
     * @param untagResourceRequest
     * @return Result of the UntagResource operation returned by the service.
     * @throws ResourceNotFoundException
     *         CodeStar Notifications can't find a resource that matches the provided ARN.
     * @throws LimitExceededException
     *         One of the CodeStar Notifications limits has been exceeded. Limits apply to accounts, notification rules,
     *         notifications, resources, and targets. For more information, see Limits.
     * @throws ValidationException
     *         One or more parameter values are not valid.
     * @throws ConcurrentModificationException
     *         CodeStar Notifications can't complete the request because the resource is being modified by another
     *         process. Wait a few minutes and try again.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws CodestarNotificationsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample CodestarNotificationsClient.UntagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/codestar-notifications-2019-10-15/UntagResource"
     *      target="_top">AWS API Documentation</a>
     */
    default UntagResourceResponse untagResource(UntagResourceRequest untagResourceRequest) throws ResourceNotFoundException,
            LimitExceededException, ValidationException, ConcurrentModificationException, AwsServiceException,
            SdkClientException, CodestarNotificationsException {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Removes the association between one or more provided tags and a notification rule.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link UntagResourceRequest.Builder} avoiding the need to
     * create one manually via {@link UntagResourceRequest#builder()}
     * </p>
     *
     * @param untagResourceRequest
     *        A {@link Consumer} that will call methods on
     *        {@link software.amazon.awssdk.services.codestarnotifications.model.UntagResourceRequest.Builder} to create
     *        a request.
     * @return Result of the UntagResource operation returned by the service.
     * @throws ResourceNotFoundException
     *         CodeStar Notifications can't find a resource that matches the provided ARN.
     * @throws LimitExceededException
     *         One of the CodeStar Notifications limits has been exceeded. Limits apply to accounts, notification rules,
     *         notifications, resources, and targets. For more information, see Limits.
     * @throws ValidationException
     *         One or more parameter values are not valid.
     * @throws ConcurrentModificationException
     *         CodeStar Notifications can't complete the request because the resource is being modified by another
     *         process. Wait a few minutes and try again.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws CodestarNotificationsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample CodestarNotificationsClient.UntagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/codestar-notifications-2019-10-15/UntagResource"
     *      target="_top">AWS API Documentation</a>
     */
    default UntagResourceResponse untagResource(Consumer<UntagResourceRequest.Builder> untagResourceRequest)
            throws ResourceNotFoundException, LimitExceededException, ValidationException, ConcurrentModificationException,
            AwsServiceException, SdkClientException, CodestarNotificationsException {
        return untagResource(UntagResourceRequest.builder().applyMutation(untagResourceRequest).build());
    }

    /**
     * <p>
     * Updates a notification rule for a resource. You can change the events that trigger the notification rule, the
     * status of the rule, and the targets that receive the notifications.
     * </p>
     * <note>
     * <p>
     * To add or remove tags for a notification rule, you must use <a>TagResource</a> and <a>UntagResource</a>.
     * </p>
     * </note>
     *
     * @param updateNotificationRuleRequest
     * @return Result of the UpdateNotificationRule operation returned by the service.
     * @throws ValidationException
     *         One or more parameter values are not valid.
     * @throws ResourceNotFoundException
     *         CodeStar Notifications can't find a resource that matches the provided ARN.
     * @throws ConfigurationException
     *         Some or all of the configuration is incomplete, missing, or not valid.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws CodestarNotificationsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample CodestarNotificationsClient.UpdateNotificationRule
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/codestar-notifications-2019-10-15/UpdateNotificationRule"
     *      target="_top">AWS API Documentation</a>
     */
    default UpdateNotificationRuleResponse updateNotificationRule(UpdateNotificationRuleRequest updateNotificationRuleRequest)
            throws ValidationException, ResourceNotFoundException, ConfigurationException, AwsServiceException,
            SdkClientException, CodestarNotificationsException {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Updates a notification rule for a resource. You can change the events that trigger the notification rule, the
     * status of the rule, and the targets that receive the notifications.
     * </p>
     * <note>
     * <p>
     * To add or remove tags for a notification rule, you must use <a>TagResource</a> and <a>UntagResource</a>.
     * </p>
     * </note><br/>
     * <p>
     * This is a convenience which creates an instance of the {@link UpdateNotificationRuleRequest.Builder} avoiding the
     * need to create one manually via {@link UpdateNotificationRuleRequest#builder()}
     * </p>
     *
     * @param updateNotificationRuleRequest
     *        A {@link Consumer} that will call methods on
     *        {@link software.amazon.awssdk.services.codestarnotifications.model.UpdateNotificationRuleRequest.Builder}
     *        to create a request.
     * @return Result of the UpdateNotificationRule operation returned by the service.
     * @throws ValidationException
     *         One or more parameter values are not valid.
     * @throws ResourceNotFoundException
     *         CodeStar Notifications can't find a resource that matches the provided ARN.
     * @throws ConfigurationException
     *         Some or all of the configuration is incomplete, missing, or not valid.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws CodestarNotificationsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample CodestarNotificationsClient.UpdateNotificationRule
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/codestar-notifications-2019-10-15/UpdateNotificationRule"
     *      target="_top">AWS API Documentation</a>
     */
    default UpdateNotificationRuleResponse updateNotificationRule(
            Consumer<UpdateNotificationRuleRequest.Builder> updateNotificationRuleRequest) throws ValidationException,
            ResourceNotFoundException, ConfigurationException, AwsServiceException, SdkClientException,
            CodestarNotificationsException {
        return updateNotificationRule(UpdateNotificationRuleRequest.builder().applyMutation(updateNotificationRuleRequest)
                .build());
    }

    /**
     * Create a {@link CodestarNotificationsClient} with the region loaded from the
     * {@link software.amazon.awssdk.regions.providers.DefaultAwsRegionProviderChain} and credentials loaded from the
     * {@link software.amazon.awssdk.auth.credentials.DefaultCredentialsProvider}.
     */
    static CodestarNotificationsClient create() {
        return builder().build();
    }

    /**
     * Create a builder that can be used to configure and create a {@link CodestarNotificationsClient}.
     */
    static CodestarNotificationsClientBuilder builder() {
        return new DefaultCodestarNotificationsClientBuilder();
    }

    static ServiceMetadata serviceMetadata() {
        return ServiceMetadata.of(SERVICE_METADATA_ID);
    }

    @Override
    default CodestarNotificationsServiceClientConfiguration serviceClientConfiguration() {
        throw new UnsupportedOperationException();
    }
}
