/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.codegurureviewer.model;

import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.awscore.AwsRequestOverrideConfiguration;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.DefaultValueTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.traits.MapTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructMap;
import software.amazon.awssdk.core.util.SdkAutoConstructMap;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 */
@Generated("software.amazon.awssdk:codegen")
public final class AssociateRepositoryRequest extends CodeGuruReviewerRequest implements
        ToCopyableBuilder<AssociateRepositoryRequest.Builder, AssociateRepositoryRequest> {
    private static final SdkField<Repository> REPOSITORY_FIELD = SdkField.<Repository> builder(MarshallingType.SDK_POJO)
            .memberName("Repository").getter(getter(AssociateRepositoryRequest::repository)).setter(setter(Builder::repository))
            .constructor(Repository::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Repository").build()).build();

    private static final SdkField<String> CLIENT_REQUEST_TOKEN_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("ClientRequestToken")
            .getter(getter(AssociateRepositoryRequest::clientRequestToken))
            .setter(setter(Builder::clientRequestToken))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ClientRequestToken").build(),
                    DefaultValueTrait.idempotencyToken()).build();

    private static final SdkField<Map<String, String>> TAGS_FIELD = SdkField
            .<Map<String, String>> builder(MarshallingType.MAP)
            .memberName("Tags")
            .getter(getter(AssociateRepositoryRequest::tags))
            .setter(setter(Builder::tags))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Tags").build(),
                    MapTrait.builder()
                            .keyLocationName("key")
                            .valueLocationName("value")
                            .valueFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("value").build()).build()).build()).build();

    private static final SdkField<KMSKeyDetails> KMS_KEY_DETAILS_FIELD = SdkField
            .<KMSKeyDetails> builder(MarshallingType.SDK_POJO).memberName("KMSKeyDetails")
            .getter(getter(AssociateRepositoryRequest::kmsKeyDetails)).setter(setter(Builder::kmsKeyDetails))
            .constructor(KMSKeyDetails::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("KMSKeyDetails").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(REPOSITORY_FIELD,
            CLIENT_REQUEST_TOKEN_FIELD, TAGS_FIELD, KMS_KEY_DETAILS_FIELD));

    private final Repository repository;

    private final String clientRequestToken;

    private final Map<String, String> tags;

    private final KMSKeyDetails kmsKeyDetails;

    private AssociateRepositoryRequest(BuilderImpl builder) {
        super(builder);
        this.repository = builder.repository;
        this.clientRequestToken = builder.clientRequestToken;
        this.tags = builder.tags;
        this.kmsKeyDetails = builder.kmsKeyDetails;
    }

    /**
     * <p>
     * The repository to associate.
     * </p>
     * 
     * @return The repository to associate.
     */
    public final Repository repository() {
        return repository;
    }

    /**
     * <p>
     * Amazon CodeGuru Reviewer uses this value to prevent the accidental creation of duplicate repository associations
     * if there are failures and retries.
     * </p>
     * 
     * @return Amazon CodeGuru Reviewer uses this value to prevent the accidental creation of duplicate repository
     *         associations if there are failures and retries.
     */
    public final String clientRequestToken() {
        return clientRequestToken;
    }

    /**
     * For responses, this returns true if the service returned a value for the Tags property. This DOES NOT check that
     * the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is useful
     * because the SDK will never return a null collection or map, but you may need to differentiate between the service
     * returning nothing (or null) and the service returning an empty collection or map. For requests, this returns true
     * if a value for the property was specified in the request builder, and false if a value was not specified.
     */
    public final boolean hasTags() {
        return tags != null && !(tags instanceof SdkAutoConstructMap);
    }

    /**
     * <p>
     * An array of key-value pairs used to tag an associated repository. A tag is a custom attribute label with two
     * parts:
     * </p>
     * <ul>
     * <li>
     * <p>
     * A <i>tag key</i> (for example, <code>CostCenter</code>, <code>Environment</code>, <code>Project</code>, or
     * <code>Secret</code>). Tag keys are case sensitive.
     * </p>
     * </li>
     * <li>
     * <p>
     * An optional field known as a <i>tag value</i> (for example, <code>111122223333</code>, <code>Production</code>,
     * or a team name). Omitting the tag value is the same as using an empty string. Like tag keys, tag values are case
     * sensitive.
     * </p>
     * </li>
     * </ul>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasTags} method.
     * </p>
     * 
     * @return An array of key-value pairs used to tag an associated repository. A tag is a custom attribute label with
     *         two parts:</p>
     *         <ul>
     *         <li>
     *         <p>
     *         A <i>tag key</i> (for example, <code>CostCenter</code>, <code>Environment</code>, <code>Project</code>,
     *         or <code>Secret</code>). Tag keys are case sensitive.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         An optional field known as a <i>tag value</i> (for example, <code>111122223333</code>,
     *         <code>Production</code>, or a team name). Omitting the tag value is the same as using an empty string.
     *         Like tag keys, tag values are case sensitive.
     *         </p>
     *         </li>
     */
    public final Map<String, String> tags() {
        return tags;
    }

    /**
     * <p>
     * A <code>KMSKeyDetails</code> object that contains:
     * </p>
     * <ul>
     * <li>
     * <p>
     * The encryption option for this repository association. It is either owned by Amazon Web Services Key Management
     * Service (KMS) (<code>AWS_OWNED_CMK</code>) or customer managed (<code>CUSTOMER_MANAGED_CMK</code>).
     * </p>
     * </li>
     * <li>
     * <p>
     * The ID of the Amazon Web Services KMS key that is associated with this repository association.
     * </p>
     * </li>
     * </ul>
     * 
     * @return A <code>KMSKeyDetails</code> object that contains:</p>
     *         <ul>
     *         <li>
     *         <p>
     *         The encryption option for this repository association. It is either owned by Amazon Web Services Key
     *         Management Service (KMS) (<code>AWS_OWNED_CMK</code>) or customer managed (
     *         <code>CUSTOMER_MANAGED_CMK</code>).
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         The ID of the Amazon Web Services KMS key that is associated with this repository association.
     *         </p>
     *         </li>
     */
    public final KMSKeyDetails kmsKeyDetails() {
        return kmsKeyDetails;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + super.hashCode();
        hashCode = 31 * hashCode + Objects.hashCode(repository());
        hashCode = 31 * hashCode + Objects.hashCode(clientRequestToken());
        hashCode = 31 * hashCode + Objects.hashCode(hasTags() ? tags() : null);
        hashCode = 31 * hashCode + Objects.hashCode(kmsKeyDetails());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return super.equals(obj) && equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof AssociateRepositoryRequest)) {
            return false;
        }
        AssociateRepositoryRequest other = (AssociateRepositoryRequest) obj;
        return Objects.equals(repository(), other.repository())
                && Objects.equals(clientRequestToken(), other.clientRequestToken()) && hasTags() == other.hasTags()
                && Objects.equals(tags(), other.tags()) && Objects.equals(kmsKeyDetails(), other.kmsKeyDetails());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("AssociateRepositoryRequest").add("Repository", repository())
                .add("ClientRequestToken", clientRequestToken()).add("Tags", hasTags() ? tags() : null)
                .add("KMSKeyDetails", kmsKeyDetails()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "Repository":
            return Optional.ofNullable(clazz.cast(repository()));
        case "ClientRequestToken":
            return Optional.ofNullable(clazz.cast(clientRequestToken()));
        case "Tags":
            return Optional.ofNullable(clazz.cast(tags()));
        case "KMSKeyDetails":
            return Optional.ofNullable(clazz.cast(kmsKeyDetails()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<AssociateRepositoryRequest, T> g) {
        return obj -> g.apply((AssociateRepositoryRequest) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends CodeGuruReviewerRequest.Builder, SdkPojo,
            CopyableBuilder<Builder, AssociateRepositoryRequest> {
        /**
         * <p>
         * The repository to associate.
         * </p>
         * 
         * @param repository
         *        The repository to associate.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder repository(Repository repository);

        /**
         * <p>
         * The repository to associate.
         * </p>
         * This is a convenience method that creates an instance of the {@link Repository.Builder} avoiding the need to
         * create one manually via {@link Repository#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link Repository.Builder#build()} is called immediately and its result
         * is passed to {@link #repository(Repository)}.
         * 
         * @param repository
         *        a consumer that will call methods on {@link Repository.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #repository(Repository)
         */
        default Builder repository(Consumer<Repository.Builder> repository) {
            return repository(Repository.builder().applyMutation(repository).build());
        }

        /**
         * <p>
         * Amazon CodeGuru Reviewer uses this value to prevent the accidental creation of duplicate repository
         * associations if there are failures and retries.
         * </p>
         * 
         * @param clientRequestToken
         *        Amazon CodeGuru Reviewer uses this value to prevent the accidental creation of duplicate repository
         *        associations if there are failures and retries.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder clientRequestToken(String clientRequestToken);

        /**
         * <p>
         * An array of key-value pairs used to tag an associated repository. A tag is a custom attribute label with two
         * parts:
         * </p>
         * <ul>
         * <li>
         * <p>
         * A <i>tag key</i> (for example, <code>CostCenter</code>, <code>Environment</code>, <code>Project</code>, or
         * <code>Secret</code>). Tag keys are case sensitive.
         * </p>
         * </li>
         * <li>
         * <p>
         * An optional field known as a <i>tag value</i> (for example, <code>111122223333</code>,
         * <code>Production</code>, or a team name). Omitting the tag value is the same as using an empty string. Like
         * tag keys, tag values are case sensitive.
         * </p>
         * </li>
         * </ul>
         * 
         * @param tags
         *        An array of key-value pairs used to tag an associated repository. A tag is a custom attribute label
         *        with two parts:</p>
         *        <ul>
         *        <li>
         *        <p>
         *        A <i>tag key</i> (for example, <code>CostCenter</code>, <code>Environment</code>, <code>Project</code>
         *        , or <code>Secret</code>). Tag keys are case sensitive.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        An optional field known as a <i>tag value</i> (for example, <code>111122223333</code>,
         *        <code>Production</code>, or a team name). Omitting the tag value is the same as using an empty string.
         *        Like tag keys, tag values are case sensitive.
         *        </p>
         *        </li>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder tags(Map<String, String> tags);

        /**
         * <p>
         * A <code>KMSKeyDetails</code> object that contains:
         * </p>
         * <ul>
         * <li>
         * <p>
         * The encryption option for this repository association. It is either owned by Amazon Web Services Key
         * Management Service (KMS) (<code>AWS_OWNED_CMK</code>) or customer managed (<code>CUSTOMER_MANAGED_CMK</code>
         * ).
         * </p>
         * </li>
         * <li>
         * <p>
         * The ID of the Amazon Web Services KMS key that is associated with this repository association.
         * </p>
         * </li>
         * </ul>
         * 
         * @param kmsKeyDetails
         *        A <code>KMSKeyDetails</code> object that contains:</p>
         *        <ul>
         *        <li>
         *        <p>
         *        The encryption option for this repository association. It is either owned by Amazon Web Services Key
         *        Management Service (KMS) (<code>AWS_OWNED_CMK</code>) or customer managed (
         *        <code>CUSTOMER_MANAGED_CMK</code>).
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        The ID of the Amazon Web Services KMS key that is associated with this repository association.
         *        </p>
         *        </li>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder kmsKeyDetails(KMSKeyDetails kmsKeyDetails);

        /**
         * <p>
         * A <code>KMSKeyDetails</code> object that contains:
         * </p>
         * <ul>
         * <li>
         * <p>
         * The encryption option for this repository association. It is either owned by Amazon Web Services Key
         * Management Service (KMS) (<code>AWS_OWNED_CMK</code>) or customer managed (<code>CUSTOMER_MANAGED_CMK</code>
         * ).
         * </p>
         * </li>
         * <li>
         * <p>
         * The ID of the Amazon Web Services KMS key that is associated with this repository association.
         * </p>
         * </li>
         * </ul>
         * This is a convenience method that creates an instance of the {@link KMSKeyDetails.Builder} avoiding the need
         * to create one manually via {@link KMSKeyDetails#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link KMSKeyDetails.Builder#build()} is called immediately and its
         * result is passed to {@link #kmsKeyDetails(KMSKeyDetails)}.
         * 
         * @param kmsKeyDetails
         *        a consumer that will call methods on {@link KMSKeyDetails.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #kmsKeyDetails(KMSKeyDetails)
         */
        default Builder kmsKeyDetails(Consumer<KMSKeyDetails.Builder> kmsKeyDetails) {
            return kmsKeyDetails(KMSKeyDetails.builder().applyMutation(kmsKeyDetails).build());
        }

        @Override
        Builder overrideConfiguration(AwsRequestOverrideConfiguration overrideConfiguration);

        @Override
        Builder overrideConfiguration(Consumer<AwsRequestOverrideConfiguration.Builder> builderConsumer);
    }

    static final class BuilderImpl extends CodeGuruReviewerRequest.BuilderImpl implements Builder {
        private Repository repository;

        private String clientRequestToken;

        private Map<String, String> tags = DefaultSdkAutoConstructMap.getInstance();

        private KMSKeyDetails kmsKeyDetails;

        private BuilderImpl() {
        }

        private BuilderImpl(AssociateRepositoryRequest model) {
            super(model);
            repository(model.repository);
            clientRequestToken(model.clientRequestToken);
            tags(model.tags);
            kmsKeyDetails(model.kmsKeyDetails);
        }

        public final Repository.Builder getRepository() {
            return repository != null ? repository.toBuilder() : null;
        }

        public final void setRepository(Repository.BuilderImpl repository) {
            this.repository = repository != null ? repository.build() : null;
        }

        @Override
        public final Builder repository(Repository repository) {
            this.repository = repository;
            return this;
        }

        public final String getClientRequestToken() {
            return clientRequestToken;
        }

        public final void setClientRequestToken(String clientRequestToken) {
            this.clientRequestToken = clientRequestToken;
        }

        @Override
        public final Builder clientRequestToken(String clientRequestToken) {
            this.clientRequestToken = clientRequestToken;
            return this;
        }

        public final Map<String, String> getTags() {
            if (tags instanceof SdkAutoConstructMap) {
                return null;
            }
            return tags;
        }

        public final void setTags(Map<String, String> tags) {
            this.tags = TagMapCopier.copy(tags);
        }

        @Override
        public final Builder tags(Map<String, String> tags) {
            this.tags = TagMapCopier.copy(tags);
            return this;
        }

        public final KMSKeyDetails.Builder getKmsKeyDetails() {
            return kmsKeyDetails != null ? kmsKeyDetails.toBuilder() : null;
        }

        public final void setKmsKeyDetails(KMSKeyDetails.BuilderImpl kmsKeyDetails) {
            this.kmsKeyDetails = kmsKeyDetails != null ? kmsKeyDetails.build() : null;
        }

        @Override
        public final Builder kmsKeyDetails(KMSKeyDetails kmsKeyDetails) {
            this.kmsKeyDetails = kmsKeyDetails;
            return this;
        }

        @Override
        public Builder overrideConfiguration(AwsRequestOverrideConfiguration overrideConfiguration) {
            super.overrideConfiguration(overrideConfiguration);
            return this;
        }

        @Override
        public Builder overrideConfiguration(Consumer<AwsRequestOverrideConfiguration.Builder> builderConsumer) {
            super.overrideConfiguration(builderConsumer);
            return this;
        }

        @Override
        public AssociateRepositoryRequest build() {
            return new AssociateRepositoryRequest(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
