/*
 * Copyright 2015-2020 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.codecommit.model;

import java.io.Serializable;
import java.nio.ByteBuffer;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkBytes;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.adapter.StandardMemberCopier;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Information about a file added or updated as part of a commit.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class PutFileEntry implements SdkPojo, Serializable, ToCopyableBuilder<PutFileEntry.Builder, PutFileEntry> {
    private static final SdkField<String> FILE_PATH_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(PutFileEntry::filePath)).setter(setter(Builder::filePath))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("filePath").build()).build();

    private static final SdkField<String> FILE_MODE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(PutFileEntry::fileModeAsString)).setter(setter(Builder::fileMode))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("fileMode").build()).build();

    private static final SdkField<SdkBytes> FILE_CONTENT_FIELD = SdkField.<SdkBytes> builder(MarshallingType.SDK_BYTES)
            .getter(getter(PutFileEntry::fileContent)).setter(setter(Builder::fileContent))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("fileContent").build()).build();

    private static final SdkField<SourceFileSpecifier> SOURCE_FILE_FIELD = SdkField
            .<SourceFileSpecifier> builder(MarshallingType.SDK_POJO).getter(getter(PutFileEntry::sourceFile))
            .setter(setter(Builder::sourceFile)).constructor(SourceFileSpecifier::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("sourceFile").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(FILE_PATH_FIELD,
            FILE_MODE_FIELD, FILE_CONTENT_FIELD, SOURCE_FILE_FIELD));

    private static final long serialVersionUID = 1L;

    private final String filePath;

    private final String fileMode;

    private final SdkBytes fileContent;

    private final SourceFileSpecifier sourceFile;

    private PutFileEntry(BuilderImpl builder) {
        this.filePath = builder.filePath;
        this.fileMode = builder.fileMode;
        this.fileContent = builder.fileContent;
        this.sourceFile = builder.sourceFile;
    }

    /**
     * <p>
     * The full path to the file in the repository, including the name of the file.
     * </p>
     * 
     * @return The full path to the file in the repository, including the name of the file.
     */
    public String filePath() {
        return filePath;
    }

    /**
     * <p>
     * The extrapolated file mode permissions for the file. Valid values include EXECUTABLE and NORMAL.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #fileMode} will
     * return {@link FileModeTypeEnum#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #fileModeAsString}.
     * </p>
     * 
     * @return The extrapolated file mode permissions for the file. Valid values include EXECUTABLE and NORMAL.
     * @see FileModeTypeEnum
     */
    public FileModeTypeEnum fileMode() {
        return FileModeTypeEnum.fromValue(fileMode);
    }

    /**
     * <p>
     * The extrapolated file mode permissions for the file. Valid values include EXECUTABLE and NORMAL.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #fileMode} will
     * return {@link FileModeTypeEnum#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #fileModeAsString}.
     * </p>
     * 
     * @return The extrapolated file mode permissions for the file. Valid values include EXECUTABLE and NORMAL.
     * @see FileModeTypeEnum
     */
    public String fileModeAsString() {
        return fileMode;
    }

    /**
     * <p>
     * The content of the file, if a source file is not specified.
     * </p>
     * 
     * @return The content of the file, if a source file is not specified.
     */
    public SdkBytes fileContent() {
        return fileContent;
    }

    /**
     * <p>
     * The name and full path of the file that contains the changes you want to make as part of the commit, if you are
     * not providing the file content directly.
     * </p>
     * 
     * @return The name and full path of the file that contains the changes you want to make as part of the commit, if
     *         you are not providing the file content directly.
     */
    public SourceFileSpecifier sourceFile() {
        return sourceFile;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(filePath());
        hashCode = 31 * hashCode + Objects.hashCode(fileModeAsString());
        hashCode = 31 * hashCode + Objects.hashCode(fileContent());
        hashCode = 31 * hashCode + Objects.hashCode(sourceFile());
        return hashCode;
    }

    @Override
    public boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof PutFileEntry)) {
            return false;
        }
        PutFileEntry other = (PutFileEntry) obj;
        return Objects.equals(filePath(), other.filePath()) && Objects.equals(fileModeAsString(), other.fileModeAsString())
                && Objects.equals(fileContent(), other.fileContent()) && Objects.equals(sourceFile(), other.sourceFile());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public String toString() {
        return ToString.builder("PutFileEntry").add("FilePath", filePath()).add("FileMode", fileModeAsString())
                .add("FileContent", fileContent()).add("SourceFile", sourceFile()).build();
    }

    public <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "filePath":
            return Optional.ofNullable(clazz.cast(filePath()));
        case "fileMode":
            return Optional.ofNullable(clazz.cast(fileModeAsString()));
        case "fileContent":
            return Optional.ofNullable(clazz.cast(fileContent()));
        case "sourceFile":
            return Optional.ofNullable(clazz.cast(sourceFile()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<PutFileEntry, T> g) {
        return obj -> g.apply((PutFileEntry) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, PutFileEntry> {
        /**
         * <p>
         * The full path to the file in the repository, including the name of the file.
         * </p>
         * 
         * @param filePath
         *        The full path to the file in the repository, including the name of the file.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder filePath(String filePath);

        /**
         * <p>
         * The extrapolated file mode permissions for the file. Valid values include EXECUTABLE and NORMAL.
         * </p>
         * 
         * @param fileMode
         *        The extrapolated file mode permissions for the file. Valid values include EXECUTABLE and NORMAL.
         * @see FileModeTypeEnum
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see FileModeTypeEnum
         */
        Builder fileMode(String fileMode);

        /**
         * <p>
         * The extrapolated file mode permissions for the file. Valid values include EXECUTABLE and NORMAL.
         * </p>
         * 
         * @param fileMode
         *        The extrapolated file mode permissions for the file. Valid values include EXECUTABLE and NORMAL.
         * @see FileModeTypeEnum
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see FileModeTypeEnum
         */
        Builder fileMode(FileModeTypeEnum fileMode);

        /**
         * <p>
         * The content of the file, if a source file is not specified.
         * </p>
         * 
         * @param fileContent
         *        The content of the file, if a source file is not specified.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder fileContent(SdkBytes fileContent);

        /**
         * <p>
         * The name and full path of the file that contains the changes you want to make as part of the commit, if you
         * are not providing the file content directly.
         * </p>
         * 
         * @param sourceFile
         *        The name and full path of the file that contains the changes you want to make as part of the commit,
         *        if you are not providing the file content directly.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder sourceFile(SourceFileSpecifier sourceFile);

        /**
         * <p>
         * The name and full path of the file that contains the changes you want to make as part of the commit, if you
         * are not providing the file content directly.
         * </p>
         * This is a convenience that creates an instance of the {@link SourceFileSpecifier.Builder} avoiding the need
         * to create one manually via {@link SourceFileSpecifier#builder()}.
         *
         * When the {@link Consumer} completes, {@link SourceFileSpecifier.Builder#build()} is called immediately and
         * its result is passed to {@link #sourceFile(SourceFileSpecifier)}.
         * 
         * @param sourceFile
         *        a consumer that will call methods on {@link SourceFileSpecifier.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #sourceFile(SourceFileSpecifier)
         */
        default Builder sourceFile(Consumer<SourceFileSpecifier.Builder> sourceFile) {
            return sourceFile(SourceFileSpecifier.builder().applyMutation(sourceFile).build());
        }
    }

    static final class BuilderImpl implements Builder {
        private String filePath;

        private String fileMode;

        private SdkBytes fileContent;

        private SourceFileSpecifier sourceFile;

        private BuilderImpl() {
        }

        private BuilderImpl(PutFileEntry model) {
            filePath(model.filePath);
            fileMode(model.fileMode);
            fileContent(model.fileContent);
            sourceFile(model.sourceFile);
        }

        public final String getFilePath() {
            return filePath;
        }

        @Override
        public final Builder filePath(String filePath) {
            this.filePath = filePath;
            return this;
        }

        public final void setFilePath(String filePath) {
            this.filePath = filePath;
        }

        public final String getFileModeAsString() {
            return fileMode;
        }

        @Override
        public final Builder fileMode(String fileMode) {
            this.fileMode = fileMode;
            return this;
        }

        @Override
        public final Builder fileMode(FileModeTypeEnum fileMode) {
            this.fileMode(fileMode == null ? null : fileMode.toString());
            return this;
        }

        public final void setFileMode(String fileMode) {
            this.fileMode = fileMode;
        }

        public final ByteBuffer getFileContent() {
            return fileContent == null ? null : fileContent.asByteBuffer();
        }

        @Override
        public final Builder fileContent(SdkBytes fileContent) {
            this.fileContent = StandardMemberCopier.copy(fileContent);
            return this;
        }

        public final void setFileContent(ByteBuffer fileContent) {
            fileContent(fileContent == null ? null : SdkBytes.fromByteBuffer(fileContent));
        }

        public final SourceFileSpecifier.Builder getSourceFile() {
            return sourceFile != null ? sourceFile.toBuilder() : null;
        }

        @Override
        public final Builder sourceFile(SourceFileSpecifier sourceFile) {
            this.sourceFile = sourceFile;
            return this;
        }

        public final void setSourceFile(SourceFileSpecifier.BuilderImpl sourceFile) {
            this.sourceFile = sourceFile != null ? sourceFile.build() : null;
        }

        @Override
        public PutFileEntry build() {
            return new PutFileEntry(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
