/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.codebuild.model;

import java.io.Serializable;
import java.time.Instant;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Information about a webhook that connects repository events to a build project in AWS CodeBuild.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class Webhook implements SdkPojo, Serializable, ToCopyableBuilder<Webhook.Builder, Webhook> {
    private static final SdkField<String> URL_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(Webhook::url)).setter(setter(Builder::url))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("url").build()).build();

    private static final SdkField<String> PAYLOAD_URL_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(Webhook::payloadUrl)).setter(setter(Builder::payloadUrl))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("payloadUrl").build()).build();

    private static final SdkField<String> SECRET_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(Webhook::secret)).setter(setter(Builder::secret))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("secret").build()).build();

    private static final SdkField<String> BRANCH_FILTER_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(Webhook::branchFilter)).setter(setter(Builder::branchFilter))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("branchFilter").build()).build();

    private static final SdkField<List<List<WebhookFilter>>> FILTER_GROUPS_FIELD = SdkField
            .<List<List<WebhookFilter>>> builder(MarshallingType.LIST)
            .getter(getter(Webhook::filterGroups))
            .setter(setter(Builder::filterGroups))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("filterGroups").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<List<WebhookFilter>> builder(MarshallingType.LIST)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build(),
                                                    ListTrait
                                                            .builder()
                                                            .memberLocationName(null)
                                                            .memberFieldInfo(
                                                                    SdkField.<WebhookFilter> builder(MarshallingType.SDK_POJO)
                                                                            .constructor(WebhookFilter::builder)
                                                                            .traits(LocationTrait.builder()
                                                                                    .location(MarshallLocation.PAYLOAD)
                                                                                    .locationName("member").build()).build())
                                                            .build()).build()).build()).build();

    private static final SdkField<Instant> LAST_MODIFIED_SECRET_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .getter(getter(Webhook::lastModifiedSecret)).setter(setter(Builder::lastModifiedSecret))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("lastModifiedSecret").build())
            .build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(URL_FIELD, PAYLOAD_URL_FIELD,
            SECRET_FIELD, BRANCH_FILTER_FIELD, FILTER_GROUPS_FIELD, LAST_MODIFIED_SECRET_FIELD));

    private static final long serialVersionUID = 1L;

    private final String url;

    private final String payloadUrl;

    private final String secret;

    private final String branchFilter;

    private final List<List<WebhookFilter>> filterGroups;

    private final Instant lastModifiedSecret;

    private Webhook(BuilderImpl builder) {
        this.url = builder.url;
        this.payloadUrl = builder.payloadUrl;
        this.secret = builder.secret;
        this.branchFilter = builder.branchFilter;
        this.filterGroups = builder.filterGroups;
        this.lastModifiedSecret = builder.lastModifiedSecret;
    }

    /**
     * <p>
     * The URL to the webhook.
     * </p>
     * 
     * @return The URL to the webhook.
     */
    public String url() {
        return url;
    }

    /**
     * <p>
     * The AWS CodeBuild endpoint where webhook events are sent.
     * </p>
     * 
     * @return The AWS CodeBuild endpoint where webhook events are sent.
     */
    public String payloadUrl() {
        return payloadUrl;
    }

    /**
     * <p>
     * The secret token of the associated repository.
     * </p>
     * <note>
     * <p>
     * A Bitbucket webhook does not support <code>secret</code>.
     * </p>
     * </note>
     * 
     * @return The secret token of the associated repository. </p> <note>
     *         <p>
     *         A Bitbucket webhook does not support <code>secret</code>.
     *         </p>
     */
    public String secret() {
        return secret;
    }

    /**
     * <p>
     * A regular expression used to determine which repository branches are built when a webhook is triggered. If the
     * name of a branch matches the regular expression, then it is built. If <code>branchFilter</code> is empty, then
     * all branches are built.
     * </p>
     * <note>
     * <p>
     * It is recommended that you use <code>filterGroups</code> instead of <code>branchFilter</code>.
     * </p>
     * </note>
     * 
     * @return A regular expression used to determine which repository branches are built when a webhook is triggered.
     *         If the name of a branch matches the regular expression, then it is built. If <code>branchFilter</code> is
     *         empty, then all branches are built.</p> <note>
     *         <p>
     *         It is recommended that you use <code>filterGroups</code> instead of <code>branchFilter</code>.
     *         </p>
     */
    public String branchFilter() {
        return branchFilter;
    }

    /**
     * Returns true if the FilterGroups property was specified by the sender (it may be empty), or false if the sender
     * did not specify the value (it will be empty). For responses returned by the SDK, the sender is the AWS service.
     */
    public boolean hasFilterGroups() {
        return filterGroups != null && !(filterGroups instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * An array of arrays of <code>WebhookFilter</code> objects used to determine which webhooks are triggered. At least
     * one <code>WebhookFilter</code> in the array must specify <code>EVENT</code> as its <code>type</code>.
     * </p>
     * <p>
     * For a build to be triggered, at least one filter group in the <code>filterGroups</code> array must pass. For a
     * filter group to pass, each of its filters must pass.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasFilterGroups()} to see if a value was sent in this field.
     * </p>
     * 
     * @return An array of arrays of <code>WebhookFilter</code> objects used to determine which webhooks are triggered.
     *         At least one <code>WebhookFilter</code> in the array must specify <code>EVENT</code> as its
     *         <code>type</code>. </p>
     *         <p>
     *         For a build to be triggered, at least one filter group in the <code>filterGroups</code> array must pass.
     *         For a filter group to pass, each of its filters must pass.
     */
    public List<List<WebhookFilter>> filterGroups() {
        return filterGroups;
    }

    /**
     * <p>
     * A timestamp that indicates the last time a repository's secret token was modified.
     * </p>
     * 
     * @return A timestamp that indicates the last time a repository's secret token was modified.
     */
    public Instant lastModifiedSecret() {
        return lastModifiedSecret;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(url());
        hashCode = 31 * hashCode + Objects.hashCode(payloadUrl());
        hashCode = 31 * hashCode + Objects.hashCode(secret());
        hashCode = 31 * hashCode + Objects.hashCode(branchFilter());
        hashCode = 31 * hashCode + Objects.hashCode(filterGroups());
        hashCode = 31 * hashCode + Objects.hashCode(lastModifiedSecret());
        return hashCode;
    }

    @Override
    public boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof Webhook)) {
            return false;
        }
        Webhook other = (Webhook) obj;
        return Objects.equals(url(), other.url()) && Objects.equals(payloadUrl(), other.payloadUrl())
                && Objects.equals(secret(), other.secret()) && Objects.equals(branchFilter(), other.branchFilter())
                && Objects.equals(filterGroups(), other.filterGroups())
                && Objects.equals(lastModifiedSecret(), other.lastModifiedSecret());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public String toString() {
        return ToString.builder("Webhook").add("Url", url()).add("PayloadUrl", payloadUrl()).add("Secret", secret())
                .add("BranchFilter", branchFilter()).add("FilterGroups", filterGroups())
                .add("LastModifiedSecret", lastModifiedSecret()).build();
    }

    public <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "url":
            return Optional.ofNullable(clazz.cast(url()));
        case "payloadUrl":
            return Optional.ofNullable(clazz.cast(payloadUrl()));
        case "secret":
            return Optional.ofNullable(clazz.cast(secret()));
        case "branchFilter":
            return Optional.ofNullable(clazz.cast(branchFilter()));
        case "filterGroups":
            return Optional.ofNullable(clazz.cast(filterGroups()));
        case "lastModifiedSecret":
            return Optional.ofNullable(clazz.cast(lastModifiedSecret()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<Webhook, T> g) {
        return obj -> g.apply((Webhook) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, Webhook> {
        /**
         * <p>
         * The URL to the webhook.
         * </p>
         * 
         * @param url
         *        The URL to the webhook.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder url(String url);

        /**
         * <p>
         * The AWS CodeBuild endpoint where webhook events are sent.
         * </p>
         * 
         * @param payloadUrl
         *        The AWS CodeBuild endpoint where webhook events are sent.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder payloadUrl(String payloadUrl);

        /**
         * <p>
         * The secret token of the associated repository.
         * </p>
         * <note>
         * <p>
         * A Bitbucket webhook does not support <code>secret</code>.
         * </p>
         * </note>
         * 
         * @param secret
         *        The secret token of the associated repository. </p> <note>
         *        <p>
         *        A Bitbucket webhook does not support <code>secret</code>.
         *        </p>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder secret(String secret);

        /**
         * <p>
         * A regular expression used to determine which repository branches are built when a webhook is triggered. If
         * the name of a branch matches the regular expression, then it is built. If <code>branchFilter</code> is empty,
         * then all branches are built.
         * </p>
         * <note>
         * <p>
         * It is recommended that you use <code>filterGroups</code> instead of <code>branchFilter</code>.
         * </p>
         * </note>
         * 
         * @param branchFilter
         *        A regular expression used to determine which repository branches are built when a webhook is
         *        triggered. If the name of a branch matches the regular expression, then it is built. If
         *        <code>branchFilter</code> is empty, then all branches are built.</p> <note>
         *        <p>
         *        It is recommended that you use <code>filterGroups</code> instead of <code>branchFilter</code>.
         *        </p>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder branchFilter(String branchFilter);

        /**
         * <p>
         * An array of arrays of <code>WebhookFilter</code> objects used to determine which webhooks are triggered. At
         * least one <code>WebhookFilter</code> in the array must specify <code>EVENT</code> as its <code>type</code>.
         * </p>
         * <p>
         * For a build to be triggered, at least one filter group in the <code>filterGroups</code> array must pass. For
         * a filter group to pass, each of its filters must pass.
         * </p>
         * 
         * @param filterGroups
         *        An array of arrays of <code>WebhookFilter</code> objects used to determine which webhooks are
         *        triggered. At least one <code>WebhookFilter</code> in the array must specify <code>EVENT</code> as its
         *        <code>type</code>. </p>
         *        <p>
         *        For a build to be triggered, at least one filter group in the <code>filterGroups</code> array must
         *        pass. For a filter group to pass, each of its filters must pass.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder filterGroups(Collection<? extends Collection<WebhookFilter>> filterGroups);

        /**
         * <p>
         * An array of arrays of <code>WebhookFilter</code> objects used to determine which webhooks are triggered. At
         * least one <code>WebhookFilter</code> in the array must specify <code>EVENT</code> as its <code>type</code>.
         * </p>
         * <p>
         * For a build to be triggered, at least one filter group in the <code>filterGroups</code> array must pass. For
         * a filter group to pass, each of its filters must pass.
         * </p>
         * 
         * @param filterGroups
         *        An array of arrays of <code>WebhookFilter</code> objects used to determine which webhooks are
         *        triggered. At least one <code>WebhookFilter</code> in the array must specify <code>EVENT</code> as its
         *        <code>type</code>. </p>
         *        <p>
         *        For a build to be triggered, at least one filter group in the <code>filterGroups</code> array must
         *        pass. For a filter group to pass, each of its filters must pass.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder filterGroups(Collection<WebhookFilter>... filterGroups);

        /**
         * <p>
         * A timestamp that indicates the last time a repository's secret token was modified.
         * </p>
         * 
         * @param lastModifiedSecret
         *        A timestamp that indicates the last time a repository's secret token was modified.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder lastModifiedSecret(Instant lastModifiedSecret);
    }

    static final class BuilderImpl implements Builder {
        private String url;

        private String payloadUrl;

        private String secret;

        private String branchFilter;

        private List<List<WebhookFilter>> filterGroups = DefaultSdkAutoConstructList.getInstance();

        private Instant lastModifiedSecret;

        private BuilderImpl() {
        }

        private BuilderImpl(Webhook model) {
            url(model.url);
            payloadUrl(model.payloadUrl);
            secret(model.secret);
            branchFilter(model.branchFilter);
            filterGroups(model.filterGroups);
            lastModifiedSecret(model.lastModifiedSecret);
        }

        public final String getUrl() {
            return url;
        }

        @Override
        public final Builder url(String url) {
            this.url = url;
            return this;
        }

        public final void setUrl(String url) {
            this.url = url;
        }

        public final String getPayloadUrl() {
            return payloadUrl;
        }

        @Override
        public final Builder payloadUrl(String payloadUrl) {
            this.payloadUrl = payloadUrl;
            return this;
        }

        public final void setPayloadUrl(String payloadUrl) {
            this.payloadUrl = payloadUrl;
        }

        public final String getSecret() {
            return secret;
        }

        @Override
        public final Builder secret(String secret) {
            this.secret = secret;
            return this;
        }

        public final void setSecret(String secret) {
            this.secret = secret;
        }

        public final String getBranchFilter() {
            return branchFilter;
        }

        @Override
        public final Builder branchFilter(String branchFilter) {
            this.branchFilter = branchFilter;
            return this;
        }

        public final void setBranchFilter(String branchFilter) {
            this.branchFilter = branchFilter;
        }

        public final Collection<? extends Collection<WebhookFilter>> getFilterGroups() {
            return filterGroups;
        }

        @Override
        public final Builder filterGroups(Collection<? extends Collection<WebhookFilter>> filterGroups) {
            this.filterGroups = FilterGroupsCopier.copy(filterGroups);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder filterGroups(Collection<WebhookFilter>... filterGroups) {
            filterGroups(Arrays.asList(filterGroups));
            return this;
        }

        public final void setFilterGroups(Collection<? extends Collection<WebhookFilter>> filterGroups) {
            this.filterGroups = FilterGroupsCopier.copy(filterGroups);
        }

        public final Instant getLastModifiedSecret() {
            return lastModifiedSecret;
        }

        @Override
        public final Builder lastModifiedSecret(Instant lastModifiedSecret) {
            this.lastModifiedSecret = lastModifiedSecret;
            return this;
        }

        public final void setLastModifiedSecret(Instant lastModifiedSecret) {
            this.lastModifiedSecret = lastModifiedSecret;
        }

        @Override
        public Webhook build() {
            return new Webhook(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
