/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.codebuild.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Information about the cache for the build project.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class ProjectCache implements SdkPojo, Serializable, ToCopyableBuilder<ProjectCache.Builder, ProjectCache> {
    private static final SdkField<String> TYPE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(ProjectCache::typeAsString)).setter(setter(Builder::type))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("type").build()).build();

    private static final SdkField<String> LOCATION_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(ProjectCache::location)).setter(setter(Builder::location))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("location").build()).build();

    private static final SdkField<List<String>> MODES_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .getter(getter(ProjectCache::modesAsStrings))
            .setter(setter(Builder::modesWithStrings))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("modes").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(TYPE_FIELD, LOCATION_FIELD,
            MODES_FIELD));

    private static final long serialVersionUID = 1L;

    private final String type;

    private final String location;

    private final List<String> modes;

    private ProjectCache(BuilderImpl builder) {
        this.type = builder.type;
        this.location = builder.location;
        this.modes = builder.modes;
    }

    /**
     * <p>
     * The type of cache used by the build project. Valid values include:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>NO_CACHE</code>: The build project does not use any cache.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>S3</code>: The build project reads and writes from and to S3.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>LOCAL</code>: The build project stores a cache locally on a build host that is only available to that build
     * host.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #type} will return
     * {@link CacheType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #typeAsString}.
     * </p>
     * 
     * @return The type of cache used by the build project. Valid values include:</p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>NO_CACHE</code>: The build project does not use any cache.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>S3</code>: The build project reads and writes from and to S3.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>LOCAL</code>: The build project stores a cache locally on a build host that is only available to
     *         that build host.
     *         </p>
     *         </li>
     * @see CacheType
     */
    public CacheType type() {
        return CacheType.fromValue(type);
    }

    /**
     * <p>
     * The type of cache used by the build project. Valid values include:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>NO_CACHE</code>: The build project does not use any cache.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>S3</code>: The build project reads and writes from and to S3.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>LOCAL</code>: The build project stores a cache locally on a build host that is only available to that build
     * host.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #type} will return
     * {@link CacheType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #typeAsString}.
     * </p>
     * 
     * @return The type of cache used by the build project. Valid values include:</p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>NO_CACHE</code>: The build project does not use any cache.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>S3</code>: The build project reads and writes from and to S3.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>LOCAL</code>: The build project stores a cache locally on a build host that is only available to
     *         that build host.
     *         </p>
     *         </li>
     * @see CacheType
     */
    public String typeAsString() {
        return type;
    }

    /**
     * <p>
     * Information about the cache location:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>NO_CACHE</code> or <code>LOCAL</code>: This value is ignored.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>S3</code>: This is the S3 bucket name/prefix.
     * </p>
     * </li>
     * </ul>
     * 
     * @return Information about the cache location: </p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>NO_CACHE</code> or <code>LOCAL</code>: This value is ignored.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>S3</code>: This is the S3 bucket name/prefix.
     *         </p>
     *         </li>
     */
    public String location() {
        return location;
    }

    /**
     * <p>
     * If you use a <code>LOCAL</code> cache, the local cache mode. You can use one or more local cache modes at the
     * same time.
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>LOCAL_SOURCE_CACHE</code> mode caches Git metadata for primary and secondary sources. After the cache is
     * created, subsequent builds pull only the change between commits. This mode is a good choice for projects with a
     * clean working directory and a source that is a large Git repository. If you choose this option and your project
     * does not use a Git repository (GitHub, GitHub Enterprise, or Bitbucket), the option is ignored.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>LOCAL_DOCKER_LAYER_CACHE</code> mode caches existing Docker layers. This mode is a good choice for projects
     * that build or pull large Docker images. It can prevent the performance issues caused by pulling large Docker
     * images down from the network.
     * </p>
     * <note>
     * <ul>
     * <li>
     * <p>
     * You can use a Docker layer cache in the Linux environment only.
     * </p>
     * </li>
     * <li>
     * <p>
     * The <code>privileged</code> flag must be set so that your project has the required Docker permissions.
     * </p>
     * </li>
     * <li>
     * <p>
     * You should consider the security implications before you use a Docker layer cache.
     * </p>
     * </li>
     * </ul>
     * </note></li>
     * </ul>
     * <ul>
     * <li>
     * <p>
     * <code>LOCAL_CUSTOM_CACHE</code> mode caches directories you specify in the buildspec file. This mode is a good
     * choice if your build scenario is not suited to one of the other three local cache modes. If you use a custom
     * cache:
     * </p>
     * <ul>
     * <li>
     * <p>
     * Only directories can be specified for caching. You cannot specify individual files.
     * </p>
     * </li>
     * <li>
     * <p>
     * Symlinks are used to reference cached directories.
     * </p>
     * </li>
     * <li>
     * <p>
     * Cached directories are linked to your build before it downloads its project sources. Cached items are overridden
     * if a source item has the same name. Directories are specified using cache paths in the buildspec file.
     * </p>
     * </li>
     * </ul>
     * </li>
     * </ul>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasModes()} to see if a value was sent in this field.
     * </p>
     * 
     * @return If you use a <code>LOCAL</code> cache, the local cache mode. You can use one or more local cache modes at
     *         the same time. </p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>LOCAL_SOURCE_CACHE</code> mode caches Git metadata for primary and secondary sources. After the
     *         cache is created, subsequent builds pull only the change between commits. This mode is a good choice for
     *         projects with a clean working directory and a source that is a large Git repository. If you choose this
     *         option and your project does not use a Git repository (GitHub, GitHub Enterprise, or Bitbucket), the
     *         option is ignored.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>LOCAL_DOCKER_LAYER_CACHE</code> mode caches existing Docker layers. This mode is a good choice for
     *         projects that build or pull large Docker images. It can prevent the performance issues caused by pulling
     *         large Docker images down from the network.
     *         </p>
     *         <note>
     *         <ul>
     *         <li>
     *         <p>
     *         You can use a Docker layer cache in the Linux environment only.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         The <code>privileged</code> flag must be set so that your project has the required Docker permissions.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         You should consider the security implications before you use a Docker layer cache.
     *         </p>
     *         </li>
     *         </ul>
     *         </note></li>
     *         </ul>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>LOCAL_CUSTOM_CACHE</code> mode caches directories you specify in the buildspec file. This mode is a
     *         good choice if your build scenario is not suited to one of the other three local cache modes. If you use
     *         a custom cache:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         Only directories can be specified for caching. You cannot specify individual files.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Symlinks are used to reference cached directories.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Cached directories are linked to your build before it downloads its project sources. Cached items are
     *         overridden if a source item has the same name. Directories are specified using cache paths in the
     *         buildspec file.
     *         </p>
     *         </li>
     *         </ul>
     *         </li>
     */
    public List<CacheMode> modes() {
        return ProjectCacheModesCopier.copyStringToEnum(modes);
    }

    /**
     * Returns true if the Modes property was specified by the sender (it may be empty), or false if the sender did not
     * specify the value (it will be empty). For responses returned by the SDK, the sender is the AWS service.
     */
    public boolean hasModes() {
        return modes != null && !(modes instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * If you use a <code>LOCAL</code> cache, the local cache mode. You can use one or more local cache modes at the
     * same time.
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>LOCAL_SOURCE_CACHE</code> mode caches Git metadata for primary and secondary sources. After the cache is
     * created, subsequent builds pull only the change between commits. This mode is a good choice for projects with a
     * clean working directory and a source that is a large Git repository. If you choose this option and your project
     * does not use a Git repository (GitHub, GitHub Enterprise, or Bitbucket), the option is ignored.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>LOCAL_DOCKER_LAYER_CACHE</code> mode caches existing Docker layers. This mode is a good choice for projects
     * that build or pull large Docker images. It can prevent the performance issues caused by pulling large Docker
     * images down from the network.
     * </p>
     * <note>
     * <ul>
     * <li>
     * <p>
     * You can use a Docker layer cache in the Linux environment only.
     * </p>
     * </li>
     * <li>
     * <p>
     * The <code>privileged</code> flag must be set so that your project has the required Docker permissions.
     * </p>
     * </li>
     * <li>
     * <p>
     * You should consider the security implications before you use a Docker layer cache.
     * </p>
     * </li>
     * </ul>
     * </note></li>
     * </ul>
     * <ul>
     * <li>
     * <p>
     * <code>LOCAL_CUSTOM_CACHE</code> mode caches directories you specify in the buildspec file. This mode is a good
     * choice if your build scenario is not suited to one of the other three local cache modes. If you use a custom
     * cache:
     * </p>
     * <ul>
     * <li>
     * <p>
     * Only directories can be specified for caching. You cannot specify individual files.
     * </p>
     * </li>
     * <li>
     * <p>
     * Symlinks are used to reference cached directories.
     * </p>
     * </li>
     * <li>
     * <p>
     * Cached directories are linked to your build before it downloads its project sources. Cached items are overridden
     * if a source item has the same name. Directories are specified using cache paths in the buildspec file.
     * </p>
     * </li>
     * </ul>
     * </li>
     * </ul>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasModes()} to see if a value was sent in this field.
     * </p>
     * 
     * @return If you use a <code>LOCAL</code> cache, the local cache mode. You can use one or more local cache modes at
     *         the same time. </p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>LOCAL_SOURCE_CACHE</code> mode caches Git metadata for primary and secondary sources. After the
     *         cache is created, subsequent builds pull only the change between commits. This mode is a good choice for
     *         projects with a clean working directory and a source that is a large Git repository. If you choose this
     *         option and your project does not use a Git repository (GitHub, GitHub Enterprise, or Bitbucket), the
     *         option is ignored.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>LOCAL_DOCKER_LAYER_CACHE</code> mode caches existing Docker layers. This mode is a good choice for
     *         projects that build or pull large Docker images. It can prevent the performance issues caused by pulling
     *         large Docker images down from the network.
     *         </p>
     *         <note>
     *         <ul>
     *         <li>
     *         <p>
     *         You can use a Docker layer cache in the Linux environment only.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         The <code>privileged</code> flag must be set so that your project has the required Docker permissions.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         You should consider the security implications before you use a Docker layer cache.
     *         </p>
     *         </li>
     *         </ul>
     *         </note></li>
     *         </ul>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>LOCAL_CUSTOM_CACHE</code> mode caches directories you specify in the buildspec file. This mode is a
     *         good choice if your build scenario is not suited to one of the other three local cache modes. If you use
     *         a custom cache:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         Only directories can be specified for caching. You cannot specify individual files.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Symlinks are used to reference cached directories.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Cached directories are linked to your build before it downloads its project sources. Cached items are
     *         overridden if a source item has the same name. Directories are specified using cache paths in the
     *         buildspec file.
     *         </p>
     *         </li>
     *         </ul>
     *         </li>
     */
    public List<String> modesAsStrings() {
        return modes;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(typeAsString());
        hashCode = 31 * hashCode + Objects.hashCode(location());
        hashCode = 31 * hashCode + Objects.hashCode(modesAsStrings());
        return hashCode;
    }

    @Override
    public boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof ProjectCache)) {
            return false;
        }
        ProjectCache other = (ProjectCache) obj;
        return Objects.equals(typeAsString(), other.typeAsString()) && Objects.equals(location(), other.location())
                && Objects.equals(modesAsStrings(), other.modesAsStrings());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public String toString() {
        return ToString.builder("ProjectCache").add("Type", typeAsString()).add("Location", location())
                .add("Modes", modesAsStrings()).build();
    }

    public <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "type":
            return Optional.ofNullable(clazz.cast(typeAsString()));
        case "location":
            return Optional.ofNullable(clazz.cast(location()));
        case "modes":
            return Optional.ofNullable(clazz.cast(modesAsStrings()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<ProjectCache, T> g) {
        return obj -> g.apply((ProjectCache) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, ProjectCache> {
        /**
         * <p>
         * The type of cache used by the build project. Valid values include:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>NO_CACHE</code>: The build project does not use any cache.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>S3</code>: The build project reads and writes from and to S3.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>LOCAL</code>: The build project stores a cache locally on a build host that is only available to that
         * build host.
         * </p>
         * </li>
         * </ul>
         * 
         * @param type
         *        The type of cache used by the build project. Valid values include:</p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>NO_CACHE</code>: The build project does not use any cache.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>S3</code>: The build project reads and writes from and to S3.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>LOCAL</code>: The build project stores a cache locally on a build host that is only available to
         *        that build host.
         *        </p>
         *        </li>
         * @see CacheType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see CacheType
         */
        Builder type(String type);

        /**
         * <p>
         * The type of cache used by the build project. Valid values include:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>NO_CACHE</code>: The build project does not use any cache.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>S3</code>: The build project reads and writes from and to S3.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>LOCAL</code>: The build project stores a cache locally on a build host that is only available to that
         * build host.
         * </p>
         * </li>
         * </ul>
         * 
         * @param type
         *        The type of cache used by the build project. Valid values include:</p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>NO_CACHE</code>: The build project does not use any cache.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>S3</code>: The build project reads and writes from and to S3.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>LOCAL</code>: The build project stores a cache locally on a build host that is only available to
         *        that build host.
         *        </p>
         *        </li>
         * @see CacheType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see CacheType
         */
        Builder type(CacheType type);

        /**
         * <p>
         * Information about the cache location:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>NO_CACHE</code> or <code>LOCAL</code>: This value is ignored.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>S3</code>: This is the S3 bucket name/prefix.
         * </p>
         * </li>
         * </ul>
         * 
         * @param location
         *        Information about the cache location: </p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>NO_CACHE</code> or <code>LOCAL</code>: This value is ignored.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>S3</code>: This is the S3 bucket name/prefix.
         *        </p>
         *        </li>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder location(String location);

        /**
         * <p>
         * If you use a <code>LOCAL</code> cache, the local cache mode. You can use one or more local cache modes at the
         * same time.
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>LOCAL_SOURCE_CACHE</code> mode caches Git metadata for primary and secondary sources. After the cache
         * is created, subsequent builds pull only the change between commits. This mode is a good choice for projects
         * with a clean working directory and a source that is a large Git repository. If you choose this option and
         * your project does not use a Git repository (GitHub, GitHub Enterprise, or Bitbucket), the option is ignored.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>LOCAL_DOCKER_LAYER_CACHE</code> mode caches existing Docker layers. This mode is a good choice for
         * projects that build or pull large Docker images. It can prevent the performance issues caused by pulling
         * large Docker images down from the network.
         * </p>
         * <note>
         * <ul>
         * <li>
         * <p>
         * You can use a Docker layer cache in the Linux environment only.
         * </p>
         * </li>
         * <li>
         * <p>
         * The <code>privileged</code> flag must be set so that your project has the required Docker permissions.
         * </p>
         * </li>
         * <li>
         * <p>
         * You should consider the security implications before you use a Docker layer cache.
         * </p>
         * </li>
         * </ul>
         * </note></li>
         * </ul>
         * <ul>
         * <li>
         * <p>
         * <code>LOCAL_CUSTOM_CACHE</code> mode caches directories you specify in the buildspec file. This mode is a
         * good choice if your build scenario is not suited to one of the other three local cache modes. If you use a
         * custom cache:
         * </p>
         * <ul>
         * <li>
         * <p>
         * Only directories can be specified for caching. You cannot specify individual files.
         * </p>
         * </li>
         * <li>
         * <p>
         * Symlinks are used to reference cached directories.
         * </p>
         * </li>
         * <li>
         * <p>
         * Cached directories are linked to your build before it downloads its project sources. Cached items are
         * overridden if a source item has the same name. Directories are specified using cache paths in the buildspec
         * file.
         * </p>
         * </li>
         * </ul>
         * </li>
         * </ul>
         * 
         * @param modes
         *        If you use a <code>LOCAL</code> cache, the local cache mode. You can use one or more local cache modes
         *        at the same time. </p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>LOCAL_SOURCE_CACHE</code> mode caches Git metadata for primary and secondary sources. After the
         *        cache is created, subsequent builds pull only the change between commits. This mode is a good choice
         *        for projects with a clean working directory and a source that is a large Git repository. If you choose
         *        this option and your project does not use a Git repository (GitHub, GitHub Enterprise, or Bitbucket),
         *        the option is ignored.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>LOCAL_DOCKER_LAYER_CACHE</code> mode caches existing Docker layers. This mode is a good choice
         *        for projects that build or pull large Docker images. It can prevent the performance issues caused by
         *        pulling large Docker images down from the network.
         *        </p>
         *        <note>
         *        <ul>
         *        <li>
         *        <p>
         *        You can use a Docker layer cache in the Linux environment only.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        The <code>privileged</code> flag must be set so that your project has the required Docker permissions.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        You should consider the security implications before you use a Docker layer cache.
         *        </p>
         *        </li>
         *        </ul>
         *        </note></li>
         *        </ul>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>LOCAL_CUSTOM_CACHE</code> mode caches directories you specify in the buildspec file. This mode
         *        is a good choice if your build scenario is not suited to one of the other three local cache modes. If
         *        you use a custom cache:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        Only directories can be specified for caching. You cannot specify individual files.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Symlinks are used to reference cached directories.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Cached directories are linked to your build before it downloads its project sources. Cached items are
         *        overridden if a source item has the same name. Directories are specified using cache paths in the
         *        buildspec file.
         *        </p>
         *        </li>
         *        </ul>
         *        </li>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder modesWithStrings(Collection<String> modes);

        /**
         * <p>
         * If you use a <code>LOCAL</code> cache, the local cache mode. You can use one or more local cache modes at the
         * same time.
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>LOCAL_SOURCE_CACHE</code> mode caches Git metadata for primary and secondary sources. After the cache
         * is created, subsequent builds pull only the change between commits. This mode is a good choice for projects
         * with a clean working directory and a source that is a large Git repository. If you choose this option and
         * your project does not use a Git repository (GitHub, GitHub Enterprise, or Bitbucket), the option is ignored.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>LOCAL_DOCKER_LAYER_CACHE</code> mode caches existing Docker layers. This mode is a good choice for
         * projects that build or pull large Docker images. It can prevent the performance issues caused by pulling
         * large Docker images down from the network.
         * </p>
         * <note>
         * <ul>
         * <li>
         * <p>
         * You can use a Docker layer cache in the Linux environment only.
         * </p>
         * </li>
         * <li>
         * <p>
         * The <code>privileged</code> flag must be set so that your project has the required Docker permissions.
         * </p>
         * </li>
         * <li>
         * <p>
         * You should consider the security implications before you use a Docker layer cache.
         * </p>
         * </li>
         * </ul>
         * </note></li>
         * </ul>
         * <ul>
         * <li>
         * <p>
         * <code>LOCAL_CUSTOM_CACHE</code> mode caches directories you specify in the buildspec file. This mode is a
         * good choice if your build scenario is not suited to one of the other three local cache modes. If you use a
         * custom cache:
         * </p>
         * <ul>
         * <li>
         * <p>
         * Only directories can be specified for caching. You cannot specify individual files.
         * </p>
         * </li>
         * <li>
         * <p>
         * Symlinks are used to reference cached directories.
         * </p>
         * </li>
         * <li>
         * <p>
         * Cached directories are linked to your build before it downloads its project sources. Cached items are
         * overridden if a source item has the same name. Directories are specified using cache paths in the buildspec
         * file.
         * </p>
         * </li>
         * </ul>
         * </li>
         * </ul>
         * 
         * @param modes
         *        If you use a <code>LOCAL</code> cache, the local cache mode. You can use one or more local cache modes
         *        at the same time. </p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>LOCAL_SOURCE_CACHE</code> mode caches Git metadata for primary and secondary sources. After the
         *        cache is created, subsequent builds pull only the change between commits. This mode is a good choice
         *        for projects with a clean working directory and a source that is a large Git repository. If you choose
         *        this option and your project does not use a Git repository (GitHub, GitHub Enterprise, or Bitbucket),
         *        the option is ignored.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>LOCAL_DOCKER_LAYER_CACHE</code> mode caches existing Docker layers. This mode is a good choice
         *        for projects that build or pull large Docker images. It can prevent the performance issues caused by
         *        pulling large Docker images down from the network.
         *        </p>
         *        <note>
         *        <ul>
         *        <li>
         *        <p>
         *        You can use a Docker layer cache in the Linux environment only.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        The <code>privileged</code> flag must be set so that your project has the required Docker permissions.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        You should consider the security implications before you use a Docker layer cache.
         *        </p>
         *        </li>
         *        </ul>
         *        </note></li>
         *        </ul>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>LOCAL_CUSTOM_CACHE</code> mode caches directories you specify in the buildspec file. This mode
         *        is a good choice if your build scenario is not suited to one of the other three local cache modes. If
         *        you use a custom cache:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        Only directories can be specified for caching. You cannot specify individual files.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Symlinks are used to reference cached directories.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Cached directories are linked to your build before it downloads its project sources. Cached items are
         *        overridden if a source item has the same name. Directories are specified using cache paths in the
         *        buildspec file.
         *        </p>
         *        </li>
         *        </ul>
         *        </li>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder modesWithStrings(String... modes);

        /**
         * <p>
         * If you use a <code>LOCAL</code> cache, the local cache mode. You can use one or more local cache modes at the
         * same time.
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>LOCAL_SOURCE_CACHE</code> mode caches Git metadata for primary and secondary sources. After the cache
         * is created, subsequent builds pull only the change between commits. This mode is a good choice for projects
         * with a clean working directory and a source that is a large Git repository. If you choose this option and
         * your project does not use a Git repository (GitHub, GitHub Enterprise, or Bitbucket), the option is ignored.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>LOCAL_DOCKER_LAYER_CACHE</code> mode caches existing Docker layers. This mode is a good choice for
         * projects that build or pull large Docker images. It can prevent the performance issues caused by pulling
         * large Docker images down from the network.
         * </p>
         * <note>
         * <ul>
         * <li>
         * <p>
         * You can use a Docker layer cache in the Linux environment only.
         * </p>
         * </li>
         * <li>
         * <p>
         * The <code>privileged</code> flag must be set so that your project has the required Docker permissions.
         * </p>
         * </li>
         * <li>
         * <p>
         * You should consider the security implications before you use a Docker layer cache.
         * </p>
         * </li>
         * </ul>
         * </note></li>
         * </ul>
         * <ul>
         * <li>
         * <p>
         * <code>LOCAL_CUSTOM_CACHE</code> mode caches directories you specify in the buildspec file. This mode is a
         * good choice if your build scenario is not suited to one of the other three local cache modes. If you use a
         * custom cache:
         * </p>
         * <ul>
         * <li>
         * <p>
         * Only directories can be specified for caching. You cannot specify individual files.
         * </p>
         * </li>
         * <li>
         * <p>
         * Symlinks are used to reference cached directories.
         * </p>
         * </li>
         * <li>
         * <p>
         * Cached directories are linked to your build before it downloads its project sources. Cached items are
         * overridden if a source item has the same name. Directories are specified using cache paths in the buildspec
         * file.
         * </p>
         * </li>
         * </ul>
         * </li>
         * </ul>
         * 
         * @param modes
         *        If you use a <code>LOCAL</code> cache, the local cache mode. You can use one or more local cache modes
         *        at the same time. </p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>LOCAL_SOURCE_CACHE</code> mode caches Git metadata for primary and secondary sources. After the
         *        cache is created, subsequent builds pull only the change between commits. This mode is a good choice
         *        for projects with a clean working directory and a source that is a large Git repository. If you choose
         *        this option and your project does not use a Git repository (GitHub, GitHub Enterprise, or Bitbucket),
         *        the option is ignored.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>LOCAL_DOCKER_LAYER_CACHE</code> mode caches existing Docker layers. This mode is a good choice
         *        for projects that build or pull large Docker images. It can prevent the performance issues caused by
         *        pulling large Docker images down from the network.
         *        </p>
         *        <note>
         *        <ul>
         *        <li>
         *        <p>
         *        You can use a Docker layer cache in the Linux environment only.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        The <code>privileged</code> flag must be set so that your project has the required Docker permissions.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        You should consider the security implications before you use a Docker layer cache.
         *        </p>
         *        </li>
         *        </ul>
         *        </note></li>
         *        </ul>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>LOCAL_CUSTOM_CACHE</code> mode caches directories you specify in the buildspec file. This mode
         *        is a good choice if your build scenario is not suited to one of the other three local cache modes. If
         *        you use a custom cache:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        Only directories can be specified for caching. You cannot specify individual files.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Symlinks are used to reference cached directories.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Cached directories are linked to your build before it downloads its project sources. Cached items are
         *        overridden if a source item has the same name. Directories are specified using cache paths in the
         *        buildspec file.
         *        </p>
         *        </li>
         *        </ul>
         *        </li>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder modes(Collection<CacheMode> modes);

        /**
         * <p>
         * If you use a <code>LOCAL</code> cache, the local cache mode. You can use one or more local cache modes at the
         * same time.
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>LOCAL_SOURCE_CACHE</code> mode caches Git metadata for primary and secondary sources. After the cache
         * is created, subsequent builds pull only the change between commits. This mode is a good choice for projects
         * with a clean working directory and a source that is a large Git repository. If you choose this option and
         * your project does not use a Git repository (GitHub, GitHub Enterprise, or Bitbucket), the option is ignored.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>LOCAL_DOCKER_LAYER_CACHE</code> mode caches existing Docker layers. This mode is a good choice for
         * projects that build or pull large Docker images. It can prevent the performance issues caused by pulling
         * large Docker images down from the network.
         * </p>
         * <note>
         * <ul>
         * <li>
         * <p>
         * You can use a Docker layer cache in the Linux environment only.
         * </p>
         * </li>
         * <li>
         * <p>
         * The <code>privileged</code> flag must be set so that your project has the required Docker permissions.
         * </p>
         * </li>
         * <li>
         * <p>
         * You should consider the security implications before you use a Docker layer cache.
         * </p>
         * </li>
         * </ul>
         * </note></li>
         * </ul>
         * <ul>
         * <li>
         * <p>
         * <code>LOCAL_CUSTOM_CACHE</code> mode caches directories you specify in the buildspec file. This mode is a
         * good choice if your build scenario is not suited to one of the other three local cache modes. If you use a
         * custom cache:
         * </p>
         * <ul>
         * <li>
         * <p>
         * Only directories can be specified for caching. You cannot specify individual files.
         * </p>
         * </li>
         * <li>
         * <p>
         * Symlinks are used to reference cached directories.
         * </p>
         * </li>
         * <li>
         * <p>
         * Cached directories are linked to your build before it downloads its project sources. Cached items are
         * overridden if a source item has the same name. Directories are specified using cache paths in the buildspec
         * file.
         * </p>
         * </li>
         * </ul>
         * </li>
         * </ul>
         * 
         * @param modes
         *        If you use a <code>LOCAL</code> cache, the local cache mode. You can use one or more local cache modes
         *        at the same time. </p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>LOCAL_SOURCE_CACHE</code> mode caches Git metadata for primary and secondary sources. After the
         *        cache is created, subsequent builds pull only the change between commits. This mode is a good choice
         *        for projects with a clean working directory and a source that is a large Git repository. If you choose
         *        this option and your project does not use a Git repository (GitHub, GitHub Enterprise, or Bitbucket),
         *        the option is ignored.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>LOCAL_DOCKER_LAYER_CACHE</code> mode caches existing Docker layers. This mode is a good choice
         *        for projects that build or pull large Docker images. It can prevent the performance issues caused by
         *        pulling large Docker images down from the network.
         *        </p>
         *        <note>
         *        <ul>
         *        <li>
         *        <p>
         *        You can use a Docker layer cache in the Linux environment only.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        The <code>privileged</code> flag must be set so that your project has the required Docker permissions.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        You should consider the security implications before you use a Docker layer cache.
         *        </p>
         *        </li>
         *        </ul>
         *        </note></li>
         *        </ul>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>LOCAL_CUSTOM_CACHE</code> mode caches directories you specify in the buildspec file. This mode
         *        is a good choice if your build scenario is not suited to one of the other three local cache modes. If
         *        you use a custom cache:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        Only directories can be specified for caching. You cannot specify individual files.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Symlinks are used to reference cached directories.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Cached directories are linked to your build before it downloads its project sources. Cached items are
         *        overridden if a source item has the same name. Directories are specified using cache paths in the
         *        buildspec file.
         *        </p>
         *        </li>
         *        </ul>
         *        </li>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder modes(CacheMode... modes);
    }

    static final class BuilderImpl implements Builder {
        private String type;

        private String location;

        private List<String> modes = DefaultSdkAutoConstructList.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(ProjectCache model) {
            type(model.type);
            location(model.location);
            modesWithStrings(model.modes);
        }

        public final String getType() {
            return type;
        }

        @Override
        public final Builder type(String type) {
            this.type = type;
            return this;
        }

        @Override
        public final Builder type(CacheType type) {
            this.type(type == null ? null : type.toString());
            return this;
        }

        public final void setType(String type) {
            this.type = type;
        }

        public final String getLocation() {
            return location;
        }

        @Override
        public final Builder location(String location) {
            this.location = location;
            return this;
        }

        public final void setLocation(String location) {
            this.location = location;
        }

        public final Collection<String> getModes() {
            return modes;
        }

        @Override
        public final Builder modesWithStrings(Collection<String> modes) {
            this.modes = ProjectCacheModesCopier.copy(modes);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder modesWithStrings(String... modes) {
            modesWithStrings(Arrays.asList(modes));
            return this;
        }

        @Override
        public final Builder modes(Collection<CacheMode> modes) {
            this.modes = ProjectCacheModesCopier.copyEnumToString(modes);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder modes(CacheMode... modes) {
            modes(Arrays.asList(modes));
            return this;
        }

        public final void setModes(Collection<String> modes) {
            this.modes = ProjectCacheModesCopier.copy(modes);
        }

        @Override
        public ProjectCache build() {
            return new ProjectCache(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
