/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.codebuild;

import java.util.function.Consumer;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.AwsRequestOverrideConfiguration;
import software.amazon.awssdk.awscore.client.handler.AwsSyncClientHandler;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.core.ApiName;
import software.amazon.awssdk.core.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.client.handler.SyncClientHandler;
import software.amazon.awssdk.core.exception.SdkClientException;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.core.util.VersionInfo;
import software.amazon.awssdk.protocols.core.ExceptionMetadata;
import software.amazon.awssdk.protocols.json.AwsJsonProtocol;
import software.amazon.awssdk.protocols.json.AwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.BaseAwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.JsonOperationMetadata;
import software.amazon.awssdk.services.codebuild.model.AccountLimitExceededException;
import software.amazon.awssdk.services.codebuild.model.BatchDeleteBuildsRequest;
import software.amazon.awssdk.services.codebuild.model.BatchDeleteBuildsResponse;
import software.amazon.awssdk.services.codebuild.model.BatchGetBuildsRequest;
import software.amazon.awssdk.services.codebuild.model.BatchGetBuildsResponse;
import software.amazon.awssdk.services.codebuild.model.BatchGetProjectsRequest;
import software.amazon.awssdk.services.codebuild.model.BatchGetProjectsResponse;
import software.amazon.awssdk.services.codebuild.model.BatchGetReportGroupsRequest;
import software.amazon.awssdk.services.codebuild.model.BatchGetReportGroupsResponse;
import software.amazon.awssdk.services.codebuild.model.BatchGetReportsRequest;
import software.amazon.awssdk.services.codebuild.model.BatchGetReportsResponse;
import software.amazon.awssdk.services.codebuild.model.CodeBuildException;
import software.amazon.awssdk.services.codebuild.model.CodeBuildRequest;
import software.amazon.awssdk.services.codebuild.model.CreateProjectRequest;
import software.amazon.awssdk.services.codebuild.model.CreateProjectResponse;
import software.amazon.awssdk.services.codebuild.model.CreateReportGroupRequest;
import software.amazon.awssdk.services.codebuild.model.CreateReportGroupResponse;
import software.amazon.awssdk.services.codebuild.model.CreateWebhookRequest;
import software.amazon.awssdk.services.codebuild.model.CreateWebhookResponse;
import software.amazon.awssdk.services.codebuild.model.DeleteProjectRequest;
import software.amazon.awssdk.services.codebuild.model.DeleteProjectResponse;
import software.amazon.awssdk.services.codebuild.model.DeleteReportGroupRequest;
import software.amazon.awssdk.services.codebuild.model.DeleteReportGroupResponse;
import software.amazon.awssdk.services.codebuild.model.DeleteReportRequest;
import software.amazon.awssdk.services.codebuild.model.DeleteReportResponse;
import software.amazon.awssdk.services.codebuild.model.DeleteResourcePolicyRequest;
import software.amazon.awssdk.services.codebuild.model.DeleteResourcePolicyResponse;
import software.amazon.awssdk.services.codebuild.model.DeleteSourceCredentialsRequest;
import software.amazon.awssdk.services.codebuild.model.DeleteSourceCredentialsResponse;
import software.amazon.awssdk.services.codebuild.model.DeleteWebhookRequest;
import software.amazon.awssdk.services.codebuild.model.DeleteWebhookResponse;
import software.amazon.awssdk.services.codebuild.model.DescribeTestCasesRequest;
import software.amazon.awssdk.services.codebuild.model.DescribeTestCasesResponse;
import software.amazon.awssdk.services.codebuild.model.GetResourcePolicyRequest;
import software.amazon.awssdk.services.codebuild.model.GetResourcePolicyResponse;
import software.amazon.awssdk.services.codebuild.model.ImportSourceCredentialsRequest;
import software.amazon.awssdk.services.codebuild.model.ImportSourceCredentialsResponse;
import software.amazon.awssdk.services.codebuild.model.InvalidInputException;
import software.amazon.awssdk.services.codebuild.model.InvalidateProjectCacheRequest;
import software.amazon.awssdk.services.codebuild.model.InvalidateProjectCacheResponse;
import software.amazon.awssdk.services.codebuild.model.ListBuildsForProjectRequest;
import software.amazon.awssdk.services.codebuild.model.ListBuildsForProjectResponse;
import software.amazon.awssdk.services.codebuild.model.ListBuildsRequest;
import software.amazon.awssdk.services.codebuild.model.ListBuildsResponse;
import software.amazon.awssdk.services.codebuild.model.ListCuratedEnvironmentImagesRequest;
import software.amazon.awssdk.services.codebuild.model.ListCuratedEnvironmentImagesResponse;
import software.amazon.awssdk.services.codebuild.model.ListProjectsRequest;
import software.amazon.awssdk.services.codebuild.model.ListProjectsResponse;
import software.amazon.awssdk.services.codebuild.model.ListReportGroupsRequest;
import software.amazon.awssdk.services.codebuild.model.ListReportGroupsResponse;
import software.amazon.awssdk.services.codebuild.model.ListReportsForReportGroupRequest;
import software.amazon.awssdk.services.codebuild.model.ListReportsForReportGroupResponse;
import software.amazon.awssdk.services.codebuild.model.ListReportsRequest;
import software.amazon.awssdk.services.codebuild.model.ListReportsResponse;
import software.amazon.awssdk.services.codebuild.model.ListSharedProjectsRequest;
import software.amazon.awssdk.services.codebuild.model.ListSharedProjectsResponse;
import software.amazon.awssdk.services.codebuild.model.ListSharedReportGroupsRequest;
import software.amazon.awssdk.services.codebuild.model.ListSharedReportGroupsResponse;
import software.amazon.awssdk.services.codebuild.model.ListSourceCredentialsRequest;
import software.amazon.awssdk.services.codebuild.model.ListSourceCredentialsResponse;
import software.amazon.awssdk.services.codebuild.model.OAuthProviderException;
import software.amazon.awssdk.services.codebuild.model.PutResourcePolicyRequest;
import software.amazon.awssdk.services.codebuild.model.PutResourcePolicyResponse;
import software.amazon.awssdk.services.codebuild.model.ResourceAlreadyExistsException;
import software.amazon.awssdk.services.codebuild.model.ResourceNotFoundException;
import software.amazon.awssdk.services.codebuild.model.StartBuildRequest;
import software.amazon.awssdk.services.codebuild.model.StartBuildResponse;
import software.amazon.awssdk.services.codebuild.model.StopBuildRequest;
import software.amazon.awssdk.services.codebuild.model.StopBuildResponse;
import software.amazon.awssdk.services.codebuild.model.UpdateProjectRequest;
import software.amazon.awssdk.services.codebuild.model.UpdateProjectResponse;
import software.amazon.awssdk.services.codebuild.model.UpdateReportGroupRequest;
import software.amazon.awssdk.services.codebuild.model.UpdateReportGroupResponse;
import software.amazon.awssdk.services.codebuild.model.UpdateWebhookRequest;
import software.amazon.awssdk.services.codebuild.model.UpdateWebhookResponse;
import software.amazon.awssdk.services.codebuild.paginators.DescribeTestCasesIterable;
import software.amazon.awssdk.services.codebuild.paginators.ListBuildsForProjectIterable;
import software.amazon.awssdk.services.codebuild.paginators.ListBuildsIterable;
import software.amazon.awssdk.services.codebuild.paginators.ListProjectsIterable;
import software.amazon.awssdk.services.codebuild.paginators.ListReportGroupsIterable;
import software.amazon.awssdk.services.codebuild.paginators.ListReportsForReportGroupIterable;
import software.amazon.awssdk.services.codebuild.paginators.ListReportsIterable;
import software.amazon.awssdk.services.codebuild.paginators.ListSharedProjectsIterable;
import software.amazon.awssdk.services.codebuild.paginators.ListSharedReportGroupsIterable;
import software.amazon.awssdk.services.codebuild.transform.BatchDeleteBuildsRequestMarshaller;
import software.amazon.awssdk.services.codebuild.transform.BatchGetBuildsRequestMarshaller;
import software.amazon.awssdk.services.codebuild.transform.BatchGetProjectsRequestMarshaller;
import software.amazon.awssdk.services.codebuild.transform.BatchGetReportGroupsRequestMarshaller;
import software.amazon.awssdk.services.codebuild.transform.BatchGetReportsRequestMarshaller;
import software.amazon.awssdk.services.codebuild.transform.CreateProjectRequestMarshaller;
import software.amazon.awssdk.services.codebuild.transform.CreateReportGroupRequestMarshaller;
import software.amazon.awssdk.services.codebuild.transform.CreateWebhookRequestMarshaller;
import software.amazon.awssdk.services.codebuild.transform.DeleteProjectRequestMarshaller;
import software.amazon.awssdk.services.codebuild.transform.DeleteReportGroupRequestMarshaller;
import software.amazon.awssdk.services.codebuild.transform.DeleteReportRequestMarshaller;
import software.amazon.awssdk.services.codebuild.transform.DeleteResourcePolicyRequestMarshaller;
import software.amazon.awssdk.services.codebuild.transform.DeleteSourceCredentialsRequestMarshaller;
import software.amazon.awssdk.services.codebuild.transform.DeleteWebhookRequestMarshaller;
import software.amazon.awssdk.services.codebuild.transform.DescribeTestCasesRequestMarshaller;
import software.amazon.awssdk.services.codebuild.transform.GetResourcePolicyRequestMarshaller;
import software.amazon.awssdk.services.codebuild.transform.ImportSourceCredentialsRequestMarshaller;
import software.amazon.awssdk.services.codebuild.transform.InvalidateProjectCacheRequestMarshaller;
import software.amazon.awssdk.services.codebuild.transform.ListBuildsForProjectRequestMarshaller;
import software.amazon.awssdk.services.codebuild.transform.ListBuildsRequestMarshaller;
import software.amazon.awssdk.services.codebuild.transform.ListCuratedEnvironmentImagesRequestMarshaller;
import software.amazon.awssdk.services.codebuild.transform.ListProjectsRequestMarshaller;
import software.amazon.awssdk.services.codebuild.transform.ListReportGroupsRequestMarshaller;
import software.amazon.awssdk.services.codebuild.transform.ListReportsForReportGroupRequestMarshaller;
import software.amazon.awssdk.services.codebuild.transform.ListReportsRequestMarshaller;
import software.amazon.awssdk.services.codebuild.transform.ListSharedProjectsRequestMarshaller;
import software.amazon.awssdk.services.codebuild.transform.ListSharedReportGroupsRequestMarshaller;
import software.amazon.awssdk.services.codebuild.transform.ListSourceCredentialsRequestMarshaller;
import software.amazon.awssdk.services.codebuild.transform.PutResourcePolicyRequestMarshaller;
import software.amazon.awssdk.services.codebuild.transform.StartBuildRequestMarshaller;
import software.amazon.awssdk.services.codebuild.transform.StopBuildRequestMarshaller;
import software.amazon.awssdk.services.codebuild.transform.UpdateProjectRequestMarshaller;
import software.amazon.awssdk.services.codebuild.transform.UpdateReportGroupRequestMarshaller;
import software.amazon.awssdk.services.codebuild.transform.UpdateWebhookRequestMarshaller;

/**
 * Internal implementation of {@link CodeBuildClient}.
 *
 * @see CodeBuildClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultCodeBuildClient implements CodeBuildClient {
    private final SyncClientHandler clientHandler;

    private final AwsJsonProtocolFactory protocolFactory;

    private final SdkClientConfiguration clientConfiguration;

    protected DefaultCodeBuildClient(SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsSyncClientHandler(clientConfiguration);
        this.clientConfiguration = clientConfiguration;
        this.protocolFactory = init(AwsJsonProtocolFactory.builder()).build();
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    /**
     * <p>
     * Deletes one or more builds.
     * </p>
     *
     * @param batchDeleteBuildsRequest
     * @return Result of the BatchDeleteBuilds operation returned by the service.
     * @throws InvalidInputException
     *         The input value that was provided is not valid.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws CodeBuildException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample CodeBuildClient.BatchDeleteBuilds
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/codebuild-2016-10-06/BatchDeleteBuilds" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public BatchDeleteBuildsResponse batchDeleteBuilds(BatchDeleteBuildsRequest batchDeleteBuildsRequest)
            throws InvalidInputException, AwsServiceException, SdkClientException, CodeBuildException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<BatchDeleteBuildsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                BatchDeleteBuildsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<BatchDeleteBuildsRequest, BatchDeleteBuildsResponse>()
                .withOperationName("BatchDeleteBuilds").withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(batchDeleteBuildsRequest)
                .withMarshaller(new BatchDeleteBuildsRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Gets information about one or more builds.
     * </p>
     *
     * @param batchGetBuildsRequest
     * @return Result of the BatchGetBuilds operation returned by the service.
     * @throws InvalidInputException
     *         The input value that was provided is not valid.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws CodeBuildException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample CodeBuildClient.BatchGetBuilds
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/codebuild-2016-10-06/BatchGetBuilds" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public BatchGetBuildsResponse batchGetBuilds(BatchGetBuildsRequest batchGetBuildsRequest) throws InvalidInputException,
            AwsServiceException, SdkClientException, CodeBuildException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<BatchGetBuildsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                BatchGetBuildsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<BatchGetBuildsRequest, BatchGetBuildsResponse>()
                .withOperationName("BatchGetBuilds").withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(batchGetBuildsRequest)
                .withMarshaller(new BatchGetBuildsRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Gets information about one or more build projects.
     * </p>
     *
     * @param batchGetProjectsRequest
     * @return Result of the BatchGetProjects operation returned by the service.
     * @throws InvalidInputException
     *         The input value that was provided is not valid.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws CodeBuildException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample CodeBuildClient.BatchGetProjects
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/codebuild-2016-10-06/BatchGetProjects" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public BatchGetProjectsResponse batchGetProjects(BatchGetProjectsRequest batchGetProjectsRequest)
            throws InvalidInputException, AwsServiceException, SdkClientException, CodeBuildException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<BatchGetProjectsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                BatchGetProjectsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<BatchGetProjectsRequest, BatchGetProjectsResponse>()
                .withOperationName("BatchGetProjects").withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(batchGetProjectsRequest)
                .withMarshaller(new BatchGetProjectsRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Returns an array of report groups.
     * </p>
     *
     * @param batchGetReportGroupsRequest
     * @return Result of the BatchGetReportGroups operation returned by the service.
     * @throws InvalidInputException
     *         The input value that was provided is not valid.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws CodeBuildException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample CodeBuildClient.BatchGetReportGroups
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/codebuild-2016-10-06/BatchGetReportGroups" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public BatchGetReportGroupsResponse batchGetReportGroups(BatchGetReportGroupsRequest batchGetReportGroupsRequest)
            throws InvalidInputException, AwsServiceException, SdkClientException, CodeBuildException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<BatchGetReportGroupsResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, BatchGetReportGroupsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<BatchGetReportGroupsRequest, BatchGetReportGroupsResponse>()
                .withOperationName("BatchGetReportGroups").withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(batchGetReportGroupsRequest)
                .withMarshaller(new BatchGetReportGroupsRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Returns an array of reports.
     * </p>
     *
     * @param batchGetReportsRequest
     * @return Result of the BatchGetReports operation returned by the service.
     * @throws InvalidInputException
     *         The input value that was provided is not valid.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws CodeBuildException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample CodeBuildClient.BatchGetReports
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/codebuild-2016-10-06/BatchGetReports" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public BatchGetReportsResponse batchGetReports(BatchGetReportsRequest batchGetReportsRequest) throws InvalidInputException,
            AwsServiceException, SdkClientException, CodeBuildException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<BatchGetReportsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                BatchGetReportsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<BatchGetReportsRequest, BatchGetReportsResponse>()
                .withOperationName("BatchGetReports").withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(batchGetReportsRequest)
                .withMarshaller(new BatchGetReportsRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Creates a build project.
     * </p>
     *
     * @param createProjectRequest
     * @return Result of the CreateProject operation returned by the service.
     * @throws InvalidInputException
     *         The input value that was provided is not valid.
     * @throws ResourceAlreadyExistsException
     *         The specified AWS resource cannot be created, because an AWS resource with the same settings already
     *         exists.
     * @throws AccountLimitExceededException
     *         An AWS service limit was exceeded for the calling AWS account.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws CodeBuildException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample CodeBuildClient.CreateProject
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/codebuild-2016-10-06/CreateProject" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CreateProjectResponse createProject(CreateProjectRequest createProjectRequest) throws InvalidInputException,
            ResourceAlreadyExistsException, AccountLimitExceededException, AwsServiceException, SdkClientException,
            CodeBuildException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<CreateProjectResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                CreateProjectResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<CreateProjectRequest, CreateProjectResponse>()
                .withOperationName("CreateProject").withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(createProjectRequest)
                .withMarshaller(new CreateProjectRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Creates a report group. A report group contains a collection of reports.
     * </p>
     *
     * @param createReportGroupRequest
     * @return Result of the CreateReportGroup operation returned by the service.
     * @throws InvalidInputException
     *         The input value that was provided is not valid.
     * @throws ResourceAlreadyExistsException
     *         The specified AWS resource cannot be created, because an AWS resource with the same settings already
     *         exists.
     * @throws AccountLimitExceededException
     *         An AWS service limit was exceeded for the calling AWS account.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws CodeBuildException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample CodeBuildClient.CreateReportGroup
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/codebuild-2016-10-06/CreateReportGroup" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CreateReportGroupResponse createReportGroup(CreateReportGroupRequest createReportGroupRequest)
            throws InvalidInputException, ResourceAlreadyExistsException, AccountLimitExceededException, AwsServiceException,
            SdkClientException, CodeBuildException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<CreateReportGroupResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                CreateReportGroupResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<CreateReportGroupRequest, CreateReportGroupResponse>()
                .withOperationName("CreateReportGroup").withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(createReportGroupRequest)
                .withMarshaller(new CreateReportGroupRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * For an existing AWS CodeBuild build project that has its source code stored in a GitHub or Bitbucket repository,
     * enables AWS CodeBuild to start rebuilding the source code every time a code change is pushed to the repository.
     * </p>
     * <important>
     * <p>
     * If you enable webhooks for an AWS CodeBuild project, and the project is used as a build step in AWS CodePipeline,
     * then two identical builds are created for each commit. One build is triggered through webhooks, and one through
     * AWS CodePipeline. Because billing is on a per-build basis, you are billed for both builds. Therefore, if you are
     * using AWS CodePipeline, we recommend that you disable webhooks in AWS CodeBuild. In the AWS CodeBuild console,
     * clear the Webhook box. For more information, see step 5 in <a
     * href="https://docs.aws.amazon.com/codebuild/latest/userguide/change-project.html#change-project-console">Change a
     * Build Project's Settings</a>.
     * </p>
     * </important>
     *
     * @param createWebhookRequest
     * @return Result of the CreateWebhook operation returned by the service.
     * @throws InvalidInputException
     *         The input value that was provided is not valid.
     * @throws OAuthProviderException
     *         There was a problem with the underlying OAuth provider.
     * @throws ResourceAlreadyExistsException
     *         The specified AWS resource cannot be created, because an AWS resource with the same settings already
     *         exists.
     * @throws ResourceNotFoundException
     *         The specified AWS resource cannot be found.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws CodeBuildException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample CodeBuildClient.CreateWebhook
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/codebuild-2016-10-06/CreateWebhook" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CreateWebhookResponse createWebhook(CreateWebhookRequest createWebhookRequest) throws InvalidInputException,
            OAuthProviderException, ResourceAlreadyExistsException, ResourceNotFoundException, AwsServiceException,
            SdkClientException, CodeBuildException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<CreateWebhookResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                CreateWebhookResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<CreateWebhookRequest, CreateWebhookResponse>()
                .withOperationName("CreateWebhook").withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(createWebhookRequest)
                .withMarshaller(new CreateWebhookRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Deletes a build project. When you delete a project, its builds are not deleted.
     * </p>
     *
     * @param deleteProjectRequest
     * @return Result of the DeleteProject operation returned by the service.
     * @throws InvalidInputException
     *         The input value that was provided is not valid.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws CodeBuildException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample CodeBuildClient.DeleteProject
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/codebuild-2016-10-06/DeleteProject" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public DeleteProjectResponse deleteProject(DeleteProjectRequest deleteProjectRequest) throws InvalidInputException,
            AwsServiceException, SdkClientException, CodeBuildException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DeleteProjectResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                DeleteProjectResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<DeleteProjectRequest, DeleteProjectResponse>()
                .withOperationName("DeleteProject").withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(deleteProjectRequest)
                .withMarshaller(new DeleteProjectRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Deletes a report.
     * </p>
     *
     * @param deleteReportRequest
     * @return Result of the DeleteReport operation returned by the service.
     * @throws InvalidInputException
     *         The input value that was provided is not valid.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws CodeBuildException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample CodeBuildClient.DeleteReport
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/codebuild-2016-10-06/DeleteReport" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public DeleteReportResponse deleteReport(DeleteReportRequest deleteReportRequest) throws InvalidInputException,
            AwsServiceException, SdkClientException, CodeBuildException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DeleteReportResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                DeleteReportResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<DeleteReportRequest, DeleteReportResponse>()
                .withOperationName("DeleteReport").withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(deleteReportRequest)
                .withMarshaller(new DeleteReportRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * <code>DeleteReportGroup</code>: Deletes a report group. Before you delete a report group, you must delete its
     * reports. Use <a
     * href="https://docs.aws.amazon.com/codebuild/latest/APIReference/API_ListReportsForReportGroup.html"
     * >ListReportsForReportGroup</a> to get the reports in a report group. Use <a
     * href="https://docs.aws.amazon.com/codebuild/latest/APIReference/API_DeleteReport.html">DeleteReport</a> to delete
     * the reports. If you call <code>DeleteReportGroup</code> for a report group that contains one or more reports, an
     * exception is thrown.
     * </p>
     *
     * @param deleteReportGroupRequest
     * @return Result of the DeleteReportGroup operation returned by the service.
     * @throws InvalidInputException
     *         The input value that was provided is not valid.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws CodeBuildException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample CodeBuildClient.DeleteReportGroup
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/codebuild-2016-10-06/DeleteReportGroup" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public DeleteReportGroupResponse deleteReportGroup(DeleteReportGroupRequest deleteReportGroupRequest)
            throws InvalidInputException, AwsServiceException, SdkClientException, CodeBuildException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DeleteReportGroupResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                DeleteReportGroupResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<DeleteReportGroupRequest, DeleteReportGroupResponse>()
                .withOperationName("DeleteReportGroup").withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(deleteReportGroupRequest)
                .withMarshaller(new DeleteReportGroupRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Deletes a resource policy that is identified by its resource ARN.
     * </p>
     *
     * @param deleteResourcePolicyRequest
     * @return Result of the DeleteResourcePolicy operation returned by the service.
     * @throws InvalidInputException
     *         The input value that was provided is not valid.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws CodeBuildException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample CodeBuildClient.DeleteResourcePolicy
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/codebuild-2016-10-06/DeleteResourcePolicy" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public DeleteResourcePolicyResponse deleteResourcePolicy(DeleteResourcePolicyRequest deleteResourcePolicyRequest)
            throws InvalidInputException, AwsServiceException, SdkClientException, CodeBuildException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DeleteResourcePolicyResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, DeleteResourcePolicyResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<DeleteResourcePolicyRequest, DeleteResourcePolicyResponse>()
                .withOperationName("DeleteResourcePolicy").withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(deleteResourcePolicyRequest)
                .withMarshaller(new DeleteResourcePolicyRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Deletes a set of GitHub, GitHub Enterprise, or Bitbucket source credentials.
     * </p>
     *
     * @param deleteSourceCredentialsRequest
     * @return Result of the DeleteSourceCredentials operation returned by the service.
     * @throws InvalidInputException
     *         The input value that was provided is not valid.
     * @throws ResourceNotFoundException
     *         The specified AWS resource cannot be found.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws CodeBuildException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample CodeBuildClient.DeleteSourceCredentials
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/codebuild-2016-10-06/DeleteSourceCredentials"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DeleteSourceCredentialsResponse deleteSourceCredentials(DeleteSourceCredentialsRequest deleteSourceCredentialsRequest)
            throws InvalidInputException, ResourceNotFoundException, AwsServiceException, SdkClientException, CodeBuildException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DeleteSourceCredentialsResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, DeleteSourceCredentialsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<DeleteSourceCredentialsRequest, DeleteSourceCredentialsResponse>()
                .withOperationName("DeleteSourceCredentials").withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(deleteSourceCredentialsRequest)
                .withMarshaller(new DeleteSourceCredentialsRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * For an existing AWS CodeBuild build project that has its source code stored in a GitHub or Bitbucket repository,
     * stops AWS CodeBuild from rebuilding the source code every time a code change is pushed to the repository.
     * </p>
     *
     * @param deleteWebhookRequest
     * @return Result of the DeleteWebhook operation returned by the service.
     * @throws InvalidInputException
     *         The input value that was provided is not valid.
     * @throws ResourceNotFoundException
     *         The specified AWS resource cannot be found.
     * @throws OAuthProviderException
     *         There was a problem with the underlying OAuth provider.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws CodeBuildException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample CodeBuildClient.DeleteWebhook
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/codebuild-2016-10-06/DeleteWebhook" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public DeleteWebhookResponse deleteWebhook(DeleteWebhookRequest deleteWebhookRequest) throws InvalidInputException,
            ResourceNotFoundException, OAuthProviderException, AwsServiceException, SdkClientException, CodeBuildException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DeleteWebhookResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                DeleteWebhookResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<DeleteWebhookRequest, DeleteWebhookResponse>()
                .withOperationName("DeleteWebhook").withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(deleteWebhookRequest)
                .withMarshaller(new DeleteWebhookRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Returns a list of details about test cases for a report.
     * </p>
     *
     * @param describeTestCasesRequest
     * @return Result of the DescribeTestCases operation returned by the service.
     * @throws InvalidInputException
     *         The input value that was provided is not valid.
     * @throws ResourceNotFoundException
     *         The specified AWS resource cannot be found.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws CodeBuildException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample CodeBuildClient.DescribeTestCases
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/codebuild-2016-10-06/DescribeTestCases" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public DescribeTestCasesResponse describeTestCases(DescribeTestCasesRequest describeTestCasesRequest)
            throws InvalidInputException, ResourceNotFoundException, AwsServiceException, SdkClientException, CodeBuildException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DescribeTestCasesResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                DescribeTestCasesResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<DescribeTestCasesRequest, DescribeTestCasesResponse>()
                .withOperationName("DescribeTestCases").withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(describeTestCasesRequest)
                .withMarshaller(new DescribeTestCasesRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Returns a list of details about test cases for a report.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #describeTestCases(software.amazon.awssdk.services.codebuild.model.DescribeTestCasesRequest)} operation.
     * The return type is a custom iterable that can be used to iterate through all the pages. SDK will internally
     * handle making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.codebuild.paginators.DescribeTestCasesIterable responses = client.describeTestCasesPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.codebuild.paginators.DescribeTestCasesIterable responses = client
     *             .describeTestCasesPaginator(request);
     *     for (software.amazon.awssdk.services.codebuild.model.DescribeTestCasesResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.codebuild.paginators.DescribeTestCasesIterable responses = client.describeTestCasesPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of maxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #describeTestCases(software.amazon.awssdk.services.codebuild.model.DescribeTestCasesRequest)}
     * operation.</b>
     * </p>
     *
     * @param describeTestCasesRequest
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws InvalidInputException
     *         The input value that was provided is not valid.
     * @throws ResourceNotFoundException
     *         The specified AWS resource cannot be found.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws CodeBuildException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample CodeBuildClient.DescribeTestCases
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/codebuild-2016-10-06/DescribeTestCases" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public DescribeTestCasesIterable describeTestCasesPaginator(DescribeTestCasesRequest describeTestCasesRequest)
            throws InvalidInputException, ResourceNotFoundException, AwsServiceException, SdkClientException, CodeBuildException {
        return new DescribeTestCasesIterable(this, applyPaginatorUserAgent(describeTestCasesRequest));
    }

    /**
     * <p>
     * Gets a resource policy that is identified by its resource ARN.
     * </p>
     *
     * @param getResourcePolicyRequest
     * @return Result of the GetResourcePolicy operation returned by the service.
     * @throws ResourceNotFoundException
     *         The specified AWS resource cannot be found.
     * @throws InvalidInputException
     *         The input value that was provided is not valid.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws CodeBuildException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample CodeBuildClient.GetResourcePolicy
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/codebuild-2016-10-06/GetResourcePolicy" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public GetResourcePolicyResponse getResourcePolicy(GetResourcePolicyRequest getResourcePolicyRequest)
            throws ResourceNotFoundException, InvalidInputException, AwsServiceException, SdkClientException, CodeBuildException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetResourcePolicyResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                GetResourcePolicyResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<GetResourcePolicyRequest, GetResourcePolicyResponse>()
                .withOperationName("GetResourcePolicy").withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(getResourcePolicyRequest)
                .withMarshaller(new GetResourcePolicyRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Imports the source repository credentials for an AWS CodeBuild project that has its source code stored in a
     * GitHub, GitHub Enterprise, or Bitbucket repository.
     * </p>
     *
     * @param importSourceCredentialsRequest
     * @return Result of the ImportSourceCredentials operation returned by the service.
     * @throws InvalidInputException
     *         The input value that was provided is not valid.
     * @throws AccountLimitExceededException
     *         An AWS service limit was exceeded for the calling AWS account.
     * @throws ResourceAlreadyExistsException
     *         The specified AWS resource cannot be created, because an AWS resource with the same settings already
     *         exists.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws CodeBuildException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample CodeBuildClient.ImportSourceCredentials
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/codebuild-2016-10-06/ImportSourceCredentials"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ImportSourceCredentialsResponse importSourceCredentials(ImportSourceCredentialsRequest importSourceCredentialsRequest)
            throws InvalidInputException, AccountLimitExceededException, ResourceAlreadyExistsException, AwsServiceException,
            SdkClientException, CodeBuildException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ImportSourceCredentialsResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, ImportSourceCredentialsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<ImportSourceCredentialsRequest, ImportSourceCredentialsResponse>()
                .withOperationName("ImportSourceCredentials").withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(importSourceCredentialsRequest)
                .withMarshaller(new ImportSourceCredentialsRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Resets the cache for a project.
     * </p>
     *
     * @param invalidateProjectCacheRequest
     * @return Result of the InvalidateProjectCache operation returned by the service.
     * @throws InvalidInputException
     *         The input value that was provided is not valid.
     * @throws ResourceNotFoundException
     *         The specified AWS resource cannot be found.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws CodeBuildException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample CodeBuildClient.InvalidateProjectCache
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/codebuild-2016-10-06/InvalidateProjectCache"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public InvalidateProjectCacheResponse invalidateProjectCache(InvalidateProjectCacheRequest invalidateProjectCacheRequest)
            throws InvalidInputException, ResourceNotFoundException, AwsServiceException, SdkClientException, CodeBuildException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<InvalidateProjectCacheResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, InvalidateProjectCacheResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<InvalidateProjectCacheRequest, InvalidateProjectCacheResponse>()
                .withOperationName("InvalidateProjectCache").withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(invalidateProjectCacheRequest)
                .withMarshaller(new InvalidateProjectCacheRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Gets a list of build IDs, with each build ID representing a single build.
     * </p>
     *
     * @param listBuildsRequest
     * @return Result of the ListBuilds operation returned by the service.
     * @throws InvalidInputException
     *         The input value that was provided is not valid.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws CodeBuildException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample CodeBuildClient.ListBuilds
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/codebuild-2016-10-06/ListBuilds" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public ListBuildsResponse listBuilds(ListBuildsRequest listBuildsRequest) throws InvalidInputException, AwsServiceException,
            SdkClientException, CodeBuildException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListBuildsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                ListBuildsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<ListBuildsRequest, ListBuildsResponse>()
                .withOperationName("ListBuilds").withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(listBuildsRequest)
                .withMarshaller(new ListBuildsRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Gets a list of build IDs, with each build ID representing a single build.
     * </p>
     * <br/>
     * <p>
     * This is a variant of {@link #listBuilds(software.amazon.awssdk.services.codebuild.model.ListBuildsRequest)}
     * operation. The return type is a custom iterable that can be used to iterate through all the pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.codebuild.paginators.ListBuildsIterable responses = client.listBuildsPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.codebuild.paginators.ListBuildsIterable responses = client.listBuildsPaginator(request);
     *     for (software.amazon.awssdk.services.codebuild.model.ListBuildsResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.codebuild.paginators.ListBuildsIterable responses = client.listBuildsPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of null won't limit the number of results you get with the paginator. It
     * only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listBuilds(software.amazon.awssdk.services.codebuild.model.ListBuildsRequest)} operation.</b>
     * </p>
     *
     * @param listBuildsRequest
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws InvalidInputException
     *         The input value that was provided is not valid.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws CodeBuildException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample CodeBuildClient.ListBuilds
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/codebuild-2016-10-06/ListBuilds" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public ListBuildsIterable listBuildsPaginator(ListBuildsRequest listBuildsRequest) throws InvalidInputException,
            AwsServiceException, SdkClientException, CodeBuildException {
        return new ListBuildsIterable(this, applyPaginatorUserAgent(listBuildsRequest));
    }

    /**
     * <p>
     * Gets a list of build IDs for the specified build project, with each build ID representing a single build.
     * </p>
     *
     * @param listBuildsForProjectRequest
     * @return Result of the ListBuildsForProject operation returned by the service.
     * @throws InvalidInputException
     *         The input value that was provided is not valid.
     * @throws ResourceNotFoundException
     *         The specified AWS resource cannot be found.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws CodeBuildException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample CodeBuildClient.ListBuildsForProject
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/codebuild-2016-10-06/ListBuildsForProject" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public ListBuildsForProjectResponse listBuildsForProject(ListBuildsForProjectRequest listBuildsForProjectRequest)
            throws InvalidInputException, ResourceNotFoundException, AwsServiceException, SdkClientException, CodeBuildException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListBuildsForProjectResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, ListBuildsForProjectResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<ListBuildsForProjectRequest, ListBuildsForProjectResponse>()
                .withOperationName("ListBuildsForProject").withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(listBuildsForProjectRequest)
                .withMarshaller(new ListBuildsForProjectRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Gets a list of build IDs for the specified build project, with each build ID representing a single build.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #listBuildsForProject(software.amazon.awssdk.services.codebuild.model.ListBuildsForProjectRequest)}
     * operation. The return type is a custom iterable that can be used to iterate through all the pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.codebuild.paginators.ListBuildsForProjectIterable responses = client.listBuildsForProjectPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.codebuild.paginators.ListBuildsForProjectIterable responses = client
     *             .listBuildsForProjectPaginator(request);
     *     for (software.amazon.awssdk.services.codebuild.model.ListBuildsForProjectResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.codebuild.paginators.ListBuildsForProjectIterable responses = client.listBuildsForProjectPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of null won't limit the number of results you get with the paginator. It
     * only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listBuildsForProject(software.amazon.awssdk.services.codebuild.model.ListBuildsForProjectRequest)}
     * operation.</b>
     * </p>
     *
     * @param listBuildsForProjectRequest
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws InvalidInputException
     *         The input value that was provided is not valid.
     * @throws ResourceNotFoundException
     *         The specified AWS resource cannot be found.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws CodeBuildException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample CodeBuildClient.ListBuildsForProject
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/codebuild-2016-10-06/ListBuildsForProject" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public ListBuildsForProjectIterable listBuildsForProjectPaginator(ListBuildsForProjectRequest listBuildsForProjectRequest)
            throws InvalidInputException, ResourceNotFoundException, AwsServiceException, SdkClientException, CodeBuildException {
        return new ListBuildsForProjectIterable(this, applyPaginatorUserAgent(listBuildsForProjectRequest));
    }

    /**
     * <p>
     * Gets information about Docker images that are managed by AWS CodeBuild.
     * </p>
     *
     * @param listCuratedEnvironmentImagesRequest
     * @return Result of the ListCuratedEnvironmentImages operation returned by the service.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws CodeBuildException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample CodeBuildClient.ListCuratedEnvironmentImages
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/codebuild-2016-10-06/ListCuratedEnvironmentImages"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListCuratedEnvironmentImagesResponse listCuratedEnvironmentImages(
            ListCuratedEnvironmentImagesRequest listCuratedEnvironmentImagesRequest) throws AwsServiceException,
            SdkClientException, CodeBuildException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListCuratedEnvironmentImagesResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, ListCuratedEnvironmentImagesResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler
                .execute(new ClientExecutionParams<ListCuratedEnvironmentImagesRequest, ListCuratedEnvironmentImagesResponse>()
                        .withOperationName("ListCuratedEnvironmentImages").withResponseHandler(responseHandler)
                        .withErrorResponseHandler(errorResponseHandler).withInput(listCuratedEnvironmentImagesRequest)
                        .withMarshaller(new ListCuratedEnvironmentImagesRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Gets a list of build project names, with each build project name representing a single build project.
     * </p>
     *
     * @param listProjectsRequest
     * @return Result of the ListProjects operation returned by the service.
     * @throws InvalidInputException
     *         The input value that was provided is not valid.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws CodeBuildException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample CodeBuildClient.ListProjects
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/codebuild-2016-10-06/ListProjects" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public ListProjectsResponse listProjects(ListProjectsRequest listProjectsRequest) throws InvalidInputException,
            AwsServiceException, SdkClientException, CodeBuildException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListProjectsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                ListProjectsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<ListProjectsRequest, ListProjectsResponse>()
                .withOperationName("ListProjects").withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(listProjectsRequest)
                .withMarshaller(new ListProjectsRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Gets a list of build project names, with each build project name representing a single build project.
     * </p>
     * <br/>
     * <p>
     * This is a variant of {@link #listProjects(software.amazon.awssdk.services.codebuild.model.ListProjectsRequest)}
     * operation. The return type is a custom iterable that can be used to iterate through all the pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.codebuild.paginators.ListProjectsIterable responses = client.listProjectsPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.codebuild.paginators.ListProjectsIterable responses = client.listProjectsPaginator(request);
     *     for (software.amazon.awssdk.services.codebuild.model.ListProjectsResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.codebuild.paginators.ListProjectsIterable responses = client.listProjectsPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of null won't limit the number of results you get with the paginator. It
     * only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listProjects(software.amazon.awssdk.services.codebuild.model.ListProjectsRequest)} operation.</b>
     * </p>
     *
     * @param listProjectsRequest
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws InvalidInputException
     *         The input value that was provided is not valid.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws CodeBuildException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample CodeBuildClient.ListProjects
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/codebuild-2016-10-06/ListProjects" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public ListProjectsIterable listProjectsPaginator(ListProjectsRequest listProjectsRequest) throws InvalidInputException,
            AwsServiceException, SdkClientException, CodeBuildException {
        return new ListProjectsIterable(this, applyPaginatorUserAgent(listProjectsRequest));
    }

    /**
     * <p>
     * Gets a list ARNs for the report groups in the current AWS account.
     * </p>
     *
     * @param listReportGroupsRequest
     * @return Result of the ListReportGroups operation returned by the service.
     * @throws InvalidInputException
     *         The input value that was provided is not valid.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws CodeBuildException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample CodeBuildClient.ListReportGroups
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/codebuild-2016-10-06/ListReportGroups" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public ListReportGroupsResponse listReportGroups(ListReportGroupsRequest listReportGroupsRequest)
            throws InvalidInputException, AwsServiceException, SdkClientException, CodeBuildException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListReportGroupsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                ListReportGroupsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<ListReportGroupsRequest, ListReportGroupsResponse>()
                .withOperationName("ListReportGroups").withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(listReportGroupsRequest)
                .withMarshaller(new ListReportGroupsRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Gets a list ARNs for the report groups in the current AWS account.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #listReportGroups(software.amazon.awssdk.services.codebuild.model.ListReportGroupsRequest)} operation. The
     * return type is a custom iterable that can be used to iterate through all the pages. SDK will internally handle
     * making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.codebuild.paginators.ListReportGroupsIterable responses = client.listReportGroupsPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.codebuild.paginators.ListReportGroupsIterable responses = client
     *             .listReportGroupsPaginator(request);
     *     for (software.amazon.awssdk.services.codebuild.model.ListReportGroupsResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.codebuild.paginators.ListReportGroupsIterable responses = client.listReportGroupsPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of maxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listReportGroups(software.amazon.awssdk.services.codebuild.model.ListReportGroupsRequest)} operation.</b>
     * </p>
     *
     * @param listReportGroupsRequest
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws InvalidInputException
     *         The input value that was provided is not valid.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws CodeBuildException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample CodeBuildClient.ListReportGroups
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/codebuild-2016-10-06/ListReportGroups" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public ListReportGroupsIterable listReportGroupsPaginator(ListReportGroupsRequest listReportGroupsRequest)
            throws InvalidInputException, AwsServiceException, SdkClientException, CodeBuildException {
        return new ListReportGroupsIterable(this, applyPaginatorUserAgent(listReportGroupsRequest));
    }

    /**
     * <p>
     * Returns a list of ARNs for the reports in the current AWS account.
     * </p>
     *
     * @param listReportsRequest
     * @return Result of the ListReports operation returned by the service.
     * @throws InvalidInputException
     *         The input value that was provided is not valid.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws CodeBuildException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample CodeBuildClient.ListReports
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/codebuild-2016-10-06/ListReports" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public ListReportsResponse listReports(ListReportsRequest listReportsRequest) throws InvalidInputException,
            AwsServiceException, SdkClientException, CodeBuildException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListReportsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                ListReportsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<ListReportsRequest, ListReportsResponse>()
                .withOperationName("ListReports").withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(listReportsRequest)
                .withMarshaller(new ListReportsRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Returns a list of ARNs for the reports in the current AWS account.
     * </p>
     * <br/>
     * <p>
     * This is a variant of {@link #listReports(software.amazon.awssdk.services.codebuild.model.ListReportsRequest)}
     * operation. The return type is a custom iterable that can be used to iterate through all the pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.codebuild.paginators.ListReportsIterable responses = client.listReportsPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.codebuild.paginators.ListReportsIterable responses = client.listReportsPaginator(request);
     *     for (software.amazon.awssdk.services.codebuild.model.ListReportsResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.codebuild.paginators.ListReportsIterable responses = client.listReportsPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of maxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listReports(software.amazon.awssdk.services.codebuild.model.ListReportsRequest)} operation.</b>
     * </p>
     *
     * @param listReportsRequest
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws InvalidInputException
     *         The input value that was provided is not valid.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws CodeBuildException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample CodeBuildClient.ListReports
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/codebuild-2016-10-06/ListReports" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public ListReportsIterable listReportsPaginator(ListReportsRequest listReportsRequest) throws InvalidInputException,
            AwsServiceException, SdkClientException, CodeBuildException {
        return new ListReportsIterable(this, applyPaginatorUserAgent(listReportsRequest));
    }

    /**
     * <p>
     * Returns a list of ARNs for the reports that belong to a <code>ReportGroup</code>.
     * </p>
     *
     * @param listReportsForReportGroupRequest
     * @return Result of the ListReportsForReportGroup operation returned by the service.
     * @throws InvalidInputException
     *         The input value that was provided is not valid.
     * @throws ResourceNotFoundException
     *         The specified AWS resource cannot be found.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws CodeBuildException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample CodeBuildClient.ListReportsForReportGroup
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/codebuild-2016-10-06/ListReportsForReportGroup"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListReportsForReportGroupResponse listReportsForReportGroup(
            ListReportsForReportGroupRequest listReportsForReportGroupRequest) throws InvalidInputException,
            ResourceNotFoundException, AwsServiceException, SdkClientException, CodeBuildException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListReportsForReportGroupResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, ListReportsForReportGroupResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler
                .execute(new ClientExecutionParams<ListReportsForReportGroupRequest, ListReportsForReportGroupResponse>()
                        .withOperationName("ListReportsForReportGroup").withResponseHandler(responseHandler)
                        .withErrorResponseHandler(errorResponseHandler).withInput(listReportsForReportGroupRequest)
                        .withMarshaller(new ListReportsForReportGroupRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Returns a list of ARNs for the reports that belong to a <code>ReportGroup</code>.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #listReportsForReportGroup(software.amazon.awssdk.services.codebuild.model.ListReportsForReportGroupRequest)}
     * operation. The return type is a custom iterable that can be used to iterate through all the pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.codebuild.paginators.ListReportsForReportGroupIterable responses = client.listReportsForReportGroupPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.codebuild.paginators.ListReportsForReportGroupIterable responses = client
     *             .listReportsForReportGroupPaginator(request);
     *     for (software.amazon.awssdk.services.codebuild.model.ListReportsForReportGroupResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.codebuild.paginators.ListReportsForReportGroupIterable responses = client.listReportsForReportGroupPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of maxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listReportsForReportGroup(software.amazon.awssdk.services.codebuild.model.ListReportsForReportGroupRequest)}
     * operation.</b>
     * </p>
     *
     * @param listReportsForReportGroupRequest
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws InvalidInputException
     *         The input value that was provided is not valid.
     * @throws ResourceNotFoundException
     *         The specified AWS resource cannot be found.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws CodeBuildException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample CodeBuildClient.ListReportsForReportGroup
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/codebuild-2016-10-06/ListReportsForReportGroup"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListReportsForReportGroupIterable listReportsForReportGroupPaginator(
            ListReportsForReportGroupRequest listReportsForReportGroupRequest) throws InvalidInputException,
            ResourceNotFoundException, AwsServiceException, SdkClientException, CodeBuildException {
        return new ListReportsForReportGroupIterable(this, applyPaginatorUserAgent(listReportsForReportGroupRequest));
    }

    /**
     * <p>
     * Gets a list of projects that are shared with other AWS accounts or users.
     * </p>
     *
     * @param listSharedProjectsRequest
     * @return Result of the ListSharedProjects operation returned by the service.
     * @throws InvalidInputException
     *         The input value that was provided is not valid.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws CodeBuildException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample CodeBuildClient.ListSharedProjects
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/codebuild-2016-10-06/ListSharedProjects" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public ListSharedProjectsResponse listSharedProjects(ListSharedProjectsRequest listSharedProjectsRequest)
            throws InvalidInputException, AwsServiceException, SdkClientException, CodeBuildException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListSharedProjectsResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, ListSharedProjectsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<ListSharedProjectsRequest, ListSharedProjectsResponse>()
                .withOperationName("ListSharedProjects").withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(listSharedProjectsRequest)
                .withMarshaller(new ListSharedProjectsRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Gets a list of projects that are shared with other AWS accounts or users.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #listSharedProjects(software.amazon.awssdk.services.codebuild.model.ListSharedProjectsRequest)} operation.
     * The return type is a custom iterable that can be used to iterate through all the pages. SDK will internally
     * handle making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.codebuild.paginators.ListSharedProjectsIterable responses = client.listSharedProjectsPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.codebuild.paginators.ListSharedProjectsIterable responses = client
     *             .listSharedProjectsPaginator(request);
     *     for (software.amazon.awssdk.services.codebuild.model.ListSharedProjectsResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.codebuild.paginators.ListSharedProjectsIterable responses = client.listSharedProjectsPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of maxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listSharedProjects(software.amazon.awssdk.services.codebuild.model.ListSharedProjectsRequest)}
     * operation.</b>
     * </p>
     *
     * @param listSharedProjectsRequest
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws InvalidInputException
     *         The input value that was provided is not valid.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws CodeBuildException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample CodeBuildClient.ListSharedProjects
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/codebuild-2016-10-06/ListSharedProjects" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public ListSharedProjectsIterable listSharedProjectsPaginator(ListSharedProjectsRequest listSharedProjectsRequest)
            throws InvalidInputException, AwsServiceException, SdkClientException, CodeBuildException {
        return new ListSharedProjectsIterable(this, applyPaginatorUserAgent(listSharedProjectsRequest));
    }

    /**
     * <p>
     * Gets a list of report groups that are shared with other AWS accounts or users.
     * </p>
     *
     * @param listSharedReportGroupsRequest
     * @return Result of the ListSharedReportGroups operation returned by the service.
     * @throws InvalidInputException
     *         The input value that was provided is not valid.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws CodeBuildException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample CodeBuildClient.ListSharedReportGroups
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/codebuild-2016-10-06/ListSharedReportGroups"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListSharedReportGroupsResponse listSharedReportGroups(ListSharedReportGroupsRequest listSharedReportGroupsRequest)
            throws InvalidInputException, AwsServiceException, SdkClientException, CodeBuildException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListSharedReportGroupsResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, ListSharedReportGroupsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<ListSharedReportGroupsRequest, ListSharedReportGroupsResponse>()
                .withOperationName("ListSharedReportGroups").withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(listSharedReportGroupsRequest)
                .withMarshaller(new ListSharedReportGroupsRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Gets a list of report groups that are shared with other AWS accounts or users.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #listSharedReportGroups(software.amazon.awssdk.services.codebuild.model.ListSharedReportGroupsRequest)}
     * operation. The return type is a custom iterable that can be used to iterate through all the pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.codebuild.paginators.ListSharedReportGroupsIterable responses = client.listSharedReportGroupsPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.codebuild.paginators.ListSharedReportGroupsIterable responses = client
     *             .listSharedReportGroupsPaginator(request);
     *     for (software.amazon.awssdk.services.codebuild.model.ListSharedReportGroupsResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.codebuild.paginators.ListSharedReportGroupsIterable responses = client.listSharedReportGroupsPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of maxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listSharedReportGroups(software.amazon.awssdk.services.codebuild.model.ListSharedReportGroupsRequest)}
     * operation.</b>
     * </p>
     *
     * @param listSharedReportGroupsRequest
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws InvalidInputException
     *         The input value that was provided is not valid.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws CodeBuildException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample CodeBuildClient.ListSharedReportGroups
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/codebuild-2016-10-06/ListSharedReportGroups"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListSharedReportGroupsIterable listSharedReportGroupsPaginator(
            ListSharedReportGroupsRequest listSharedReportGroupsRequest) throws InvalidInputException, AwsServiceException,
            SdkClientException, CodeBuildException {
        return new ListSharedReportGroupsIterable(this, applyPaginatorUserAgent(listSharedReportGroupsRequest));
    }

    /**
     * <p>
     * Returns a list of <code>SourceCredentialsInfo</code> objects.
     * </p>
     *
     * @param listSourceCredentialsRequest
     * @return Result of the ListSourceCredentials operation returned by the service.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws CodeBuildException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample CodeBuildClient.ListSourceCredentials
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/codebuild-2016-10-06/ListSourceCredentials"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListSourceCredentialsResponse listSourceCredentials(ListSourceCredentialsRequest listSourceCredentialsRequest)
            throws AwsServiceException, SdkClientException, CodeBuildException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListSourceCredentialsResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, ListSourceCredentialsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<ListSourceCredentialsRequest, ListSourceCredentialsResponse>()
                .withOperationName("ListSourceCredentials").withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(listSourceCredentialsRequest)
                .withMarshaller(new ListSourceCredentialsRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Stores a resource policy for the ARN of a <code>Project</code> or <code>ReportGroup</code> object.
     * </p>
     *
     * @param putResourcePolicyRequest
     * @return Result of the PutResourcePolicy operation returned by the service.
     * @throws ResourceNotFoundException
     *         The specified AWS resource cannot be found.
     * @throws InvalidInputException
     *         The input value that was provided is not valid.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws CodeBuildException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample CodeBuildClient.PutResourcePolicy
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/codebuild-2016-10-06/PutResourcePolicy" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public PutResourcePolicyResponse putResourcePolicy(PutResourcePolicyRequest putResourcePolicyRequest)
            throws ResourceNotFoundException, InvalidInputException, AwsServiceException, SdkClientException, CodeBuildException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<PutResourcePolicyResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                PutResourcePolicyResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<PutResourcePolicyRequest, PutResourcePolicyResponse>()
                .withOperationName("PutResourcePolicy").withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(putResourcePolicyRequest)
                .withMarshaller(new PutResourcePolicyRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Starts running a build.
     * </p>
     *
     * @param startBuildRequest
     * @return Result of the StartBuild operation returned by the service.
     * @throws InvalidInputException
     *         The input value that was provided is not valid.
     * @throws ResourceNotFoundException
     *         The specified AWS resource cannot be found.
     * @throws AccountLimitExceededException
     *         An AWS service limit was exceeded for the calling AWS account.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws CodeBuildException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample CodeBuildClient.StartBuild
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/codebuild-2016-10-06/StartBuild" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public StartBuildResponse startBuild(StartBuildRequest startBuildRequest) throws InvalidInputException,
            ResourceNotFoundException, AccountLimitExceededException, AwsServiceException, SdkClientException, CodeBuildException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<StartBuildResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                StartBuildResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<StartBuildRequest, StartBuildResponse>()
                .withOperationName("StartBuild").withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(startBuildRequest)
                .withMarshaller(new StartBuildRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Attempts to stop running a build.
     * </p>
     *
     * @param stopBuildRequest
     * @return Result of the StopBuild operation returned by the service.
     * @throws InvalidInputException
     *         The input value that was provided is not valid.
     * @throws ResourceNotFoundException
     *         The specified AWS resource cannot be found.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws CodeBuildException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample CodeBuildClient.StopBuild
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/codebuild-2016-10-06/StopBuild" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public StopBuildResponse stopBuild(StopBuildRequest stopBuildRequest) throws InvalidInputException,
            ResourceNotFoundException, AwsServiceException, SdkClientException, CodeBuildException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<StopBuildResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                StopBuildResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<StopBuildRequest, StopBuildResponse>()
                .withOperationName("StopBuild").withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(stopBuildRequest)
                .withMarshaller(new StopBuildRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Changes the settings of a build project.
     * </p>
     *
     * @param updateProjectRequest
     * @return Result of the UpdateProject operation returned by the service.
     * @throws InvalidInputException
     *         The input value that was provided is not valid.
     * @throws ResourceNotFoundException
     *         The specified AWS resource cannot be found.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws CodeBuildException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample CodeBuildClient.UpdateProject
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/codebuild-2016-10-06/UpdateProject" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public UpdateProjectResponse updateProject(UpdateProjectRequest updateProjectRequest) throws InvalidInputException,
            ResourceNotFoundException, AwsServiceException, SdkClientException, CodeBuildException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UpdateProjectResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                UpdateProjectResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<UpdateProjectRequest, UpdateProjectResponse>()
                .withOperationName("UpdateProject").withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(updateProjectRequest)
                .withMarshaller(new UpdateProjectRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Updates a report group.
     * </p>
     *
     * @param updateReportGroupRequest
     * @return Result of the UpdateReportGroup operation returned by the service.
     * @throws InvalidInputException
     *         The input value that was provided is not valid.
     * @throws ResourceNotFoundException
     *         The specified AWS resource cannot be found.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws CodeBuildException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample CodeBuildClient.UpdateReportGroup
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/codebuild-2016-10-06/UpdateReportGroup" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public UpdateReportGroupResponse updateReportGroup(UpdateReportGroupRequest updateReportGroupRequest)
            throws InvalidInputException, ResourceNotFoundException, AwsServiceException, SdkClientException, CodeBuildException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UpdateReportGroupResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                UpdateReportGroupResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<UpdateReportGroupRequest, UpdateReportGroupResponse>()
                .withOperationName("UpdateReportGroup").withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(updateReportGroupRequest)
                .withMarshaller(new UpdateReportGroupRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Updates the webhook associated with an AWS CodeBuild build project.
     * </p>
     * <note>
     * <p>
     * If you use Bitbucket for your repository, <code>rotateSecret</code> is ignored.
     * </p>
     * </note>
     *
     * @param updateWebhookRequest
     * @return Result of the UpdateWebhook operation returned by the service.
     * @throws InvalidInputException
     *         The input value that was provided is not valid.
     * @throws ResourceNotFoundException
     *         The specified AWS resource cannot be found.
     * @throws OAuthProviderException
     *         There was a problem with the underlying OAuth provider.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws CodeBuildException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample CodeBuildClient.UpdateWebhook
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/codebuild-2016-10-06/UpdateWebhook" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public UpdateWebhookResponse updateWebhook(UpdateWebhookRequest updateWebhookRequest) throws InvalidInputException,
            ResourceNotFoundException, OAuthProviderException, AwsServiceException, SdkClientException, CodeBuildException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UpdateWebhookResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                UpdateWebhookResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<UpdateWebhookRequest, UpdateWebhookResponse>()
                .withOperationName("UpdateWebhook").withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(updateWebhookRequest)
                .withMarshaller(new UpdateWebhookRequestMarshaller(protocolFactory)));
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata) {
        return protocolFactory.createErrorResponseHandler(operationMetadata);
    }

    private <T extends BaseAwsJsonProtocolFactory.Builder<T>> T init(T builder) {
        return builder
                .clientConfiguration(clientConfiguration)
                .defaultServiceExceptionSupplier(CodeBuildException::builder)
                .protocol(AwsJsonProtocol.AWS_JSON)
                .protocolVersion("1.1")
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("OAuthProviderException")
                                .exceptionBuilderSupplier(OAuthProviderException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ResourceNotFoundException")
                                .exceptionBuilderSupplier(ResourceNotFoundException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidInputException")
                                .exceptionBuilderSupplier(InvalidInputException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("AccountLimitExceededException")
                                .exceptionBuilderSupplier(AccountLimitExceededException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ResourceAlreadyExistsException")
                                .exceptionBuilderSupplier(ResourceAlreadyExistsException::builder).build());
    }

    @Override
    public void close() {
        clientHandler.close();
    }

    private <T extends CodeBuildRequest> T applyPaginatorUserAgent(T request) {
        Consumer<AwsRequestOverrideConfiguration.Builder> userAgentApplier = b -> b.addApiName(ApiName.builder()
                .version(VersionInfo.SDK_VERSION).name("PAGINATED").build());
        AwsRequestOverrideConfiguration overrideConfiguration = request.overrideConfiguration()
                .map(c -> c.toBuilder().applyMutation(userAgentApplier).build())
                .orElse((AwsRequestOverrideConfiguration.builder().applyMutation(userAgentApplier).build()));
        return (T) request.toBuilder().overrideConfiguration(overrideConfiguration).build();
    }
}
