/*
 * Copyright 2014-2019 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.cloudfront.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * A complex type that specifies the request headers, if any, that you want CloudFront to base caching on for this cache
 * behavior.
 * </p>
 * <p>
 * For the headers that you specify, CloudFront caches separate versions of a specified object based on the header
 * values in viewer requests. For example, suppose viewer requests for <code>logo.jpg</code> contain a custom
 * <code>product</code> header that has a value of either <code>acme</code> or <code>apex</code>, and you configure
 * CloudFront to cache your content based on values in the <code>product</code> header. CloudFront forwards the
 * <code>product</code> header to the origin and caches the response from the origin once for each header value. For
 * more information about caching based on header values, see <a
 * href="https://docs.aws.amazon.com/AmazonCloudFront/latest/DeveloperGuide/header-caching.html">How CloudFront Forwards
 * and Caches Headers</a> in the <i>Amazon CloudFront Developer Guide</i>.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class Headers implements SdkPojo, Serializable, ToCopyableBuilder<Headers.Builder, Headers> {
    private static final SdkField<Integer> QUANTITY_FIELD = SdkField
            .<Integer> builder(MarshallingType.INTEGER)
            .getter(getter(Headers::quantity))
            .setter(setter(Builder::quantity))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Quantity")
                    .unmarshallLocationName("Quantity").build()).build();

    private static final SdkField<List<String>> ITEMS_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .getter(getter(Headers::items))
            .setter(setter(Builder::items))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Items")
                    .unmarshallLocationName("Items").build(),
                    ListTrait
                            .builder()
                            .memberLocationName("Name")
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("Name").unmarshallLocationName("Name").build()).build())
                            .build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(QUANTITY_FIELD, ITEMS_FIELD));

    private static final long serialVersionUID = 1L;

    private final Integer quantity;

    private final List<String> items;

    private Headers(BuilderImpl builder) {
        this.quantity = builder.quantity;
        this.items = builder.items;
    }

    /**
     * <p>
     * The number of different headers that you want CloudFront to base caching on for this cache behavior. You can
     * configure each cache behavior in a web distribution to do one of the following:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <b>Forward all headers to your origin</b>: Specify <code>1</code> for <code>Quantity</code> and <code>*</code>
     * for <code>Name</code>.
     * </p>
     * <important>
     * <p>
     * CloudFront doesn't cache the objects that are associated with this cache behavior. Instead, CloudFront sends
     * every request to the origin.
     * </p>
     * </important></li>
     * <li>
     * <p>
     * <b>Forward a whitelist of headers you specify</b>: Specify the number of headers that you want CloudFront to base
     * caching on. Then specify the header names in <code>Name</code> elements. CloudFront caches your objects based on
     * the values in the specified headers.
     * </p>
     * </li>
     * <li>
     * <p>
     * <b>Forward only the default headers</b>: Specify <code>0</code> for <code>Quantity</code> and omit
     * <code>Items</code>. In this configuration, CloudFront doesn't cache based on the values in the request headers.
     * </p>
     * </li>
     * </ul>
     * <p>
     * Regardless of which option you choose, CloudFront forwards headers to your origin based on whether the origin is
     * an S3 bucket or a custom origin. See the following documentation:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <b>S3 bucket</b>: See <a href=
     * "https://docs.aws.amazon.com/AmazonCloudFront/latest/DeveloperGuide/RequestAndResponseBehaviorS3Origin.html#request-s3-removed-headers"
     * >HTTP Request Headers That CloudFront Removes or Updates</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <b>Custom origin</b>: See <a href=
     * "https://docs.aws.amazon.com/AmazonCloudFront/latest/DeveloperGuide/RequestAndResponseBehaviorCustomOrigin.html#request-custom-headers-behavior"
     * >HTTP Request Headers and CloudFront Behavior</a>
     * </p>
     * </li>
     * </ul>
     * 
     * @return The number of different headers that you want CloudFront to base caching on for this cache behavior. You
     *         can configure each cache behavior in a web distribution to do one of the following:</p>
     *         <ul>
     *         <li>
     *         <p>
     *         <b>Forward all headers to your origin</b>: Specify <code>1</code> for <code>Quantity</code> and
     *         <code>*</code> for <code>Name</code>.
     *         </p>
     *         <important>
     *         <p>
     *         CloudFront doesn't cache the objects that are associated with this cache behavior. Instead, CloudFront
     *         sends every request to the origin.
     *         </p>
     *         </important></li>
     *         <li>
     *         <p>
     *         <b>Forward a whitelist of headers you specify</b>: Specify the number of headers that you want CloudFront
     *         to base caching on. Then specify the header names in <code>Name</code> elements. CloudFront caches your
     *         objects based on the values in the specified headers.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b>Forward only the default headers</b>: Specify <code>0</code> for <code>Quantity</code> and omit
     *         <code>Items</code>. In this configuration, CloudFront doesn't cache based on the values in the request
     *         headers.
     *         </p>
     *         </li>
     *         </ul>
     *         <p>
     *         Regardless of which option you choose, CloudFront forwards headers to your origin based on whether the
     *         origin is an S3 bucket or a custom origin. See the following documentation:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         <b>S3 bucket</b>: See <a href=
     *         "https://docs.aws.amazon.com/AmazonCloudFront/latest/DeveloperGuide/RequestAndResponseBehaviorS3Origin.html#request-s3-removed-headers"
     *         >HTTP Request Headers That CloudFront Removes or Updates</a>
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b>Custom origin</b>: See <a href=
     *         "https://docs.aws.amazon.com/AmazonCloudFront/latest/DeveloperGuide/RequestAndResponseBehaviorCustomOrigin.html#request-custom-headers-behavior"
     *         >HTTP Request Headers and CloudFront Behavior</a>
     *         </p>
     *         </li>
     */
    public Integer quantity() {
        return quantity;
    }

    /**
     * <p>
     * A list that contains one <code>Name</code> element for each header that you want CloudFront to use for caching in
     * this cache behavior. If <code>Quantity</code> is <code>0</code>, omit <code>Items</code>.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * 
     * @return A list that contains one <code>Name</code> element for each header that you want CloudFront to use for
     *         caching in this cache behavior. If <code>Quantity</code> is <code>0</code>, omit <code>Items</code>.
     */
    public List<String> items() {
        return items;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(quantity());
        hashCode = 31 * hashCode + Objects.hashCode(items());
        return hashCode;
    }

    @Override
    public boolean equals(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof Headers)) {
            return false;
        }
        Headers other = (Headers) obj;
        return Objects.equals(quantity(), other.quantity()) && Objects.equals(items(), other.items());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public String toString() {
        return ToString.builder("Headers").add("Quantity", quantity()).add("Items", items()).build();
    }

    public <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "Quantity":
            return Optional.ofNullable(clazz.cast(quantity()));
        case "Items":
            return Optional.ofNullable(clazz.cast(items()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<Headers, T> g) {
        return obj -> g.apply((Headers) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, Headers> {
        /**
         * <p>
         * The number of different headers that you want CloudFront to base caching on for this cache behavior. You can
         * configure each cache behavior in a web distribution to do one of the following:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <b>Forward all headers to your origin</b>: Specify <code>1</code> for <code>Quantity</code> and
         * <code>*</code> for <code>Name</code>.
         * </p>
         * <important>
         * <p>
         * CloudFront doesn't cache the objects that are associated with this cache behavior. Instead, CloudFront sends
         * every request to the origin.
         * </p>
         * </important></li>
         * <li>
         * <p>
         * <b>Forward a whitelist of headers you specify</b>: Specify the number of headers that you want CloudFront to
         * base caching on. Then specify the header names in <code>Name</code> elements. CloudFront caches your objects
         * based on the values in the specified headers.
         * </p>
         * </li>
         * <li>
         * <p>
         * <b>Forward only the default headers</b>: Specify <code>0</code> for <code>Quantity</code> and omit
         * <code>Items</code>. In this configuration, CloudFront doesn't cache based on the values in the request
         * headers.
         * </p>
         * </li>
         * </ul>
         * <p>
         * Regardless of which option you choose, CloudFront forwards headers to your origin based on whether the origin
         * is an S3 bucket or a custom origin. See the following documentation:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <b>S3 bucket</b>: See <a href=
         * "https://docs.aws.amazon.com/AmazonCloudFront/latest/DeveloperGuide/RequestAndResponseBehaviorS3Origin.html#request-s3-removed-headers"
         * >HTTP Request Headers That CloudFront Removes or Updates</a>
         * </p>
         * </li>
         * <li>
         * <p>
         * <b>Custom origin</b>: See <a href=
         * "https://docs.aws.amazon.com/AmazonCloudFront/latest/DeveloperGuide/RequestAndResponseBehaviorCustomOrigin.html#request-custom-headers-behavior"
         * >HTTP Request Headers and CloudFront Behavior</a>
         * </p>
         * </li>
         * </ul>
         * 
         * @param quantity
         *        The number of different headers that you want CloudFront to base caching on for this cache behavior.
         *        You can configure each cache behavior in a web distribution to do one of the following:</p>
         *        <ul>
         *        <li>
         *        <p>
         *        <b>Forward all headers to your origin</b>: Specify <code>1</code> for <code>Quantity</code> and
         *        <code>*</code> for <code>Name</code>.
         *        </p>
         *        <important>
         *        <p>
         *        CloudFront doesn't cache the objects that are associated with this cache behavior. Instead, CloudFront
         *        sends every request to the origin.
         *        </p>
         *        </important></li>
         *        <li>
         *        <p>
         *        <b>Forward a whitelist of headers you specify</b>: Specify the number of headers that you want
         *        CloudFront to base caching on. Then specify the header names in <code>Name</code> elements. CloudFront
         *        caches your objects based on the values in the specified headers.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <b>Forward only the default headers</b>: Specify <code>0</code> for <code>Quantity</code> and omit
         *        <code>Items</code>. In this configuration, CloudFront doesn't cache based on the values in the request
         *        headers.
         *        </p>
         *        </li>
         *        </ul>
         *        <p>
         *        Regardless of which option you choose, CloudFront forwards headers to your origin based on whether the
         *        origin is an S3 bucket or a custom origin. See the following documentation:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        <b>S3 bucket</b>: See <a href=
         *        "https://docs.aws.amazon.com/AmazonCloudFront/latest/DeveloperGuide/RequestAndResponseBehaviorS3Origin.html#request-s3-removed-headers"
         *        >HTTP Request Headers That CloudFront Removes or Updates</a>
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <b>Custom origin</b>: See <a href=
         *        "https://docs.aws.amazon.com/AmazonCloudFront/latest/DeveloperGuide/RequestAndResponseBehaviorCustomOrigin.html#request-custom-headers-behavior"
         *        >HTTP Request Headers and CloudFront Behavior</a>
         *        </p>
         *        </li>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder quantity(Integer quantity);

        /**
         * <p>
         * A list that contains one <code>Name</code> element for each header that you want CloudFront to use for
         * caching in this cache behavior. If <code>Quantity</code> is <code>0</code>, omit <code>Items</code>.
         * </p>
         * 
         * @param items
         *        A list that contains one <code>Name</code> element for each header that you want CloudFront to use for
         *        caching in this cache behavior. If <code>Quantity</code> is <code>0</code>, omit <code>Items</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder items(Collection<String> items);

        /**
         * <p>
         * A list that contains one <code>Name</code> element for each header that you want CloudFront to use for
         * caching in this cache behavior. If <code>Quantity</code> is <code>0</code>, omit <code>Items</code>.
         * </p>
         * 
         * @param items
         *        A list that contains one <code>Name</code> element for each header that you want CloudFront to use for
         *        caching in this cache behavior. If <code>Quantity</code> is <code>0</code>, omit <code>Items</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder items(String... items);
    }

    static final class BuilderImpl implements Builder {
        private Integer quantity;

        private List<String> items = DefaultSdkAutoConstructList.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(Headers model) {
            quantity(model.quantity);
            items(model.items);
        }

        public final Integer getQuantity() {
            return quantity;
        }

        @Override
        public final Builder quantity(Integer quantity) {
            this.quantity = quantity;
            return this;
        }

        public final void setQuantity(Integer quantity) {
            this.quantity = quantity;
        }

        public final Collection<String> getItems() {
            return items;
        }

        @Override
        public final Builder items(Collection<String> items) {
            this.items = HeaderListCopier.copy(items);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder items(String... items) {
            items(Arrays.asList(items));
            return this;
        }

        public final void setItems(Collection<String> items) {
            this.items = HeaderListCopier.copy(items);
        }

        @Override
        public Headers build() {
            return new Headers(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
