/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.cloudfront.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.Mutable;
import software.amazon.awssdk.annotations.NotThreadSafe;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * A custom origin. A custom origin is any origin that is <i>not</i> an Amazon S3 bucket, with one exception. An Amazon
 * S3 bucket that is <a href="https://docs.aws.amazon.com/AmazonS3/latest/dev/WebsiteHosting.html">configured with
 * static website hosting</a> <i>is</i> a custom origin.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class CustomOriginConfig implements SdkPojo, Serializable,
        ToCopyableBuilder<CustomOriginConfig.Builder, CustomOriginConfig> {
    private static final SdkField<Integer> HTTP_PORT_FIELD = SdkField
            .<Integer> builder(MarshallingType.INTEGER)
            .memberName("HTTPPort")
            .getter(getter(CustomOriginConfig::httpPort))
            .setter(setter(Builder::httpPort))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("HTTPPort")
                    .unmarshallLocationName("HTTPPort").build()).build();

    private static final SdkField<Integer> HTTPS_PORT_FIELD = SdkField
            .<Integer> builder(MarshallingType.INTEGER)
            .memberName("HTTPSPort")
            .getter(getter(CustomOriginConfig::httpsPort))
            .setter(setter(Builder::httpsPort))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("HTTPSPort")
                    .unmarshallLocationName("HTTPSPort").build()).build();

    private static final SdkField<String> ORIGIN_PROTOCOL_POLICY_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("OriginProtocolPolicy")
            .getter(getter(CustomOriginConfig::originProtocolPolicyAsString))
            .setter(setter(Builder::originProtocolPolicy))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("OriginProtocolPolicy")
                    .unmarshallLocationName("OriginProtocolPolicy").build()).build();

    private static final SdkField<OriginSslProtocols> ORIGIN_SSL_PROTOCOLS_FIELD = SdkField
            .<OriginSslProtocols> builder(MarshallingType.SDK_POJO)
            .memberName("OriginSslProtocols")
            .getter(getter(CustomOriginConfig::originSslProtocols))
            .setter(setter(Builder::originSslProtocols))
            .constructor(OriginSslProtocols::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("OriginSslProtocols")
                    .unmarshallLocationName("OriginSslProtocols").build()).build();

    private static final SdkField<Integer> ORIGIN_READ_TIMEOUT_FIELD = SdkField
            .<Integer> builder(MarshallingType.INTEGER)
            .memberName("OriginReadTimeout")
            .getter(getter(CustomOriginConfig::originReadTimeout))
            .setter(setter(Builder::originReadTimeout))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("OriginReadTimeout")
                    .unmarshallLocationName("OriginReadTimeout").build()).build();

    private static final SdkField<Integer> ORIGIN_KEEPALIVE_TIMEOUT_FIELD = SdkField
            .<Integer> builder(MarshallingType.INTEGER)
            .memberName("OriginKeepaliveTimeout")
            .getter(getter(CustomOriginConfig::originKeepaliveTimeout))
            .setter(setter(Builder::originKeepaliveTimeout))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("OriginKeepaliveTimeout")
                    .unmarshallLocationName("OriginKeepaliveTimeout").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(HTTP_PORT_FIELD,
            HTTPS_PORT_FIELD, ORIGIN_PROTOCOL_POLICY_FIELD, ORIGIN_SSL_PROTOCOLS_FIELD, ORIGIN_READ_TIMEOUT_FIELD,
            ORIGIN_KEEPALIVE_TIMEOUT_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = memberNameToFieldInitializer();

    private static final long serialVersionUID = 1L;

    private final Integer httpPort;

    private final Integer httpsPort;

    private final String originProtocolPolicy;

    private final OriginSslProtocols originSslProtocols;

    private final Integer originReadTimeout;

    private final Integer originKeepaliveTimeout;

    private CustomOriginConfig(BuilderImpl builder) {
        this.httpPort = builder.httpPort;
        this.httpsPort = builder.httpsPort;
        this.originProtocolPolicy = builder.originProtocolPolicy;
        this.originSslProtocols = builder.originSslProtocols;
        this.originReadTimeout = builder.originReadTimeout;
        this.originKeepaliveTimeout = builder.originKeepaliveTimeout;
    }

    /**
     * <p>
     * The HTTP port that CloudFront uses to connect to the origin. Specify the HTTP port that the origin listens on.
     * </p>
     * 
     * @return The HTTP port that CloudFront uses to connect to the origin. Specify the HTTP port that the origin
     *         listens on.
     */
    public final Integer httpPort() {
        return httpPort;
    }

    /**
     * <p>
     * The HTTPS port that CloudFront uses to connect to the origin. Specify the HTTPS port that the origin listens on.
     * </p>
     * 
     * @return The HTTPS port that CloudFront uses to connect to the origin. Specify the HTTPS port that the origin
     *         listens on.
     */
    public final Integer httpsPort() {
        return httpsPort;
    }

    /**
     * <p>
     * Specifies the protocol (HTTP or HTTPS) that CloudFront uses to connect to the origin. Valid values are:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>http-only</code> – CloudFront always uses HTTP to connect to the origin.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>match-viewer</code> – CloudFront connects to the origin using the same protocol that the viewer used to
     * connect to CloudFront.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>https-only</code> – CloudFront always uses HTTPS to connect to the origin.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version,
     * {@link #originProtocolPolicy} will return {@link OriginProtocolPolicy#UNKNOWN_TO_SDK_VERSION}. The raw value
     * returned by the service is available from {@link #originProtocolPolicyAsString}.
     * </p>
     * 
     * @return Specifies the protocol (HTTP or HTTPS) that CloudFront uses to connect to the origin. Valid values
     *         are:</p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>http-only</code> – CloudFront always uses HTTP to connect to the origin.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>match-viewer</code> – CloudFront connects to the origin using the same protocol that the viewer
     *         used to connect to CloudFront.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>https-only</code> – CloudFront always uses HTTPS to connect to the origin.
     *         </p>
     *         </li>
     * @see OriginProtocolPolicy
     */
    public final OriginProtocolPolicy originProtocolPolicy() {
        return OriginProtocolPolicy.fromValue(originProtocolPolicy);
    }

    /**
     * <p>
     * Specifies the protocol (HTTP or HTTPS) that CloudFront uses to connect to the origin. Valid values are:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>http-only</code> – CloudFront always uses HTTP to connect to the origin.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>match-viewer</code> – CloudFront connects to the origin using the same protocol that the viewer used to
     * connect to CloudFront.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>https-only</code> – CloudFront always uses HTTPS to connect to the origin.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version,
     * {@link #originProtocolPolicy} will return {@link OriginProtocolPolicy#UNKNOWN_TO_SDK_VERSION}. The raw value
     * returned by the service is available from {@link #originProtocolPolicyAsString}.
     * </p>
     * 
     * @return Specifies the protocol (HTTP or HTTPS) that CloudFront uses to connect to the origin. Valid values
     *         are:</p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>http-only</code> – CloudFront always uses HTTP to connect to the origin.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>match-viewer</code> – CloudFront connects to the origin using the same protocol that the viewer
     *         used to connect to CloudFront.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>https-only</code> – CloudFront always uses HTTPS to connect to the origin.
     *         </p>
     *         </li>
     * @see OriginProtocolPolicy
     */
    public final String originProtocolPolicyAsString() {
        return originProtocolPolicy;
    }

    /**
     * <p>
     * Specifies the minimum SSL/TLS protocol that CloudFront uses when connecting to your origin over HTTPS. Valid
     * values include <code>SSLv3</code>, <code>TLSv1</code>, <code>TLSv1.1</code>, and <code>TLSv1.2</code>.
     * </p>
     * <p>
     * For more information, see <a href=
     * "https://docs.aws.amazon.com/AmazonCloudFront/latest/DeveloperGuide/DownloadDistValuesOrigin.html#DownloadDistValuesOriginSSLProtocols"
     * >Minimum Origin SSL Protocol</a> in the <i>Amazon CloudFront Developer Guide</i>.
     * </p>
     * 
     * @return Specifies the minimum SSL/TLS protocol that CloudFront uses when connecting to your origin over HTTPS.
     *         Valid values include <code>SSLv3</code>, <code>TLSv1</code>, <code>TLSv1.1</code>, and
     *         <code>TLSv1.2</code>.</p>
     *         <p>
     *         For more information, see <a href=
     *         "https://docs.aws.amazon.com/AmazonCloudFront/latest/DeveloperGuide/DownloadDistValuesOrigin.html#DownloadDistValuesOriginSSLProtocols"
     *         >Minimum Origin SSL Protocol</a> in the <i>Amazon CloudFront Developer Guide</i>.
     */
    public final OriginSslProtocols originSslProtocols() {
        return originSslProtocols;
    }

    /**
     * <p>
     * Specifies how long, in seconds, CloudFront waits for a response from the origin. This is also known as the
     * <i>origin response timeout</i>. The minimum timeout is 1 second, the maximum is 120 seconds, and the default (if
     * you don't specify otherwise) is 30 seconds.
     * </p>
     * <p>
     * For more information, see <a href=
     * "https://docs.aws.amazon.com/AmazonCloudFront/latest/DeveloperGuide/DownloadDistValuesOrigin.html#DownloadDistValuesOriginResponseTimeout"
     * >Response timeout</a> in the <i>Amazon CloudFront Developer Guide</i>.
     * </p>
     * 
     * @return Specifies how long, in seconds, CloudFront waits for a response from the origin. This is also known as
     *         the <i>origin response timeout</i>. The minimum timeout is 1 second, the maximum is 120 seconds, and the
     *         default (if you don't specify otherwise) is 30 seconds.</p>
     *         <p>
     *         For more information, see <a href=
     *         "https://docs.aws.amazon.com/AmazonCloudFront/latest/DeveloperGuide/DownloadDistValuesOrigin.html#DownloadDistValuesOriginResponseTimeout"
     *         >Response timeout</a> in the <i>Amazon CloudFront Developer Guide</i>.
     */
    public final Integer originReadTimeout() {
        return originReadTimeout;
    }

    /**
     * <p>
     * Specifies how long, in seconds, CloudFront persists its connection to the origin. The minimum timeout is 1
     * second, the maximum is 120 seconds, and the default (if you don't specify otherwise) is 5 seconds.
     * </p>
     * <p>
     * For more information, see <a href=
     * "https://docs.aws.amazon.com/AmazonCloudFront/latest/DeveloperGuide/DownloadDistValuesOrigin.html#DownloadDistValuesOriginKeepaliveTimeout"
     * >Keep-alive timeout (custom origins only)</a> in the <i>Amazon CloudFront Developer Guide</i>.
     * </p>
     * 
     * @return Specifies how long, in seconds, CloudFront persists its connection to the origin. The minimum timeout is
     *         1 second, the maximum is 120 seconds, and the default (if you don't specify otherwise) is 5 seconds.</p>
     *         <p>
     *         For more information, see <a href=
     *         "https://docs.aws.amazon.com/AmazonCloudFront/latest/DeveloperGuide/DownloadDistValuesOrigin.html#DownloadDistValuesOriginKeepaliveTimeout"
     *         >Keep-alive timeout (custom origins only)</a> in the <i>Amazon CloudFront Developer Guide</i>.
     */
    public final Integer originKeepaliveTimeout() {
        return originKeepaliveTimeout;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(httpPort());
        hashCode = 31 * hashCode + Objects.hashCode(httpsPort());
        hashCode = 31 * hashCode + Objects.hashCode(originProtocolPolicyAsString());
        hashCode = 31 * hashCode + Objects.hashCode(originSslProtocols());
        hashCode = 31 * hashCode + Objects.hashCode(originReadTimeout());
        hashCode = 31 * hashCode + Objects.hashCode(originKeepaliveTimeout());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof CustomOriginConfig)) {
            return false;
        }
        CustomOriginConfig other = (CustomOriginConfig) obj;
        return Objects.equals(httpPort(), other.httpPort()) && Objects.equals(httpsPort(), other.httpsPort())
                && Objects.equals(originProtocolPolicyAsString(), other.originProtocolPolicyAsString())
                && Objects.equals(originSslProtocols(), other.originSslProtocols())
                && Objects.equals(originReadTimeout(), other.originReadTimeout())
                && Objects.equals(originKeepaliveTimeout(), other.originKeepaliveTimeout());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("CustomOriginConfig").add("HTTPPort", httpPort()).add("HTTPSPort", httpsPort())
                .add("OriginProtocolPolicy", originProtocolPolicyAsString()).add("OriginSslProtocols", originSslProtocols())
                .add("OriginReadTimeout", originReadTimeout()).add("OriginKeepaliveTimeout", originKeepaliveTimeout()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "HTTPPort":
            return Optional.ofNullable(clazz.cast(httpPort()));
        case "HTTPSPort":
            return Optional.ofNullable(clazz.cast(httpsPort()));
        case "OriginProtocolPolicy":
            return Optional.ofNullable(clazz.cast(originProtocolPolicyAsString()));
        case "OriginSslProtocols":
            return Optional.ofNullable(clazz.cast(originSslProtocols()));
        case "OriginReadTimeout":
            return Optional.ofNullable(clazz.cast(originReadTimeout()));
        case "OriginKeepaliveTimeout":
            return Optional.ofNullable(clazz.cast(originKeepaliveTimeout()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static Map<String, SdkField<?>> memberNameToFieldInitializer() {
        Map<String, SdkField<?>> map = new HashMap<>();
        map.put("HTTPPort", HTTP_PORT_FIELD);
        map.put("HTTPSPort", HTTPS_PORT_FIELD);
        map.put("OriginProtocolPolicy", ORIGIN_PROTOCOL_POLICY_FIELD);
        map.put("OriginSslProtocols", ORIGIN_SSL_PROTOCOLS_FIELD);
        map.put("OriginReadTimeout", ORIGIN_READ_TIMEOUT_FIELD);
        map.put("OriginKeepaliveTimeout", ORIGIN_KEEPALIVE_TIMEOUT_FIELD);
        return Collections.unmodifiableMap(map);
    }

    private static <T> Function<Object, T> getter(Function<CustomOriginConfig, T> g) {
        return obj -> g.apply((CustomOriginConfig) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    @Mutable
    @NotThreadSafe
    public interface Builder extends SdkPojo, CopyableBuilder<Builder, CustomOriginConfig> {
        /**
         * <p>
         * The HTTP port that CloudFront uses to connect to the origin. Specify the HTTP port that the origin listens
         * on.
         * </p>
         * 
         * @param httpPort
         *        The HTTP port that CloudFront uses to connect to the origin. Specify the HTTP port that the origin
         *        listens on.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder httpPort(Integer httpPort);

        /**
         * <p>
         * The HTTPS port that CloudFront uses to connect to the origin. Specify the HTTPS port that the origin listens
         * on.
         * </p>
         * 
         * @param httpsPort
         *        The HTTPS port that CloudFront uses to connect to the origin. Specify the HTTPS port that the origin
         *        listens on.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder httpsPort(Integer httpsPort);

        /**
         * <p>
         * Specifies the protocol (HTTP or HTTPS) that CloudFront uses to connect to the origin. Valid values are:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>http-only</code> – CloudFront always uses HTTP to connect to the origin.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>match-viewer</code> – CloudFront connects to the origin using the same protocol that the viewer used to
         * connect to CloudFront.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>https-only</code> – CloudFront always uses HTTPS to connect to the origin.
         * </p>
         * </li>
         * </ul>
         * 
         * @param originProtocolPolicy
         *        Specifies the protocol (HTTP or HTTPS) that CloudFront uses to connect to the origin. Valid values
         *        are:</p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>http-only</code> – CloudFront always uses HTTP to connect to the origin.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>match-viewer</code> – CloudFront connects to the origin using the same protocol that the viewer
         *        used to connect to CloudFront.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>https-only</code> – CloudFront always uses HTTPS to connect to the origin.
         *        </p>
         *        </li>
         * @see OriginProtocolPolicy
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see OriginProtocolPolicy
         */
        Builder originProtocolPolicy(String originProtocolPolicy);

        /**
         * <p>
         * Specifies the protocol (HTTP or HTTPS) that CloudFront uses to connect to the origin. Valid values are:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>http-only</code> – CloudFront always uses HTTP to connect to the origin.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>match-viewer</code> – CloudFront connects to the origin using the same protocol that the viewer used to
         * connect to CloudFront.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>https-only</code> – CloudFront always uses HTTPS to connect to the origin.
         * </p>
         * </li>
         * </ul>
         * 
         * @param originProtocolPolicy
         *        Specifies the protocol (HTTP or HTTPS) that CloudFront uses to connect to the origin. Valid values
         *        are:</p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>http-only</code> – CloudFront always uses HTTP to connect to the origin.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>match-viewer</code> – CloudFront connects to the origin using the same protocol that the viewer
         *        used to connect to CloudFront.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>https-only</code> – CloudFront always uses HTTPS to connect to the origin.
         *        </p>
         *        </li>
         * @see OriginProtocolPolicy
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see OriginProtocolPolicy
         */
        Builder originProtocolPolicy(OriginProtocolPolicy originProtocolPolicy);

        /**
         * <p>
         * Specifies the minimum SSL/TLS protocol that CloudFront uses when connecting to your origin over HTTPS. Valid
         * values include <code>SSLv3</code>, <code>TLSv1</code>, <code>TLSv1.1</code>, and <code>TLSv1.2</code>.
         * </p>
         * <p>
         * For more information, see <a href=
         * "https://docs.aws.amazon.com/AmazonCloudFront/latest/DeveloperGuide/DownloadDistValuesOrigin.html#DownloadDistValuesOriginSSLProtocols"
         * >Minimum Origin SSL Protocol</a> in the <i>Amazon CloudFront Developer Guide</i>.
         * </p>
         * 
         * @param originSslProtocols
         *        Specifies the minimum SSL/TLS protocol that CloudFront uses when connecting to your origin over HTTPS.
         *        Valid values include <code>SSLv3</code>, <code>TLSv1</code>, <code>TLSv1.1</code>, and
         *        <code>TLSv1.2</code>.</p>
         *        <p>
         *        For more information, see <a href=
         *        "https://docs.aws.amazon.com/AmazonCloudFront/latest/DeveloperGuide/DownloadDistValuesOrigin.html#DownloadDistValuesOriginSSLProtocols"
         *        >Minimum Origin SSL Protocol</a> in the <i>Amazon CloudFront Developer Guide</i>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder originSslProtocols(OriginSslProtocols originSslProtocols);

        /**
         * <p>
         * Specifies the minimum SSL/TLS protocol that CloudFront uses when connecting to your origin over HTTPS. Valid
         * values include <code>SSLv3</code>, <code>TLSv1</code>, <code>TLSv1.1</code>, and <code>TLSv1.2</code>.
         * </p>
         * <p>
         * For more information, see <a href=
         * "https://docs.aws.amazon.com/AmazonCloudFront/latest/DeveloperGuide/DownloadDistValuesOrigin.html#DownloadDistValuesOriginSSLProtocols"
         * >Minimum Origin SSL Protocol</a> in the <i>Amazon CloudFront Developer Guide</i>.
         * </p>
         * This is a convenience method that creates an instance of the {@link OriginSslProtocols.Builder} avoiding the
         * need to create one manually via {@link OriginSslProtocols#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link OriginSslProtocols.Builder#build()} is called immediately and its
         * result is passed to {@link #originSslProtocols(OriginSslProtocols)}.
         * 
         * @param originSslProtocols
         *        a consumer that will call methods on {@link OriginSslProtocols.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #originSslProtocols(OriginSslProtocols)
         */
        default Builder originSslProtocols(Consumer<OriginSslProtocols.Builder> originSslProtocols) {
            return originSslProtocols(OriginSslProtocols.builder().applyMutation(originSslProtocols).build());
        }

        /**
         * <p>
         * Specifies how long, in seconds, CloudFront waits for a response from the origin. This is also known as the
         * <i>origin response timeout</i>. The minimum timeout is 1 second, the maximum is 120 seconds, and the default
         * (if you don't specify otherwise) is 30 seconds.
         * </p>
         * <p>
         * For more information, see <a href=
         * "https://docs.aws.amazon.com/AmazonCloudFront/latest/DeveloperGuide/DownloadDistValuesOrigin.html#DownloadDistValuesOriginResponseTimeout"
         * >Response timeout</a> in the <i>Amazon CloudFront Developer Guide</i>.
         * </p>
         * 
         * @param originReadTimeout
         *        Specifies how long, in seconds, CloudFront waits for a response from the origin. This is also known as
         *        the <i>origin response timeout</i>. The minimum timeout is 1 second, the maximum is 120 seconds, and
         *        the default (if you don't specify otherwise) is 30 seconds.</p>
         *        <p>
         *        For more information, see <a href=
         *        "https://docs.aws.amazon.com/AmazonCloudFront/latest/DeveloperGuide/DownloadDistValuesOrigin.html#DownloadDistValuesOriginResponseTimeout"
         *        >Response timeout</a> in the <i>Amazon CloudFront Developer Guide</i>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder originReadTimeout(Integer originReadTimeout);

        /**
         * <p>
         * Specifies how long, in seconds, CloudFront persists its connection to the origin. The minimum timeout is 1
         * second, the maximum is 120 seconds, and the default (if you don't specify otherwise) is 5 seconds.
         * </p>
         * <p>
         * For more information, see <a href=
         * "https://docs.aws.amazon.com/AmazonCloudFront/latest/DeveloperGuide/DownloadDistValuesOrigin.html#DownloadDistValuesOriginKeepaliveTimeout"
         * >Keep-alive timeout (custom origins only)</a> in the <i>Amazon CloudFront Developer Guide</i>.
         * </p>
         * 
         * @param originKeepaliveTimeout
         *        Specifies how long, in seconds, CloudFront persists its connection to the origin. The minimum timeout
         *        is 1 second, the maximum is 120 seconds, and the default (if you don't specify otherwise) is 5
         *        seconds.</p>
         *        <p>
         *        For more information, see <a href=
         *        "https://docs.aws.amazon.com/AmazonCloudFront/latest/DeveloperGuide/DownloadDistValuesOrigin.html#DownloadDistValuesOriginKeepaliveTimeout"
         *        >Keep-alive timeout (custom origins only)</a> in the <i>Amazon CloudFront Developer Guide</i>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder originKeepaliveTimeout(Integer originKeepaliveTimeout);
    }

    static final class BuilderImpl implements Builder {
        private Integer httpPort;

        private Integer httpsPort;

        private String originProtocolPolicy;

        private OriginSslProtocols originSslProtocols;

        private Integer originReadTimeout;

        private Integer originKeepaliveTimeout;

        private BuilderImpl() {
        }

        private BuilderImpl(CustomOriginConfig model) {
            httpPort(model.httpPort);
            httpsPort(model.httpsPort);
            originProtocolPolicy(model.originProtocolPolicy);
            originSslProtocols(model.originSslProtocols);
            originReadTimeout(model.originReadTimeout);
            originKeepaliveTimeout(model.originKeepaliveTimeout);
        }

        public final Integer getHttpPort() {
            return httpPort;
        }

        public final void setHttpPort(Integer httpPort) {
            this.httpPort = httpPort;
        }

        @Override
        public final Builder httpPort(Integer httpPort) {
            this.httpPort = httpPort;
            return this;
        }

        public final Integer getHttpsPort() {
            return httpsPort;
        }

        public final void setHttpsPort(Integer httpsPort) {
            this.httpsPort = httpsPort;
        }

        @Override
        public final Builder httpsPort(Integer httpsPort) {
            this.httpsPort = httpsPort;
            return this;
        }

        public final String getOriginProtocolPolicy() {
            return originProtocolPolicy;
        }

        public final void setOriginProtocolPolicy(String originProtocolPolicy) {
            this.originProtocolPolicy = originProtocolPolicy;
        }

        @Override
        public final Builder originProtocolPolicy(String originProtocolPolicy) {
            this.originProtocolPolicy = originProtocolPolicy;
            return this;
        }

        @Override
        public final Builder originProtocolPolicy(OriginProtocolPolicy originProtocolPolicy) {
            this.originProtocolPolicy(originProtocolPolicy == null ? null : originProtocolPolicy.toString());
            return this;
        }

        public final OriginSslProtocols.Builder getOriginSslProtocols() {
            return originSslProtocols != null ? originSslProtocols.toBuilder() : null;
        }

        public final void setOriginSslProtocols(OriginSslProtocols.BuilderImpl originSslProtocols) {
            this.originSslProtocols = originSslProtocols != null ? originSslProtocols.build() : null;
        }

        @Override
        public final Builder originSslProtocols(OriginSslProtocols originSslProtocols) {
            this.originSslProtocols = originSslProtocols;
            return this;
        }

        public final Integer getOriginReadTimeout() {
            return originReadTimeout;
        }

        public final void setOriginReadTimeout(Integer originReadTimeout) {
            this.originReadTimeout = originReadTimeout;
        }

        @Override
        public final Builder originReadTimeout(Integer originReadTimeout) {
            this.originReadTimeout = originReadTimeout;
            return this;
        }

        public final Integer getOriginKeepaliveTimeout() {
            return originKeepaliveTimeout;
        }

        public final void setOriginKeepaliveTimeout(Integer originKeepaliveTimeout) {
            this.originKeepaliveTimeout = originKeepaliveTimeout;
        }

        @Override
        public final Builder originKeepaliveTimeout(Integer originKeepaliveTimeout) {
            this.originKeepaliveTimeout = originKeepaliveTimeout;
            return this;
        }

        @Override
        public CustomOriginConfig build() {
            return new CustomOriginConfig(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
