/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.cloudfront.model;

import java.beans.Transient;
import java.nio.ByteBuffer;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.awscore.AwsRequestOverrideConfiguration;
import software.amazon.awssdk.core.SdkBytes;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 */
@Generated("software.amazon.awssdk:codegen")
public final class TestFunctionRequest extends CloudFrontRequest implements
        ToCopyableBuilder<TestFunctionRequest.Builder, TestFunctionRequest> {
    private static final SdkField<String> NAME_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("Name")
            .getter(getter(TestFunctionRequest::name))
            .setter(setter(Builder::name))
            .traits(LocationTrait.builder().location(MarshallLocation.PATH).locationName("Name").unmarshallLocationName("Name")
                    .build()).build();

    private static final SdkField<String> IF_MATCH_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("IfMatch")
            .getter(getter(TestFunctionRequest::ifMatch))
            .setter(setter(Builder::ifMatch))
            .traits(LocationTrait.builder().location(MarshallLocation.HEADER).locationName("If-Match")
                    .unmarshallLocationName("If-Match").build()).build();

    private static final SdkField<String> STAGE_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("Stage")
            .getter(getter(TestFunctionRequest::stageAsString))
            .setter(setter(Builder::stage))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Stage")
                    .unmarshallLocationName("Stage").build()).build();

    private static final SdkField<SdkBytes> EVENT_OBJECT_FIELD = SdkField
            .<SdkBytes> builder(MarshallingType.SDK_BYTES)
            .memberName("EventObject")
            .getter(getter(TestFunctionRequest::eventObject))
            .setter(setter(Builder::eventObject))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("EventObject")
                    .unmarshallLocationName("EventObject").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(NAME_FIELD, IF_MATCH_FIELD,
            STAGE_FIELD, EVENT_OBJECT_FIELD));

    private final String name;

    private final String ifMatch;

    private final String stage;

    private final SdkBytes eventObject;

    private TestFunctionRequest(BuilderImpl builder) {
        super(builder);
        this.name = builder.name;
        this.ifMatch = builder.ifMatch;
        this.stage = builder.stage;
        this.eventObject = builder.eventObject;
    }

    /**
     * <p>
     * The name of the function that you are testing.
     * </p>
     * 
     * @return The name of the function that you are testing.
     */
    public final String name() {
        return name;
    }

    /**
     * <p>
     * The current version (<code>ETag</code> value) of the function that you are testing, which you can get using
     * <code>DescribeFunction</code>.
     * </p>
     * 
     * @return The current version (<code>ETag</code> value) of the function that you are testing, which you can get
     *         using <code>DescribeFunction</code>.
     */
    public final String ifMatch() {
        return ifMatch;
    }

    /**
     * <p>
     * The stage of the function that you are testing, either <code>DEVELOPMENT</code> or <code>LIVE</code>.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #stage} will return
     * {@link FunctionStage#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #stageAsString}.
     * </p>
     * 
     * @return The stage of the function that you are testing, either <code>DEVELOPMENT</code> or <code>LIVE</code>.
     * @see FunctionStage
     */
    public final FunctionStage stage() {
        return FunctionStage.fromValue(stage);
    }

    /**
     * <p>
     * The stage of the function that you are testing, either <code>DEVELOPMENT</code> or <code>LIVE</code>.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #stage} will return
     * {@link FunctionStage#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #stageAsString}.
     * </p>
     * 
     * @return The stage of the function that you are testing, either <code>DEVELOPMENT</code> or <code>LIVE</code>.
     * @see FunctionStage
     */
    public final String stageAsString() {
        return stage;
    }

    /**
     * <p>
     * The event object to test the function with. For more information about the structure of the event object, see <a
     * href="https://docs.aws.amazon.com/AmazonCloudFront/latest/DeveloperGuide/managing-functions.html#test-function">
     * Testing functions</a> in the <i>Amazon CloudFront Developer Guide</i>.
     * </p>
     * 
     * @return The event object to test the function with. For more information about the structure of the event object,
     *         see <a href=
     *         "https://docs.aws.amazon.com/AmazonCloudFront/latest/DeveloperGuide/managing-functions.html#test-function"
     *         >Testing functions</a> in the <i>Amazon CloudFront Developer Guide</i>.
     */
    public final SdkBytes eventObject() {
        return eventObject;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + super.hashCode();
        hashCode = 31 * hashCode + Objects.hashCode(name());
        hashCode = 31 * hashCode + Objects.hashCode(ifMatch());
        hashCode = 31 * hashCode + Objects.hashCode(stageAsString());
        hashCode = 31 * hashCode + Objects.hashCode(eventObject());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return super.equals(obj) && equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof TestFunctionRequest)) {
            return false;
        }
        TestFunctionRequest other = (TestFunctionRequest) obj;
        return Objects.equals(name(), other.name()) && Objects.equals(ifMatch(), other.ifMatch())
                && Objects.equals(stageAsString(), other.stageAsString()) && Objects.equals(eventObject(), other.eventObject());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("TestFunctionRequest").add("Name", name()).add("IfMatch", ifMatch())
                .add("Stage", stageAsString())
                .add("EventObject", eventObject() == null ? null : "*** Sensitive Data Redacted ***").build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "Name":
            return Optional.ofNullable(clazz.cast(name()));
        case "IfMatch":
            return Optional.ofNullable(clazz.cast(ifMatch()));
        case "Stage":
            return Optional.ofNullable(clazz.cast(stageAsString()));
        case "EventObject":
            return Optional.ofNullable(clazz.cast(eventObject()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<TestFunctionRequest, T> g) {
        return obj -> g.apply((TestFunctionRequest) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends CloudFrontRequest.Builder, SdkPojo, CopyableBuilder<Builder, TestFunctionRequest> {
        /**
         * <p>
         * The name of the function that you are testing.
         * </p>
         * 
         * @param name
         *        The name of the function that you are testing.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder name(String name);

        /**
         * <p>
         * The current version (<code>ETag</code> value) of the function that you are testing, which you can get using
         * <code>DescribeFunction</code>.
         * </p>
         * 
         * @param ifMatch
         *        The current version (<code>ETag</code> value) of the function that you are testing, which you can get
         *        using <code>DescribeFunction</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder ifMatch(String ifMatch);

        /**
         * <p>
         * The stage of the function that you are testing, either <code>DEVELOPMENT</code> or <code>LIVE</code>.
         * </p>
         * 
         * @param stage
         *        The stage of the function that you are testing, either <code>DEVELOPMENT</code> or <code>LIVE</code>.
         * @see FunctionStage
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see FunctionStage
         */
        Builder stage(String stage);

        /**
         * <p>
         * The stage of the function that you are testing, either <code>DEVELOPMENT</code> or <code>LIVE</code>.
         * </p>
         * 
         * @param stage
         *        The stage of the function that you are testing, either <code>DEVELOPMENT</code> or <code>LIVE</code>.
         * @see FunctionStage
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see FunctionStage
         */
        Builder stage(FunctionStage stage);

        /**
         * <p>
         * The event object to test the function with. For more information about the structure of the event object, see
         * <a href=
         * "https://docs.aws.amazon.com/AmazonCloudFront/latest/DeveloperGuide/managing-functions.html#test-function"
         * >Testing functions</a> in the <i>Amazon CloudFront Developer Guide</i>.
         * </p>
         * 
         * @param eventObject
         *        The event object to test the function with. For more information about the structure of the event
         *        object, see <a href=
         *        "https://docs.aws.amazon.com/AmazonCloudFront/latest/DeveloperGuide/managing-functions.html#test-function"
         *        >Testing functions</a> in the <i>Amazon CloudFront Developer Guide</i>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder eventObject(SdkBytes eventObject);

        @Override
        Builder overrideConfiguration(AwsRequestOverrideConfiguration overrideConfiguration);

        @Override
        Builder overrideConfiguration(Consumer<AwsRequestOverrideConfiguration.Builder> builderConsumer);
    }

    static final class BuilderImpl extends CloudFrontRequest.BuilderImpl implements Builder {
        private String name;

        private String ifMatch;

        private String stage;

        private SdkBytes eventObject;

        private BuilderImpl() {
        }

        private BuilderImpl(TestFunctionRequest model) {
            super(model);
            name(model.name);
            ifMatch(model.ifMatch);
            stage(model.stage);
            eventObject(model.eventObject);
        }

        public final String getName() {
            return name;
        }

        public final void setName(String name) {
            this.name = name;
        }

        @Override
        @Transient
        public final Builder name(String name) {
            this.name = name;
            return this;
        }

        public final String getIfMatch() {
            return ifMatch;
        }

        public final void setIfMatch(String ifMatch) {
            this.ifMatch = ifMatch;
        }

        @Override
        @Transient
        public final Builder ifMatch(String ifMatch) {
            this.ifMatch = ifMatch;
            return this;
        }

        public final String getStage() {
            return stage;
        }

        public final void setStage(String stage) {
            this.stage = stage;
        }

        @Override
        @Transient
        public final Builder stage(String stage) {
            this.stage = stage;
            return this;
        }

        @Override
        @Transient
        public final Builder stage(FunctionStage stage) {
            this.stage(stage == null ? null : stage.toString());
            return this;
        }

        public final ByteBuffer getEventObject() {
            return eventObject == null ? null : eventObject.asByteBuffer();
        }

        public final void setEventObject(ByteBuffer eventObject) {
            eventObject(eventObject == null ? null : SdkBytes.fromByteBuffer(eventObject));
        }

        @Override
        @Transient
        public final Builder eventObject(SdkBytes eventObject) {
            this.eventObject = eventObject;
            return this;
        }

        @Override
        public Builder overrideConfiguration(AwsRequestOverrideConfiguration overrideConfiguration) {
            super.overrideConfiguration(overrideConfiguration);
            return this;
        }

        @Override
        public Builder overrideConfiguration(Consumer<AwsRequestOverrideConfiguration.Builder> builderConsumer) {
            super.overrideConfiguration(builderConsumer);
            return this;
        }

        @Override
        public TestFunctionRequest build() {
            return new TestFunctionRequest(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
