/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.cloudfront.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * This object determines the values that CloudFront includes in the cache key. These values can include HTTP headers,
 * cookies, and URL query strings. CloudFront uses the cache key to find an object in its cache that it can return to
 * the viewer.
 * </p>
 * <p>
 * The headers, cookies, and query strings that are included in the cache key are automatically included in requests
 * that CloudFront sends to the origin. CloudFront sends a request when it can’t find an object in its cache that
 * matches the request’s cache key. If you want to send values to the origin but <i>not</i> include them in the cache
 * key, use <code>OriginRequestPolicy</code>.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class ParametersInCacheKeyAndForwardedToOrigin implements SdkPojo, Serializable,
        ToCopyableBuilder<ParametersInCacheKeyAndForwardedToOrigin.Builder, ParametersInCacheKeyAndForwardedToOrigin> {
    private static final SdkField<Boolean> ENABLE_ACCEPT_ENCODING_GZIP_FIELD = SdkField
            .<Boolean> builder(MarshallingType.BOOLEAN)
            .memberName("EnableAcceptEncodingGzip")
            .getter(getter(ParametersInCacheKeyAndForwardedToOrigin::enableAcceptEncodingGzip))
            .setter(setter(Builder::enableAcceptEncodingGzip))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("EnableAcceptEncodingGzip")
                    .unmarshallLocationName("EnableAcceptEncodingGzip").build()).build();

    private static final SdkField<Boolean> ENABLE_ACCEPT_ENCODING_BROTLI_FIELD = SdkField
            .<Boolean> builder(MarshallingType.BOOLEAN)
            .memberName("EnableAcceptEncodingBrotli")
            .getter(getter(ParametersInCacheKeyAndForwardedToOrigin::enableAcceptEncodingBrotli))
            .setter(setter(Builder::enableAcceptEncodingBrotli))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("EnableAcceptEncodingBrotli")
                    .unmarshallLocationName("EnableAcceptEncodingBrotli").build()).build();

    private static final SdkField<CachePolicyHeadersConfig> HEADERS_CONFIG_FIELD = SdkField
            .<CachePolicyHeadersConfig> builder(MarshallingType.SDK_POJO)
            .memberName("HeadersConfig")
            .getter(getter(ParametersInCacheKeyAndForwardedToOrigin::headersConfig))
            .setter(setter(Builder::headersConfig))
            .constructor(CachePolicyHeadersConfig::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("HeadersConfig")
                    .unmarshallLocationName("HeadersConfig").build()).build();

    private static final SdkField<CachePolicyCookiesConfig> COOKIES_CONFIG_FIELD = SdkField
            .<CachePolicyCookiesConfig> builder(MarshallingType.SDK_POJO)
            .memberName("CookiesConfig")
            .getter(getter(ParametersInCacheKeyAndForwardedToOrigin::cookiesConfig))
            .setter(setter(Builder::cookiesConfig))
            .constructor(CachePolicyCookiesConfig::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("CookiesConfig")
                    .unmarshallLocationName("CookiesConfig").build()).build();

    private static final SdkField<CachePolicyQueryStringsConfig> QUERY_STRINGS_CONFIG_FIELD = SdkField
            .<CachePolicyQueryStringsConfig> builder(MarshallingType.SDK_POJO)
            .memberName("QueryStringsConfig")
            .getter(getter(ParametersInCacheKeyAndForwardedToOrigin::queryStringsConfig))
            .setter(setter(Builder::queryStringsConfig))
            .constructor(CachePolicyQueryStringsConfig::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("QueryStringsConfig")
                    .unmarshallLocationName("QueryStringsConfig").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(
            ENABLE_ACCEPT_ENCODING_GZIP_FIELD, ENABLE_ACCEPT_ENCODING_BROTLI_FIELD, HEADERS_CONFIG_FIELD, COOKIES_CONFIG_FIELD,
            QUERY_STRINGS_CONFIG_FIELD));

    private static final long serialVersionUID = 1L;

    private final Boolean enableAcceptEncodingGzip;

    private final Boolean enableAcceptEncodingBrotli;

    private final CachePolicyHeadersConfig headersConfig;

    private final CachePolicyCookiesConfig cookiesConfig;

    private final CachePolicyQueryStringsConfig queryStringsConfig;

    private ParametersInCacheKeyAndForwardedToOrigin(BuilderImpl builder) {
        this.enableAcceptEncodingGzip = builder.enableAcceptEncodingGzip;
        this.enableAcceptEncodingBrotli = builder.enableAcceptEncodingBrotli;
        this.headersConfig = builder.headersConfig;
        this.cookiesConfig = builder.cookiesConfig;
        this.queryStringsConfig = builder.queryStringsConfig;
    }

    /**
     * <p>
     * A flag that can affect whether the <code>Accept-Encoding</code> HTTP header is included in the cache key and
     * included in requests that CloudFront sends to the origin.
     * </p>
     * <p>
     * This field is related to the <code>EnableAcceptEncodingBrotli</code> field. If one or both of these fields is
     * <code>true</code> <i>and</i> the viewer request includes the <code>Accept-Encoding</code> header, then CloudFront
     * does the following:
     * </p>
     * <ul>
     * <li>
     * <p>
     * Normalizes the value of the viewer’s <code>Accept-Encoding</code> header
     * </p>
     * </li>
     * <li>
     * <p>
     * Includes the normalized header in the cache key
     * </p>
     * </li>
     * <li>
     * <p>
     * Includes the normalized header in the request to the origin, if a request is necessary
     * </p>
     * </li>
     * </ul>
     * <p>
     * For more information, see <a href=
     * "https://docs.aws.amazon.com/AmazonCloudFront/latest/DeveloperGuide/controlling-the-cache-key.html#cache-policy-compressed-objects"
     * >Compression support</a> in the <i>Amazon CloudFront Developer Guide</i>.
     * </p>
     * <p>
     * If you set this value to <code>true</code>, and this cache behavior also has an origin request policy attached,
     * do not include the <code>Accept-Encoding</code> header in the origin request policy. CloudFront always includes
     * the <code>Accept-Encoding</code> header in origin requests when the value of this field is <code>true</code>, so
     * including this header in an origin request policy has no effect.
     * </p>
     * <p>
     * If both of these fields are <code>false</code>, then CloudFront treats the <code>Accept-Encoding</code> header
     * the same as any other HTTP header in the viewer request. By default, it’s not included in the cache key and it’s
     * not included in origin requests. In this case, you can manually add <code>Accept-Encoding</code> to the headers
     * whitelist like any other HTTP header.
     * </p>
     * 
     * @return A flag that can affect whether the <code>Accept-Encoding</code> HTTP header is included in the cache key
     *         and included in requests that CloudFront sends to the origin.</p>
     *         <p>
     *         This field is related to the <code>EnableAcceptEncodingBrotli</code> field. If one or both of these
     *         fields is <code>true</code> <i>and</i> the viewer request includes the <code>Accept-Encoding</code>
     *         header, then CloudFront does the following:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         Normalizes the value of the viewer’s <code>Accept-Encoding</code> header
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Includes the normalized header in the cache key
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Includes the normalized header in the request to the origin, if a request is necessary
     *         </p>
     *         </li>
     *         </ul>
     *         <p>
     *         For more information, see <a href=
     *         "https://docs.aws.amazon.com/AmazonCloudFront/latest/DeveloperGuide/controlling-the-cache-key.html#cache-policy-compressed-objects"
     *         >Compression support</a> in the <i>Amazon CloudFront Developer Guide</i>.
     *         </p>
     *         <p>
     *         If you set this value to <code>true</code>, and this cache behavior also has an origin request policy
     *         attached, do not include the <code>Accept-Encoding</code> header in the origin request policy. CloudFront
     *         always includes the <code>Accept-Encoding</code> header in origin requests when the value of this field
     *         is <code>true</code>, so including this header in an origin request policy has no effect.
     *         </p>
     *         <p>
     *         If both of these fields are <code>false</code>, then CloudFront treats the <code>Accept-Encoding</code>
     *         header the same as any other HTTP header in the viewer request. By default, it’s not included in the
     *         cache key and it’s not included in origin requests. In this case, you can manually add
     *         <code>Accept-Encoding</code> to the headers whitelist like any other HTTP header.
     */
    public final Boolean enableAcceptEncodingGzip() {
        return enableAcceptEncodingGzip;
    }

    /**
     * <p>
     * A flag that can affect whether the <code>Accept-Encoding</code> HTTP header is included in the cache key and
     * included in requests that CloudFront sends to the origin.
     * </p>
     * <p>
     * This field is related to the <code>EnableAcceptEncodingGzip</code> field. If one or both of these fields is
     * <code>true</code> <i>and</i> the viewer request includes the <code>Accept-Encoding</code> header, then CloudFront
     * does the following:
     * </p>
     * <ul>
     * <li>
     * <p>
     * Normalizes the value of the viewer’s <code>Accept-Encoding</code> header
     * </p>
     * </li>
     * <li>
     * <p>
     * Includes the normalized header in the cache key
     * </p>
     * </li>
     * <li>
     * <p>
     * Includes the normalized header in the request to the origin, if a request is necessary
     * </p>
     * </li>
     * </ul>
     * <p>
     * For more information, see <a href=
     * "https://docs.aws.amazon.com/AmazonCloudFront/latest/DeveloperGuide/controlling-the-cache-key.html#cache-policy-compressed-objects"
     * >Compression support</a> in the <i>Amazon CloudFront Developer Guide</i>.
     * </p>
     * <p>
     * If you set this value to <code>true</code>, and this cache behavior also has an origin request policy attached,
     * do not include the <code>Accept-Encoding</code> header in the origin request policy. CloudFront always includes
     * the <code>Accept-Encoding</code> header in origin requests when the value of this field is <code>true</code>, so
     * including this header in an origin request policy has no effect.
     * </p>
     * <p>
     * If both of these fields are <code>false</code>, then CloudFront treats the <code>Accept-Encoding</code> header
     * the same as any other HTTP header in the viewer request. By default, it’s not included in the cache key and it’s
     * not included in origin requests. In this case, you can manually add <code>Accept-Encoding</code> to the headers
     * whitelist like any other HTTP header.
     * </p>
     * 
     * @return A flag that can affect whether the <code>Accept-Encoding</code> HTTP header is included in the cache key
     *         and included in requests that CloudFront sends to the origin.</p>
     *         <p>
     *         This field is related to the <code>EnableAcceptEncodingGzip</code> field. If one or both of these fields
     *         is <code>true</code> <i>and</i> the viewer request includes the <code>Accept-Encoding</code> header, then
     *         CloudFront does the following:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         Normalizes the value of the viewer’s <code>Accept-Encoding</code> header
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Includes the normalized header in the cache key
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Includes the normalized header in the request to the origin, if a request is necessary
     *         </p>
     *         </li>
     *         </ul>
     *         <p>
     *         For more information, see <a href=
     *         "https://docs.aws.amazon.com/AmazonCloudFront/latest/DeveloperGuide/controlling-the-cache-key.html#cache-policy-compressed-objects"
     *         >Compression support</a> in the <i>Amazon CloudFront Developer Guide</i>.
     *         </p>
     *         <p>
     *         If you set this value to <code>true</code>, and this cache behavior also has an origin request policy
     *         attached, do not include the <code>Accept-Encoding</code> header in the origin request policy. CloudFront
     *         always includes the <code>Accept-Encoding</code> header in origin requests when the value of this field
     *         is <code>true</code>, so including this header in an origin request policy has no effect.
     *         </p>
     *         <p>
     *         If both of these fields are <code>false</code>, then CloudFront treats the <code>Accept-Encoding</code>
     *         header the same as any other HTTP header in the viewer request. By default, it’s not included in the
     *         cache key and it’s not included in origin requests. In this case, you can manually add
     *         <code>Accept-Encoding</code> to the headers whitelist like any other HTTP header.
     */
    public final Boolean enableAcceptEncodingBrotli() {
        return enableAcceptEncodingBrotli;
    }

    /**
     * <p>
     * An object that determines whether any HTTP headers (and if so, which headers) are included in the cache key and
     * automatically included in requests that CloudFront sends to the origin.
     * </p>
     * 
     * @return An object that determines whether any HTTP headers (and if so, which headers) are included in the cache
     *         key and automatically included in requests that CloudFront sends to the origin.
     */
    public final CachePolicyHeadersConfig headersConfig() {
        return headersConfig;
    }

    /**
     * <p>
     * An object that determines whether any cookies in viewer requests (and if so, which cookies) are included in the
     * cache key and automatically included in requests that CloudFront sends to the origin.
     * </p>
     * 
     * @return An object that determines whether any cookies in viewer requests (and if so, which cookies) are included
     *         in the cache key and automatically included in requests that CloudFront sends to the origin.
     */
    public final CachePolicyCookiesConfig cookiesConfig() {
        return cookiesConfig;
    }

    /**
     * <p>
     * An object that determines whether any URL query strings in viewer requests (and if so, which query strings) are
     * included in the cache key and automatically included in requests that CloudFront sends to the origin.
     * </p>
     * 
     * @return An object that determines whether any URL query strings in viewer requests (and if so, which query
     *         strings) are included in the cache key and automatically included in requests that CloudFront sends to
     *         the origin.
     */
    public final CachePolicyQueryStringsConfig queryStringsConfig() {
        return queryStringsConfig;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(enableAcceptEncodingGzip());
        hashCode = 31 * hashCode + Objects.hashCode(enableAcceptEncodingBrotli());
        hashCode = 31 * hashCode + Objects.hashCode(headersConfig());
        hashCode = 31 * hashCode + Objects.hashCode(cookiesConfig());
        hashCode = 31 * hashCode + Objects.hashCode(queryStringsConfig());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof ParametersInCacheKeyAndForwardedToOrigin)) {
            return false;
        }
        ParametersInCacheKeyAndForwardedToOrigin other = (ParametersInCacheKeyAndForwardedToOrigin) obj;
        return Objects.equals(enableAcceptEncodingGzip(), other.enableAcceptEncodingGzip())
                && Objects.equals(enableAcceptEncodingBrotli(), other.enableAcceptEncodingBrotli())
                && Objects.equals(headersConfig(), other.headersConfig())
                && Objects.equals(cookiesConfig(), other.cookiesConfig())
                && Objects.equals(queryStringsConfig(), other.queryStringsConfig());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("ParametersInCacheKeyAndForwardedToOrigin")
                .add("EnableAcceptEncodingGzip", enableAcceptEncodingGzip())
                .add("EnableAcceptEncodingBrotli", enableAcceptEncodingBrotli()).add("HeadersConfig", headersConfig())
                .add("CookiesConfig", cookiesConfig()).add("QueryStringsConfig", queryStringsConfig()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "EnableAcceptEncodingGzip":
            return Optional.ofNullable(clazz.cast(enableAcceptEncodingGzip()));
        case "EnableAcceptEncodingBrotli":
            return Optional.ofNullable(clazz.cast(enableAcceptEncodingBrotli()));
        case "HeadersConfig":
            return Optional.ofNullable(clazz.cast(headersConfig()));
        case "CookiesConfig":
            return Optional.ofNullable(clazz.cast(cookiesConfig()));
        case "QueryStringsConfig":
            return Optional.ofNullable(clazz.cast(queryStringsConfig()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<ParametersInCacheKeyAndForwardedToOrigin, T> g) {
        return obj -> g.apply((ParametersInCacheKeyAndForwardedToOrigin) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, ParametersInCacheKeyAndForwardedToOrigin> {
        /**
         * <p>
         * A flag that can affect whether the <code>Accept-Encoding</code> HTTP header is included in the cache key and
         * included in requests that CloudFront sends to the origin.
         * </p>
         * <p>
         * This field is related to the <code>EnableAcceptEncodingBrotli</code> field. If one or both of these fields is
         * <code>true</code> <i>and</i> the viewer request includes the <code>Accept-Encoding</code> header, then
         * CloudFront does the following:
         * </p>
         * <ul>
         * <li>
         * <p>
         * Normalizes the value of the viewer’s <code>Accept-Encoding</code> header
         * </p>
         * </li>
         * <li>
         * <p>
         * Includes the normalized header in the cache key
         * </p>
         * </li>
         * <li>
         * <p>
         * Includes the normalized header in the request to the origin, if a request is necessary
         * </p>
         * </li>
         * </ul>
         * <p>
         * For more information, see <a href=
         * "https://docs.aws.amazon.com/AmazonCloudFront/latest/DeveloperGuide/controlling-the-cache-key.html#cache-policy-compressed-objects"
         * >Compression support</a> in the <i>Amazon CloudFront Developer Guide</i>.
         * </p>
         * <p>
         * If you set this value to <code>true</code>, and this cache behavior also has an origin request policy
         * attached, do not include the <code>Accept-Encoding</code> header in the origin request policy. CloudFront
         * always includes the <code>Accept-Encoding</code> header in origin requests when the value of this field is
         * <code>true</code>, so including this header in an origin request policy has no effect.
         * </p>
         * <p>
         * If both of these fields are <code>false</code>, then CloudFront treats the <code>Accept-Encoding</code>
         * header the same as any other HTTP header in the viewer request. By default, it’s not included in the cache
         * key and it’s not included in origin requests. In this case, you can manually add <code>Accept-Encoding</code>
         * to the headers whitelist like any other HTTP header.
         * </p>
         * 
         * @param enableAcceptEncodingGzip
         *        A flag that can affect whether the <code>Accept-Encoding</code> HTTP header is included in the cache
         *        key and included in requests that CloudFront sends to the origin.</p>
         *        <p>
         *        This field is related to the <code>EnableAcceptEncodingBrotli</code> field. If one or both of these
         *        fields is <code>true</code> <i>and</i> the viewer request includes the <code>Accept-Encoding</code>
         *        header, then CloudFront does the following:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        Normalizes the value of the viewer’s <code>Accept-Encoding</code> header
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Includes the normalized header in the cache key
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Includes the normalized header in the request to the origin, if a request is necessary
         *        </p>
         *        </li>
         *        </ul>
         *        <p>
         *        For more information, see <a href=
         *        "https://docs.aws.amazon.com/AmazonCloudFront/latest/DeveloperGuide/controlling-the-cache-key.html#cache-policy-compressed-objects"
         *        >Compression support</a> in the <i>Amazon CloudFront Developer Guide</i>.
         *        </p>
         *        <p>
         *        If you set this value to <code>true</code>, and this cache behavior also has an origin request policy
         *        attached, do not include the <code>Accept-Encoding</code> header in the origin request policy.
         *        CloudFront always includes the <code>Accept-Encoding</code> header in origin requests when the value
         *        of this field is <code>true</code>, so including this header in an origin request policy has no
         *        effect.
         *        </p>
         *        <p>
         *        If both of these fields are <code>false</code>, then CloudFront treats the
         *        <code>Accept-Encoding</code> header the same as any other HTTP header in the viewer request. By
         *        default, it’s not included in the cache key and it’s not included in origin requests. In this case,
         *        you can manually add <code>Accept-Encoding</code> to the headers whitelist like any other HTTP header.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder enableAcceptEncodingGzip(Boolean enableAcceptEncodingGzip);

        /**
         * <p>
         * A flag that can affect whether the <code>Accept-Encoding</code> HTTP header is included in the cache key and
         * included in requests that CloudFront sends to the origin.
         * </p>
         * <p>
         * This field is related to the <code>EnableAcceptEncodingGzip</code> field. If one or both of these fields is
         * <code>true</code> <i>and</i> the viewer request includes the <code>Accept-Encoding</code> header, then
         * CloudFront does the following:
         * </p>
         * <ul>
         * <li>
         * <p>
         * Normalizes the value of the viewer’s <code>Accept-Encoding</code> header
         * </p>
         * </li>
         * <li>
         * <p>
         * Includes the normalized header in the cache key
         * </p>
         * </li>
         * <li>
         * <p>
         * Includes the normalized header in the request to the origin, if a request is necessary
         * </p>
         * </li>
         * </ul>
         * <p>
         * For more information, see <a href=
         * "https://docs.aws.amazon.com/AmazonCloudFront/latest/DeveloperGuide/controlling-the-cache-key.html#cache-policy-compressed-objects"
         * >Compression support</a> in the <i>Amazon CloudFront Developer Guide</i>.
         * </p>
         * <p>
         * If you set this value to <code>true</code>, and this cache behavior also has an origin request policy
         * attached, do not include the <code>Accept-Encoding</code> header in the origin request policy. CloudFront
         * always includes the <code>Accept-Encoding</code> header in origin requests when the value of this field is
         * <code>true</code>, so including this header in an origin request policy has no effect.
         * </p>
         * <p>
         * If both of these fields are <code>false</code>, then CloudFront treats the <code>Accept-Encoding</code>
         * header the same as any other HTTP header in the viewer request. By default, it’s not included in the cache
         * key and it’s not included in origin requests. In this case, you can manually add <code>Accept-Encoding</code>
         * to the headers whitelist like any other HTTP header.
         * </p>
         * 
         * @param enableAcceptEncodingBrotli
         *        A flag that can affect whether the <code>Accept-Encoding</code> HTTP header is included in the cache
         *        key and included in requests that CloudFront sends to the origin.</p>
         *        <p>
         *        This field is related to the <code>EnableAcceptEncodingGzip</code> field. If one or both of these
         *        fields is <code>true</code> <i>and</i> the viewer request includes the <code>Accept-Encoding</code>
         *        header, then CloudFront does the following:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        Normalizes the value of the viewer’s <code>Accept-Encoding</code> header
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Includes the normalized header in the cache key
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Includes the normalized header in the request to the origin, if a request is necessary
         *        </p>
         *        </li>
         *        </ul>
         *        <p>
         *        For more information, see <a href=
         *        "https://docs.aws.amazon.com/AmazonCloudFront/latest/DeveloperGuide/controlling-the-cache-key.html#cache-policy-compressed-objects"
         *        >Compression support</a> in the <i>Amazon CloudFront Developer Guide</i>.
         *        </p>
         *        <p>
         *        If you set this value to <code>true</code>, and this cache behavior also has an origin request policy
         *        attached, do not include the <code>Accept-Encoding</code> header in the origin request policy.
         *        CloudFront always includes the <code>Accept-Encoding</code> header in origin requests when the value
         *        of this field is <code>true</code>, so including this header in an origin request policy has no
         *        effect.
         *        </p>
         *        <p>
         *        If both of these fields are <code>false</code>, then CloudFront treats the
         *        <code>Accept-Encoding</code> header the same as any other HTTP header in the viewer request. By
         *        default, it’s not included in the cache key and it’s not included in origin requests. In this case,
         *        you can manually add <code>Accept-Encoding</code> to the headers whitelist like any other HTTP header.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder enableAcceptEncodingBrotli(Boolean enableAcceptEncodingBrotli);

        /**
         * <p>
         * An object that determines whether any HTTP headers (and if so, which headers) are included in the cache key
         * and automatically included in requests that CloudFront sends to the origin.
         * </p>
         * 
         * @param headersConfig
         *        An object that determines whether any HTTP headers (and if so, which headers) are included in the
         *        cache key and automatically included in requests that CloudFront sends to the origin.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder headersConfig(CachePolicyHeadersConfig headersConfig);

        /**
         * <p>
         * An object that determines whether any HTTP headers (and if so, which headers) are included in the cache key
         * and automatically included in requests that CloudFront sends to the origin.
         * </p>
         * This is a convenience that creates an instance of the {@link CachePolicyHeadersConfig.Builder} avoiding the
         * need to create one manually via {@link CachePolicyHeadersConfig#builder()}.
         *
         * When the {@link Consumer} completes, {@link CachePolicyHeadersConfig.Builder#build()} is called immediately
         * and its result is passed to {@link #headersConfig(CachePolicyHeadersConfig)}.
         * 
         * @param headersConfig
         *        a consumer that will call methods on {@link CachePolicyHeadersConfig.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #headersConfig(CachePolicyHeadersConfig)
         */
        default Builder headersConfig(Consumer<CachePolicyHeadersConfig.Builder> headersConfig) {
            return headersConfig(CachePolicyHeadersConfig.builder().applyMutation(headersConfig).build());
        }

        /**
         * <p>
         * An object that determines whether any cookies in viewer requests (and if so, which cookies) are included in
         * the cache key and automatically included in requests that CloudFront sends to the origin.
         * </p>
         * 
         * @param cookiesConfig
         *        An object that determines whether any cookies in viewer requests (and if so, which cookies) are
         *        included in the cache key and automatically included in requests that CloudFront sends to the origin.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder cookiesConfig(CachePolicyCookiesConfig cookiesConfig);

        /**
         * <p>
         * An object that determines whether any cookies in viewer requests (and if so, which cookies) are included in
         * the cache key and automatically included in requests that CloudFront sends to the origin.
         * </p>
         * This is a convenience that creates an instance of the {@link CachePolicyCookiesConfig.Builder} avoiding the
         * need to create one manually via {@link CachePolicyCookiesConfig#builder()}.
         *
         * When the {@link Consumer} completes, {@link CachePolicyCookiesConfig.Builder#build()} is called immediately
         * and its result is passed to {@link #cookiesConfig(CachePolicyCookiesConfig)}.
         * 
         * @param cookiesConfig
         *        a consumer that will call methods on {@link CachePolicyCookiesConfig.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #cookiesConfig(CachePolicyCookiesConfig)
         */
        default Builder cookiesConfig(Consumer<CachePolicyCookiesConfig.Builder> cookiesConfig) {
            return cookiesConfig(CachePolicyCookiesConfig.builder().applyMutation(cookiesConfig).build());
        }

        /**
         * <p>
         * An object that determines whether any URL query strings in viewer requests (and if so, which query strings)
         * are included in the cache key and automatically included in requests that CloudFront sends to the origin.
         * </p>
         * 
         * @param queryStringsConfig
         *        An object that determines whether any URL query strings in viewer requests (and if so, which query
         *        strings) are included in the cache key and automatically included in requests that CloudFront sends to
         *        the origin.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder queryStringsConfig(CachePolicyQueryStringsConfig queryStringsConfig);

        /**
         * <p>
         * An object that determines whether any URL query strings in viewer requests (and if so, which query strings)
         * are included in the cache key and automatically included in requests that CloudFront sends to the origin.
         * </p>
         * This is a convenience that creates an instance of the {@link CachePolicyQueryStringsConfig.Builder} avoiding
         * the need to create one manually via {@link CachePolicyQueryStringsConfig#builder()}.
         *
         * When the {@link Consumer} completes, {@link CachePolicyQueryStringsConfig.Builder#build()} is called
         * immediately and its result is passed to {@link #queryStringsConfig(CachePolicyQueryStringsConfig)}.
         * 
         * @param queryStringsConfig
         *        a consumer that will call methods on {@link CachePolicyQueryStringsConfig.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #queryStringsConfig(CachePolicyQueryStringsConfig)
         */
        default Builder queryStringsConfig(Consumer<CachePolicyQueryStringsConfig.Builder> queryStringsConfig) {
            return queryStringsConfig(CachePolicyQueryStringsConfig.builder().applyMutation(queryStringsConfig).build());
        }
    }

    static final class BuilderImpl implements Builder {
        private Boolean enableAcceptEncodingGzip;

        private Boolean enableAcceptEncodingBrotli;

        private CachePolicyHeadersConfig headersConfig;

        private CachePolicyCookiesConfig cookiesConfig;

        private CachePolicyQueryStringsConfig queryStringsConfig;

        private BuilderImpl() {
        }

        private BuilderImpl(ParametersInCacheKeyAndForwardedToOrigin model) {
            enableAcceptEncodingGzip(model.enableAcceptEncodingGzip);
            enableAcceptEncodingBrotli(model.enableAcceptEncodingBrotli);
            headersConfig(model.headersConfig);
            cookiesConfig(model.cookiesConfig);
            queryStringsConfig(model.queryStringsConfig);
        }

        public final Boolean getEnableAcceptEncodingGzip() {
            return enableAcceptEncodingGzip;
        }

        @Override
        public final Builder enableAcceptEncodingGzip(Boolean enableAcceptEncodingGzip) {
            this.enableAcceptEncodingGzip = enableAcceptEncodingGzip;
            return this;
        }

        public final void setEnableAcceptEncodingGzip(Boolean enableAcceptEncodingGzip) {
            this.enableAcceptEncodingGzip = enableAcceptEncodingGzip;
        }

        public final Boolean getEnableAcceptEncodingBrotli() {
            return enableAcceptEncodingBrotli;
        }

        @Override
        public final Builder enableAcceptEncodingBrotli(Boolean enableAcceptEncodingBrotli) {
            this.enableAcceptEncodingBrotli = enableAcceptEncodingBrotli;
            return this;
        }

        public final void setEnableAcceptEncodingBrotli(Boolean enableAcceptEncodingBrotli) {
            this.enableAcceptEncodingBrotli = enableAcceptEncodingBrotli;
        }

        public final CachePolicyHeadersConfig.Builder getHeadersConfig() {
            return headersConfig != null ? headersConfig.toBuilder() : null;
        }

        @Override
        public final Builder headersConfig(CachePolicyHeadersConfig headersConfig) {
            this.headersConfig = headersConfig;
            return this;
        }

        public final void setHeadersConfig(CachePolicyHeadersConfig.BuilderImpl headersConfig) {
            this.headersConfig = headersConfig != null ? headersConfig.build() : null;
        }

        public final CachePolicyCookiesConfig.Builder getCookiesConfig() {
            return cookiesConfig != null ? cookiesConfig.toBuilder() : null;
        }

        @Override
        public final Builder cookiesConfig(CachePolicyCookiesConfig cookiesConfig) {
            this.cookiesConfig = cookiesConfig;
            return this;
        }

        public final void setCookiesConfig(CachePolicyCookiesConfig.BuilderImpl cookiesConfig) {
            this.cookiesConfig = cookiesConfig != null ? cookiesConfig.build() : null;
        }

        public final CachePolicyQueryStringsConfig.Builder getQueryStringsConfig() {
            return queryStringsConfig != null ? queryStringsConfig.toBuilder() : null;
        }

        @Override
        public final Builder queryStringsConfig(CachePolicyQueryStringsConfig queryStringsConfig) {
            this.queryStringsConfig = queryStringsConfig;
            return this;
        }

        public final void setQueryStringsConfig(CachePolicyQueryStringsConfig.BuilderImpl queryStringsConfig) {
            this.queryStringsConfig = queryStringsConfig != null ? queryStringsConfig.build() : null;
        }

        @Override
        public ParametersInCacheKeyAndForwardedToOrigin build() {
            return new ParametersInCacheKeyAndForwardedToOrigin(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
